﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.ComponentModel.DataAnnotations;
using System.Web.Mvc;

namespace BMS.Web.Controllers.Shared
{
    public class RequiredIfForActionAttribute : ValidationAttribute, IClientValidatable
    {
        private RequiredAttribute _innerAttribute = new RequiredAttribute();

        public string DependentProperty { get; set; }
        public string DependentAction { get; set; }
        public object TargetValue { get; set; }

        public RequiredIfForActionAttribute(string dependentProperty, object targetValue, string dependentAction)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this.DependentProperty = dependentProperty;
                this.DependentAction = dependentAction;
                this.TargetValue = targetValue;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected override ValidationResult IsValid(object value, ValidationContext validationContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // get a reference to the property this validation depends upon
                var containerType = validationContext.ObjectInstance.GetType();
                var field = containerType.GetProperty(this.DependentProperty);
                var action = containerType.GetProperty(this.DependentAction);

                if (field != null)
                {
                    // get the value of the dependent property
                    var dependentvalue = field.GetValue(validationContext.ObjectInstance, null);
                    var actionvalue = action.GetValue(validationContext.ObjectInstance, null);

                    //check if the action context is matched
                    if (actionvalue != null)
                        // compare the value against the target value
                        if ((dependentvalue == null && this.TargetValue == null) ||
                            (dependentvalue != null && dependentvalue.Equals(this.TargetValue)))
                        {
                            // match => means we should try validating this field
                            if (!_innerAttribute.IsValid(value))
                                // validation failed - return an error
                                return new ValidationResult(this.ErrorMessage, new[] { validationContext.MemberName });
                        }
                }

                return ValidationResult.Success;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IEnumerable<ModelClientValidationRule> GetClientValidationRules(ModelMetadata metadata, ControllerContext context)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                var rule = new ModelClientValidationRule()
                {
                    ErrorMessage = FormatErrorMessage(metadata.GetDisplayName()),
                    ValidationType = "requiredifaction",
                };

                string depProp = BuildDependentPropertyId(metadata, context as ViewContext);
                string depAction = BuildDependentActionId(metadata, context as ViewContext);

                // find the value on the control we depend on;
                // if it's a bool, format it javascript style 
                // (the default is True or False!)
                string targetValue = (this.TargetValue ?? "").ToString();
                if (this.TargetValue.GetType() == typeof(bool))
                    targetValue = targetValue.ToLower();

                rule.ValidationParameters.Add("dependentproperty", depProp);
                rule.ValidationParameters.Add("dependentaction", depAction);
                rule.ValidationParameters.Add("targetvalue", targetValue);

                yield return rule;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private string BuildDependentPropertyId(ModelMetadata metadata, ViewContext viewContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // build the ID of the property
                string depProp = viewContext.ViewData.TemplateInfo.GetFullHtmlFieldId(this.DependentProperty);
                // unfortunately this will have the name of the current field appended to the beginning,
                // because the TemplateInfo's context has had this fieldname appended to it. Instead, we
                // want to get the context as though it was one level higher (i.e. outside the current property,
                // which is the containing object (our Person), and hence the same level as the dependent property.
                var thisField = metadata.PropertyName + "_";
                if (depProp.StartsWith(thisField))
                    // strip it off again
                    depProp = depProp.Substring(thisField.Length);
                return depProp;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private string BuildDependentActionId(ModelMetadata metadata, ViewContext viewContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // build the ID of the property
                string depAction = viewContext.ViewData.TemplateInfo.GetFullHtmlFieldId(this.DependentAction);
                // unfortunately this will have the name of the current field appended to the beginning,
                // because the TemplateInfo's context has had this fieldname appended to it. Instead, we
                // want to get the context as though it was one level higher (i.e. outside the current property,
                // which is the containing object (our Person), and hence the same level as the dependent property.
                var thisField = metadata.PropertyName + "_";
                if (depAction.StartsWith(thisField))
                    // strip it off again
                    depAction = depAction.Substring(thisField.Length);
                return depAction;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}