﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using Mdws2ORM.QuerySvcService;
using System.ServiceModel;
using BMS.VistaIntegration.VistA;
using BMS.Utils;
using BMS.VistaIntegration.Data.WF;
using BMS.VistaIntegration.Mdws.Commands.EIS;
using BMS.VistaIntegration.Mdws.Commands;
using BMS.VistaIntegration.Mdws.Commands.EVS;
using BMS.VistaIntegration.Mdws.Commands.WF;
using BMS.VistaIntegration.Mdws.Commands.Concrete.WF;
using BMS.VistaIntegration.Mdws.Commands.Concrete.EIS;
using Mdws2ORM;
using Mdws2ORM.Core;

namespace BMS.VistaIntegration.Mdws
{
    public class MdwsVistAQuery : IVistAQuery
    {
        private static readonly BmsLogger logger = new BmsLogger("MdwsVistAQuery:");

        private const int DEFAULT_BULK_COUNT = 1000;
        private readonly MdwsVistASession session;
        private readonly ListHospitalLocationCommand hospitalLocationCommand = new ListHospitalLocationCommand();
        private readonly ListPatientCommand patientCommand = new ListPatientCommand();
        private readonly ListRoomBedCommand roomBedCommand = new ListRoomBedCommand();
        private readonly ListWardLocationCommand wardLocationCommand = new ListWardLocationCommand();
        private readonly ListNewPersonCommand newPersonCommand = new ListNewPersonCommand();
        private readonly ListFacilityTreatingSpecialtyCommand facilityTreatingSpecialtyCommand = new ListFacilityTreatingSpecialtyCommand();
        private readonly ListFacilityMovementTypeCommand facilityMovementTypeCommand = new ListFacilityMovementTypeCommand();
        private readonly ListOrderableItemCommand orderableItemCommand = new ListOrderableItemCommand();
        private readonly ListSpecialtyCommand specialtyCommand = new ListSpecialtyCommand();
        private readonly ListOrderStatusCommand orderStatusCommand = new ListOrderStatusCommand();
        private readonly ListOrdersCommand ordersFilterCommand = new ListOrdersCommand();
        private readonly ListPatientMovementCommand patientMovementCommand = new ListPatientMovementCommand();
        private readonly ListScheduledAdmissionCommand scheduledAdmissionCommand = new ListScheduledAdmissionCommand();
        private readonly ListMedicalCenterDivisionCommand medicalCenterDivisionCommand = new ListMedicalCenterDivisionCommand();
        private readonly ListPatientAppointmentFromClinicsCommand patAppFromClinics = new ListPatientAppointmentFromClinicsCommand();
        private readonly ListPatientAppointmentFromClinicsAndPatientCommand patAppFromClinicsAndPatient = new ListPatientAppointmentFromClinicsAndPatientCommand();
        private readonly ListPatientMovementsByAdmissionCommand patientMovementsByAdmissionCommand = new ListPatientMovementsByAdmissionCommand();
        private readonly ListAdmittedPatientsCommand admittedPatientsCommand = new ListAdmittedPatientsCommand();
        private readonly ListPatientMovementIensCommand patientMovementIensCommand = new ListPatientMovementIensCommand();
        private readonly ListAdmittedPatientsForUpdateCommand admittedPatientsForUpdateCommand = new ListAdmittedPatientsForUpdateCommand();

        private ISession querySession;
        private bool isOrderStatusCached = false;
        private bool isNewPersonCached = false;
        private bool isTreatingSpecialtyCached = false;
        private bool isTypeOfMovementCached = false;
        private bool isWardLocationCached = false;        
        private bool isMedicalCenterDivisionCached = false;
        private bool isSpecialtyCached = false;

        public int BulkCount { get; set; }

        public MdwsVistAQuery(MdwsVistASession session)
        {
            this.session = session;
            BulkCount = DEFAULT_BULK_COUNT;
            bool bedHoldSupported = VistasBedHoldSupported.IsBedHoldSupported(session.VistASite.Name);
            this.querySession = BmsSessionFactory.CreateSession(session.Client, bedHoldSupported, session.VistASite);
            InitCache();
        }

        private IList<T> GetResult<T>(BaseListCommand<T> command, bool isCached = false) where T : class , new()
        {
            if (!command.HasMore)
                return null;

            if (!string.IsNullOrEmpty(command.Last))
                session.RunClientAction(() => session.IsAlive());
            command.MaxCount = BulkCount;
            if (!isCached)
                querySession.SetCacheForType<T>(false);
            IList<T> result = null;
            try
            {
                session.RunClientAction(() => result = command.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = command.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetResult - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetResult - exception");
                    Tracer.TraceException(ex);
                }
            }
            command.From = command.Last;
            if (!isCached)
                querySession.SetCacheForType<T>(true);
            return result;
        }

        private IList<T> GetResult<T>(BaseListPeriodCommand<T> command, DateTime? startDate, DateTime? endDate, bool isCached = false) where T : class , new()
        {
            command.StartDate = startDate;
            command.EndDate = endDate;
            return GetResult(command, isCached);
        }

        #region Cache

        private void InitCache()
        {
            querySession.SetCacheForType<PatientAppointment>(false);
            querySession.SetCacheForType<PatientMovement>(false);
            querySession.SetCacheForType<Order>(false);
            querySession.SetCacheForType<OrderAction>(false);
            querySession.SetCacheForType<ScheduledAdmission>(false);
        }        

        private void CacheType<T, V>(ref bool isCached)
            where T : BaseListCommand<V>, new()
            where V : class
        {
            if (isCached) return;
            isCached = true;
            T command = new T();
            try
            {
                session.RunClientAction(() =>
                {
                    command.MaxCount = BulkCount;
                    while (command.HasMore)
                    {
                        command.Execute(querySession);
                        command.From = command.Last;
                    }
                });
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() =>
                        {
                            command.MaxCount = BulkCount;
                            while (command.HasMore)
                            {
                                command.Execute(querySession);
                                command.From = command.Last;
                            }
                        });
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery CacheType - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery CacheType - exception");
                    Tracer.TraceException(ex);
                }
            }
        }

        private void CacheOrderStatus()
        {
            CacheType<ListOrderStatusCommand, OrderStatus>(ref isOrderStatusCached);
        }

        private void CacheNewPersons()
        {
            CacheType<ListNewPersonCommand, NewPerson>(ref isNewPersonCached);
        }

        private void CacheTreatingSpecialty()
        {
            CacheType<ListFacilityTreatingSpecialtyCommand, FacilityTreatingSpecialty>(ref isTreatingSpecialtyCached);
        }

        private void CacheTypeOfMovement()
        {
            CacheType<ListFacilityMovementTypeCommand, FacilityMovementType>(ref isTypeOfMovementCached);
        }

        private void CacheSpecialty()
        {
            CacheType<ListSpecialtyCommand, Specialty>(ref isSpecialtyCached);
        }

        private void CacheWardLocation()
        {
            CacheSpecialty();
            CacheType<ListWardLocationCommand, WardLocation>(ref isWardLocationCached);
        }        

        private void CacheMedicalCenterDivision()
        {
            CacheType<ListMedicalCenterDivisionCommand, MedicalCenterDivision>(ref isMedicalCenterDivisionCached);
        }

        #endregion        

        #region EIS

        public IList<HospitalLocation> GetHospitalLocations()
        {            
            CacheWardLocation();
            return GetResult(hospitalLocationCommand);
        }

        public IList<WardLocation> GetWardLocations()
        {
            CacheSpecialty();
            isWardLocationCached = true;
            return GetResult(wardLocationCommand, true);
        }

        private WardLocation GetWardLocationByIen(string ien)
        {
            GetWardLocationCommand getWardLocationCommand = new GetWardLocationCommand();
            WardLocation result = null;
            getWardLocationCommand.Ien = ien;
            try
            {
                session.RunClientAction(() => result = getWardLocationCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = getWardLocationCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetWardLocationByIen - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetWardLocationByIen - exception");
                    Tracer.TraceException(ex);
                }
            }
            return result;
        }

        public IList<RoomBed> GetRoomBeds()
        {
            if (roomBedCommand.SecondTimeCall)
                return null;
            roomBedCommand.MaxCount = BulkCount;
            List<RoomBed> result = null;
            try
            {
                session.RunClientAction(() => result = roomBedCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = roomBedCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetRoomBeds - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetRoomBeds - exception");
                    Tracer.TraceException(ex);
                }
            }
            if (result != null && result.Count > 0)
            {
                IList<WardLocation> wards = GetWardLocations();
                foreach (RoomBed rb in result)
                {
                    if (rb.WardsWhichCanAssign != null && rb.WardsWhichCanAssign.Count > 0)
                    {
                        rb.Wards = new List<WardLocation>();
                        rb.WardsWhichCanAssign.ForEach(a => { WardLocation w = wards.Where(b => b.IEN.Equals(a)).FirstOrDefault(); if (w != null) rb.Wards.Add(w); });
                    }
                }
            }
            roomBedCommand.SecondTimeCall = true;
            return result;
        }

        private RoomBed GetRoomBedByIen(string ien)
        {
            roomBedCommand.MaxCount = 1;
            roomBedCommand.Ien = ien;
            List<RoomBed> result = null;
            try
            {
                session.RunClientAction(() => result = roomBedCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = roomBedCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetRoomBeds - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetRoomBeds - exception");
                    Tracer.TraceException(ex);
                }
            }
            if (result != null && result.Count > 0)
                return result[0];
            return null;
        }

        public IList<NewPerson> GetNewPersons(DateTime? startDate = null, DateTime? endDate = null)
        {
            return GetResult(newPersonCommand, startDate, endDate);
        }

        public IList<Patient> GetPatients(DateTime? startDate = null, DateTime? endDate = null, string vistaPatientIen = null)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - MdwsVistaQuery: Start GetPatients", session.VistASite.Name);
            CacheTreatingSpecialty();            
            patientCommand.StartTime = startDate; 
            patientCommand.EndTime = endDate;
            patientCommand.MaxCount = BulkCount;
            patientCommand.Client = session.Client;
            patientCommand.VistaSite = session.VistASite;
            patientCommand.VistaPatientIen = vistaPatientIen;
            querySession.SetCacheForType<Patient>(false);
            IList<Patient> result = null;
            try
            {
                session.RunClientAction(() => result = patientCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = patientCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatients - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatients - exception");
                    Tracer.TraceException(ex);
                }
            }
            querySession.SetCacheForType<Patient>(true);
            logger.LogFormat(BmsLogger.Level.Info, "{0} - MdwsVistaQuery: End GetPatients", session.VistASite.Name);
            return result;
        }

        public Patient GetPatientByIen(string ien)
        {
            GetPatientCommand getPatientCommand = new GetPatientCommand();
            Patient result = null;
            getPatientCommand.Ien = ien;
            try
            {
                session.RunClientAction(() => result = getPatientCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = getPatientCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatientByIen - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatientByIen - exception");
                    Tracer.TraceException(ex);
                }
            }
            return result;
        }

        public IList<Patient> GetPatients(IEnumerable<string> ssns)
        {
            ListPatientBySsnCommand command = new ListPatientBySsnCommand();
            command.SSNs = ssns;
            IList<Patient> result = null;
            try
            {
                session.RunClientAction(() => result = command.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = command.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatients - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatients - exception");
                    Tracer.TraceException(ex);
                }
            }
            return result;
        }

        public IList<Patient> GetAdmittedPatientsForUpdate()
        {
            CacheTreatingSpecialty();
            admittedPatientsForUpdateCommand.MaxCount = BulkCount;
            querySession.SetCacheForType<Patient>(false);
            IList<Patient> result = null;
            try
            {
                session.RunClientAction(() => result = admittedPatientsForUpdateCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = admittedPatientsForUpdateCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetAdmittedPatientsForUpdate - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetAdmittedPatientsForUpdate - exception");
                    Tracer.TraceException(ex);
                }
            }
            return result;
        }

        #endregion

        #region EVS

        public IList<FacilityTreatingSpecialty> GetFacilityTreatingSpecialties()
        {
            IList<FacilityTreatingSpecialty> result = GetResult(facilityTreatingSpecialtyCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;
        }

        public IList<FacilityMovementType> GetFacilityMovementTypes()
        {
            IList<FacilityMovementType> result = GetResult(facilityMovementTypeCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;
        }

        public IList<OrderableItem> GetOrderableItems()
        {
            IList<OrderableItem> result = GetResult(orderableItemCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;            
        }

        public IList<Specialty> GetSpecialties()
        {
            IList<Specialty> result = GetResult(specialtyCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim()) && !a.Name.StartsWith("z", StringComparison.InvariantCultureIgnoreCase)).ToList();
            else
                return result;
        }

        public IList<OrderStatus> GetOrderStatuses()
        {
            return GetResult(orderStatusCommand);
        }

        public IList<MedicalCenterDivision> GetMedicalCenterDivisions()
        {
            IList<MedicalCenterDivision> result = GetResult(medicalCenterDivisionCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;
        }

        #endregion

        #region WF

        public IList<OrderAction> GetOrderActions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> orderableItemsIen = null)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - MdwsVistaQuery: Start GetOrderActions", session.VistASite.Name);
            CacheWardLocation();
            CacheOrderStatus();
            if (!orderableItemsIen.Any())
                return new List<OrderAction>();

            ordersFilterCommand.PatientIen = patientIen;
            ordersFilterCommand.OrderableItemsIen = orderableItemsIen;
            ordersFilterCommand.StartTime = startDate;
            ordersFilterCommand.EndTime = endDate;
            ordersFilterCommand.Client = session.Client;
            ordersFilterCommand.MaxCount = BulkCount;
            ordersFilterCommand.VistaSite = session.VistASite;
            IList<OrderAction> result = null;
            try
            {
                session.RunClientAction(() => result = ordersFilterCommand.Execute(querySession));

            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = ordersFilterCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetOrderActions - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetOrderActions - exception");
                    Tracer.TraceException(ex);
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - MdwsVistaQuery: End GetOrderActions", session.VistASite.Name);
            return result;
        }

        public IList<PatientAppointment> GetPatientAppointments(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> clinicIens = null)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetPatientAppointments", session.VistASite.Name);
            IList<PatientAppointment> result = null;

            if (string.IsNullOrEmpty(patientIen))
            {
                patAppFromClinics.ClinicIens = clinicIens;
                patAppFromClinics.StartDate = startDate;
                patAppFromClinics.EndDate = endDate;
                patAppFromClinics.MaxCount = BulkCount;
                try
                {
                    session.RunClientAction(() => result = patAppFromClinics.Execute(querySession));
                }
                catch (Exception ex)
                {
                    if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                    {
                        session.RunClientAction(() => session.IsAlive());
                        try
                        {
                            session.RunClientAction(() => result = patAppFromClinics.Execute(querySession));
                        }
                        catch (Exception ex2)
                        {
                            if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                                session.RunClientAction(() => session.IsAlive());
                            else
                            {
                                Tracer.TraceMessage("MdwsVistaQuery GetPatientAppointments - exception (second try)");
                                Tracer.TraceException(ex2);
                            }
                        }
                    }
                    else
                    {
                        Tracer.TraceMessage("MdwsVistaQuery GetPatientAppointments - exception");
                        Tracer.TraceException(ex);
                    }
                }
            }
            else
            {
                patAppFromClinicsAndPatient.ClinicIens = clinicIens;
                patAppFromClinicsAndPatient.PatientIen = patientIen;
                patAppFromClinicsAndPatient.StartTime = startDate;
                patAppFromClinicsAndPatient.EndTime = endDate;
                try
                {
                    session.RunClientAction(() => result = patAppFromClinicsAndPatient.Execute(querySession).ToList());
                }
                catch (Exception ex)
                {
                    if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                    {
                        session.RunClientAction(() => session.IsAlive());
                        try
                        {
                            session.RunClientAction(() => result = patAppFromClinicsAndPatient.Execute(querySession).ToList());
                        }
                        catch (Exception ex2)
                        {
                            if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                                session.RunClientAction(() => session.IsAlive());
                            else
                            {
                                Tracer.TraceMessage("MdwsVistaQuery GetPatientAppointments 2 - exception (second try)");
                                Tracer.TraceException(ex2);
                            }
                        }
                    }
                    else
                    {
                        Tracer.TraceMessage("MdwsVistaQuery GetPatientAppointments 2 - exception");
                        Tracer.TraceException(ex);
                    }
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetPatientAppointments", session.VistASite.Name);
            return result;
        }

        public IList<PatientMovement> GetPatientMovements(DateTime? startDate = null, DateTime? endDate = null)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetPatientMovements", session.VistASite.Name);
            CacheTreatingSpecialty();
            CacheTypeOfMovement();            
            patientMovementCommand.StartTime = startDate;
            patientMovementCommand.EndTime = endDate;
            patientMovementCommand.MaxCount = BulkCount;
            IList<PatientMovement> result = null;
            try
            {
                session.RunClientAction(() => result = patientMovementCommand.Execute(querySession));
            }
            catch (Exception ex)
            {                
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = patientMovementCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatientMovements - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatientMovements - exception");
                    Tracer.TraceException(ex);
                }
            }
            if (result != null && result.Count > 0)
            {
                IList<WardLocation> wards = GetWardLocations();
                foreach (PatientMovement pm in result)
                {
                    if (!string.IsNullOrEmpty(pm.WardLocationId))
                        pm.Ward = wards.Where(a => a.IEN.Equals(pm.WardLocationId)).FirstOrDefault();
                    if (!string.IsNullOrEmpty(pm.RoomBedId))
                    {
                        pm.Bed = GetRoomBedByIen((long.Parse(pm.RoomBedId) - 1).ToString());
                        if (pm.Bed.WardsWhichCanAssign != null && pm.Bed.WardsWhichCanAssign.Count > 0)
                        {
                            pm.Bed.Wards = new List<WardLocation>();
                            pm.Bed.WardsWhichCanAssign.ForEach(a => { WardLocation w = wards.Where(b => b.IEN.Equals(a)).FirstOrDefault(); if (w != null) pm.Bed.Wards.Add(w); });
                        }
                    }
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetPatientMovements", session.VistASite.Name);
            return result;
        }

        public IList<ScheduledAdmission> GetScheduledAdmissions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetScheduledAdmissions", session.VistASite.Name);
            CacheMedicalCenterDivision();
            CacheTreatingSpecialty();
            CacheWardLocation();
            scheduledAdmissionCommand.PatientIen = patientIen;
            scheduledAdmissionCommand.StartTime = startDate;
            scheduledAdmissionCommand.EndTime = endDate;
            scheduledAdmissionCommand.MaxCount = BulkCount;
            IList<ScheduledAdmission> result = null;
            try
            {
                session.RunClientAction(() => result = scheduledAdmissionCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = scheduledAdmissionCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetScheduledAdmissions - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetScheduledAdmissions - exception");
                    Tracer.TraceException(ex);
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetScheduledAdmissions", session.VistASite.Name);
            return result;
        }                

        public PatientMovement GetPatientMovement(string patientIen, DateTime dateTime, MovementTransactionType movementType)
        {
            GetPatientMovementCommand getPatientMovementCommand = new GetPatientMovementCommand();
            PatientMovement result = null;
            getPatientMovementCommand.PatientIen = patientIen;
            getPatientMovementCommand.DateTime = dateTime;
            getPatientMovementCommand.MovementType = movementType;
            try
            {
                session.RunClientAction(() => result = getPatientMovementCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = getPatientMovementCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatientMovement - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatientMovement - exception");
                    Tracer.TraceException(ex);
                }
            }
            if (result != null)
            {
                if (!string.IsNullOrEmpty(result.WardLocationId))
                    result.Ward = GetWardLocationByIen(result.WardLocationId);
                if (!string.IsNullOrEmpty(result.RoomBedId))
                {
                    result.Bed = GetRoomBedByIen((long.Parse(result.RoomBedId) - 1).ToString());
                    if (result.Bed.WardsWhichCanAssign != null && result.Bed.WardsWhichCanAssign.Count > 0)
                    {
                        result.Bed.Wards = new List<WardLocation>();
                        foreach (string wardIen in result.Bed.WardsWhichCanAssign)
                            result.Bed.Wards.Add(GetWardLocationByIen(wardIen));
                    }
                }
            }
            return result;
        }

        public PatientMovement GetPatientMovementByIen(string patientMovementIen)
        {
            GetPatientMovementCommand getPatientMovementCommand = new GetPatientMovementCommand();
            PatientMovement result = null;
            getPatientMovementCommand.PatientMovementIen = patientMovementIen;
            try
            {
                session.RunClientAction(() => result = getPatientMovementCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = getPatientMovementCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatientMovementByIen - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatientMovementByIen - exception");
                    Tracer.TraceException(ex);
                }
            }
            if (result != null)
            {
                if (!string.IsNullOrEmpty(result.WardLocationId))
                    result.Ward = GetWardLocationByIen(result.WardLocationId);
                if (!string.IsNullOrEmpty(result.RoomBedId))
                {
                    result.Bed = GetRoomBedByIen((long.Parse(result.RoomBedId) - 1).ToString());
                    if (result.Bed.WardsWhichCanAssign != null && result.Bed.WardsWhichCanAssign.Count > 0)
                    {
                        result.Bed.Wards = new List<WardLocation>();
                        foreach (string wardIen in result.Bed.WardsWhichCanAssign)
                            result.Bed.Wards.Add(GetWardLocationByIen(wardIen));
                    }
                }
            }
            return result;
        }

        public IList<BedSwitch> GetBedsSwitch(IEnumerable<string> iens)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetBedsSwitch", session.VistASite.Name);
            ListBedSwitchCommand bedSwitchCommand = new ListBedSwitchCommand();
            IList<BedSwitch> result = null;
            bedSwitchCommand.PatientMovementsIEn = iens;
            try
            {
                session.RunClientAction(() => result = bedSwitchCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = bedSwitchCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetBedsSwitch - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetBedsSwitch - exception");
                    Tracer.TraceException(ex);
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetBedsSwitch", session.VistASite.Name);
            return result;
        }

        public IList<string> GetCanceledOrders(IEnumerable<string> iens)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetCanceledOrders", session.VistASite.Name);
            ListCancelOrdersCommand command = new ListCancelOrdersCommand();
            IList<string> result = null;
            command.OrderIens = iens;
            try
            {
                session.RunClientAction(() => result = command.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = command.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetCanceledOrders - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetCanceledOrders - exception");
                    Tracer.TraceException(ex);
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetCanceledOrders", session.VistASite.Name);
            return result;
        }        

        public IList<PatientMovement> GetPatientMovementsForAdmission(string admissionIen)
        {            
            patientMovementsByAdmissionCommand.AdmissionIen = admissionIen;
            patientMovementsByAdmissionCommand.MaxCount = BulkCount/20;
            IList<PatientMovement> result = null;
            try
            {
                session.RunClientAction(() => result = patientMovementsByAdmissionCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = patientMovementsByAdmissionCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatientMovementsForAdmission - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatientMovementsForAdmission - exception");
                    Tracer.TraceException(ex);
                }
            }
            if (result != null && result.Count > 0)
            {
                foreach (PatientMovement pm in result)
                {
                    if (!string.IsNullOrEmpty(pm.WardLocationId))
                        pm.Ward = GetWardLocationByIen(pm.WardLocationId);
                    if (!string.IsNullOrEmpty(pm.RoomBedId))
                    {
                        pm.Bed = GetRoomBedByIen((long.Parse(pm.RoomBedId) - 1).ToString());
                        if (pm.Bed.WardsWhichCanAssign != null && pm.Bed.WardsWhichCanAssign.Count > 0)
                        {
                            pm.Bed.Wards = new List<WardLocation>();
                            foreach (string wardIen in pm.Bed.WardsWhichCanAssign)
                                pm.Bed.Wards.Add(GetWardLocationByIen(wardIen));
                        }
                    }
                }
            }
            return result;
        }

        public IList<PatientMovementIen> GetPatientMovementIens(DateTime startDate, DateTime endDate)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetPatientMovementIens", session.VistASite.Name);
            patientMovementIensCommand.StartTime = startDate;
            patientMovementIensCommand.EndTime = endDate;
            patientMovementIensCommand.MaxCount = BulkCount;
            IList<PatientMovementIen> result = null;
            try
            {
                session.RunClientAction(() => result = patientMovementIensCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = patientMovementIensCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetPatientMovementIens - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetPatientMovementIens - exception");
                    Tracer.TraceException(ex);
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetPatientMovementIens", session.VistASite.Name);
            return result;
        }

        public IList<AdmittedPatient> GetAdmittedPatients(string lastPatientMovementIen)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start GetAdmittedPatients", session.VistASite.Name);
            admittedPatientsCommand.AdmissionIen = lastPatientMovementIen;
            admittedPatientsCommand.MaxCount = BulkCount;
            IList<AdmittedPatient> result = null;
            try
            {
                session.RunClientAction(() => result = admittedPatientsCommand.Execute(querySession));
            }
            catch (Exception ex)
            {
                if (ex.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                {
                    session.RunClientAction(() => session.IsAlive());
                    try
                    {
                        session.RunClientAction(() => result = admittedPatientsCommand.Execute(querySession));
                    }
                    catch (Exception ex2)
                    {
                        if (ex2.Message.Contains(Constants.Mdws_Fault_Exception_Text))
                            session.RunClientAction(() => session.IsAlive());
                        else
                        {
                            Tracer.TraceMessage("MdwsVistaQuery GetAdmittedPatients - exception (second try)");
                            Tracer.TraceException(ex2);
                        }
                    }
                }
                else
                {
                    Tracer.TraceMessage("MdwsVistaQuery GetAdmittedPatients - exception");
                    Tracer.TraceException(ex);
                }
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End GetAdmittedPatients", session.VistASite.Name);
            return result;
        }

        #endregion
    }
}
