﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using BMS.Utils;
using InfoWorld.HL7.ITS;
using InfoWorld.Security.Authorization.PolicyAdministrationPoint;
using System.Collections;
using System.ServiceModel.Channels;
using BMS.ServicesWrapper.Proxy.PDP;
using InfoWorld.Security.Xacml.Saml;
using Lagash.Xacml.Core.Context;
using BMS.ServicesWrapper.EIS;
using System.Threading;
using System.ServiceModel;
using System.Configuration;


namespace BMS.ServicesWrapper.Security
{
    class SecurityWrapper : ISecurityWrapper
    {
        private const string BUILD_IN = "Built-in";
        private const string DOMAIN_RESOURCE_TYPE = "HL7-RIM-V3-DOMAIN";
        private const string ORGANIZATION_RESOURCE_TYPE = "HL7-RIM-V3-ORGANIZATION"; //for facility
        private const string MEMBER_OF_PERMISSION = "HM, Member of"; //BMS, Read si BMS, Update
        private const string OPERATIONS_RESOURCE_EXTENSION = "HM3Services";

        private const string READ_DATA_PERMISSION = "BMS, Read";
        private const string WRITE_DATA_PERMISSION = "BMS, Write";

        private const string ADMINISTRATORS_ROLE = "Administrators";
        private const string EMS_STAFF_ROLE = "EmsStaff";
        private const string GUESTS_ROLE = "Guests";
        private const string NATIONAL_USERS_ROLE = "NationalUsers";
        private const string REGIONAL_USERS_ROLE = "RegionalUsers";
        private const string SITE_USERS_ROLE = "SiteUsers";
        private const string SUPPORT_USERS_ROLE = "SupportUsers";
        private const string VISN_USERS_ROLE = "VisnUsers";
        private const string SUPER_USERS_ROLE = "SuperUsers";
        private const string AUDIT_LOG_USERS_ROLE = "AuditLogUsers";
        private const string EMS_SUPERVISOR_USERS_ROLE = "EMSSupervisorUsers";
        private bool _isWindows = false;
        private SecurityCache _cache;
        private bool isIISSingleInstance = bool.Parse(ConfigurationManager.AppSettings["Is_IIS_Single_Instance"]);

        public AdministrativeFunctionsClient AdministrativeFunctionsClient        //PAP
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (_isWindows)
                        return SecurityFactory.AdministrativeFunctionsClientWindows;
                    return SecurityFactory.AdministrativeFunctionsClient;
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        private Proxy.PDP.AuthorizationServiceClient AuthorizationServiceClient    //check PDP permission
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (_isWindows)
                        return SecurityFactory.AuthorizationServiceClientWindows;
                    return SecurityFactory.AuthorizationServiceClient;
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        private enum OperationMode { Reset, Add, Revoke }

        public SecurityWrapper(bool isWindows)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this._isWindows = isWindows;
                _cache = new SecurityCache(this);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region IPAPWrapper Members

        public IList<string> GetAvailableDomains()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<string> result = _cache.GetAvailableDomains();
                if (result == null || result.Count == 0)
                {
                    _cache.CacheAvailableDomains();
                    result = _cache.GetAvailableDomains();
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<string> GetAllUserRoles()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    return AdministrativeFunctionsClient.GetAllUserRoles();
                }
                catch
                {
                    return null;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Facade.Data.User GetUser(string userId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string currentDomain = GetCurrentDomain();
                BMS.Facade.Data.User user = ToUser(AdministrativeFunctionsClient.GetUserBySid(userId), currentDomain);
                return user;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public BMS.Facade.Data.User GetUserByUserName(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UserGroup ug = AdministrativeFunctionsClient.GetUserByUserName(userName);
                if (ug != null)
                    return new BMS.Facade.Data.User()
                    {
                        Domain = ug.Domain,
                        FullName = ug.FullName,
                        UserName = ug.Domain + @"\" + ug.Name,
                        Id = new II() { root = ug.Domain, extension = ug.ID }
                    };
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Facade.Data.User GetDefaultUserProfile()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new Facade.Data.User()
                {
                    Domain = GetCurrentDomain(),
                    UserName = Constants.DEFAULT_USER_PROFILE,
                    Id = new II() { root = Constants.DEFAULT_USER_PROFILE, extension = Constants.DEFAULT_USER_PROFILE }
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string GetCurrentDomain()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domain = _cache.GetCurrentDomain();
                if (string.IsNullOrEmpty(domain))
                {
                    _cache.CacheCurrentDomain();
                    domain = _cache.GetCurrentDomain();
                }
                return domain;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<string> GetAvailableDomains(string username, string userid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CheckUserRoleExists(username, userid);
                return AdministrativeFunctionsClient.GetAvailableDomainsForUser(userid);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }                

        //Get the list of users with permissions(read/write) on the supplied facility
        public IList<UserOperations> GetUserOperations(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DefinitionBase readPermission = null;
                DefinitionBase writePermission = null;
                List<DefinitionBase> permissions = _cache.GetPermissionsByResourceType(ORGANIZATION_RESOURCE_TYPE);
                if (permissions != null && permissions.Count > 0)
                {
                    readPermission = permissions.Where(a => a.Name.Equals(READ_DATA_PERMISSION, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    writePermission = permissions.Where(a => a.Name.Equals(WRITE_DATA_PERMISSION, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
                if (readPermission == null)
                    readPermission = GetPermission(ORGANIZATION_RESOURCE_TYPE, READ_DATA_PERMISSION);
                if (writePermission == null)
                    writePermission = GetPermission(ORGANIZATION_RESOURCE_TYPE, WRITE_DATA_PERMISSION);

                string resourceId = GetResourceId(facilityId, ORGANIZATION_RESOURCE_TYPE);
                List<Role_Resource> rr = _cache.GetPoliciesForOrganizations();
                List<Role> policies = new List<Role>();
                if (rr != null && rr.Count > 0)
                {
                    Role_Resource resource = rr.Where(a => a.Root.Equals(facilityId.root, StringComparison.InvariantCultureIgnoreCase) && a.Extension.Equals(facilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (resource != null)
                        policies = resource.RoleList;
                }
                if (policies == null || policies.Count == 0)
                    policies = AdministrativeFunctionsClient.GetPolicy(resourceId);

                IList<UserOperations> result = new List<UserOperations>();
                foreach (Role role in policies)
                {
                    if (string.IsNullOrEmpty(role.Name))
                        continue;

                    UserOperations user = new UserOperations()
                    {
                        UserName = role.Name,
                        BmsRead = role.Permissions.Contains(readPermission.Id),
                        BmsWrite = role.Permissions.Contains(writePermission.Id),
                        RoleId = role.Id,
                    };
                    result.Add(user);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public UserRoles GetUserRoles(String userName, string userId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                InfoWorld.Security.Authorization.PolicyAdministrationPoint.User user = new InfoWorld.Security.Authorization.PolicyAdministrationPoint.User(userId, userName, false);
                List<Role> roles = _cache.GetUserRoles(user.Id);
                if (roles == null || roles.Count == 0)
                {
                    roles = AdministrativeFunctionsClient.GetUserRoles(user);
                    _cache.CacheUserRoles(user, roles);
                }
                UserRoles userRoles = new UserRoles() { UserName = userName, UserId = userId };
                userRoles.IsAdministrator = GetUserRole(roles, ADMINISTRATORS_ROLE);
                userRoles.IsEMSUser = GetUserRole(roles, EMS_STAFF_ROLE);
                userRoles.IsGuestUser = GetUserRole(roles, GUESTS_ROLE);
                userRoles.IsNationalUser = GetUserRole(roles, NATIONAL_USERS_ROLE);
                userRoles.IsRegionalUser = GetUserRole(roles, REGIONAL_USERS_ROLE);
                userRoles.IsSiteUser = GetUserRole(roles, SITE_USERS_ROLE);
                userRoles.IsSupportUser = GetUserRole(roles, SUPPORT_USERS_ROLE);
                userRoles.IsVisnUser = GetUserRole(roles, VISN_USERS_ROLE);
                userRoles.IsAuditLogUser = GetUserRole(roles, AUDIT_LOG_USERS_ROLE);
                userRoles.IsEMSSupervisorUser = GetUserRole(roles, EMS_SUPERVISOR_USERS_ROLE);
                return userRoles;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<string> GetActiveDirectoryDomains()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<string> result = _cache.GetDomainsOrWorkGroups();
                if (result == null || result.Count == 0)
                {
                    _cache.CacheDomainsOrWorkGroups();
                    result = _cache.GetDomainsOrWorkGroups();
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public bool GetUserIsSuperUser(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    Facade.Data.User user = GetUserByUserName(userName);
                    List<Role> roles = _cache.GetUserRoles(user.Id.extension);
                    if (roles == null || roles.Count == 0)
                    {
                        InfoWorld.Security.Authorization.PolicyAdministrationPoint.User usr = new InfoWorld.Security.Authorization.PolicyAdministrationPoint.User(user.Id.extension, userName, false);
                        roles = AdministrativeFunctionsClient.GetUserRoles(usr);
                        _cache.CacheUserRoles(usr, roles);
                    }
                    return GetUserRole(roles, SUPER_USERS_ROLE);
                }
                catch
                {
                    return false;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<BMS.Facade.Data.User> GetUsersByRole(string roleName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Role role = _cache.GetRoleByName(roleName);
                if (role == null)
                    role = AdministrativeFunctionsClient.GetRoleByName(roleName);
                List<InfoWorld.Security.Authorization.PolicyAdministrationPoint.User> users = AdministrativeFunctionsClient.GetRoleUsers(Guid.Parse(role.Id));
                List<BMS.Facade.Data.User> result = new List<Facade.Data.User>();
                foreach (InfoWorld.Security.Authorization.PolicyAdministrationPoint.User user in users)
                    result.Add(new BMS.Facade.Data.User() { Id = new II(string.Empty, user.Id), UserName = user.Name });
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<BMS.Facade.Data.User> GetUsersByRoleAndFacility(string roleName, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<BMS.Facade.Data.User> result = new List<Facade.Data.User>();
                Role role = _cache.GetRoleByName(roleName);
                if (role == null)
                    role = AdministrativeFunctionsClient.GetRoleByName(roleName);
                List<InfoWorld.Security.Authorization.PolicyAdministrationPoint.User> users = AdministrativeFunctionsClient.GetRoleUsers(Guid.Parse(role.Id));

                List<Role_Resource> rr = _cache.GetPoliciesForOrganizations();
                if (rr == null || rr.Count == 0)
                {
                    _cache.CachePoliciesForOrganizations();
                    rr = _cache.GetPoliciesForOrganizations();
                }
                List<Role> rolesWithPolicy = null;
                if (rr != null && rr.Count > 0)
                {
                    Role_Resource resource = rr.Where(a => a.Root.Equals(facilityId.root, StringComparison.InvariantCultureIgnoreCase) && a.Extension.Equals(facilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (resource != null)
                        rolesWithPolicy = resource.RoleList;
                }
                if (rolesWithPolicy != null)
                {
                    Role userRole = null;
                    foreach (InfoWorld.Security.Authorization.PolicyAdministrationPoint.User u in users)
                    {
                        userRole = rolesWithPolicy.Where(a => a.Name.Equals(u.Name, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (userRole != null && userRole.Permissions != null && userRole.Permissions.Count > 0)
                            result.Add(new BMS.Facade.Data.User() { Id = new II(string.Empty, u.Id), UserName = u.Name });
                    }
                }
                                
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string GetFullUserName(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UserGroup ug = AdministrativeFunctionsClient.GetUserByUserName(userName);
                if (ug != null)
                    return ug.Name + "@" + ug.Domain;
                else
                    return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facade.Data.User> SearchUsers(string domain, string searchText)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<UserGroup> domainUsers = AdministrativeFunctionsClient.SearchUsers(domain, searchText);
                return MapUserGroupToUser(domainUsers);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Authorization(check access)

        public bool CheckPermission(string userName, string permissionName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return CheckPermission(userName, OPERATIONS_RESOURCE_EXTENSION, permissionName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public bool CheckBMSReadPermission(string userName, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return CheckPermission(userName, facilityId.extension, READ_DATA_PERMISSION);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public bool CheckBMSWritePermission(string userName, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return CheckPermission(userName, facilityId.extension, WRITE_DATA_PERMISSION);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<PermissionFacility> GetFacilitiesForReadAndWrite(string userName, List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<PermissionFacility> permissionFacilities = new List<PermissionFacility>();
                if (GetUserIsSuperUser(userName))
                {
                    foreach (Facility f in facilities)
                        permissionFacilities.Add(new PermissionFacility() { Facility = f, ReadAccess = true, WriteAccess = true });

                    return permissionFacilities;
                }
                else
                {
                    List<Role_Resource> result = _cache.GetPoliciesForOrganizations();
                    if (result == null || result.Count == 0)
                    {
                        _cache.CachePoliciesForOrganizations();
                        result = _cache.GetPoliciesForOrganizations();
                    }
                    result = (from p in result
                              where (from f in facilities
                                     select f.Id.extension).Contains(p.Extension, StringComparer.InvariantCultureIgnoreCase)
                              select p).ToList();

                    DefinitionBase readPermission = null;
                    DefinitionBase writePermission = null;
                    List<DefinitionBase> permissions = _cache.GetPermissionsByResourceType(ORGANIZATION_RESOURCE_TYPE);
                    if (permissions != null && permissions.Count > 0)
                    {
                        readPermission = permissions.Where(a => a.Name.Equals(READ_DATA_PERMISSION, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        writePermission = permissions.Where(a => a.Name.Equals(WRITE_DATA_PERMISSION, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    }
                    if (readPermission == null)
                        readPermission = GetPermission(ORGANIZATION_RESOURCE_TYPE, READ_DATA_PERMISSION);
                    if (writePermission == null)
                        writePermission = GetPermission(ORGANIZATION_RESOURCE_TYPE, WRITE_DATA_PERMISSION);
                    List<Role> roles = null;
                    PermissionFacility pf = null;
                    foreach (Role_Resource rr in result)
                    {
                        roles = rr.RoleList.Where(a => a.Name.Equals(userName, StringComparison.InvariantCultureIgnoreCase)).ToList();
                        if (roles != null && roles.Count > 0)
                        {
                            pf = new PermissionFacility();
                            pf.Facility = facilities.Where(a => a.Id.extension.Equals(rr.Extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                            if (roles.Where(a => a.Permissions.Where(b => b.Equals(readPermission.Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null).FirstOrDefault() != null)
                                pf.ReadAccess = true;
                            if (roles.Where(a => a.Permissions.Where(b => b.Equals(writePermission.Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null).FirstOrDefault() != null)
                                pf.WriteAccess = true;
                            if (pf.ReadAccess || pf.WriteAccess)
                                permissionFacilities.Add(pf);
                        }
                    }
                    return permissionFacilities;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<Facility> GetFacilitiesWithReadAccess(string userName, List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Facility> result = new List<Facility>();
                foreach (Facility f in facilities)
                {
                    if (CheckPermission(userName, f.Id.extension, READ_DATA_PERMISSION))
                        result.Add(f);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Save Methods

        public void SetAvailableDomains(string username, string userid, IList<string> domains)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Operation operation = AdministrativeFunctionsClient.GetOperationByName(MEMBER_OF_PERMISSION);
                Role role = _cache.GetRoleByName(username);
                if (role == null)
                    role = AdministrativeFunctionsClient.GetRoleByName(username);             
                domains.ForEach(domain => AdministrativeFunctionsClient.InsertPermissionForResourceAndOperation(GetResourceId(new II(domain, domain), DOMAIN_RESOURCE_TYPE), role.Id, operation.Id));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void AddUserOperations(string userName, List<PermissionFacility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SetUserOperations(facilities, userName, OperationMode.Add);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void RemoveUserOperations(string userName, List<PermissionFacility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SetUserOperations(facilities, userName, OperationMode.Revoke);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveUserRoles(UserRoles userRoles)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {                
                List<Role> roles = _cache.GetRoles();
                if (roles == null || roles.Count == 0)
                    roles = AdministrativeFunctionsClient.GetRoles(false);
                SetUserRole(userRoles, roles, ADMINISTRATORS_ROLE, userRoles.IsAdministrator);
                SetUserRole(userRoles, roles, EMS_STAFF_ROLE, userRoles.IsEMSUser);
                SetUserRole(userRoles, roles, GUESTS_ROLE, userRoles.IsGuestUser);
                SetUserRole(userRoles, roles, NATIONAL_USERS_ROLE, userRoles.IsNationalUser);
                SetUserRole(userRoles, roles, REGIONAL_USERS_ROLE, userRoles.IsRegionalUser);
                SetUserRole(userRoles, roles, SITE_USERS_ROLE, userRoles.IsSiteUser);
                SetUserRole(userRoles, roles, SUPPORT_USERS_ROLE, userRoles.IsSupportUser);
                SetUserRole(userRoles, roles, VISN_USERS_ROLE, userRoles.IsVisnUser);
                SetUserRole(userRoles, roles, AUDIT_LOG_USERS_ROLE, userRoles.IsAuditLogUser);
                SetUserRole(userRoles, roles, EMS_SUPERVISOR_USERS_ROLE, userRoles.IsEMSSupervisorUser);
                InfoWorld.Security.Authorization.PolicyAdministrationPoint.User user = new InfoWorld.Security.Authorization.PolicyAdministrationPoint.User(userRoles.UserId, userRoles.UserName, false);
                CacheUserRoles(user);
                if (!isIISSingleInstance)
                {
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateUserRolesCache(user);
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateUserRolesCache(user);
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security SaveUserRoles: user - " + user.Name);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateUserRolesCache(user);
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security SaveUserRoles: user - " + user.Name);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("Security SaveUserRoles: user - " + user.Name);
                            Tracer.TraceException(ex);
                        }
                    });
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        #endregion

        #endregion

        #region Private Methods

        private bool CheckPermission(string userName, string resourceExtension, string permissionName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(userName))
                    throw new ArgumentNullException("userName");

                if (string.IsNullOrEmpty(permissionName))
                    throw new ArgumentNullException("permissionName");                

                bool result = false;
                List<Permission> permissions = _cache.GetUserPermissionsCache(userName);
                Permission permission = null;
                bool resultFound = false;

                if (permissions != null && permissions.Count > 0)
                {
                    permission = permissions.Where(a => a.Operation.Equals(permissionName, StringComparison.InvariantCultureIgnoreCase) && a.ResourceExtension.Equals(resourceExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (permission != null)
                    {
                        result = permission.HasAccess;
                        resultFound = true;
                    }
                }

                if (!resultFound)
                {
                    Message stmtMessage = CreateMessage(userName, resourceExtension, permissionName);
                    result = GetPermission(stmtMessage);
                    _cache.AddUserPermissionsToCache(userName, new Permission() { Operation = permissionName, ResourceExtension = resourceExtension, HasAccess = result });
                }

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SetUserOperations(List<PermissionFacility> facilities, string userName, OperationMode mode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DefinitionBase readPermission = null;
                DefinitionBase writePermission = null;
                List<DefinitionBase> permissions = _cache.GetPermissionsByResourceType(ORGANIZATION_RESOURCE_TYPE);
                if (permissions != null && permissions.Count > 0)
                {
                    readPermission = permissions.Where(a => a.Name.Equals(READ_DATA_PERMISSION, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    writePermission = permissions.Where(a => a.Name.Equals(WRITE_DATA_PERMISSION, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
                if (readPermission == null)
                    readPermission = GetPermission(ORGANIZATION_RESOURCE_TYPE, READ_DATA_PERMISSION);
                if (writePermission == null)
                    writePermission = GetPermission(ORGANIZATION_RESOURCE_TYPE, WRITE_DATA_PERMISSION);

                Role role = _cache.GetRoleByName(userName);
                if (role == null)
                {
                    role = AdministrativeFunctionsClient.GetRoleByName(userName);
                    if (role == null)
                    {
                        AdministrativeFunctionsClient.AddRole(userName, BUILD_IN);
                        role = AdministrativeFunctionsClient.GetRoleByName(userName);
                    }
                    _cache.CacheNewRole(role);
                }

                int count = facilities.Where(a => a.ReadAccess).ToList().Count;
                count = count + facilities.Where(a => a.WriteAccess).ToList().Count;

                string[] resources = new string[count];
                string[] roles = new string[count];
                string[] actions = new string[count];
                int i = 0;
                string resourceId = string.Empty;
                foreach (PermissionFacility pf in facilities)
                {
                    if (mode == OperationMode.Revoke)
                        resourceId = pf.Facility.Id.root + "|" + pf.Facility.Id.extension;
                    else
                        resourceId = GetResourceId(pf.Facility.Id, ORGANIZATION_RESOURCE_TYPE);
                    resources[i] = resourceId;
                    roles[i] = role.Id;
                    if (pf.ReadAccess)
                        actions[i] = readPermission.Id;
                    if (pf.WriteAccess)
                    {
                        if (pf.ReadAccess)
                            i = i + 1;
                        resources[i] = resourceId;
                        roles[i] = role.Id;
                        actions[i] = writePermission.Id;
                    }
                    i = i + 1;
                }

                if (mode == OperationMode.Add || mode == OperationMode.Reset)
                    AdministrativeFunctionsClient.InsertPermissionsForResourcesAndOperations(resources, roles, actions);
                else if (mode == OperationMode.Revoke)
                    AdministrativeFunctionsClient.DeletePermissionsForResourcesAndOperations(resources, roles, actions);
                _cache.ClearUserPermissionsCache(userName);
                if (mode == OperationMode.Revoke)
                    _cache.ManagePoliciesForOrganizationsByRole(role, facilities, true, null, null);
                else
                    _cache.ManagePoliciesForOrganizationsByRole(role, facilities, false, readPermission, writePermission);
                if (!isIISSingleInstance)
                {
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.ClearUserPermissionCache(userName);
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.ClearUserPermissionCache(userName);
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security SetUserOperations: user - " + userName);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.ClearUserPermissionCache(userName);
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security SetUserOperations: user - " + userName);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("Security SetUserOperations: user - " + userName);
                            Tracer.TraceException(ex);
                        }
                    });
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdatePoliciesForOrganizationsCache();
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdatePoliciesForOrganizationsCache();
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security SetUserOperations: user - " + userName);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdatePoliciesForOrganizationsCache();
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security SetUserOperations: user - " + userName);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("Security SetUserOperations: user - " + userName);
                            Tracer.TraceException(ex);
                        }
                    });
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SetUserRole(UserRoles user, List<Role> roles, string roleName, bool assign)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Role role = roles.First(r => r.Name == roleName);
                if (assign)
                    AdministrativeFunctionsClient.AssignUser(user.UserId, role.Id);
                else
                    AdministrativeFunctionsClient.DeassignUser(user.UserId, role.Id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool GetUserRole(List<Role> roles, string roleName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return roles.FirstOrDefault(r => r.Name == roleName) != null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<Facade.Data.User> MapUserGroupToUser(IList<UserGroup> userGroups)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string currentDomain = GetCurrentDomain();
                IList<Facade.Data.User> result = userGroups.Select(u => ToUser(u, currentDomain)).ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private Facade.Data.User ToUser(UserGroup ug, string currentDomain)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new Facade.Data.User()
                {
                    Domain = currentDomain,
                    UserName = ug.Domain + "\\" + ug.Name,
                    Id = new II() { root = ug.Domain, extension = ug.ID },
                    FullName = string.IsNullOrEmpty(ug.FullName) ? ug.Name : ug.FullName,
                    IsGroup = ug.IsGroup
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static string GetResourceId(II id, string resourceType)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return string.Format("{0}|{1}|{2}", id.root, id.extension, resourceType);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private System.ServiceModel.Channels.Message CreateRequestMesage(AuthorizationServiceClient clientAutorizare, String subject, String resources, String action)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                XACMLAuthzDecisionQueryType query = new XACMLAuthzDecisionQueryType();
                Lagash.Xacml.Core.Context.AttributeCollection subjAttributes = new Lagash.Xacml.Core.Context.AttributeCollection();
                subjAttributes.Add(new AttributeElement("urn:oasis:names:tc:xacml:1.0:subject:subject-id",
                                                        "http://www.w3.org/2001/XMLSchema#string",
                                                        null,
                                                        null,
                                                        subject, Lagash.Xacml.Core.XacmlVersion.Version20));

                Lagash.Xacml.Core.Context.AttributeCollection resAttributes = new Lagash.Xacml.Core.Context.AttributeCollection();
                resAttributes.Add(new AttributeElement("urn:oasis:names:tc:xacml:1.0:resource:resource-id",
                                                        "http://www.w3.org/2001/XMLSchema#string",
                                                        null,
                                                        null,
                                                        resources, Lagash.Xacml.Core.XacmlVersion.Version20));

                Lagash.Xacml.Core.Context.AttributeCollection actAttributes = new Lagash.Xacml.Core.Context.AttributeCollection();
                actAttributes.Add(new AttributeElement("urn:oasis:names:tc:xacml:1.0:action:action-id",
                                                        "http://www.w3.org/2001/XMLSchema#string",
                                                        null,
                                                        null,
                                                        action, Lagash.Xacml.Core.XacmlVersion.Version20));

                SubjectCollection subjects = new SubjectCollection();
                subjects.Add(new SubjectElement(null, subjAttributes, Lagash.Xacml.Core.XacmlVersion.Version20));
                ResourceCollection resourcesCol = new ResourceCollection();
                resourcesCol.Add(new ResourceElement(null, ResourceScope.Immediate, resAttributes, Lagash.Xacml.Core.XacmlVersion.Version20));
                ActionElement actionEl = new ActionElement(actAttributes, Lagash.Xacml.Core.XacmlVersion.Version20);
                EnvironmentElement environment = new EnvironmentElement(new Lagash.Xacml.Core.Context.AttributeCollection(), Lagash.Xacml.Core.XacmlVersion.Version20);

                RequestElement request = new RequestElement(subjects, resourcesCol, actionEl, environment, Lagash.Xacml.Core.XacmlVersion.Version20);// (reader, Lagash.Xacml.Core.XacmlVersion.Version20);
                query.Request = request;

                return System.ServiceModel.Channels.Message.CreateMessage(clientAutorizare.Endpoint.Binding.MessageVersion, "", query);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool GetPermission(Message stmtMessage)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                XACMLAuthzDecisionStatementType statement = XACMLAuthzDecisionStatementType.CreateFrom(stmtMessage.GetReaderAtBodyContents());
                bool permission = (statement.Response.Results[0].Decision.ToString().ToUpper().CompareTo("PERMIT") == 0);
                return permission;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private Message CreateMessage(string userName, string resourceExtension, string permissionName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Message stmtMessage = null;
                AuthorizationServiceClient clientAutorizare = AuthorizationServiceClient;
                try
                {
                    Message msg = CreateRequestMesage(clientAutorizare, userName, resourceExtension, permissionName);
                    stmtMessage = clientAutorizare.CheckAccess(msg);
                }
                catch (InvalidOperationException ex)
                {
                    if (ex.Message == "This message cannot support the operation because it has been written."
                        || ex.Message == "This message cannot support the operation because it has been copied."
                        || ex.Message == "This message cannot support the operation because it has been read.")
                    {
                        Message msg = CreateRequestMesage(clientAutorizare, userName, resourceExtension, permissionName);
                        stmtMessage = clientAutorizare.CheckAccess(msg);
                    }
                    else
                        throw;
                }
                return stmtMessage;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string CheckUserRoleExists(string username, string userid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Role role = _cache.GetRoleByName(username);
                if (role == null)
                    role = AdministrativeFunctionsClient.GetRoleByName(username);
                string roleId = null;
                if (role == null)
                {
                    roleId = AdministrativeFunctionsClient.AddRole(username, BUILD_IN);
                    if (roleId.Equals(Guid.Empty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                        throw new InvalidOperationException("roleId empty in AdministrativeFunctionsClient.Addrole function for " + username);
                    AdministrativeFunctionsClient.AssignUser(userid, roleId);
                    InfoWorld.Security.Authorization.PolicyAdministrationPoint.User user = new InfoWorld.Security.Authorization.PolicyAdministrationPoint.User(userid, username, false);
                    CacheUserRoles(user);
                    if (!isIISSingleInstance)
                    {
                        ThreadPool.QueueUserWorkItem((o) =>
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateUserRolesCache(user);
                            }
                            catch (CommunicationObjectAbortedException)
                            {
                                try
                                {
                                    BMService.BMSFactory.BedManagerCacheClientWindows.UpdateUserRolesCache(user);
                                }
                                catch (Exception ex)
                                {
                                    Tracer.TraceMessage("Security CheckUserRoleExists: user - " + username);
                                    Tracer.TraceException(ex);
                                }
                            }
                            catch (ObjectDisposedException)
                            {
                                try
                                {
                                    BMService.BMSFactory.BedManagerCacheClientWindows.UpdateUserRolesCache(user);
                                }
                                catch (Exception ex)
                                {
                                    Tracer.TraceMessage("Security CheckUserRoleExists: user - " + username);
                                    Tracer.TraceException(ex);
                                }
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("Security CheckUserRoleExists: user - " + username);
                                Tracer.TraceException(ex);
                            }
                        });
                    }
                }
                else
                    roleId = role.Id;
                return roleId;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private DefinitionBase GetPermission(string resourceType, string permissionName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<DefinitionBase> list = _cache.GetPermissionsByResourceType(resourceType);
                if (list == null || list.Count == 0)                    
                    list = AdministrativeFunctionsClient.GetPermissionsByResourceType(resourceType);
                return list.FirstOrDefault(d => d.Name == permissionName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }                

        private bool IsAvailableRole(Role role, string username, string perimissionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (role.Name == null)
                    return false;
                if (!role.Name.Equals(username, StringComparison.InvariantCultureIgnoreCase))
                    return false;
                bool result = role.Permissions.Contains(perimissionId);
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Cache

        public void FillCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _cache.FillCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void ClearUserPermissionCache(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _cache.ClearUserPermissionsCache(userName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public void CachePoliciesForOrganizations()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _cache.CachePoliciesForOrganizations();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CacheUserRoles(InfoWorld.Security.Authorization.PolicyAdministrationPoint.User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _cache.CacheUserRoles(user, AdministrativeFunctionsClient.GetUserRoles(user));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }                

        #endregion        
    }

}
