﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using InfoWorld.Security.Authorization.PolicyAdministrationPoint;
using System.Threading;
using System.ServiceModel.Channels;

namespace BMS.ServicesWrapper.Security
{
    internal class SecurityCache
    {
        ISecurityWrapper _SecurityWrapper;
        
        internal SecurityCache(ISecurityWrapper SecurityWrapper) 
        {
            _SecurityWrapper = SecurityWrapper;            
        }

        private static readonly ReaderWriterLockSlim cacheLock = new ReaderWriterLockSlim();
        private const string DOMAIN_RESOURCE_TYPE = "HL7-RIM-V3-DOMAIN";
        private const string ORGANIZATION_RESOURCE_TYPE = "HL7-RIM-V3-ORGANIZATION"; //for facility        
        private const string OPERATIONS_RESOURCE_EXTENSION = "HM3Services";

        static List<string> _domains = new List<string>();
        static List<string> _domainsOrWorkGroups = null;        
        static string _currentDomain;

        static Dictionary<string, List<DefinitionBase>> _permissionsByResourceType = new Dictionary<string,List<DefinitionBase>>(StringComparer.InvariantCultureIgnoreCase);        
        static List<Role_Resource> _policiesByOrganizations = new List<Role_Resource>();
        static Dictionary<string, Role> _rolesByNames = new Dictionary<string, Role>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<Role>> _userRoles = new Dictionary<string, List<Role>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<Permission>> _userPermissionsCache = new Dictionary<string, List<Permission>>(StringComparer.InvariantCultureIgnoreCase);

        internal void FlushCache()
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_domains != null || _domains.Count > 0)
                    _domains.Clear();
                if (_permissionsByResourceType != null || _permissionsByResourceType.Count > 0)
                    _permissionsByResourceType.Clear();                
                if (_policiesByOrganizations != null || _policiesByOrganizations.Count > 0)
                    _policiesByOrganizations.Clear();
                if (_rolesByNames != null || _rolesByNames.Count > 0)
                    _rolesByNames.Clear();
                if (_userRoles != null || _userRoles.Count > 0)
                    _userRoles.Clear();
                if (_userPermissionsCache != null || _userPermissionsCache.Count > 0)
                    _userPermissionsCache.Clear();
                _domainsOrWorkGroups = null;                
                _currentDomain = null;
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void FillCache()
        {
            FlushCache();
            cacheLock.EnterWriteLock();
            try
            {
                _domains = _SecurityWrapper.AdministrativeFunctionsClient.GetAvailableDomains();
                _currentDomain = _SecurityWrapper.AdministrativeFunctionsClient.GetCurrentDomain();
                _domainsOrWorkGroups = _SecurityWrapper.AdministrativeFunctionsClient.GetDomains();
                _permissionsByResourceType.Add(DOMAIN_RESOURCE_TYPE, _SecurityWrapper.AdministrativeFunctionsClient.GetPermissionsByResourceType(DOMAIN_RESOURCE_TYPE));
                _permissionsByResourceType.Add(ORGANIZATION_RESOURCE_TYPE, _SecurityWrapper.AdministrativeFunctionsClient.GetPermissionsByResourceType(ORGANIZATION_RESOURCE_TYPE));
                
                List<Facade.Data.Facility> facilities = EIS.EISFactory.InstanceFromWCF.GetFacilities().ToList();
                StringBuilder resourceIds = new StringBuilder();
                foreach (Facade.Data.Facility f in facilities)
                {
                    resourceIds.Append(f.Id.root);
                    resourceIds.Append(";");
                    resourceIds.Append(f.Id.extension);
                    resourceIds.Append(";");
                    resourceIds.Append(ORGANIZATION_RESOURCE_TYPE);
                    resourceIds.Append(";");
                }
                _policiesByOrganizations = _SecurityWrapper.AdministrativeFunctionsClient.GetBulkPolicies(resourceIds.ToString());
                List<Role> roles = _SecurityWrapper.AdministrativeFunctionsClient.GetRoles(false);
                foreach (Role r in roles)
                    _rolesByNames.Add(r.Name, r);                                
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        #region Get Methods

        internal List<string> GetAvailableDomains()
        {
            return _domains;            
        }

        internal List<DefinitionBase> GetPermissionsByResourceType(string resourceType) 
        {
            if (_permissionsByResourceType.ContainsKey(resourceType))
                return _permissionsByResourceType[resourceType];                                
            return null;
        }

        internal List<Role_Resource> GetPoliciesForOrganizations()
        {
            return _policiesByOrganizations;            
        }

        internal Role GetRoleByName(string name) 
        {
            if (_rolesByNames.ContainsKey(name))
                return _rolesByNames[name];            
            return null;
        }

        internal List<Role> GetRoles()
        {
            List<Role> list = new List<Role>();
            foreach (string name in _rolesByNames.Keys)
                list.Add(_rolesByNames[name]);
            return list;            
        }
        
        internal List<Role> GetUserRoles(string id) 
        {
            if (_userRoles.ContainsKey(id))
                return _userRoles[id];            
            return null;
        }

        internal List<Permission> GetUserPermissionsCache(string userName)
        {
            if (_userPermissionsCache.ContainsKey(userName))
                return _userPermissionsCache[userName];            
            return null;            
        }

        internal List<string> GetDomainsOrWorkGroups()
        {
            return _domainsOrWorkGroups;
        }

        internal string GetCurrentDomain()
        {
            return _currentDomain;
        }

        #endregion

        #region Manage Cache

        internal void CacheAvailableDomains()
        {
            cacheLock.EnterWriteLock();
            try
            {
                _domains = _SecurityWrapper.AdministrativeFunctionsClient.GetAvailableDomains();
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void ClearUserPermissionsCache(string userName)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_userPermissionsCache.ContainsKey(userName))
                    _userPermissionsCache.Remove(userName);
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void AddUserPermissionsToCache(string userName, Permission permission)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (!_userPermissionsCache.ContainsKey(userName))
                    _userPermissionsCache.Add(userName, new List<Permission>());
                Permission tempPermission = _userPermissionsCache[userName].Where(a => a.Operation.Equals(permission.Operation, StringComparison.InvariantCultureIgnoreCase) && a.ResourceExtension.Equals(permission.ResourceExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (tempPermission != null)
                    _userPermissionsCache[userName].Remove(tempPermission);
                _userPermissionsCache[userName].Add(permission);                
            }
            finally { cacheLock.ExitWriteLock(); }
        }        

        internal void CachePoliciesForOrganizations()
        {
            cacheLock.EnterWriteLock();
            try
            {
                List<Facade.Data.Facility> facilities = EIS.EISFactory.InstanceFromWCF.GetFacilities().ToList();
                StringBuilder resourceIds = new StringBuilder();
                foreach (Facade.Data.Facility f in facilities)
                {
                    resourceIds.Append(f.Id.root);
                    resourceIds.Append(";");
                    resourceIds.Append(f.Id.extension);
                    resourceIds.Append(";");
                    resourceIds.Append(ORGANIZATION_RESOURCE_TYPE);
                    resourceIds.Append(";");
                }
                _policiesByOrganizations = _SecurityWrapper.AdministrativeFunctionsClient.GetBulkPolicies(resourceIds.ToString());
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void ManagePoliciesForOrganizationsByRole(Role role, List<Facade.Data.PermissionFacility> facilities, bool isRemove, DefinitionBase readPermission, DefinitionBase writePermission)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_policiesByOrganizations == null && _policiesByOrganizations.Count == 0)
                    CachePoliciesForOrganizations();
                else
                {
                    Role_Resource rr = null;
                    List<string> permissions = null;
                    foreach (BMS.Facade.Data.PermissionFacility pf in facilities)
                    {
                        rr = _policiesByOrganizations.Where(a => a.Extension.Equals(pf.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (rr != null && rr.RoleList != null && rr.RoleList.Count > 0)
                        {
                            if (isRemove)
                                rr.RoleList.RemoveAll(a => a.Id.Equals(role.Id, StringComparison.InvariantCultureIgnoreCase));
                            else
                            {
                                permissions = new List<string>();
                                if (pf.ReadAccess)
                                    permissions.Add(readPermission.Id);
                                if (pf.WriteAccess)
                                    permissions.Add(writePermission.Id);
                                rr.RoleList.Add(new Role() { Id = role.Id, Name = role.Name, Description = role.Description, ExtensionData = role.ExtensionData, Permissions = permissions });
                            }
                        }
                    }
                }
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheNewRole(Role role)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_rolesByNames != null)
                {
                    if (!_rolesByNames.ContainsKey(role.Name))
                        _rolesByNames.Add(role.Name, role);
                    else
                        _rolesByNames[role.Name] = role;
                }   
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheUserRoles(User user, List<Role> roles)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_userRoles.ContainsKey(user.Id))
                    _userRoles.Remove(user.Id);
                if (roles != null && roles.Count > 0)
                    _userRoles.Add(user.Id, roles);
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheCurrentDomain()
        {
            cacheLock.EnterWriteLock();
            try
            {
                _currentDomain = _SecurityWrapper.AdministrativeFunctionsClient.GetCurrentDomain();
            }
            catch (Exception) 
            {
                _currentDomain = _SecurityWrapper.AdministrativeFunctionsClient.GetCurrentDomain();
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheDomainsOrWorkGroups()
        {
            cacheLock.EnterWriteLock();
            try
            {
                _domainsOrWorkGroups = _SecurityWrapper.AdministrativeFunctionsClient.GetDomains();
            }
            finally { cacheLock.ExitWriteLock(); }            
        }

        #endregion
    }
}
