﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using FC = BMS.Facade.Data;
using DC = BMS.DataContracts;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.WF;
using BMS.Utils;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.VistaWorker.Exceptions;
using BMS.ServicesWrapper.BMService;

namespace BMS.VistaWorker.Writer.Concrete.WF
{
    /// <summary>
    /// The scheduled admission writer.
    /// </summary>
    class ScheduledAdmissionWriter : BaseWorkFlowWriter<ScheduledAdmission>
    {

        /// <summary>
        /// Sends the event.
        /// </summary>
        /// <param name="entity">The entity.</param>
        protected override void SendEvent(ScheduledAdmission entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FC.Ward ward = (entity.WardLocationId == null) ? null : EIS.GetWard(entity.WardLocationId, VistaSite.Id);
                IList<CD> specialityList = EVS.GetCodes(new ServicesWrapper.EVS.CodeFilterParameters() { VocabularyDomain = FC.Util.Vocabulary.VistaTreatingSpecialty.ToString() });
                IList<CD> medicalCenterDivisionList = EVS.GetCodes(new ServicesWrapper.EVS.CodeFilterParameters() { VocabularyDomain = FC.Util.Vocabulary.VistaMedicalCenterDivision.ToString() });

                Dictionary<string, string> providerNames = null;
                if (entity.Provider != null)
                    providerNames = Utilities.SplitPersonFullName(entity.Provider.Name);

                Dictionary<string, string> schedulerNames = null;
                if (entity.Scheduler != null)
                    schedulerNames = Utilities.SplitPersonFullName(entity.Scheduler.Name);

                string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();

                DC.ScheduledAdmission scheduledAdmission = new DC.ScheduledAdmission()
                {
                    Id = new II(domainId, null),
                    Diagnosis = entity.AdmittingDiagnosis,
                    CanceledDate = entity.DateTimeCancelled,
                    WardId = ward.Id,
                    IEN = entity.IEN,
                    LOSExpected = (entity.LengthOfStayExpected.HasValue) ? entity.LengthOfStayExpected.Value : 0,
                    PatientId = MakeWfPatient(entity.Patient),
                    ReservationDate = entity.ReservationDateTime,
                    Surgery = (entity.Surgery.ToUpper().Equals("N") || entity.Surgery.ToUpper().Equals("NO")) ? false : true,
                    Speciality = (entity.TreatingSpecialtyId.HasValue) ? specialityList.Where(a => a.code == VistaSite.Code + "_" + entity.TreatingSpecialtyId.Value.ToString()).FirstOrDefault() : null,
                    VistaDivision = (entity.MedicalCenterDivisionId.HasValue) ? medicalCenterDivisionList.Where(a => a.code == VistaSite.Code + "_" + entity.MedicalCenterDivisionId.Value.ToString()).FirstOrDefault() : null,
                    SchedulerId = null,
                    ProviderId = null,
                    VistaSiteId = (VistaSite != null) ? VistaSite.Id : null
                };
                ConvertToUtc(scheduledAdmission);

                try
                {
                    BMSFactory.BedManagerOperationsClientWindows.CreateScheduledAdmission(scheduledAdmission);
                    string msj = string.Format("Create scheduled admission for patient {0} with succes", entity.Patient.Name);
                    Logger.LogInformation(msj);
                }
                catch (FaultException<GenericWFServiceFault> e)
                {
                    string format = "Create admission order for patient {0} failed:\n{1}";
                    string msj = string.Format(format, entity.Patient.Name, e.Detail);
                    throw new AbortEntityProcessingException(msj);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
