﻿using System;
using System.ServiceModel;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.HL7.ServiceContracts;
using BMS.VistaWorker2.Writer;
using BMS.Utils;
using FD = BMS.Facade.Data;
using VI = BMS.VistaIntegration.FacadeContracts;
using InfoWorld.HL7.ITS;
using BMS.VistaIntegration.VistA;
using System.Linq;
using System.Collections.Generic;
using InfoWorld.EVS.CTSMAPI;
using BMS.Facade;
using BMS.VistaIntegration.HL7.DataContracts;
using System.Globalization;
using BMS.VistaIntegration.LogCheckers;

namespace BMS.VistaIntegration.HL7.ServiceImplementation
{
    [ServiceBehavior(IgnoreExtensionDataObject = true, InstanceContextMode = InstanceContextMode.PerSession, IncludeExceptionDetailInFaults = true)]
    public class HL7OperationsCore : HL7OperationsBase, IHL7Operations
    {
        public void NotifyEvent(HL7Event hl7Event)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            IVistASession vistaSession = null;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            try
            {
                IWriterManager writerManager = GetWriterManager(hl7Event.SendingFacilityIdentifier);
                if (writerManager == null)
                    return;

                ActType previousActType = ActType.All;
                MovementTransactionType transactionType = (MovementTransactionType)0;
                string previousActOrderableItemTypeCode = null;
                bool checkPatientAppointment = false;

                CultureInfo provider = CultureInfo.InvariantCulture;
                string formatWithTimeZone = "yyyyMMddHHmmsszzz";
                string format = "yyyyMMddHHmmss";
                bool eventDateTimeHasValue = false, birthTimeHasValue = false;
                DateTimeOffset eventDateTime = DateTimeOffset.UtcNow;
                DateTime patientDateOfBirth;

                if (!(birthTimeHasValue = DateTime.TryParseExact(hl7Event.PatientDateOfBirth, formatWithTimeZone, provider, DateTimeStyles.None, out patientDateOfBirth)))
                    birthTimeHasValue = DateTime.TryParseExact(hl7Event.PatientDateOfBirth, format, provider, DateTimeStyles.None, out patientDateOfBirth);


                NoLogChecker noLogChecker = new NoLogChecker();

                switch (hl7Event.EventType)
                {
                    case Constants.AdmissionEventMessageType:
                        previousActType = ActType.AdmissionOrder;   //ScheduledAdmission has the same value, as it is treated the same.
                        previousActOrderableItemTypeCode = Constants.ORDERABLE_ITEM_TYPE_ADMISSION_CODE;
                        transactionType = MovementTransactionType.ADMISSION;
                        if (!(eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.AdmissionDateTime, formatWithTimeZone, provider, DateTimeStyles.None, out eventDateTime)))
                            eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.AdmissionDateTime, format, provider, DateTimeStyles.None, out eventDateTime);
                        break;
                    case Constants.TransferEventMessageType:
                        previousActType = ActType.MovementOrder;
                        previousActOrderableItemTypeCode = Constants.ORDERABLE_ITEM_TYPE_TRANSFER_CODE;
                        transactionType = MovementTransactionType.TRANSFER;
                        if (!(eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.AdmissionDateTime, formatWithTimeZone, provider, DateTimeStyles.None, out eventDateTime)))
                            eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.AdmissionDateTime, format, provider, DateTimeStyles.None, out eventDateTime);
                        break;
                    case Constants.DischargeEventMessageType:
                        previousActType = ActType.DischargeOrder;   //special case, must check the patient appointment too.
                        checkPatientAppointment = true;
                        previousActOrderableItemTypeCode = Constants.ORDERABLE_ITEM_TYPE_DISCHARGE_CODE;
                        transactionType = MovementTransactionType.DISCHARGE;
                        if (!(eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.DischargeDateTime, formatWithTimeZone, provider, DateTimeStyles.None, out eventDateTime)))
                            eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.DischargeDateTime, format, provider, DateTimeStyles.None, out eventDateTime);
                        break;
                    case Constants.RegisterPatientEventMessageType:
                        previousActType = (ActType)0;
                        break;
                    default:
                        return;
                }

                if (!eventDateTimeHasValue)
                    if (!(eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.RecordedDateTime, formatWithTimeZone, provider, DateTimeStyles.None, out eventDateTime)))
                        eventDateTimeHasValue = DateTimeOffset.TryParseExact(hl7Event.RecordedDateTime, format, provider, DateTimeStyles.None, out eventDateTime);

                if (previousActType == (ActType)0)
                    vistaSession = GetVistASession(VI.Admin.VistaDataType.Patient, hl7Event.SendingFacilityIdentifier);
                else
                    vistaSession = GetVistASession(VI.Admin.VistaDataType.ADT, hl7Event.SendingFacilityIdentifier);
                if (vistaSession == null)
                    return;
                //if the event is a register patient event, get the patient from vista and update the patient entity in BMS.
                IVistAQuery vistaQuery = vistaSession.MakeQuery();

                if (previousActType == (ActType)0)
                {
                    Patient vistaPatient = vistaQuery.GetPatientByIen(hl7Event.PatientIEN);

                    if (vistaPatient == null)
                    {
                        Tracer.TraceMessage("Error! An HL7 event concerning patient with SSN " + hl7Event.PatientSSN + " was received. The patient wasn't found in VistA though.)");
                        return;
                    }
                    using (var entityWriter = writerManager.MakeEntityWriter<Patient>())
                    {
                        entityWriter.LogChecker = noLogChecker;
                        entityWriter.Update(vistaPatient);
                    }
                    return;
                }

                if (checkPatientAppointment)
                {
                    FD.Act previousBMSAppointment = Facade.FacadeManager.ADTInterface.GetLastAct(hl7Event.PatientSSN, ActType.PatientAppointment, hl7Event.SendingFacilityIdentifier);
                    IList<PatientAppointment> patientAppointments = vistaQuery.GetPatientAppointments(startDate: previousBMSAppointment.CreationDate + TimeSpan.FromSeconds(1), patientIen: hl7Event.PatientIEN);
                    if (patientAppointments != null && patientAppointments.Count > 0)
                    {
                        //the patient appointment found must be inserted into BMS before processing anything else.
                        PatientAppointment patientAppointment = patientAppointments.OrderByDescending(pa => pa.DateAppointmentMade).First();
                        using (var entityWriter = writerManager.MakeEntityWriter<PatientAppointment>())
                        {
                            entityWriter.LogChecker = noLogChecker;
                            entityWriter.Insert(patientAppointment);
                        }
      
                    }

                    //if 2 or more patient appointments are found, there is an underlying error.
                    if (patientAppointments != null && patientAppointments.Count > 1)
                        Tracer.TraceMessage("Two or more patient appointments were found since the last HL7 VistA synchronization, there is an underlying error.");
                }

                //get the appropriate orders from VistA and check if they were inserted in BMS.
                FD.Act previousBMSAct = Facade.FacadeManager.ADTInterface.GetLastAct(hl7Event.PatientSSN, previousActType, hl7Event.SendingFacilityIdentifier);
                IList<CDWithProperties> orderableItems = FacadeUtil.GetOrderableItemByType(previousActOrderableItemTypeCode, BMS.ServicesWrapper.EVS.EVSFactory.InstanceFromWCF);
                IList<string> orderableItemsIen = new List<string>();
                foreach (CDWithProperties cdwp in orderableItems)
                {
                    string code = cdwp.code;
                    string ien = code.Substring(code.LastIndexOf('_') + 1);
                    if (!orderableItemsIen.Contains(ien))
                        orderableItemsIen.Add(ien);
                }

                //start time to search in vistA initialized to the lowest possible recording date.
                DateTime vistASearchStartTime = new DateTime(1870, 12, 31);
                if (previousBMSAct != null && previousBMSAct.CreationDate.HasValue)
                    vistASearchStartTime = previousBMSAct.CreationDate.Value + TimeSpan.FromSeconds(1);
                else
                    if (previousBMSAct == null && birthTimeHasValue)
                        vistASearchStartTime = patientDateOfBirth;

                IList<OrderAction> orderActions = vistaQuery.GetOrderActions(
                    startDate: vistASearchStartTime,
                    patientIen: hl7Event.PatientIEN,
                    orderableItemsIen: orderableItemsIen);

                //if the order hasn't been inserted, process it now.
                if (orderActions != null && orderActions.Count > 0)
                {
                    OrderAction order = orderActions.OrderByDescending(oa => oa.DateTimeOrdered).First();
                    using (var entityWriter = writerManager.MakeEntityWriter<OrderAction>())
                    {
                        entityWriter.LogChecker = noLogChecker;
                        entityWriter.Insert(order);
                    }
                   
                }

                if (orderActions != null && orderActions.Count > 1)
                    Tracer.TraceMessage("Two or more order actions were found since the last HL7 VistA synchronization, there is an underlying error.");

                //get the event by mdws or cache and insert it into BMS
                if (!eventDateTimeHasValue)
                {
                    Tracer.TraceMessage("The received ADT event has no creation date. The data will not be saved. Event: " + GetHL7EventString(hl7Event));
                    return;
                }
                PatientMovement patientMovement = vistaQuery.GetPatientMovement(hl7Event.PatientIEN, eventDateTime.DateTime, transactionType);
                if (patientMovement != null)
                {
                    using (var entityWriter = writerManager.MakeEntityWriter<PatientMovement>())
                    {
                        entityWriter.LogChecker = new DatabaseLogChecker();
                        entityWriter.Insert(patientMovement);
                    }

                }
            }
            finally
            {
                if (vistaSession != null)
                    vistaSession.Dispose();
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
            }
        }
    }
}
