﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using InterSystems.Data.CacheClient;
using BMS.VistaIntegration.Exceptions;
using BMS.VistaIntegration.Cache.Commands;

namespace BMS.VistaIntegration.Cache
{
    public class Utils
    {
        private const string SERVER = "Server";
        private const string PORT = "Port";
        private const string DATABASE = "Database";
        private const string LOG_FILE = "Log File";
        private const string SCHEMA = "Schema";

        public static CacheConnection ParseConnectionString(string connectionString, string user, string password, out string schema)
        {
            CacheConnection odbcConnection = null;
            schema = string.Empty;
            string server = string.Empty, port = string.Empty, logFile = string.Empty, database = string.Empty;
            string[] elements = connectionString.Split(';');
            if (elements == null || elements.Length == 0)
                throw new VistAConnectionException("CACHE ODBC connection string is invalid!");
            string[] parts = null;
            for (int i = 0; i < elements.Length; i++)
            {
                if (elements[i] != null && elements[i].Length > 0)
                {
                    parts = elements[i].Split('=');
                    if (parts != null && parts.Length == 2)
                    {
                        if (parts[0].Equals(SERVER, StringComparison.InvariantCultureIgnoreCase))
                            server = parts[1];
                        else if (parts[0].Equals(PORT, StringComparison.InvariantCultureIgnoreCase))
                            port = parts[1];
                        else if (parts[0].Equals(LOG_FILE, StringComparison.InvariantCultureIgnoreCase))
                            logFile = parts[1];
                        else if (parts[0].Equals(DATABASE, StringComparison.InvariantCultureIgnoreCase))
                            database = parts[1];
                        else if (parts[0].Equals(SCHEMA, StringComparison.InvariantCultureIgnoreCase))
                            schema = parts[1];
                    }
                    else
                        throw new VistAConnectionException("CACHE ODBC connection string is invalid!");
                    parts = null;
                }
            }
            if (string.IsNullOrEmpty(server))
                throw new VistAConnectionException("CACHE ODBC connection string: Server is missing!");
            if (string.IsNullOrEmpty(port))
                throw new VistAConnectionException("CACHE ODBC connection string: Port is missing!");
            if (string.IsNullOrEmpty(database))
                throw new VistAConnectionException("CACHE ODBC connection string: Database is missing!");
            if (string.IsNullOrEmpty(schema))
                throw new VistAConnectionException("CACHE ODBC connection string: Schema is missing!");

            if (string.IsNullOrEmpty(logFile))
                odbcConnection = new CacheConnection(server, port, database, user, password);
            else
                odbcConnection = new CacheConnection(server, port, database, user, password, logFile);
            return odbcConnection;
        }

        public static string ReplaceQuery(CacheCommandInfo commandInfo, string commandQuery)
        {
            string result = commandQuery;
            if (result.Contains(Constants.ROW_COUNT))
                result = result.Replace(Constants.ROW_COUNT, commandInfo.RowCount.ToString());
            if (result.Contains(Constants.FROM_IEN))
                result = result.Replace(Constants.FROM_IEN, commandInfo.FromIen);
            if (commandInfo.StartDate.HasValue && result.Contains(Constants.START_DATE))
                result = result.Replace(Constants.START_DATE, "'" + commandInfo.StartDate.Value.ToString() + "'");
            if (commandInfo.EndDate.HasValue && result.Contains(Constants.END_DATE))
                result = result.Replace(Constants.END_DATE, "'" + commandInfo.EndDate.Value.ToString() + "'");            
            if (commandInfo.StartDate.HasValue && result.Contains(Constants.PATIENT_MOVEMENT_DATE_TIME))
                result = result.Replace(Constants.PATIENT_MOVEMENT_DATE_TIME, "'" + commandInfo.StartDate.Value.ToString() + "'");            
            if (result.Contains(Constants.PATIENT_MOVEMENT_TRANSACTION))
                result = result.Replace(Constants.PATIENT_MOVEMENT_TRANSACTION, ((int)commandInfo.TransactionType).ToString());
            if (result.Contains(Constants.PATIENT_MOVEMENT_PATIENT_IEN))
                result = result.Replace(Constants.PATIENT_MOVEMENT_PATIENT_IEN, commandInfo.Ien);
            if (result.Contains(Constants.PATIENT_IEN))
                result = result.Replace(Constants.PATIENT_IEN, commandInfo.Ien);
            if (result.Contains(Constants.PATIENT_MOVEMENT_IENS))
                result = result.Replace(Constants.PATIENT_MOVEMENT_IENS, commandInfo.Ien);
            if (result.Contains(Constants.CANCELLED_ORDER_IENS))
                result = result.Replace(Constants.CANCELLED_ORDER_IENS, commandInfo.Ien);
            if (!commandInfo.IsBedHoldSupported && result.Contains("BED_HOLD"))
                result = result.Replace("RB.BED_HOLD, RB.BED_HOLD_DATE,", string.Empty);
            if (result.Contains(Constants.PATIENT_SSNS))
                result = result.Replace(Constants.PATIENT_SSNS, commandInfo.PatientSsns);

            result = ReplaceQueryIen(commandInfo, result);
            result = ReplaceQueryOrderableItemsIen(commandInfo, result);
            result = ReplaceQueryClinicsIen(commandInfo, result);
            return result;
        }

        private static string ReplaceQueryIen(CacheCommandInfo commandInfo, string commandQuery)
        {
            string result = commandQuery;
            if (!string.IsNullOrEmpty(commandInfo.Ien))
            {                
                if (result.Contains(Constants.PATIENT_MOVEMENT_ADMISSION_IEN))
                    result = result.Replace(Constants.PATIENT_MOVEMENT_ADMISSION_IEN, commandInfo.Ien);
                else if (result.Contains(Constants.CURRENT_ADMISSION_IEN))
                    result = result.Replace(Constants.CURRENT_ADMISSION_IEN, commandInfo.Ien);
                else if (result.Contains(Constants.PATIENT_MOVEMENT_IEN))
                    result = result.Replace(Constants.PATIENT_MOVEMENT_IEN, commandInfo.Ien);
                else
                {
                    StringBuilder sb = new StringBuilder();
                    sb.Append(" AND P.IEN = ");
                    sb.Append(commandInfo.Ien);
                    sb.Append(" ");
                    sb.Append(Constants.ORDER_BY);
                    result = result.Replace(Constants.ORDER_BY, sb.ToString());
                }
            }
            return result;
        }

        private static string ReplaceQueryOrderableItemsIen(CacheCommandInfo commandInfo, string commandQuery)
        {
            string result = commandQuery;
            if (result.Contains("ORDER_ACTIONS AS OA") && result.Contains("ORDER_TEXT AS OT"))
            {
                StringBuilder sb2 = new StringBuilder();
                if (commandInfo.OrderableItemsIen != null && commandInfo.OrderableItemsIen.Count() > 0)
                {
                    StringBuilder items = new StringBuilder();
                    foreach (string item in commandInfo.OrderableItemsIen)
                    {
                        items.Append(item);
                        items.Append(",");
                    }
                    items.Remove(items.Length - 1, 1);
                    sb2.Append(" AND (OI.ORDERABLE_ITEM IN (");
                    sb2.Append(items.ToString());
                    sb2.Append(") OR ((UPPER(ORDER_TEXT) LIKE '%ANTICIPATE%' AND UPPER(ORDER_TEXT) LIKE '%DISCHARGE%') OR (UPPER(ORDER_TEXT) LIKE '%PLANNED%' AND UPPER(ORDER_TEXT) LIKE '%DISCHARGE%'))) ");
                }
                else
                    sb2.Append(" AND ((UPPER(ORDER_TEXT) LIKE '%ANTICIPATE%' AND UPPER(ORDER_TEXT) LIKE '%DISCHARGE%') OR (UPPER(ORDER_TEXT) LIKE '%PLANNED%' AND UPPER(ORDER_TEXT) LIKE '%DISCHARGE%')) ");
                sb2.Append(Constants.ORDER_BY);
                result = result.Replace(Constants.ORDER_BY, sb2.ToString());
            }
            return result;
        }

        private static string ReplaceQueryClinicsIen(CacheCommandInfo commandInfo, string commandQuery)
        {
            string result = commandQuery;
            if (commandInfo.ClinicsIen != null && commandInfo.ClinicsIen.Count() > 0)
            {
                StringBuilder clinics = new StringBuilder();
                foreach (string item in commandInfo.ClinicsIen)
                {
                    clinics.Append(item);
                    clinics.Append(",");
                }
                clinics.Remove(clinics.Length - 1, 1);
                StringBuilder sb2 = new StringBuilder();
                sb2.Append(" AND AL.HOSPITAL_LOCATION IN (");
                sb2.Append(clinics.ToString());
                sb2.Append(") ");
                sb2.Append(Constants.ORDER_BY);
                result = result.Replace(Constants.ORDER_BY, sb2.ToString());
            }
            return result;
        }
    }
}
