﻿using System;
using System.Collections.Generic;
using System.Linq;
using InfoWorld.HL7.ITS;
using BMS.Facade.Service;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.ServicesWrapper.BMService;
using DC = BMS.DataContracts;
using BMS.Facade.Translators;
using BMS.Facade.Fault;
using BMS.Utils.Properties;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.Security;
using System.Threading;

namespace BMS.Facade.Implementation
{
    public class IconOperationsImplementation : IIconOperations
    {
        /// <summary>
        /// Creates the icon detail.
        /// </summary>
        /// <param name="iconDetail">The icon detail.</param>
        /// <returns></returns>
        public int CreateIconDetail(IconDetail iconDetail)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                int id = BMSFactory.BedManagerOperationsClient.CreateIconDetail(iconDetail.ToDataContract());
                if (iconDetail.Facility != null && iconDetail.Facility.Id != null)
                    _cache.ClearFacilityIcons(iconDetail.Facility.Id.extension);
                else
                {
                    if (iconDetail.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase))
                    {
                        _cache.ClearGlobalIcons(iconDetail.Subgroup);
                        _cache.ClearFacilityIcons();
                    }
                    else
                        _cache.ClearGlobalIcons(iconDetail.Group);
                }
                return id;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the icon detail.
        /// </summary>
        /// <param name="iconDetailId">The icon detail id.</param>
        /// <param name="facility">The facility.</param>
        /// <returns></returns>
        public IconDetail GetIconDetail(int iconDetailId, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BMSFactory.BedManagerOperationsClient.GetIconDetail(iconDetailId, facility == null ? null : facility.Id).ToFacadeContract(); 
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the icon detail.
        /// </summary>
        /// <param name="iconDetail">The icon detail.</param>
        public void UpdateIconDetail(IconDetail iconDetail)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.UpdateIconDetail(iconDetail.ToDataContract());
                if (iconDetail.Facility != null && iconDetail.Facility.Id != null)
                    _cache.ClearFacilityIcons(iconDetail.Facility.Id.extension);
                else
                {
                    if (iconDetail.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase))
                    {
                        _cache.ClearGlobalIcons(iconDetail.Subgroup);
                        _cache.ClearFacilityIcons();
                    }
                    else
                        _cache.ClearGlobalIcons(iconDetail.Group);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the icon detail.
        /// </summary>
        /// <param name="iconDetailId">The icon detail id.</param>
        /// <param name="facility">The facility.</param>
        public void DeleteIconDetail(int iconDetailId, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.DeleteIconDetail(iconDetailId, facility == null ? null : facility.Id);
                _cache.FlushCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Switches the icons.
        /// </summary>
        /// <param name="iconId1">The icon id1.</param>
        /// <param name="iconId2">The icon id2.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="group">The group.</param>
        public void SwitchIcons(int iconId1, int iconId2, Guid? facilityUid, string group)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> icons = new List<IconInfo>();
                if (facilityUid == null)
                    icons = _cache.GetGroupIcons(group);
                else
                    icons = _cache.GetFacilityIcons(facilityUid.ToString());
                IconInfo icon1 = icons.Where(a => a.IconId == iconId1).FirstOrDefault();
                IconInfo icon2 = icons.Where(a => a.IconId == iconId2).FirstOrDefault();
                BMSFactory.BedManagerOperationsClient.SwitchIconOrder(icon1.ToDataContract(), icon2.ToDataContract());
                int order = icon1.OrderIndex;
                icon1.OrderIndex = icon2.OrderIndex;
                icon2.OrderIndex = order;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the icon order bulk.
        /// </summary>
        /// <param name="icons">The icons.</param>
        /// <param name="facility">The facility.</param>
        public List<IconInfo> UpdateIconOrderBulk(List<IconInfo> icons, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (icons != null && icons.Count > 0)
                {
                    List<DC.IconOrder> iconOrderList = new List<DC.IconOrder>();                    
                    foreach (IconInfo info in icons)
                    {
                        if (info.Facility == null)
                        {
                            info.Facility = facility;
                            BMSFactory.BedManagerOperationsClient.CreateIconDetail(info.ToIconDetailDataContract());
                        }
                        else
                        {
                            iconOrderList.Add(new DC.IconOrder() { FacilityId = facility.Id, IconId = info.IconId, OrderIndex = info.OrderIndex });
                        }
                    }
                    if (iconOrderList.Count > 0)
                        BMSFactory.BedManagerOperationsClient.UpdateIconOrderBulk(iconOrderList);
                    _cache.CacheFacilityIcons(facility.Id.extension, icons);
                }
                return icons;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets all global icons(all subgroups, including SiteConfigurable defined in Admin section).
        /// </summary>
        /// <returns></returns>
        public List<IconInfo> GetAllGlobalIcons()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> result = new List<IconInfo>();
                if (_cache.GetGroupIcons(Constants.APPLICATION_ICON) != null && _cache.GetGroupIcons(Constants.WHITEBOARD_ICON) != null && _cache.GetGroupIcons(Constants.SITE_CONFIGURABLE_ICON) != null)
                {
                    result.AddRange(_cache.GetGroupIcons(Constants.APPLICATION_ICON));
                    result.AddRange(_cache.GetGroupIcons(Constants.WHITEBOARD_ICON));
                    result.AddRange(_cache.GetGroupIcons(Constants.SITE_CONFIGURABLE_ICON));
                }
                else
                {
                    result = FilterIconInfo(Constants.ALL, null, null);
                    _cache.FillCache(Constants.ALL, null, result);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets all icons by facility(all subgroups, including SiteConfigurable defined for the specified facility).
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        public List<IconInfo> GetAllIconsByFacility(Guid facilityUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> result = new List<IconInfo>();
                if (_cache.GetGroupIcons(Constants.APPLICATION_ICON) != null && _cache.GetGroupIcons(Constants.WHITEBOARD_ICON) != null && _cache.GetFacilityIcons(facilityUid.ToString()) != null)
                {
                    result.AddRange(_cache.GetGroupIcons(Constants.APPLICATION_ICON));
                    result.AddRange(_cache.GetGroupIcons(Constants.WHITEBOARD_ICON));
                    result.AddRange(_cache.GetFacilityIcons(facilityUid.ToString()));
                }
                else
                {
                    result = FilterIconInfo(Constants.ALL, null, facilityUid);
                    _cache.FillCache(Constants.ALL, facilityUid, result);
                }
                return result.Where(a => a.Published.HasValue && a.Published.Value).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the whiteboard icons(all whiteboard subgroups, including SiteConfigurable defined for the specified facility).
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        public List<IconInfo> GetWhiteboardIcons(Guid facilityUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> result = new List<IconInfo>();
                if (_cache.GetGroupIcons(Constants.WHITEBOARD_ICON) != null && _cache.GetFacilityIcons(facilityUid.ToString()) != null)
                {
                    result.AddRange(_cache.GetGroupIcons(Constants.WHITEBOARD_ICON));
                    result.AddRange(_cache.GetFacilityIcons(facilityUid.ToString()));
                }
                else
                {
                    result = FilterIconInfo(Constants.WHITEBOARD, null, facilityUid);
                    _cache.FillCache(Constants.WHITEBOARD, facilityUid, result);
                }
                return result.Where(a => a.Published.HasValue && a.Published.Value).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the site configurable icons.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        public List<IconInfo> GetSiteConfigurableIcons(Guid facilityUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> result = new List<IconInfo>();
                if (_cache.GetFacilityIcons(facilityUid.ToString()) != null)
                    result.AddRange(_cache.GetFacilityIcons(facilityUid.ToString()));
                else
                {
                    result = FilterIconInfo(string.Empty, null, facilityUid);
                    _cache.CacheFacilityIcons(facilityUid.ToString(), result);
                }
                return result.Where(a => a.Published.HasValue && a.Published.Value).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Searches the icons.
        /// </summary>
        /// <param name="iconFlagIds">The icon flag ids.</param>
        /// <param name="sortBy">The sort by.</param>
        /// <returns></returns>
        public List<IconInfo> SearchIcons(string iconFlagIds)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> result = FilterIconInfo(Constants.SEARCH, iconFlagIds, null);
                List<Facility> facilities = FacadeManager.EntityInterface.GetFacilities().ToList();
                Facility facility = null;
                foreach (IconInfo icon in result)
                {
                    facility = facilities.Where(a => a.Id.extension.Equals(icon.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (facility != null)
                        icon.Facility = new Place() { Id = facility.Id, Name = facility.Code };
                }
                result = (from p in result
                            orderby p.Facility.Name, p.IconName
                            select p).ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the icon by code.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        public IconInfo GetIconByCode(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IconInfo result = null;

                if (_cache.GetGroupIcons(Constants.APPLICATION_ICON) != null)
                    result = _cache.GetGroupIcons(Constants.APPLICATION_ICON).Where(a => a.IconType.code.Equals(code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();

                if (result == null)
                {
                    List<IconInfo> list = FilterIconInfo(Constants.ALL, null, null);
                    _cache.FillCache(Constants.ALL, null, list);
                    result = list.Where(a => a.IconType.code.Equals(code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the icons by codes.
        /// </summary>
        /// <param name="codes">The codes.</param>
        /// <returns></returns>
        public Dictionary<string, IconInfo> GetIconsByCodes(List<string> codes)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Dictionary<string, IconInfo> result = new Dictionary<string, IconInfo>();
                List<IconInfo> icons = _cache.GetGroupIcons(Constants.APPLICATION_ICON);
                if (icons == null)
                {
                    icons = FilterIconInfo(Constants.ALL, null, null);
                    _cache.FillCache(Constants.ALL, null, icons);
                }
                foreach (string code in codes)
                    result.Add(code, icons.Where(a => a.Group.Equals(Constants.APPLICATION_ICON, StringComparison.InvariantCultureIgnoreCase) && a.IconType.code.Equals(code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault());                    
                
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<IconInfo> FilterIconInfo(string listType, string iconFlagIds, Guid? facilityUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domain = null;
                if (facilityUid != null)                
                    domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
                    
                return BMSFactory.BedManagerQueryClientFromWCF.FilterIconInfo(listType, iconFlagIds, facilityUid == null ? null : new II(domain, facilityUid.ToString())).Select<DC.IconInfo, IconInfo>(a => a.ToFacadeContract()).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Icon Associations

        /// <summary>
        /// Filters the bed patient icon association.
        /// </summary>
        /// <param name="bedUid">The bed uid.</param>
        /// <param name="patientUid">The patient uid.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        public Dictionary<string, List<IconAssociationInfo>> FilterBedPatientIconAssociation(Guid bedUid, Guid? patientUid, Guid facilityUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Dictionary<string, List<IconAssociationInfo>> result = new Dictionary<string, List<IconAssociationInfo>>();
                List<DC.IconAssociationInfo> iconAssociatedList = BMSFactory.BedManagerQueryClient.FilterBedPatientIconAssociation(bedUid, patientUid, facilityUid).ToList();
                IEnumerable<IconInfo> iconInfoList;
                if (patientUid != null)
                    iconInfoList = GetWhiteboardIcons(facilityUid).Where(a => a.Active == true);
                else
                    iconInfoList = GetWhiteboardIcons(facilityUid).Where(a => a.PatientOrRoomBed.Value == 'R' && a.Active == true).ToList();
                
                List<string> iconCodes = new List<string>();
                iconCodes.Add(Constants.PATIENT_ICON_CODE);
                iconCodes.Add(Constants.ROOM_BED_ICON_CODE);
                Dictionary<string, IconInfo> icons = GetIconsByCodes(iconCodes);
                IconInfo iconInfoPatient = icons[Constants.PATIENT_ICON_CODE];
                IconInfo iconInfoRoomOrBed = icons[Constants.ROOM_BED_ICON_CODE]; 

                //add STANDARD_ICON    
                result.Add(Constants.STANDARD_ICON, GetIconsAssociationBySubgroup(iconAssociatedList.Where(a => a.Subgroup == Constants.STANDARD_ICON), iconInfoList.Where(a => a.Subgroup == Constants.STANDARD_ICON), iconInfoPatient, iconInfoRoomOrBed));
                //add EMERGENCY_ICON
                result.Add(Constants.EMERGENCY_ICON, GetIconsAssociationBySubgroup(iconAssociatedList.Where(a => a.Subgroup == Constants.EMERGENCY_ICON), iconInfoList.Where(a => a.Subgroup == Constants.EMERGENCY_ICON), iconInfoPatient, iconInfoRoomOrBed));
                //add SITE_CONFIGURABLE_ICON
                result.Add(Constants.SITE_CONFIGURABLE_ICON, GetIconsAssociationBySubgroup(iconAssociatedList.Where(a => a.Subgroup == Constants.SITE_CONFIGURABLE_ICON), iconInfoList.Where(a => a.Subgroup == Constants.SITE_CONFIGURABLE_ICON), iconInfoPatient, iconInfoRoomOrBed));

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }

        }

        /// <summary>
        /// Inserts the icon association.
        /// </summary>
        /// <param name="iconAssociation">The icon association.</param>
        public void InsertIconAssociation(IconAssociation iconAssociation)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                BMSFactory.BedManagerOperationsClient.InsertIconAssociation(iconAssociation.ToDataContract());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<IconAssociation> FilterBedIconAssociation(string bedUids, Guid facilityUID)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BMSFactory.BedManagerQueryClient.FilterBedIconAssociation(bedUids, facilityUID).Select<DC.IconAssociation, IconAssociation>(dcIA => dcIA.ToFacadeContract()).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<IconAssociationInfo> GetIconsAssociationBySubgroup(IEnumerable<DC.IconAssociationInfo> iconAssociatedList, IEnumerable<IconInfo> iconInfoList, IconInfo iconInfoPatient, IconInfo iconInfoRoomOrBed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconAssociationInfo> iconAssociationResultByGroup = new List<IconAssociationInfo>();
                IEnumerable<IconAssociationInfo> iconAssociatedWithABedOrAPatientList = (from icon in iconInfoList
                                                                                         where (from id in iconAssociatedList
                                                                                                select id.IconId).Contains(icon.IconId)
                                                                                         select new IconAssociationInfo
                                                                                         {
                                                                                             Icon = icon,
                                                                                             Code = Constants.Yes,
                                                                                             IconPatientOrRoomBedDescription = icon.PatientOrRoomBed.ToString() == "P" ? iconInfoPatient == null ? string.Empty : iconInfoPatient.IconDescription : iconInfoRoomOrBed == null ? string.Empty : iconInfoRoomOrBed.IconDescription,
                                                                                             IconPatientOrRoomBedImageBytes = icon.PatientOrRoomBed.ToString() == "P" ? iconInfoPatient == null ? string.Empty : iconInfoPatient.ImageBytes : iconInfoRoomOrBed == null ? string.Empty : iconInfoRoomOrBed.ImageBytes,
                                                                                             IconPatientOrRoomBedMouseOverText = icon.PatientOrRoomBed.ToString() == "P" ? iconInfoPatient == null ? string.Empty : iconInfoPatient.MouseOverText : iconInfoRoomOrBed == null ? string.Empty : iconInfoRoomOrBed.MouseOverText
                                                                                         });

                IEnumerable<IconAssociationInfo> iconNotAssociatedWithABedOrAPatientList = (from icon in iconInfoList
                                                                                            where !(from id in iconAssociatedList
                                                                                                    select id.IconId).Contains(icon.IconId)
                                                                                            select new IconAssociationInfo
                                                                                            {
                                                                                                Icon = icon,
                                                                                                Code = Constants.No,
                                                                                                IconPatientOrRoomBedDescription = icon.PatientOrRoomBed.ToString() == "P" ? iconInfoPatient == null ? string.Empty : iconInfoPatient.IconDescription : iconInfoRoomOrBed == null ? string.Empty : iconInfoRoomOrBed.IconDescription,
                                                                                                IconPatientOrRoomBedImageBytes = icon.PatientOrRoomBed.ToString() == "P" ? iconInfoPatient == null ? string.Empty : iconInfoPatient.ImageBytes : iconInfoRoomOrBed == null ? string.Empty : iconInfoRoomOrBed.ImageBytes,
                                                                                                IconPatientOrRoomBedMouseOverText = icon.PatientOrRoomBed.ToString() == "P" ? iconInfoPatient == null ? string.Empty : iconInfoPatient.MouseOverText : iconInfoRoomOrBed == null ? string.Empty : iconInfoRoomOrBed.MouseOverText
                                                                                            });

                iconAssociationResultByGroup.AddRange(iconAssociatedWithABedOrAPatientList);
                iconAssociationResultByGroup.AddRange(iconNotAssociatedWithABedOrAPatientList);
                return iconAssociationResultByGroup.OrderBy(a => a.Icon.OrderIndex).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion        

        #region Caching

        static IconCache _cache = new IconCache();
        
        class IconCache
        {

            /// <summary>
            /// dictionary used to store the global icons(not associated with a facility) per group.
            /// </summary>
            Dictionary<string, List<IconInfo>> _cacheGlobalIcons = new Dictionary<string, List<IconInfo>>(StringComparer.InvariantCultureIgnoreCase);
            /// <summary>
            /// dictionary used to store the icons per facility.
            /// </summary>
            Dictionary<string, List<IconInfo>> _cacheFacilityIcons = new Dictionary<string, List<IconInfo>>(StringComparer.InvariantCultureIgnoreCase);
            ReaderWriterLockSlim _cacheIconsLock = new ReaderWriterLockSlim();

            public List<IconInfo> GetGroupIcons(string group)
            {
                if (string.IsNullOrEmpty(group)) return null;

                if (_cacheGlobalIcons.ContainsKey(group))
                    return _cacheGlobalIcons[group];
                return null;
            }

            public List<IconInfo> GetFacilityIcons(string facilityUid)
            {
                if (string.IsNullOrEmpty(facilityUid) || facilityUid.Equals(Guid.Empty.ToString())) return null;

                if (_cacheFacilityIcons.ContainsKey(facilityUid))
                    return _cacheFacilityIcons[facilityUid];
                return null;
            }

            public void CacheGroupIcons(string group, List<IconInfo> list)
            {
                if (string.IsNullOrEmpty(group)) return;

                _cacheIconsLock.EnterWriteLock();
                try
                {
                    _cacheGlobalIcons[group] = list;
                }
                finally { _cacheIconsLock.ExitWriteLock(); }
            }

            public void CacheFacilityIcons(string facilityUid, List<IconInfo> list)
            {
                if (string.IsNullOrEmpty(facilityUid) || facilityUid.Equals(Guid.Empty.ToString())) return;

                _cacheIconsLock.EnterWriteLock();
                try
                {
                    _cacheFacilityIcons[facilityUid] = list;                    
                }
                finally { _cacheIconsLock.ExitWriteLock(); }
            }

            public void FillCache(string listType, Guid? facilityUid, List<IconInfo> list)
            {
                if (listType.Equals(Constants.ALL))
                {
                    _cache.CacheGroupIcons(Constants.APPLICATION_ICON, list.Where(a => a.Group.Equals(Constants.APPLICATION_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                    _cache.CacheGroupIcons(Constants.WHITEBOARD_ICON, list.Where(a => a.Group.Equals(Constants.WHITEBOARD_ICON, StringComparison.InvariantCultureIgnoreCase) && !a.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                    if (facilityUid == null || facilityUid == Guid.Empty)
                        _cache.CacheGroupIcons(Constants.SITE_CONFIGURABLE_ICON, list.Where(a => a.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                    else
                        _cache.CacheFacilityIcons(facilityUid.ToString(), list.Where(a => a.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                }
                else if (listType.Equals(Constants.WHITEBOARD))
                {
                    _cache.CacheGroupIcons(Constants.WHITEBOARD_ICON, list.Where(a => a.Group.Equals(Constants.WHITEBOARD_ICON, StringComparison.InvariantCultureIgnoreCase) && !a.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                    if (facilityUid == null || facilityUid == Guid.Empty)
                        _cache.CacheGroupIcons(Constants.SITE_CONFIGURABLE_ICON, list.Where(a => a.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                    else
                        _cache.CacheFacilityIcons(facilityUid.ToString(), list.Where(a => a.Subgroup.Equals(Constants.SITE_CONFIGURABLE_ICON, StringComparison.InvariantCultureIgnoreCase)).ToList());
                }
            }

            public void ClearGlobalIcons(string group)
            {
                if (string.IsNullOrEmpty(group)) return;

                _cacheIconsLock.EnterWriteLock();
                try
                {
                    _cacheGlobalIcons.Remove(group);
                }
                finally { _cacheIconsLock.ExitWriteLock(); }
            }

            public void ClearFacilityIcons(string facilityUid)
            {
                if (string.IsNullOrEmpty(facilityUid) || facilityUid.Equals(Guid.Empty.ToString())) return;

                _cacheIconsLock.EnterWriteLock();
                try
                {
                    _cacheFacilityIcons.Remove(facilityUid);
                }
                finally { _cacheIconsLock.ExitWriteLock(); }
            }

            public void ClearGlobalIcons()
            {
                _cacheIconsLock.EnterWriteLock();
                try
                {
                    _cacheGlobalIcons.Clear();
                }
                finally { _cacheIconsLock.ExitWriteLock(); }
            }

            public void ClearFacilityIcons()
            {
                _cacheIconsLock.EnterWriteLock();
                try
                {
                    _cacheFacilityIcons.Clear();
                }
                finally { _cacheIconsLock.ExitWriteLock(); }
            }

            internal void FlushCache()
            {
                ClearGlobalIcons();
                ClearFacilityIcons();
            }
        }

        #endregion
    }
}
