﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;

namespace BMS.Facade.Service
{
    /// <summary>
    /// Interface used to manipulate entities operations.
    /// </summary>
    public interface IEntityOperations
    {
        #region Bed

        /// <summary>
        /// Gets the beds in the specified division.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <returns></returns>
        IList<Bed> GetBedsInDivision(II divisionId, II vistaSiteId);

        #endregion

        #region Ward

        /// <summary>
        /// Gets the wards.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <returns></returns>
        IList<Ward> GetWardsByDivisionId(II divisionId);

        /// <summary>
        /// Gets the ward by id.
        /// </summary>
        /// <param name="divisionId">The ward id.</param>
        /// <returns>Ward</returns>
        Ward GetWardById(II wardId);

        /// <summary>
        /// Gets the vista wards.
        /// </summary>
        /// <param name="vistaSiteID">The vista site Id.</param>
        /// <returns></returns>
        IList<Ward> GetWardsByVista(II vistaSiteId);

        /// <summary>
        /// Updates the ward.
        /// </summary>
        /// <param name="ward">The ward.</param>
        /// <returns></returns>
        IList<ValidationAlert> UpdateWard(Ward ward);

        /// <summary>
        /// Deletes the ward.
        /// </summary>
        /// <param name="id">The ward identifier.</param>
        /// <returns></returns>
        IList<ValidationAlert> DeleteWard(II id);

        /// <summary>
        /// Gets the wards in a facility.
        /// </summary>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        IList<Ward> GetWardsByFacility(II facilityId);

        #endregion

        #region Division

        /// <summary>
        /// Gets the divisions of the selected Facility
        /// </summary>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        IList<Division> GetDivisions(II facilityId);

        /// <summary>
        /// Gets the division.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        Division GetDivisionByName(string name);

        /// <summary>
        /// Gets the division by id.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <returns>Division</returns>
        Division GetDivisionById(II divisionId);

        /// <summary>
        /// Gets the divisions by bed and facility.
        /// </summary>
        /// <param name="bedId">The bed id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        IList<Division> GetDivisionsByBedAndFacility(II bedId, II facilityId);

        /// <summary>
        /// Updates the division.
        /// </summary>
        /// <param name="division">The division.</param>
        void UpdateDivision(Division division);

        #endregion

        #region Facility

        /// <summary>
        /// Gets the facilities.
        /// </summary>
        /// <param name="visnId">The visn id.</param>
        /// <returns></returns>
        IList<Facility> GetFacilitiesByVisn(II visnId);

        /// <summary>
        /// Gets the facilities.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        IList<Facility> GetFacilities(II vistaSiteId);

        /// <summary>
        /// Retrieves the facility.
        /// </summary>
        /// <param name="id">The internal identifier of the facility.</param>
        /// <returns>the facility or null</returns>
        Facility GetFacility(II id);

        /// <summary>
        /// Gets the facility by code.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        Facility GetFacilityByCode(string code);

        /// <summary>
        /// Gets the facilities.
        /// </summary>
        /// <returns></returns>
        IList<Facility> GetFacilities();

        /// <summary>
        /// Returns the list of facilities. It uses the IIS credential to authenticate in STS/EIS.
        /// </summary>
        /// <returns>All facilities.</returns>
        IList<Facility> GetFacilitiesUsingWindowsAuthentication();

        /// <summary>
        /// Saves the facility.
        /// </summary>
        /// <param name="visnName">Name of the visn.</param>
        /// <param name="visnIen">The visn ien.</param>
        /// <param name="facilityName">Name of the facility.</param>
        /// <param name="facilityIen">The facility ien.</param>
        /// <returns></returns>
        IList<ValidationAlert> SaveFacility(Facility facility);

        /// <summary>
        /// Gets the facility by division.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <returns></returns>
        Facility GetFacilityByDivisionId(II divisionId);

        #endregion

        #region Hospital Location

        /// <summary>
        /// Gets the hospital locations.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        IList<HospitalLocation> GetHospitalLocations(II vistaSiteId);

        /// <summary>
        /// Gets the hospital location.
        /// </summary>
        /// <param name="hospitalLocationId">The hospital location id.</param>
        /// <returns></returns>
        HospitalLocation GetHospitalLocation(II id, II vistaSiteId);

        #endregion

        #region VistaSite

        /// <summary>
        /// Gets the vista sites.
        /// </summary>
        /// <returns></returns>
        IList<VistaSite> GetVistaSites();

        /// <summary>
        /// Gets the vista sites.
        /// </summary>
        /// <param name="visnId">The visn id.</param>
        /// <returns></returns>
        IList<VistaSite> GetVistaSites(II visnId);

        /// <summary>
        /// Gets the vista site.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        VistaSite GetVistaSite(II vistaSiteId);

        /// <summary>
        /// Gets the vista sites using windows authentication.
        /// </summary>
        /// <returns></returns>
        IList<VistaSite> GetVistaSitesUsingWindowsAuthentication();

        #endregion

        #region Visn

        /// <summary>
        /// Gets the visns by region.
        /// </summary>
        /// <param name="regionId">The region id.</param>
        /// <returns></returns>
        IList<Visn> GetVisns(II regionId);

        /// <summary>
        /// Gets the visns.
        /// </summary>
        /// <returns></returns>
        IList<Visn> GetVisns();

        /// <summary>
        /// Returns the list of visns. It uses the IIS credential to authenticate in STS/EIS.
        /// </summary>
        /// <returns>All visns.</returns>
        IList<Visn> GetVisnsUsingWindowsAuthentication();

        /// <summary>
        /// Gets the visn.
        /// </summary>
        /// <param name="visnId">The visn id.</param>
        /// <returns></returns>
        Visn GetVisn(II visnId);

        #endregion

        #region Region

        /// <summary>
        /// Gets the regions.
        /// </summary>
        /// <returns>A list of regions.</returns>
        IList<Region> GetRegions();

        /// <summary>
        /// Gets the region.
        /// </summary>
        /// <param name="regionId">The region id.</param>
        /// <returns></returns>
        Region GetRegion(II regionId);

        #endregion

        #region Patient

        /// <summary>
        /// Gets the patient by SSN.
        /// </summary>
        /// <param name="ssn">The SSN.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        Patient GetPatientBySsn(II ssn, string vistaCode);

        /// <summary>
        /// Gets the patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        Patient GetPatientById(II id, string vistaCode);

        /// <summary>
        /// Gets the last name of the patients by.
        /// </summary>
        /// <param name="lastName">The last name.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        IList<Patient> GetPatientsByLastName(string lastName, string vistaCode);

        /// <summary>
        /// Gets the patients by last name initial and last four SSN.
        /// </summary>
        /// <param name="filter">The filter.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        IList<Patient> GetPatientsByLastNameInitialAndLastFourSSN(string filter, string vistaCode);

        /// <summary>
        /// Creates the specified patient.
        /// </summary>
        /// <param name="patient">the patient to save</param> 
        void CreatePatient(Patient patient);

        /// <summary>
        /// Updates the specified patient.
        /// </summary>
        /// <param name="patient">the patient to save, identified by internal id or ien</param> 
        void UpdatePatient(Patient patient);

        /// <summary>
        /// Deletes the patient with the specified id.
        /// </summary>
        /// <param name="id">the id of the patient to delete</param> 
        void DeletePatient(II id);

        #endregion                

        #region DischargeClinics

        /// <summary>
        /// Saves the discharge clinic.
        /// </summary>
        /// <param name="dischargeClinic">The discharge clinic.</param>
        void SaveDischargeClinic(DischargeClinic dischargeClinic);

        /// <summary>
        /// Gets the discharge clinics.
        /// </summary>
        /// <param name="facility">The facility.</param>
        /// <param name="vistaSite">The vista site.</param>
        /// <returns></returns>
        List<DischargeClinic> GetDischargeClinics(Facility facility, VistaSite vistaSite);

        /// <summary>
        /// Deletes the discharge clinic.
        /// </summary>
        /// <param name="dischargeClinicId">The discharge clinic id.</param>
        void DeleteDischargeClinic(II dischargeClinicId);        

        #endregion

        /// <summary>
        /// Gets all domains.
        /// </summary>
        /// <returns></returns>
        IList<string> GetDomains();
    }
}
