﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Security.Principal;
using InfoWorld.WMI_UserGroup.DTO;
using System.Threading;
using System.DirectoryServices;
using System.Collections;
using System.Linq.Expressions;
using System.Linq;
using System.DirectoryServices.ActiveDirectory;
using System.Net;
using System.DirectoryServices.AccountManagement;

namespace InfoWorld.WMI_UserGroup
{
    /// <summary>
    /// This class is used for insert all the necessary information from AD to database.
    /// </summary>
    public class AdministrativeFunctions
    {
        static List<IAccount> domainUserGroupMappings = new List<IAccount>();
        static List<IAccount> domainUsers = new List<IAccount>();
        static List<IAccount> domainGroups = new List<IAccount>();

        #region Public Methods

        /// <summary>
        /// Gets all groups and users from Active Directory.
        /// </summary>
        /// <returns></returns>
        public Dictionary<string, List<IAccount>> GetAllGroupUsersAndDomain(List<string> sids)
        {
            Dictionary<string, List<IAccount>> diAccounts = new Dictionary<string, List<IAccount>>();

            diAccounts.Add("local_users", new List<IAccount>());
            diAccounts.Add("local_groups", new List<IAccount>());
            diAccounts.Add("domain_users", new List<IAccount>());
            diAccounts.Add("domain_groups", new List<IAccount>());
            diAccounts.Add("user_group_local", new List<IAccount>());
            diAccounts.Add("user_group_domain", new List<IAccount>());

            bool isRunningOnLocal = false;
            string localMachine = Environment.MachineName;
            try
            {
                isRunningOnLocal = bool.Parse(System.Configuration.ConfigurationManager.AppSettings["IsRunningOnLocalMachine"]);
                if (isRunningOnLocal)
                    localMachine = System.Configuration.ConfigurationManager.AppSettings["LocalMachineName"];
            }
            catch { }
            
            Forest currentForest = Forest.GetCurrentForest();
            DomainCollection domains = currentForest.Domains;  

            diAccounts["domain_groups"] = domainGroups;
            diAccounts["domain_users"] = domainUsers;
            diAccounts["user_group_domain"] = domainUserGroupMappings;

            string machine = Environment.MachineName;

            List<string> localSids = FillGroupsAndMappings(sids);

            GetLocalAccounts(diAccounts["local_users"], diAccounts["local_groups"], diAccounts["user_group_local"], localMachine, localSids);

            return diAccounts;
        }

        /// <summary>
        /// Create a  new thread to insert the information into database.
        /// </summary>
        public void InsertDBAllGroupUsersAndDomainOnThread()
        {
            AutoResetEvent autoResetEvent = new AutoResetEvent(false);
            StringBuilder stringBuilder = new StringBuilder();
            ThreadPool.QueueUserWorkItem(new WaitCallback(this.PoolWorker), new object[] { autoResetEvent, stringBuilder });
            WaitHandle.WaitAll(new WaitHandle[] { autoResetEvent });

            if (stringBuilder.Length > 0)
            {
                throw new Exception(stringBuilder.ToString());
            }
        }

        /// <summary>
        /// Insert all the the groups, users and association between a user group into database.
        /// </summary>
        public void InsertDBAllGroupUsersAndDomain()
        {
            try
            {

                DateTime ds = DateTime.Now;
                DTOUserGroupDomain dataSet = new DTOUserGroupDomain();
                DTOPermissions usersWithPermissions = new DTOPermissions();
                List<string> sids = new List<string>();

                dataSet.DELETE_ALL_USERS_FROM_ACTIVE_DIRECTORY.AddDELETE_ALL_USERS_FROM_ACTIVE_DIRECTORYRow("");

                SqlUtil.FillData(null, usersWithPermissions);

                foreach (var row in usersWithPermissions.DIM_PERMISSION) 
                {
                    string sid = row.SID;
                    sids.Add(sid);
                }

                Dictionary<string, List<IAccount>> accounts = GetAllGroupUsersAndDomain(sids);

                foreach (User u in accounts["domain_users"])
                    dataSet.DIM_DOMAIN_USER.AddDIM_DOMAIN_USERRow(u.Name, u.Sid, u.Domain);

                foreach (User u in accounts["local_users"])
                    dataSet.DIM_DOMAIN_USER.AddDIM_DOMAIN_USERRow(u.Name, u.Sid, u.Domain);

                foreach (Group u in accounts["domain_groups"])
                    dataSet.DIM_DOMAIN_GROUP.AddDIM_DOMAIN_GROUPRow(u.Name, u.Sid, u.Domain);

                foreach (Group u in accounts["local_groups"])
                    dataSet.DIM_DOMAIN_GROUP.AddDIM_DOMAIN_GROUPRow(u.Name, u.Sid, u.Domain);

                foreach (UserGroup u in accounts["user_group_domain"])
                    dataSet.DOMAIN_USER_GROUP.AddDOMAIN_USER_GROUPRow(u.Group.Name, u.Group.Domain, u.User.Name, u.User.Domain);

                foreach (UserGroup u in accounts["user_group_local"])
                    dataSet.DOMAIN_USER_GROUP.AddDOMAIN_USER_GROUPRow(u.Group.Name, u.Group.Domain, u.User.Name, u.User.Domain);


                SqlUtil.ExecuteUpdate(dataSet);

                string message = "Insert Group-Users (" + ds.ToString() + " - " + DateTime.Now.ToString() + ")\n";

                int domain_users = accounts["domain_users"].Count, local_users = accounts["local_users"].Count,
                    domain_groups = accounts["domain_groups"].Count, local_groups = accounts["local_groups"].Count,
                    ug_assoc = accounts["user_group_domain"].Count + accounts["user_group_local"].Count;

                message += "Users: " + (domain_users + local_users) +
                      " (domain = " + domain_users + ", local = " + local_users + ");\n";
                message += "Groups: " + (domain_groups + local_groups) +
                      " (domain = " + domain_groups + ", local = " + local_groups + ");\n";
                message += "Users-Groups: " + ug_assoc + ".";

                Log.LogMessage(message);
            }
            catch (Exception ex)
            {
                Log.LogException(ex);
                throw ex;
            }

        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Fills the groups and mappings with domain data.
        /// </summary>
        /// <param name="sids">The sids.</param>
        /// <returns></returns>
        static private List<string> FillGroupsAndMappings(List<string> sids)
        {
            List<string> sidsNotFound = new List<string>();
            Forest currentForest = Forest.GetCurrentForest();
            GlobalCatalog globalCatalog = currentForest.FindGlobalCatalog();

            foreach (string sid in sids)
            {
                using (var searcher = globalCatalog.GetDirectorySearcher())
                {
                    searcher.PageSize = 500;
                    searcher.Filter = string.Format("(&(|(objectClass=user)(objectClass=group))(|(objectCategory=group)(objectCategory=user)) (objectSid={0}))", sid);
                    SearchResult result = searcher.FindOne();
                    if (result != null)
                    {
                        DirectoryEntry entry = new DirectoryEntry(result.Path);
                        string entryDomain = GetDomainName(entry);

                        SecurityIdentifier entrySid = new SecurityIdentifier((byte[])result.Properties["objectSid"][0], 0);
                        string entryAccountName = (string)result.Properties["samAccountName"][0];

                        //it is a user SID
                        if (result.Properties["objectCategory"][0].ToString().IndexOf("CN=Group", StringComparison.InvariantCultureIgnoreCase) < 0)
                        {
                            User u = new User(entrySid.Value, entryAccountName, entryDomain);
                            if (!domainUsers.Contains(u))
                                domainUsers.Add(u);
                        }
                        //it is a group SID
                        else
                        {
                            Group g = new Group(entrySid.Value, entryAccountName, entryDomain);
                            PrincipalContext ctx = new PrincipalContext(ContextType.Domain, entryDomain);
                            GroupPrincipal grp = GroupPrincipal.FindByIdentity(ctx, IdentityType.SamAccountName, entryAccountName);

                            //get all group members
                            foreach (Principal p in grp.GetMembers(true))
                                //if the member is a user, process it
                                if (!p.StructuralObjectClass.Equals("group", StringComparison.InvariantCultureIgnoreCase))
                                {
                                    User u = new User(p.Sid.Value, p.SamAccountName, GetDomainName(p.DistinguishedName));
                                    if (!domainUsers.Contains(u))
                                        domainUsers.Add(u);
                                    if (domainUserGroupMappings.FirstOrDefault(ug => (ug as UserGroup).Group.Equals(g) && (ug as UserGroup).User.Equals(u)) == null)
                                        domainUserGroupMappings.Add(new UserGroup(u, g));
                                }
                            //add the group to the list of groups
                            if (!domainGroups.Contains(g))
                                domainGroups.Add(g);
                        }
                    }
                    else
                        sidsNotFound.Add(sid);
                }
            }
            return sidsNotFound;
        }

        private static string GetDomainName(SearchResult entry)
        {
            string distinguishedName = entry.Properties["distinguishedname"][0].ToString();
            return GetDomainName(distinguishedName);
        }

        private static string GetDomainName(DirectoryEntry entry)
        {
            string distinguishedName = entry.Properties["distinguishedname"][0].ToString();
            return GetDomainName(distinguishedName);
        }

        private static string GetDomainName(string distinguishedName)
        {
            StringBuilder domain = new StringBuilder();
            foreach (string item in distinguishedName.Split(','))
            {
                if (item.StartsWith("DC="))
                {
                    domain.Append(item.Replace("DC=", string.Empty));
                    domain.Append(".");
                }
            }

            if (domain.Length > 0)
                domain = domain.Remove(domain.Length - 1, 1);

            return domain.ToString();
        }

        #region get domain members
        /// <summary>
        /// Gets all the groups in the specified domain.
        /// </summary>
        /// <param name="domain">The domain.</param>
        /// <returns></returns>
        private List<IAccount> GetDomainGroups(string domain)
        {
            List<IAccount> retList = new List<IAccount>();
            using (DirectoryEntry ad = new DirectoryEntry("LDAP://" + domain))
            {
                using (DirectorySearcher ds = new DirectorySearcher(ad))
                {
                    ds.Filter = "(objectClass=group)";
                    ds.PageSize = 2000;
                    SearchResultCollection src = ds.FindAll();
                    foreach (SearchResult sr in src)
                    {
                        DirectoryEntry de = sr.GetDirectoryEntry();
                        SecurityIdentifier sid = new SecurityIdentifier((byte[])de.Properties["objectSid"].Value, 0);
                        string accountName = (string)de.Properties["samAccountName"].Value;
                        retList.Add(new Group(sid.Value, accountName, domain));
                    }
                }
            }
            if (retList.Count > 0)
                return retList;
            return null;
        }

        /// <summary>
        /// Gets all the users in the specified domain.
        /// </summary>
        /// <param name="userGroup">The user group.</param>
        /// <param name="domain">The domain.</param>
        /// <returns></returns>
        private List<IAccount> GetDomainUsers(ref List<IAccount> userGroup, string domain)
        {
            List<IAccount> retList = new List<IAccount>();
            using (DirectoryEntry ad = new DirectoryEntry("LDAP://" + domain))
            {
                using (DirectorySearcher ds = new DirectorySearcher(ad))
                {
                    ds.Filter = "(&(objectClass=user)(samAccountName=mihaia))";
                    ds.PageSize = 2000;
                    SearchResultCollection src = ds.FindAll();
                    foreach (SearchResult sr in src)
                    {
                        DirectoryEntry de = sr.GetDirectoryEntry();
                        SecurityIdentifier sid = new SecurityIdentifier((byte[])de.Properties["objectSid"].Value, 0);
                        string accountName = (string)de.Properties["samAccountName"].Value;

                        retList.Add(new User(sid.Value, accountName, domain));
                        UserGroups(de, userGroup, accountName, domain);
                    }
                }
            }
            return retList;
        }
        #endregion

        #region hierarchy methods
        /// <summary>
        /// Get list of groups that a user belongs to.
        /// </summary>
        /// <param name="de">The de.</param>
        /// <param name="userGroups">The user groups.</param>
        /// <param name="userAccount">The user account.</param>
        /// <param name="domain">The domain.</param>
        private void UserGroups(DirectoryEntry de, List<IAccount> userGroups, string userAccount, string domain)
        {
            object members = null;
            try
            {
                members = de.Properties["memberOf"];
            }
            catch { }
            if (members != null)
                foreach (object member in (IEnumerable)members)
                {
                    using (DirectoryEntry newDe = new DirectoryEntry("LDAP://" + member.ToString()))
                    {
                        string accountName;
                        try
                        {
                            accountName = (string)newDe.Properties["samAccountName"].Value;
                        }
                        catch (Exception e)
                        {
                            string msj = string.Format("Exception occured for user '{0}' when searching for property 'samAccountName' in directory entry '{1}'", userAccount, member);
                            Log.LogMessage(msj);
                            Log.LogException(e);
                            continue;
                        }
                        List<IAccount>.Enumerator en = userGroups.GetEnumerator();
                        bool alreadyExistsUserGroup = false;
                        while (en.MoveNext())
                        {
                            UserGroup ug = en.Current as UserGroup;
                            if (ug != null && ug.User != null && ug.Group != null)
                            {
                                if (ug.User.Domain == domain &&
                                    ug.User.Name == userAccount &&
                                    ug.Group.Domain == domain &&
                                    ug.Group.Name == accountName)
                                {
                                    alreadyExistsUserGroup = true;
                                    break;
                                }
                            }
                        }
                        if (!alreadyExistsUserGroup)
                        {
                            userGroups.Add(new UserGroup(new User("", userAccount, domain), new Group("", accountName, domain)));
                            UserGroups(newDe, userGroups, userAccount, domain);
                        }
                    }
                }
        }

        /// <summary>
        /// Get users in a group.
        /// </summary>
        /// <param name="de">The directoryEntry</param>
        /// <param name="groupUsers">The group users.</param>
        private void GroupUsers(DirectoryEntry de, List<IAccount> groupUsers)
        {
            object members = null;
            try
            {
                members = de.Invoke("Members", null);
            }
            catch { }
            if (members != null)
                foreach (object member in (IEnumerable)members)
                {
                    using (DirectoryEntry newDe = new DirectoryEntry(member))
                    {
                        if (newDe.SchemaClassName.ToLower() == "user")
                            groupUsers.Add(new UserGroup(new User("", newDe.Name, newDe.Parent.Name), new Group("", de.Name, de.Parent.Name)));
                    }
                }
        }
        #endregion
        /// <summary>
        /// Gets the local accounts.
        /// </summary>
        /// <param name="users">The users.</param>
        /// <param name="groups">The groups.</param>
        /// <param name="userGroups">The user groups.</param>
        /// <param name="machine">The machine.</param>
        private void GetLocalAccounts(List<IAccount> users, List<IAccount> groups, List<IAccount> userGroups, string machine, List<string> sids)
        {
            using (DirectoryEntry ad = new DirectoryEntry("WinNT://" + machine))
            {
                foreach (DirectoryEntry de in ad.Children)
                    if (de.SchemaClassName.Equals("user", StringComparison.InvariantCultureIgnoreCase) || de.SchemaClassName.Equals("group", StringComparison.InvariantCultureIgnoreCase))
                    {
                        SecurityIdentifier sid = new SecurityIdentifier((byte[])de.Properties["objectSid"].Value, 0);
                        if (sids.Contains(sid.Value, StringComparer.InvariantCultureIgnoreCase))
                        {
                            if (de.SchemaClassName.Equals("user", StringComparison.InvariantCultureIgnoreCase))
                                users.Add(new User(sid.Value, de.Name, de.Parent.Name));
                            else if (de.SchemaClassName.ToLower() == "group")
                            {
                                groups.Add(new Group(sid.Value, de.Name, de.Parent.Name));
                                GroupUsers(de, userGroups);
                            }
                        }

                    }
            }
        }

        
        /// <summary>
        /// The Thread Pool.
        /// </summary>
        /// <param name="state">The state.</param>
        private void PoolWorker(object state)
        {
            object[] param = state as object[];
            AutoResetEvent autoResetEvent = param[0] as AutoResetEvent;
            StringBuilder errors = param[1] as StringBuilder;
            try
            {
                this.InsertDBAllGroupUsersAndDomain();
            }
            catch (Exception e)
            {
                errors.AppendLine(e.ToString());
                errors.AppendLine(e.Message);
            }
            finally
            {
                autoResetEvent.Set();
            }
            return;
        }

        #endregion
    }

    /// <summary>
    /// interface - Active Directory Account.
    /// </summary>
    public interface IAccount
    { }

    /// <summary>
    /// Class which stores account in accordance with AD.
    /// </summary>
    public class Account : IAccount
    {
        /// <summary>
        /// Initializes a new instance of the Account class.
        /// </summary>
        public Account() { }

        /// <summary>
        /// Initializes a new instance of the Account class.
        /// </summary>
        /// <param name="sid">The sid.</param>
        /// <param name="name">The name.</param>
        /// <param name="domain">The domain.</param>
        public Account(string sid, string name, string domain)
        {
            _sid = sid;
            _name = name;
            _domain = domain;
        }

        private string _sid;
        /// <summary>
        /// Gets or sets the sid.
        /// </summary>
        /// <value>
        /// The sid.
        /// </value>
        public string Sid { get { return _sid; } set { _sid = value; } }

        private string _name;
        /// <summary>
        /// Gets or sets the name.
        /// </summary>
        /// <value>
        /// The name.
        /// </value>
        public string Name { get { return _name; } set { _name = value; } }

        private string _domain;
        /// <summary>
        /// Gets or sets the domain.
        /// </summary>
        /// <value>
        /// The domain.
        /// </value>
        public string Domain { get { return _domain; } set { _domain = value; } }
    }

    /// <summary>
    /// Class which stores the user.
    /// </summary>
    public class User : Account
    {
        /// <summary>
        /// Initializes a new instance of the User class.
        /// </summary>
        /// <param name="sid">The sid.</param>
        /// <param name="name">The name.</param>
        /// <param name="domain">The domain.</param>
        public User(string sid, string name, string domain)
            : base(sid, name, domain)
        { }
    }

    /// <summary>
    /// Class which stores the group.
    /// </summary>
    public class Group : Account
    {
        /// <summary>
        /// Initializes a new instance of the Group class.
        /// </summary>
        /// <param name="sid">The sid.</param>
        /// <param name="name">The name.</param>
        /// <param name="domain">The domain.</param>
        public Group(string sid, string name, string domain)
            : base(sid, name, domain)
        { }
    }

    /// <summary>
    /// Class which stores the usergroup.
    /// </summary>
    public class UserGroup : IAccount
    {
        private readonly string _groupComponent;
        private readonly string _partComponent;
        private User _user;
        private Group _group;

        /// <summary>
        /// Initializes a new instance of the UserGroup class.
        /// </summary>
        /// <param name="_groupComponent">The group component.</param>
        /// <param name="_partComponent">The part component.</param>
        public UserGroup(string groupComponent, string partComponent)
        {
            this._groupComponent = groupComponent;
            this._partComponent = partComponent;
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="UserGroup"/> class.
        /// </summary>
        /// <param name="user">The user.</param>
        /// <param name="group">The group.</param>
        public UserGroup(User user, Group group)
        {
            this._user = user;
            this._group = group;
        }

        /// <summary>
        /// Gets the part component.
        /// </summary>
        public string PartComponent
        {
            get { return _partComponent; }
        }

        /// <summary>
        /// Gets the group component.
        /// </summary>
        public string GroupComponent
        {
            get { return _groupComponent; }
        }

        /// <summary>
        /// Gets or sets the user.
        /// </summary>
        /// <value>
        /// The user.
        /// </value>
        public User User
        {
            get { return _user; }
            set { _user = value; }
        }

        /// <summary>
        /// Gets or sets the group.
        /// </summary>
        /// <value>
        /// The group.
        /// </value>
        public Group Group
        {
            get { return _group; }
            set { _group = value; }
        }
    }
}