﻿using BMS.ServiceContracts;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using BMS.DataContracts;
using System.Collections.Generic;
using BMS.ServicesWrapper.BMService;
using InfoWorld.HL7.ITS;
using System.Linq;
using BMS.ServicesWrapper.EIS;

namespace BMS.UnitTesting
{


    /// <summary>
    ///This is a test class for BedManagerQuery and is intended
    ///to contain all BedManagerQuery Unit Tests
    ///</summary>
    [TestClass()]
    public class BedManagerQuery
    {


        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {
            ServicesControl.Init();
        }
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        internal virtual IBedManagerQuery CreateIBedManagerQuery()
        {
            IBedManagerQuery target = BMSFactory.BedManagerQueryClient;
            return target;
        }

        internal IBedManagerOperations CreateIBedManagerOperations()
        {
            IBedManagerOperations target = BMSFactory.BedManagerOperationsClient;
            return target;
        }


        /// <summary>
        /// Test method for querying the 30-days waiting list.
        /// </summary>
        [TestMethod]
        [Description("Test for querying the 30-days waiting list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterAdmissions30()
        {
            FilterAdmissions(Guid.Parse("C9411463-BF14-4D18-B662-C469228278CA"), new TimeSpan(30, 0, 0, 0));
        }

        /// <summary>
        /// Test method for querying the 60-days waiting list.
        /// </summary>
        [TestMethod]
        [Description("Test for querying the 60-days waiting list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterAdmissions60()
        {
            FilterAdmissions(Guid.Parse("C9411463-BF14-4D18-B662-C469228278CA"), new TimeSpan(60, 0, 0, 0));
        }

        /// <summary>
        /// Test method for querying the 90-days waiting list.
        /// </summary>
        [TestMethod]
        [Description("Test for querying for the 90-days waiting list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterAdmissions90()
        {
            FilterAdmissions(Guid.Parse("C9411463-BF14-4D18-B662-C469228278CA"), new TimeSpan(90, 0, 0, 0));
        }

        /// <summary>
        /// Test method for querying the current waiting list.
        /// </summary>
        [TestMethod]
        [Description("Test for querying the current waiting list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterAdmissionsCurrent()
        {
            FilterAdmissions(Guid.Parse("C9411463-BF14-4D18-B662-C469228278CA"), new TimeSpan(8, 0, 0));
        }

        void FilterAdmissions(Guid facilityUid, TimeSpan waitingTime)
        {
            IBedManagerQuery target = CreateIBedManagerQuery();
            DateTime creationTime = DateTime.UtcNow - waitingTime;
            IList<AdmissionInfo> actual = target.FilterAdmissions(facilityUid, creationTime, true, false);
            foreach (AdmissionInfo admissionInfo in actual)
                Assert.IsTrue(IsWaiting(admissionInfo, creationTime));
        }

        /// <summary>
        /// Test method for querying the bed occupancy list.
        /// </summary>
        [TestMethod()]
        [Description("Test for querying the bed occupancy list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterBedOccupancy()
        {
            IBedManagerQuery query = CreateIBedManagerQuery();
            IBedManagerOperations operations = CreateIBedManagerOperations();
            //create the intent
            Admission admissionIntent = Builders.BuildAdmissionIntent(1);
            II id = operations.CreateAdmission(admissionIntent);
            admissionIntent.Id = id;
            //create admission request
            Admission admissionRequest = Builders.BuildAdmissionRequest(1);
            admissionRequest.Parent = new Act() { Id = admissionIntent.Id };
            id = operations.CreateAdmission(admissionRequest);
            admissionRequest.Id = id;
            //create the linked event
            Admission admissionEvent = Builders.BuildAdmissionEvent(1);
            admissionEvent.Parent = new Act() { Id = admissionRequest.Id };
            //create the discharge intent
            Discharge dischargeIntent = Builders.BuildDischargeIntent(1);
            //create a bed cleaning operation for the bed
            BedCleaningOperation bedCleaningOperation = Builders.BuildBedCleaningOperation(1);
            bedCleaningOperation.Bed = admissionEvent.Bed;
            //create a bed out of service event for the bed
            BedUnavailable bedOutOfServiceEvent = Builders.BuildBedOutOfServiceEvent(1);
            bedOutOfServiceEvent.Bed = admissionEvent.Bed;
            //link operations through the flowid
            admissionEvent.FlowId = admissionIntent.FlowId;
            dischargeIntent.FlowId = admissionIntent.FlowId;
            //submit the entities
            II id2 = operations.CreateAdmission(admissionEvent);
            II id3 = operations.CreateDischarge(dischargeIntent);
            II id4= operations.CreateBedCleaningOperation(bedCleaningOperation);
            II id5 = operations.CreateBedUnavailable(bedOutOfServiceEvent);
            //get the divisionId for querying
            Guid divisionUid = Guid.Parse(EISFactory.Instance.GetWard(admissionEvent.Ward.Id).Division.Id.extension);            
            BedOccupancyInfo expected = Builders.BuildBedOccupancy(admissionEvent, dischargeIntent, bedOutOfServiceEvent, bedCleaningOperation);
            expected.Id = id2;
            IList<BedOccupancyInfo> actualList = query.FilterBedOccupancy(string.Empty, null);
            BedOccupancyInfo actual = Find(actualList, id2);
            expected.LosAdmin = actual.LosAdmin;
            expected.Reason = actual.Reason;
            Assert.IsTrue( new BedOccupancyInfoComparer().Equals( expected, actual) );
        }

        private BedOccupancyInfo Find(IList<BedOccupancyInfo> list, II id)
        {
            return list.Single(b => b.Id.root == id.root && b.Id.extension == id.extension); 
        }

        /// <summary>
        /// Test method for querying the transfers list.
        /// </summary>
        [TestMethod()]
        [Description("Test for querying the transfers list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterTransfers()
        {
            IBedManagerQuery query = CreateIBedManagerQuery();
            IBedManagerOperations operations = CreateIBedManagerOperations();
            //transfer intent
            Transfer transferIntent = Builders.BuildTransferIntent(1);

            II id = operations.CreateTransfer(transferIntent);
            transferIntent.Id = id;
            //transfer request
            Transfer transferRequest = Builders.BuildTransferRequest(1);
            transferRequest.Parent = new Act() { Id = transferIntent.Id };
            id = operations.CreateTransfer(transferRequest);
            //filter
            Guid facilityUid = Guid.Parse(transferIntent.Facility.Id.extension);
            TransferInfo expected = Builders.BuildTransferInfo(transferIntent, transferRequest);
            IList<TransferInfo> actual = query.FilterTransfers(facilityUid, null);
            expected.AdmissionFacilityId = null;
            expected.Comment = null;
            Assert.IsTrue(actual.Contains(expected, new TransferInfoComparer()));
        }

        [TestMethod()]
        [Description("Test for querying the transfers list.")]
        [TestCategory(Constants.IBedManagerQuery)]
        public void FilterPatientWaiting()
        {
            IBedManagerQuery query = CreateIBedManagerQuery();
            IList<PatientWaitingStandardView> actual = query.FilterNationalPatientWaitingStandardView(true, "", "", "ccuc", "sdsd", DateTime.Now.AddMonths(-5), true);
        }

        //[TestMethod()]
        //[Description("Test for filter bed cleaning.")]
        //[TestCategory(Constants.IBedManagerQuery)]
        //public void FilterBedClean()
        //{
        //    IBedManagerQuery query = CreateIBedManagerQuery();
        //    Guid guid = new Guid();
        //    guid =  Guid.Parse("348E3C89-5F89-485D-BAA2-A9AFCDA7A379");
        //    IList<BedCleanInfo> actual = query.FilterBedClean(guid, DateTime.Now.AddMonths(-2), DateTime.Now, new Utils.PagingContext() { PageNumber = 1, PageSize = 50, OrderBy = "max(data)" });
        //}

        #region private methods

        public bool IsWaiting(AdmissionInfo admissionInfo, DateTime waitingTime)
        {
            if (!admissionInfo.CreationDate.HasValue)
                return false;
            if (admissionInfo.CreationDate.Value.ToUniversalTime() < waitingTime)
                return false;
            return true;
        }

        #endregion
    }
}
