﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.Utils;
using BMS.VistaWorker2.Writer;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.VistaIntegration.Operations.Concrete.EntryProcessors;
using BMS.VistaIntegration.VistA;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using BMS.Facade;
using BMS.ServicesWrapper.WF;
using BMS.ServicesWrapper.Security;
using BMS.ServicesWrapper.EIS;
using BMS.VistaWorker2.Writer.Implementation.Concrete.EIS;
using System.Configuration;
using BMS.ServicesWrapper.BMService;
using DC = BMS.DataContracts;
using System.Threading;
using System.Threading.Tasks;

namespace BMS.VistaIntegration.Operations.Concrete
{
    public class Operation : BaseOperation
    {
        private static readonly bool IsUpdatePatientsEnabled;
        private const string UpdateBmsPatientsKey = "VistaIntegration.UpdateBmsPatients";

        static Operation()
        {
            IsUpdatePatientsEnabled = bool.Parse(ConfigurationManager.AppSettings[UpdateBmsPatientsKey]);
        }

        public Operation(VistASite site, IList<Parameter> parameters)
            : base(site, parameters) { }
        
        private ProcessTypeCallback processTypeCallback;
        private BmsLogger logger = new BmsLogger("Operation:");        

        private void ProcessCommonType<T>(VistaDataType type, Func<Parameter, IList<T>> fetchFunc)
        {
            Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    var result = new SingleEntryProcessor<T>(base.writer, param);
                    result.Init(fetchFunc);
                    return result;
                };
            processTypeCallback(type, processorMaker);
        }

        private II GetVistAId()
        {
            string domain = null;
            try
            {
                domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
            }
            catch (Exception)
            {
                domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
            }
            II id = new InfoWorld.HL7.ITS.II(domain, base.writer.Site.Id);
            return id;
        }

        private void UpdateAuditLog(VistaDataType dataType, string errorMessage)
        {
            lock (eventArgsList)
            {
                eventArgsList[dataType].CompletedOperations += 1;
                eventArgsList[dataType].State = ParameterEventArgs.OperationState.Failed;
                if (string.IsNullOrEmpty(eventArgsList[dataType].ErrorMessage))
                    eventArgsList[dataType].ErrorMessage = errorMessage;
                else
                    eventArgsList[dataType].ErrorMessage += " \r\n " + errorMessage;
                if ((dataType == VistaDataType.ADT && eventArgsList[dataType].CompletedOperations == 5)
                        || (dataType == VistaDataType.Patient && eventArgsList[dataType].CompletedOperations == 2))
                {
                    FinaliseAuditLog(eventArgsList[dataType]);
                    eventArgsList.Remove(dataType);
                }
            }
        }

        private static List<OccupiedBedInfo> GetBmsBedSwitch(II id)
        {
            List<OccupiedBedInfo> beds = FacadeManager.BedInterface.GetOccupiedBeds(id).ToList();
            return beds;
        }

        private void ChangePatientBedSwitch(OccupiedBedInfo bmsOccupiedBed, Data.WF.BedSwitch vistaBedInfo, ParameterEventArgs.Info info)
        {
            eventArgsList[VistaDataType.ADT].RetrievedEntries++;
            try
            {
                string domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
                II vistaSiteId = new II(domain, VistASite.Id);
                Bed bed = EISFactory.InstanceWindows.GetBed(vistaBedInfo.RoomBedIen, vistaSiteId);
                Ward ward = EISFactory.InstanceWindows.GetWard(vistaBedInfo.WardIen, vistaSiteId);

                if (ward != null && bed != null)
                {
                    Bed oldBed = null;
                    if (!string.IsNullOrEmpty(bmsOccupiedBed.BedIEN))
                        oldBed = EISFactory.InstanceWindows.GetBed(bmsOccupiedBed.BedIEN, vistaSiteId);

                    if (bmsOccupiedBed.WardIEN != vistaBedInfo.WardIen)
                    {
                        BMS.DataContracts.AdmissionEvent adm = null;
                        if (bmsOccupiedBed.IsAdmission)
                            adm = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEvent(bmsOccupiedBed.Id);
                        else
                        {
                            BMS.DataContracts.MovementEvent mov = BMSFactory.BedManagerOperationsClientWindows.GetMovementEvent(bmsOccupiedBed.Id);
                            adm = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEvent(mov.AdmissionId);
                            mov.WardId = ward.Id;
                            mov.BedId = bed.Id;
                            BMSFactory.BedManagerOperationsClientWindows.UpdateMovementEvent(mov);
                        }
                        adm.WardId = ward.Id;
                        adm.BedId = bed.Id;
                        BMSFactory.BedManagerOperationsClientWindows.UpdateAdmissionEvent(adm);
                        logger.LogFormat(BmsLogger.Level.Info, "Bed switch with ward changed - update movement for patient ssn {0}", bmsOccupiedBed.PatientUID);
                    }
                    else
                    {
                        DataContracts.BedSwitch bedSwitch = new DataContracts.BedSwitch()
                        {
                            Id = new II(domain, null),
                            BedId = bed.Id,
                            OldBedId = oldBed != null ? oldBed.Id : null,
                            EnteredDate = DateTime.UtcNow,
                            PatientId = new II(domain, bmsOccupiedBed.PatientUID),
                            WardId = ward.Id,
                            VistaSiteId = vistaSiteId,
                            Ien = vistaBedInfo.Ien,
                        };
                        BMSFactory.BedManagerOperationsClientWindows.CreateBedSwitch(bedSwitch);
                        if (!bmsOccupiedBed.IsAdmission)
                        {
                            BMS.DataContracts.MovementEvent mov = BMSFactory.BedManagerOperationsClientWindows.GetMovementEvent(bmsOccupiedBed.Id);
                            mov.WardId = bedSwitch.WardId;
                            mov.BedId = bedSwitch.BedId;
                            BMSFactory.BedManagerOperationsClientWindows.UpdateMovementEvent(mov);
                        }
                        if (bedSwitch.OldBedId != null)
                            base.writer.UtilsInstance.GenerateBedCleanRequest(vistaBedInfo.Ien, bedSwitch.OldBedId, bedSwitch.WardId, bedSwitch.VistaSiteId, VistASite.TimeZone, DateTime.UtcNow, DateTime.UtcNow, "BSW");
                        base.writer.UtilsInstance.SendMailNotification(bedSwitch, bedSwitch.VistaSiteId, VistASite.TimeZone, Constants.BED_SWITCH, null, null);
                        logger.LogFormat(BmsLogger.Level.Info, "Bed switch for patient ssn {0}", bmsOccupiedBed.PatientUID);
                    }
                    BMSFactory.BedManagerOperationsClientFromWCF.DeleteIconAssociation(Guid.Parse(bmsOccupiedBed.PatientUID));
                    info.ProcessedWithSuccesEntries++;
                }
            }
            catch (WriterException we)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                    info.ProcessedWithFaildEntries++;
                    eventArgsList[VistaDataType.ADT].WriterExceptions.Add(we);
                    logger.LogFormat(BmsLogger.Level.Error, "Bed switch for patient ssn {0} failed: {1}", bmsOccupiedBed.PatientUID, we);
                }
            }
            catch (Exception e)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                    info.ProcessedWithFaildEntries++;
                    logger.LogFormat(BmsLogger.Level.Error, "Bed switch for patient ssn {0} failed: {1}", bmsOccupiedBed.PatientUID, e);
                }
            }
        }

        private void ProcessBedSwitch(II vistAId)
        {            
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessBedSwitch", VistASite.Name);            
            List<OccupiedBedInfo> list = GetBmsBedSwitch(vistAId);            
            Dictionary<string, OccupiedBedInfo> bmsDictionaryBedSwitch = list.ToDictionary(a => a.ActIEN);
            long maxIen = 0; long currentIen = 0;
            foreach (string ien in bmsDictionaryBedSwitch.Keys)
            {
                currentIen = long.Parse(ien);
                if (currentIen > maxIen)
                    maxIen = currentIen;
            }
            if (!bmsDictionaryBedSwitch.Keys.Any()) return;            
            List<Data.AdmittedPatient> vistaAdmittedPatients = base.query.GetAdmittedPatients((maxIen + 1).ToString()).ToList();            
            vistaAdmittedPatients.RemoveAll(a => string.IsNullOrEmpty(a.WardName) || string.IsNullOrEmpty(a.BedName));
            vistaAdmittedPatients.RemoveAll(a => !bmsDictionaryBedSwitch.ContainsKey(a.MovementIen));
            var bedSwitchChanged = from b in vistaAdmittedPatients
                                   let bmsBedSwitch = bmsDictionaryBedSwitch[b.MovementIen]
                                   where (b.BedName != bmsBedSwitch.BedName || b.WardName != bmsBedSwitch.WardName)
                                   select new { bmsBedSwitch, b };

            if (bedSwitchChanged != null && bedSwitchChanged.Count() > 0)
            {
                List<string> bedSwitchIens = new List<string>();
                bedSwitchChanged.ForEach(a => bedSwitchIens.Add(a.b.MovementIen));
                List<Data.WF.BedSwitch> vistaBedSwitches = base.query.GetBedsSwitch(bedSwitchIens).ToList();

                ParameterEventArgs.Info info = null;
                Data.WF.BedSwitch vistaBedSwitch = null;
                bedSwitchChanged.ForEach(s =>
                {
                    vistaBedSwitch = vistaBedSwitches.Where(a => a.Ien == s.b.MovementIen).FirstOrDefault();
                    if (s.bmsBedSwitch.BedIEN != vistaBedSwitch.RoomBedIen || s.bmsBedSwitch.WardIEN != vistaBedSwitch.WardIen)
                    {
                        if (info == null)
                            info = GetInfo("BedSwitch", eventArgsList[VistaDataType.ADT]);
                        ChangePatientBedSwitch(s.bmsBedSwitch, vistaBedSwitch, info);
                    }
                });
                vistaBedSwitches = null;
            }
            list = null;
            bmsDictionaryBedSwitch = null;
            vistaAdmittedPatients = null;
            bedSwitchChanged = null;
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessBedSwitch", VistASite.Name);            
        }

        private void ProcessCancelOrders(II vistAId)
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessCancelOrders", VistASite.Name);            
            IList<DataContracts.Order> acts = ServicesWrapper.BMService.BMSFactory.BedManagerQueryClientWindows.GetCancelableOrders(vistAId);            
            if (!acts.Any()) return;            
            IEnumerable<string> actIens = acts.Select(s => s.Ien);            
            IList<string> canceledIens = base.query.GetCanceledOrders(actIens);            
            if (canceledIens == null || canceledIens.Count == 0)
                return;
            ParameterEventArgs.Info info = null;
            acts.Where(s => s != null && !string.IsNullOrEmpty(s.Ien) && canceledIens.Contains(s.Ien)).ForEach(c =>
                {
                    if (info == null)
                        info = GetInfo("CancelOrder", eventArgsList[VistaDataType.ADT]);
                    CancelOrder(c, info);
                });
            acts = null;
            actIens = null;
            canceledIens = null;
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessCancelOrders", VistASite.Name);            
        }

        private void CancelOrder(DataContracts.Order order, ParameterEventArgs.Info info)
        {
            eventArgsList[VistaDataType.ADT].RetrievedEntries++;

            DataContracts.AdmissionOrder admission = order as DataContracts.AdmissionOrder;
            if (admission != null)
            {
                try
                {
                    admission = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionOrder(admission.Id);
                    admission.IsActive= false;
                    BMSFactory.BedManagerOperationsClientWindows.UpdateAdmissionOrder(admission);
                    info.ProcessedWithSuccesEntries++;                    
                    logger.LogFormat(BmsLogger.Level.Info, "Cancel admission order with ien {0} with succes", order.Ien);
                }
                catch (Exception e)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                        info.ProcessedWithFaildEntries++;
                        logger.LogFormat(BmsLogger.Level.Error, "Cancel admission order with ien {0} failed: {1}", order.Ien, e);
                    }
                }
                return;
            }

            DataContracts.MovementOrder movement = order as DataContracts.MovementOrder;
            if (movement != null)
            {
                try
                {
                    movement = BMSFactory.BedManagerOperationsClientWindows.GetMovementOrder(movement.Id);
                    movement.IsActive = false;
                    BMSFactory.BedManagerOperationsClientWindows.UpdateMovementOrder(movement);
                    info.ProcessedWithSuccesEntries++;
                    logger.LogFormat(BmsLogger.Level.Info, "Cancel movement order with ien {0} with succes", order.Ien);
                }
                catch (Exception e)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                        info.ProcessedWithFaildEntries++;
                        logger.LogFormat(BmsLogger.Level.Error, "Cancel movement order with ien {0} failed: {1}", order.Ien, e);
                    }
                }
                return;
            }

            DataContracts.DischargeOrder discharge = order as DataContracts.DischargeOrder;
            if (discharge != null)
            {
                try
                {
                    discharge = BMSFactory.BedManagerOperationsClientWindows.GetDischargeOrder(discharge.Id);
                    discharge.IsActive = false;
                    BMSFactory.BedManagerOperationsClientWindows.UpdateDischargeOrder(discharge);
                    info.ProcessedWithSuccesEntries++;
                    logger.LogFormat(BmsLogger.Level.Info, "Cancel discharge order with ien {0} with succes", order.Ien);
                }
                catch (Exception e)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                        info.ProcessedWithFaildEntries++;
                        logger.LogFormat(BmsLogger.Level.Error, "Cancel discharge order with ien {0} failed: {1}", order.Ien, e);
                    }
                }
                return;
            }

        }

        private void ProcessWorkFlowTypes()
        {
            Task[] tasks = new Task[] {
                Task.Factory.StartNew(() => ProcessOrders()),
                Task.Factory.StartNew(() => TaskPatientMovements()),
                Task.Factory.StartNew(() => ProcessPatientAppointments()),
                Task.Factory.StartNew(() => ProcessScheduledAdmissions())
            };
            logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - Operation ADT - Before wait tasks", VistASite.Name, DateTime.Now.ToString("MM/dd/yyyy hh:mm:ss.fff tt"));            
            try
            {
                Task.WaitAll(tasks, 43200000);
            }
            catch (AggregateException ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessWorkflowTypes wait tasks");
                ex.Handle((x) => { logger.LogError(x.ToString()); return true; });
            }
            finally
            {
                for (int i = 0; i < tasks.Length; i++)
                {
                    tasks[i].Dispose();
                    tasks[i] = null;
                }
                tasks = null;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - Operation ADT - After wait tasks", VistASite.Name, DateTime.Now.ToString("MM/dd/yyyy hh:mm:ss.fff tt"));            
        }

        private void TaskPatientMovements()
        {
            ProcessPatientMovements();
            ProcessDeletedPatientMovements();
        }

        private void ProcessOrders()
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessOrders", VistASite.Name);
            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    II vistAId = GetVistAId();
                    ProcessCancelOrders(vistAId);
                    return new OrderProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.Order, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessOrders - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                throw ex;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessOrders", VistASite.Name);
        }

        private void ProcessPatientMovements()
        {            
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessPatientMovements", VistASite.Name);
            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    II vistAId = GetVistAId();
                    ProcessBedSwitch(vistAId);
                    return new PatientMovementProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.PatientMovement, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessPatientMovements - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                throw ex;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessPatientMovements", VistASite.Name);            
        }

        private void ProcessDeletedPatientMovements()
        {            
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessDeletedPatientMovements", VistASite.Name);
            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new DeletedPatientMovementsProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.PatientMovementIen, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessDeletedPatientMovements - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                throw ex;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessDeletedPatientMovements", VistASite.Name);            
        }

        private void ProcessPatientAppointments()
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessPatientAppointments", VistASite.Name);
            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new PatientAppointmentsProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.PatientAppointment, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessPatientAppointments - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                throw ex;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessPatientAppointments", VistASite.Name);
        }

        private void ProcessScheduledAdmissions()
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessScheduledAdmissions", VistASite.Name);
            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new ScheduledAdmissionProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.ScheduledAdmission, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessScheduledAdmissions - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                throw ex;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessScheduledAdmissions", VistASite.Name);
        }

        private void ProcessWaitingList()
        {
            Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    var result = new WaitingListProcessor(base.writer, param, base.query);
                    result.Init();
                    return result;
                };
            processTypeCallback(VistaDataType.WaitingList, processorMaker);
        }

        private void ProcessPatients()
        {
            Task[] tasks = new Task[] {
                Task.Factory.StartNew(() => 
                                {
                                    if (IsUpdatePatientsEnabled)
                                        ProcessUpdatingPatients();
                                }),
                Task.Factory.StartNew(() => TaskNewPatients())
            };            
            logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - Operation Patients - Before wait tasks", VistASite.Name, DateTime.Now.ToString("MM/dd/yyyy hh:mm:ss.fff tt")); 
            try
            {
                Task.WaitAll(tasks);
            }
            catch (AggregateException ex)
            {
                logger.LogError(VistASite.Name + " - Exception ProcessPatient wait tasks");
                ex.Handle((x) => { logger.LogError(x.ToString()); return true; });
            }
            finally
            {
                foreach (Task t in tasks)
                    t.Dispose();
                tasks = null;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - Operation Patients - After wait tasks", VistASite.Name, DateTime.Now.ToString("MM/dd/yyyy hh:mm:ss.fff tt")); 
        }

        private void TaskNewPatients()
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation TaskNewPatients", VistASite.Name);
            try
            {
                string patientIen = BMSFactory.BedManagerOperationsClientWindows.GetVistaPatientIen(VistASite.Name);
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    var result = new SingleEntryProcessor<BMS.VistaIntegration.Data.Patient>(base.writer, param);
                    result.Init(p => base.query.GetPatients(p.StartDateParam, p.EndDateParam, patientIen));
                    return result;
                };
                processTypeCallback(VistaDataType.Patient, processorMaker);             
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception TaskNewPatients - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.Patient))
                    UpdateAuditLog(VistaDataType.Patient, ex.ToString());
                throw ex;
            }
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation TaskNewPatients", VistASite.Name);
        }

        private void ProcessUpdatingPatients()
        {
            logger.LogFormat(BmsLogger.Level.Info, "{0} - Start Operation ProcessUpdatingPatients", VistASite.Name);
            try
            {
                Parameter param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.Patient);
                if (param == null)
                    return;
                lock (eventArgsList)
                {
                    if (!eventArgsList.ContainsKey(VistaDataType.Patient))
                    {                    
                        eventArgsList.Add(param.Type, new ParameterEventArgs(param));
                        eventArgsList[param.Type].State = ParameterEventArgs.OperationState.Running;
                        eventArgsList[param.Type].CompletedOperations = 0;
                        OnTypeUpdating(eventArgsList[param.Type]);
                    }
                }

                II vistaSite = GetVistAId();
                // get admitted and waiting list patients from BMS
                IList<II> patientIds = ServicesWrapper.BMService.BMSFactory.BedManagerQueryClientWindows.FilterUpdatingPatients(vistaSite);
                if (!patientIds.Any()) return;
                IList<Facade.Data.Patient> bmsPatients = EISFactory.InstanceWindows.GetPatientsByIds(patientIds, null);
                List<string> iens = new List<string>();
                II ien = null;
                foreach (Facade.Data.Patient patient in bmsPatients)
                {
                    if (patient.IENList != null && patient.IENList.Count > 0)
                    {
                        ien = patient.IENList.Where(a => a.root.Equals(VistASite.Name, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (ien != null)
                            iens.Add(ien.extension);
                    }
                }
                //get admitted patients from Vista
                IList<VistaIntegration.Data.Patient> vistaPatients = base.query.GetAdmittedPatientsForUpdate();
                // remove the iens for admitted patients
                foreach (VistaIntegration.Data.Patient p in vistaPatients)
                {
                    if (iens.Contains(p.IEN))
                        iens.Remove(p.IEN);
                }
                // get the rest of the patients by ien from Vista                
                foreach (string i in iens)
                    vistaPatients.Add(base.query.GetPatientByIen(i));

                IList<Person> attendingPhysicians = EISFactory.InstanceWindows.GetMedicalPersonsByIds(bmsPatients.Select<Facade.Data.Patient, II>(a => a.AttendingPhysicianId).Where(b => b != null && !string.IsNullOrEmpty(b.extension) && !b.extension.Equals(Guid.Empty.ToString())).Distinct().ToList());
                Facade.Data.Patient bmsPatient = null;
                string bmsAttendingIen = null, vistaAttendingIen = null;
                using (var patientWriter = base.writer.MakeEntityWriter<Data.Patient>() as PatientWriter)
                {
                    foreach (VistaIntegration.Data.Patient p in vistaPatients)
                    {
                        if (p == null)
                            continue;
                        bmsPatient = bmsPatients.Where(a => a.SSN.extension == p.SocialSecurityNumber).FirstOrDefault();
                        if (bmsPatient != null)
                        {
                            if (bmsPatient.AttendingPhysicianId != null)
                            {
                                try
                                {
                                    bmsAttendingIen = attendingPhysicians.Where(a => a.Id.extension.Equals(bmsPatient.AttendingPhysicianId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Ien;
                                }
                                catch { bmsAttendingIen = null; }
                            }
                            else
                                bmsAttendingIen = null;
                            if (p.AttendingPhysician != null)
                                vistaAttendingIen = p.AttendingPhysician.IEN;
                            else
                                vistaAttendingIen = null;

                            if (p.AttendingPhysician != null && bmsAttendingIen == vistaAttendingIen)
                                p.AttendingPhysician = new NewPerson() { IEN = "-100", Name = p.AttendingPhysician.Name }; //do not upate attending physician
                            CheckForUpdatePatient(patientWriter, p, bmsPatient);
                        }
                    }
                }
                patientIds = null;
                bmsPatients = null;
                vistaPatients = null;
                attendingPhysicians = null;
            }
            catch (Exception ex)
            {
                lock (eventArgsList)
                {
                    logger.LogError(VistASite.Name + " - Exception ProcessUpdatingPatients - " + ex.ToString());
                    eventArgsList[VistaDataType.Patient].State = ParameterEventArgs.OperationState.Failed;
                    if (string.IsNullOrEmpty(eventArgsList[VistaDataType.Patient].ErrorMessage))
                        eventArgsList[VistaDataType.Patient].ErrorMessage = ex.ToString();
                    else
                        eventArgsList[VistaDataType.Patient].ErrorMessage += " \r\n " + ex.ToString();
                }
                throw ex;
            }
            finally
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.Patient].CompletedOperations += 1;
                    if (eventArgsList[VistaDataType.Patient].CompletedOperations == 2)
                    {
                        FinaliseAuditLog(eventArgsList[VistaDataType.Patient]);
                        eventArgsList.Remove(VistaDataType.Patient);
                    }
                }
                logger.LogFormat(BmsLogger.Level.Info, "{0} - End Operation ProcessUpdatingPatients", VistASite.Name);
            }
        }

        private void CheckForUpdatePatient(PatientWriter patientWriter, Data.Patient vistaPatient, Facade.Data.Patient bmsPatient)
        {
            try
            {                
                bool isUpdated = patientWriter.UpdateOldEntity(vistaPatient, bmsPatient);             
                if (isUpdated)
                {
                    ParameterEventArgs.Info info = GetInfo("UpdatePacients", eventArgsList[VistaDataType.Patient]);
                    info.ProcessedWithSuccesEntries++;
                }
            }
            catch (Exception e)
            {
                ParameterEventArgs.Info info = GetInfo("UpdatePacients", eventArgsList[VistaDataType.Patient]);
                info.ProcessedWithFaildEntries++;
                logger.LogFormat(BmsLogger.Level.Error, "Update patient data with ssn {0} failed: {1}", bmsPatient.SSN.extension, e);

            }
        }

        protected override void ProcessTypes(ProcessTypeCallback callback)
        {
            this.processTypeCallback = callback;
            Parameter param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.Specialty);
            if (param != null)
                ProcessCommonType(VistaDataType.Specialty, p => base.query.GetSpecialties());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.TreatingSpecialty);
            if (param != null)
                ProcessCommonType(VistaDataType.TreatingSpecialty, p => base.query.GetFacilityTreatingSpecialties());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.FacilityMovementType);
            if (param != null)
                ProcessCommonType(VistaDataType.FacilityMovementType, p => base.query.GetFacilityMovementTypes());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.OrderableItem);
            if (param != null)
                ProcessCommonType(VistaDataType.OrderableItem, p => base.query.GetOrderableItems());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.HospitalLocation);
            if (param != null)
                ProcessCommonType(VistaDataType.HospitalLocation, p => base.query.GetHospitalLocations());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.WardLocation);
            if (param != null)
                ProcessCommonType(VistaDataType.WardLocation, p => base.query.GetWardLocations());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.RoomBed);
            if (param != null)
                ProcessCommonType(VistaDataType.RoomBed, p => base.query.GetRoomBeds());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.MedicalCenterDivision);
            if (param != null)
                ProcessCommonType(VistaDataType.MedicalCenterDivision, p => base.query.GetMedicalCenterDivisions());
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.Patient);
            if (param != null)
                ProcessPatients();
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.WaitingList);
            if (param != null)
                ProcessWaitingList();
            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.ADT);
            if (param != null)
                ProcessWorkFlowTypes();            
        }
    }
}
