﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Operations;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaWorker2.Writer;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.ServicesWrapper.BMService;


namespace BMS.VistaIntegration.Audit
{
    static class AuditUtilities
    {
        public static BMS.DataContracts.JobLogInfo MakeJobLog(Parameter param, VistASite site, DataRetrievalMethod dataRetrievalMethod)
        {
            return new BMS.DataContracts.JobLogInfo()
            {
                StartDate = site.ConvertToSiteDateTime(DateTime.UtcNow),
                EndDate = null,
                RetrievedDataType = param.Type,
                Status = JobStatus.Running,
                VistaId = site.Id,
                VistaName = site.Name,
                RetrievalMethod = dataRetrievalMethod,
                QueryStartDate = param.StartDateParam.HasValue ?
                    DateTime.SpecifyKind(param.StartDateParam.Value, DateTimeKind.Utc)
                    : new Nullable<DateTime>(),
                QueryEndDate = param.EndDateParam.HasValue ?
                    DateTime.SpecifyKind(param.EndDateParam.Value, DateTimeKind.Utc)
                    : new Nullable<DateTime>()
            };
        }



        public static void OperationFailed(VistASite site, BMS.DataContracts.JobLogInfo jobLog, String errorMsg, string details)
        {
            bool newEntry = jobLog.Id == 0;
            jobLog.EndDate = site.ConvertToSiteDateTime(DateTime.UtcNow);
            jobLog.Status = JobStatus.Failed;
            jobLog.ErrorMessage = errorMsg;
            jobLog.Details = details;
            if (newEntry)
            {
                int id = BMSFactory.BedManagerOperationsClientWindows.AddLogEntry(jobLog);
                jobLog.Id = id;
            }
            else
                BMSFactory.BedManagerOperationsClientWindows.UpdateLogEntry(jobLog);
        }

        private static void SetDetails(BMS.DataContracts.JobLogInfo jobLog, ParameterEventArgs e)
        {

            IList<WriterException> writerExceptions = e.WriterExceptions;

            StringBuilder sb = new StringBuilder();

            if (!string.IsNullOrEmpty(jobLog.Details))
            {
                sb.AppendLine(jobLog.Details);
            }
            if (e.RetrievedEntries == 0)
            {
                sb.AppendLine("No entries were processed.");
            }
            else
            {
                string msj = null;
                foreach (var info in e.InfoDictionary)
                {
                    sb.Append(info.Key);
                    sb.Append(":");
                    if (info.Value.ProcessedWithSuccesEntries == 0 && info.Value.ProcessedWithFaildEntries == 0)
                        sb.AppendLine("No entries were processed.");
                    else
                    {
                        msj = (info.Value.ProcessedWithSuccesEntries == 0) ? "No entries were processed with success." : string.Format("{0} entries were processed with success.", info.Value.ProcessedWithSuccesEntries);
                        sb.Append(msj);
                    }
                    if (info.Value.ProcessedWithFaildEntries != 0)
                    {
                        msj = string.Format("{0} entries were not processed due to errors.", info.Value.ProcessedWithFaildEntries);
                        sb.Append(msj);
                    }
                    sb.AppendLine();
                }
            }

            if (writerExceptions != null && writerExceptions.Count > 0)
            {
                sb.AppendLine();
                string msj = string.Format("FIRST {0} WRITER ERRORS:", e.WriterExceptions.Count);
                sb.Append(msj);

                for (int i = 0; i < writerExceptions.Count; i++)
                {
                    sb.AppendLine();
                    sb.AppendLine();
                    sb.Append(i);
                    sb.Append(": ");
                    WriterException writerException = writerExceptions[i];
                    sb.Append(writerException.Message);
                    if (writerException.InnerException != null)
                    {
                        sb.AppendLine();
                        sb.Append(writerException.InnerException.ToString());
                    }
                }

            }

            jobLog.Details = sb.ToString();

        }

        public static void OperationUpdated(VistASite site, BMS.DataContracts.JobLogInfo jobLog, ParameterEventArgs e)
        {
            DateTime now = site.ConvertToSiteDateTime(DateTime.UtcNow);
            jobLog.RetrievedEntriesNumber = e.RetrievedEntries;
            jobLog.Id = e.JobLogId;
            jobLog.StartDate = e.JobLogStartDate;
            bool newEntry = e.JobLogId == 0;
            switch (e.State)
            {
                case ParameterEventArgs.OperationState.Running:
                    jobLog.StartDate = now;
                    break;
                case ParameterEventArgs.OperationState.Succeeded:
                    jobLog.Status = JobStatus.Succeeded;
                    jobLog.EndDate = now;
                    SetDetails(jobLog, e);
                    break;
                case ParameterEventArgs.OperationState.PartiallySucceeded:
                    jobLog.Status = JobStatus.PartiallySucceeded;
                    jobLog.ErrorMessage = e.ErrorMessage;
                    jobLog.EndDate = now;
                    SetDetails(jobLog, e);
                    break;
                case ParameterEventArgs.OperationState.Failed:
                    jobLog.Status = JobStatus.Failed;
                    jobLog.ErrorMessage = e.ErrorMessage;
                    jobLog.EndDate = now;
                    SetDetails(jobLog, e);
                    break;
            }
            e.JobLogStartDate = jobLog.StartDate;
            if (newEntry)
            {
                e.JobLogId = BMSFactory.BedManagerOperationsClientWindows.AddLogEntry(jobLog);
            }
            else
                BMSFactory.BedManagerOperationsClientWindows.UpdateLogEntry(jobLog);

        }
    }
}
