﻿using System;
using System.Collections.Generic;
using System.Linq;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using BMS.ServicesWrapper.EIS;
using BMS.Utils;
using System.Threading;
using InfoWorld.HL7.ITS.Extension.EIS;

namespace BMS.ServicesWrapper
{
    /// <summary>
    /// Used to cache the eis entities.
    /// </summary>
    internal class EISCache
    {
        IEISWrapper _EISWrapper;
        internal EISCache(IEISWrapper EISWrapper)
        {
            _EISWrapper = EISWrapper;
        }

        private static readonly ReaderWriterLockSlim bedCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim wardCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim hlCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim cacheLock = new ReaderWriterLockSlim();
        // cache by Entity extension as key and Entity object as value.
        static Dictionary<string, Bed> _beds = new Dictionary<string, Bed>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, Ward> _wards = new Dictionary<string, Ward>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, Division> _divisions = new Dictionary<string, Division>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, Facility> _facilities = new Dictionary<string, Facility>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, VistaSite> _vistaSites = new Dictionary<string, VistaSite>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, Visn> _visns = new Dictionary<string, Visn>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, Region> _regions = new Dictionary<string, Region>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, HospitalLocation> _hospitalLocations = new Dictionary<string, HospitalLocation>(StringComparer.InvariantCultureIgnoreCase);
        static List<string> _cacheDomains = null;
        // cache for linking the entities (the key is the parent)
        static Dictionary<string, List<string>> _cacheBedsByWard = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheBedsByDivision = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheBedsByVista = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheWardsByDivision = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheWardsByVista = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheDivisionsByFacility = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheFacilitiesByVista = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheVistaSitesByVisn = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheVisnsByRegion = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);
        static Dictionary<string, List<string>> _cacheHospitalLocationsByVista = new Dictionary<string, List<string>>(StringComparer.InvariantCultureIgnoreCase);

        //lists for storing vista identifiers to indicate if the cache was loaded
        static List<string> _loadedBedsCacheByVista = new List<string>();
        static List<string> _loadedHospitalLocationsCacheByVista = new List<string>();

        #region Get Methods by Id

        internal Bed GetBedById(string bedId, II vistaId)
        {
            if (string.IsNullOrEmpty(bedId) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                FillBedsByVistaCache(vistaId);

            if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                if (_beds.ContainsKey(bedId))
                    return _beds[bedId];
            }
            return null;
        }

        internal Bed GetBedByIenAndVista(string bedIen, II vistaId)
        {
            if (string.IsNullOrEmpty(bedIen) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                FillBedsByVistaCache(vistaId);

            if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                try
                {
                    return _beds.Values.Where(a => a.Ien == bedIen && a.VistaSite.Id.extension.Equals(vistaId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
                catch (InvalidOperationException) { return null; }
            }                        
            return null;            
        }

        internal IEnumerable<Bed> GetBedsByIds(List<II> bedsIds, II vistaId)
        {
            if (bedsIds == null || bedsIds.Count == 0 || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                FillBedsByVistaCache(vistaId);

            if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                List<Bed> result = new List<Bed>();
                foreach (II id in bedsIds)
                {
                    if (_beds.ContainsKey(id.extension))
                        result.Add(_beds[id.extension]);
                }
                return result;
            }
            return new List<Bed>();
        }

        internal Ward GetWardById(string wardId)
        {
            if (string.IsNullOrEmpty(wardId)) return null;
            if (wardCacheLock.TryEnterReadLock(0))
            {
                try
                {
                    if (_wards.ContainsKey(wardId))
                        return _wards[wardId];
                }
                finally { wardCacheLock.ExitReadLock(); }
            }
            return null;
        }

        internal Ward GetWardByIenAndVista(string wardIen, II vistaId)
        {
            if (string.IsNullOrEmpty(wardIen) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (wardCacheLock.TryEnterReadLock(0))
            {
                try
                {
                    return _wards.Values.Where(a => a.Ien == wardIen && a.VistaSite.Id.extension.Equals(vistaId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();                    
                }
                finally { wardCacheLock.ExitReadLock(); }
            }
            return null;
        }

        internal List<Ward> GetWardsByIds(List<II> wardsIds)
        {
            if (wardsIds == null || wardsIds.Count == 0) return null;
            try
            {
                List<Ward> result = new List<Ward>();
                foreach (II id in wardsIds)
                {
                    if (_wards.ContainsKey(id.extension))
                        result.Add(_wards[id.extension]);
                }
                return result;
            }
            catch (Exception) { return new List<Ward>(); }
        }

        internal Division GetDivisionById(string divisionId)
        {
            if (string.IsNullOrEmpty(divisionId)) return null;
            if (_divisions.ContainsKey(divisionId))
                return _divisions[divisionId];                
            return null;
        }

        internal Facility GetFacilityById(string facilityId)
        {
            if (string.IsNullOrEmpty(facilityId)) return null;
            if (_facilities.ContainsKey(facilityId))
                return _facilities[facilityId];                
            return null;
        }

        internal VistaSite GetVistaSiteById(string vistaSiteId)
        {
            if (string.IsNullOrEmpty(vistaSiteId)) return null;
            if (_vistaSites.ContainsKey(vistaSiteId))
                return _vistaSites[vistaSiteId];                
            return null;
        }

        internal Visn GetVisnById(string visnId)
        {
            if (string.IsNullOrEmpty(visnId)) return null;
            if (_visns.ContainsKey(visnId))
                return _visns[visnId];                
            return null;
        }

        internal Region GetRegionById(string regionId)
        {
            if (string.IsNullOrEmpty(regionId)) return null;
            if (_regions.ContainsKey(regionId))
                return _regions[regionId];                
            return null;
        }        

        internal HospitalLocation GetHospitalLocationById(string hospitalLocationId, II vistaId)
        {
            if (string.IsNullOrEmpty(hospitalLocationId) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheHospitalLocationsByVista.ContainsKey(vistaId.extension))
                FillHospitalLocationsByVistaCache(vistaId);
                
            if (_loadedHospitalLocationsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                if (_hospitalLocations.ContainsKey(hospitalLocationId))
                    return _hospitalLocations[hospitalLocationId];                
            }
            return null;
        }

        internal HospitalLocation GetHospitalLocationByIenAndVista(string hospitalLocationIen, II vistaId)
        {
            if (string.IsNullOrEmpty(hospitalLocationIen) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheHospitalLocationsByVista.ContainsKey(vistaId.extension))
                FillHospitalLocationsByVistaCache(vistaId);

            if (_loadedHospitalLocationsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                try
                {
                    return _hospitalLocations.Values.Where(a => a.Ien == hospitalLocationIen && a.VistaSite.Id.extension.Equals(vistaId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
                catch (InvalidOperationException) { return null; }
            }            
            return null;
        }

        internal IEnumerable<HospitalLocation> GetHospitalLocationsByIds(List<II> hospitalLocationsIds, II vistaId)
        {
            if (hospitalLocationsIds == null || hospitalLocationsIds.Count == 0 || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheHospitalLocationsByVista.ContainsKey(vistaId.extension))            
                FillHospitalLocationsByVistaCache(vistaId);
                
            if (_loadedHospitalLocationsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                try
                {
                    List<HospitalLocation> result = new List<HospitalLocation>();
                    foreach (II id in hospitalLocationsIds)
                    {
                        if (_hospitalLocations.ContainsKey(id.extension))
                            result.Add(_hospitalLocations[id.extension]);
                    }
                    return result;
                }
                catch (InvalidOperationException) { return new List<HospitalLocation>(); }
            }
            return new List<HospitalLocation>();
        }

        #endregion

        #region Get Methods By Parent

        internal List<Bed> GetBedsByWard(string parentId, II vistaId)
        {
            if (string.IsNullOrEmpty(parentId) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                FillBedsByVistaCache(vistaId);

            if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {                
                if (_cacheBedsByWard.ContainsKey(parentId))
                {
                    List<string> bedsByWard = new List<string>();
                    bedsByWard.AddRange(_cacheBedsByWard[parentId]);
                    List<Bed> result = new List<Bed>();
                    foreach (string id in bedsByWard)
                        if (_beds.ContainsKey(id))
                            result.Add(_beds[id]);
                    bedsByWard = null;
                    return result;
                }
            }
            return null;
        }

        internal List<Bed> GetBedsByDivision(string parentId, II vistaId)
        {
            if (string.IsNullOrEmpty(parentId) || vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;
            if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                FillBedsByVistaCache(vistaId);
                
            if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                if (_cacheBedsByDivision.ContainsKey(parentId))
                {
                    List<string> bedsByDivision = new List<string>();
                    bedsByDivision.AddRange(_cacheBedsByDivision[parentId]);
                    List<Bed> result = new List<Bed>();
                    foreach (string id in bedsByDivision)
                        if (_beds.ContainsKey(id))
                            result.Add(_beds[id]);
                    bedsByDivision = null;
                    return result;
                }
            }
            return null;
        }

        internal List<Bed> GetBedsByVista(II vistaId)
        {
            if (vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;            
            if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                FillBedsByVistaCache(vistaId);
                
            if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                try
                {
                    List<string> bedsByVista = new List<string>();
                    bedsByVista.AddRange(_cacheBedsByVista[vistaId.extension]);
                    List<Bed> result = new List<Bed>();
                    foreach (string id in bedsByVista)
                        if (_beds.ContainsKey(id))
                            result.Add(_beds[id]);
                    bedsByVista = null;
                    return result;
                }
                catch (KeyNotFoundException) { return null; }
                catch (InvalidOperationException) { return null; }
            }                        
            return null;            
        }

        internal List<Ward> GetWardsByDivision(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            if (wardCacheLock.TryEnterReadLock(0))
            {
                try
                {
                    if (_cacheWardsByDivision.ContainsKey(parentId))
                    {
                        List<Ward> result = new List<Ward>();
                        foreach (string id in _cacheWardsByDivision[parentId])
                            if (_wards.ContainsKey(id))
                                result.Add(_wards[id]);
                        return result;
                    }
                }
                finally { wardCacheLock.ExitReadLock(); }
            }
            return null;
        }

        internal List<Ward> GetWardsByVistaSite(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            if (wardCacheLock.TryEnterReadLock(0))
            {
                try
                {
                    if (_cacheWardsByVista.ContainsKey(parentId))
                    {
                        List<Ward> result = new List<Ward>();
                        foreach (string id in _cacheWardsByVista[parentId])
                            if (_wards.ContainsKey(id))
                                result.Add(_wards[id]);
                        return result;
                    }
                }
                finally { wardCacheLock.ExitReadLock(); }
            }
            return null;
        }

        internal List<Division> GetDivisionsByFacility(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            List<Division> result = new List<Division>();
            if (_cacheDivisionsByFacility.ContainsKey(parentId))
            {                
                foreach (string id in _cacheDivisionsByFacility[parentId])
                    if (_divisions.ContainsKey(id))
                        result.Add(_divisions[id]);                
            }            
            return result;
        }

        internal List<Facility> GetFacilitiesByVistaSite(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            if (_cacheFacilitiesByVista.ContainsKey(parentId))
            {
                List<Facility> result = new List<Facility>();
                foreach (string id in _cacheFacilitiesByVista[parentId])
                    if (_facilities.ContainsKey(id))
                        result.Add(_facilities[id]);
                return result;
            }            
            return null;
        }

        internal List<Facility> GetFacilitiesByVisn(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            if (_cacheVistaSitesByVisn.ContainsKey(parentId))
            {
                List<Facility> result = new List<Facility>();
                foreach (string id in _cacheVistaSitesByVisn[parentId])
                    result.AddRange(GetFacilitiesByVistaSite(id));
                return result;
            }            
            return null;
        }

        internal List<VistaSite> GetVistaSitesByVisn(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            if (_cacheVistaSitesByVisn.ContainsKey(parentId))
            {
                List<VistaSite> result = new List<VistaSite>();
                foreach (string id in _cacheVistaSitesByVisn[parentId])
                    if (_vistaSites.ContainsKey(id))
                        result.Add(_vistaSites[id]);
                return result;
            }            
            return null;
        }

        internal List<Visn> GetVisnsByRegion(string parentId)
        {
            if (string.IsNullOrEmpty(parentId)) return null;
            if (_cacheVisnsByRegion.ContainsKey(parentId))
            {
                List<Visn> result = new List<Visn>();
                foreach (string id in _cacheVisnsByRegion[parentId])
                    if (_visns.ContainsKey(id))
                        result.Add(_visns[id]);
                return result;
            }
            return null;
        }

        internal List<HospitalLocation> GetHospitalLocationsByVista(II vistaId)
        {
            if (vistaId == null || string.IsNullOrEmpty(vistaId.extension) || vistaId.extension.Equals(Guid.Empty.ToString())) return null;            
            if (!_cacheHospitalLocationsByVista.ContainsKey(vistaId.extension))
                FillHospitalLocationsByVistaCache(vistaId);
                
            if (_loadedHospitalLocationsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
            {
                List<string> hlsByVista = new List<string>();
                hlsByVista.AddRange(_cacheHospitalLocationsByVista[vistaId.extension]);
                List<HospitalLocation> result = new List<HospitalLocation>();
                foreach (string id in hlsByVista)
                    if (_hospitalLocations.ContainsKey(id))
                        result.Add(_hospitalLocations[id]);
                hlsByVista = null;
                return result;
            }            
            return null;
        }

        #endregion

        #region Get Methods All Data

        internal List<Facility> GetFacilities()
        {
            if (_facilities != null && _facilities.Count > 0)
            {
                List<Facility> result = new List<Facility>();
                foreach (string key in _facilities.Keys)
                    result.Add(_facilities[key]);
                return result;
            }
            return null;
        }

        internal List<VistaSite> GetVistaSites()
        {
            if (_vistaSites != null && _vistaSites.Count > 0)
            {
                List<VistaSite> result = new List<VistaSite>();
                foreach (string key in _vistaSites.Keys)
                    result.Add(_vistaSites[key]);
                return result;
            }
            return null;
        }

        internal List<Visn> GetVisns()
        {
            if (_visns != null && _visns.Count > 0)
            {
                List<Visn> result = new List<Visn>();
                foreach (string key in _visns.Keys)
                    result.Add(_visns[key]);
                return result;
            }
            return null;
        }

        internal List<Region> GetRegions()
        {
            if (_regions != null && _regions.Count > 0)
            {
                List<Region> result = new List<Region>();
                foreach (string key in _regions.Keys)
                    result.Add(_regions[key]);
                return result;
            }
            return null;
        }

        internal List<string> GetDomains()
        {
            if (_cacheDomains == null)
                _cacheDomains = _EISWrapper.GetDomains().ToList();
            return _cacheDomains;
        }

        #endregion

        #region Cache

        internal void FillCache()
        {
            FlushCache();
            string domain = null;
            cacheLock.EnterWriteLock();
            BmsLogger bmsLog = new BmsLogger("FillCache EIS: ");
            try
            {
                
                bmsLog.LogInformation("Started fillCache");
                _cacheDomains = _EISWrapper.GetDomains().ToList();
                // regions
                List<Region> regions = _EISWrapper.GetPlaces(new Region()).Cast<Region>().ToList();
                domain = regions[0].Id.root;
                foreach (Region r in regions)
                    _regions.Add(r.Id.extension, r);
                regions = null;
                bmsLog.LogInformation("Loaded regions in cache");
                // visns      
                List<Visn> visns = _EISWrapper.GetPlaces(new Visn()).Cast<Visn>().ToList();
                foreach (Visn v in visns)
                {
                    _visns.Add(v.Id.extension, v);
                    if (!_cacheVisnsByRegion.ContainsKey(v.Region.Id.extension))
                        _cacheVisnsByRegion.Add(v.Region.Id.extension, new List<string>());
                    _cacheVisnsByRegion[v.Region.Id.extension].Add(v.Id.extension);
                }                
                visns = null;
                bmsLog.LogInformation("Loaded visns in cache");
                // vista sites
                List<VistaSite> vistaSites = _EISWrapper.GetPlaces(new VistaSite()).Cast<VistaSite>().ToList();
                foreach (VistaSite vs in vistaSites)
                {
                    _vistaSites.Add(vs.Id.extension, vs);
                    if (!_cacheVistaSitesByVisn.ContainsKey(vs.Visn.Id.extension))
                        _cacheVistaSitesByVisn.Add(vs.Visn.Id.extension, new List<string>());
                    _cacheVistaSitesByVisn[vs.Visn.Id.extension].Add(vs.Id.extension);
                }
                vistaSites = null;
                bmsLog.LogInformation("Loaded vista sites in cache");
                //facilities
                List<Facility> facilities = _EISWrapper.GetPlaces(new Facility()).Cast<Facility>().ToList();
                foreach (Facility f in facilities)
                {
                    _facilities.Add(f.Id.extension, f);
                    if (!_cacheFacilitiesByVista.ContainsKey(f.VistaSite.Id.extension))
                        _cacheFacilitiesByVista.Add(f.VistaSite.Id.extension, new List<string>());
                    _cacheFacilitiesByVista[f.VistaSite.Id.extension].Add(f.Id.extension);
                }                
                facilities = null;
                bmsLog.LogInformation("Loaded facilities in cache");
                // divisions
                List<Division> divisions = _EISWrapper.GetPlaces(new Division()).Cast<Division>().ToList();
                foreach (Division d in divisions)
                {
                    _divisions.Add(d.Id.extension, d);
                    if (!_cacheDivisionsByFacility.ContainsKey(d.Facility.Id.extension))
                        _cacheDivisionsByFacility.Add(d.Facility.Id.extension, new List<string>());
                    _cacheDivisionsByFacility[d.Facility.Id.extension].Add(d.Id.extension);
                }                
                divisions = null;
                bmsLog.LogInformation("Loaded divisions in cache");
                // wards
                List<Ward> wards = _EISWrapper.GetPlaces(new Ward()).Cast<Ward>().ToList();
                foreach (Ward w in wards)
                {
                    _wards.Add(w.Id.extension, w);
                    if (!_cacheWardsByVista.ContainsKey(w.VistaSite.Id.extension))
                        _cacheWardsByVista.Add(w.VistaSite.Id.extension, new List<string>());
                    _cacheWardsByVista[w.VistaSite.Id.extension].Add(w.Id.extension);                    
                }                
                foreach (string d in _divisions.Keys)
                    _cacheWardsByDivision.Add(d, wards.Where(a => a.DivisionList != null && a.DivisionList.Where(b => b.Id.extension.Equals(d, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null).Select<Ward, string>(c => c.Id.extension).ToList());
                wards = null;
                bmsLog.LogInformation("Loaded wards in cache");

                bmsLog.LogInformation("Finished fillCache");
            }
            catch (Exception e)
            {
                bmsLog.LogError(e.ToString());
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheBed(II id)
        {

            Bed obj = Translator.TranslateBed(_EISWrapper.GetEntityById(new Bed() { Id = id }));            
            if (!_cacheBedsByVista.ContainsKey(obj.VistaSite.Id.extension))            
                FillBedsByVistaCache(obj.VistaSite.Id);            
            else
            {
                bedCacheLock.EnterWriteLock();
                try
                {
                    // bed
                    if (_beds.ContainsKey(obj.Id.extension))
                        _beds[obj.Id.extension] = obj;
                    else
                        _beds.Add(obj.Id.extension, obj);
                    //ward parents
                    foreach (string key in _cacheBedsByWard.Keys)                                            
                        _cacheBedsByWard[key].Remove(obj.Id.extension);                    
                    if (obj.WardList != null && obj.WardList.Count > 0)
                    {
                        foreach (Ward w in obj.WardList)
                        {
                            if (!_cacheBedsByWard.ContainsKey(w.Id.extension))
                                _cacheBedsByWard.Add(w.Id.extension, new List<string>());
                            _cacheBedsByWard[w.Id.extension].Add(obj.Id.extension);
                        }
                    }
                    // division parents
                    foreach (string key in _cacheBedsByDivision.Keys)
                        _cacheBedsByDivision[key].Remove(obj.Id.extension);                    
                    if (obj.CommentList != null && obj.CommentList.Count > 0)
                    {
                        foreach (BedComment bc in obj.CommentList)
                        {
                            if (!_cacheBedsByDivision.ContainsKey(bc.Division.Id.extension))
                                _cacheBedsByDivision.Add(bc.Division.Id.extension, new List<string>());
                            _cacheBedsByDivision[bc.Division.Id.extension].Add(obj.Id.extension);
                        }
                    }
                    // vista site parent
                    if (!_cacheBedsByVista[obj.VistaSite.Id.extension].Contains(obj.Id.extension))
                        _cacheBedsByVista[obj.VistaSite.Id.extension].Add(obj.Id.extension);
                }                    
                finally { bedCacheLock.ExitWriteLock(); }
            }            
            obj = null;            
        }

        private void FillBedsByVistaCache(II vistaId)
        {
            bedCacheLock.EnterWriteLock();
            try
            {
                if (!_cacheBedsByVista.ContainsKey(vistaId.extension))
                {
                    List<Bed> beds = _EISWrapper.GetBedsByVistaTrait(vistaId);

                    if (_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
                        _loadedBedsCacheByVista.Remove(vistaId.extension);
                    _cacheBedsByVista.Add(vistaId.extension, new List<string>());
                    
                    foreach (Bed b in beds)
                    {
                        // beds
                        if (_beds.ContainsKey(b.Id.extension))
                            _beds[b.Id.extension] = b;
                        else
                            _beds.Add(b.Id.extension, b);
                        //ward parents
                        foreach (string key in _cacheBedsByWard.Keys)
                            _cacheBedsByWard[key].Remove(b.Id.extension);
                        if (b.WardList != null && b.WardList.Count > 0)
                        {                            
                            foreach (Ward w in b.WardList)
                            {
                                if (!_cacheBedsByWard.ContainsKey(w.Id.extension))
                                    _cacheBedsByWard.Add(w.Id.extension, new List<string>());                               
                                _cacheBedsByWard[w.Id.extension].Add(b.Id.extension);
                            }
                        }
                        foreach (string key in _cacheBedsByDivision.Keys)
                            _cacheBedsByDivision[key].Remove(b.Id.extension);
                        // division parents
                        if (b.CommentList != null && b.CommentList.Count > 0)
                        {                            
                            foreach (BedComment bc in b.CommentList)
                            {
                                if (!_cacheBedsByDivision.ContainsKey(bc.Division.Id.extension))
                                    _cacheBedsByDivision.Add(bc.Division.Id.extension, new List<string>());
                                _cacheBedsByDivision[bc.Division.Id.extension].Add(b.Id.extension);
                            }
                        }
                        // vista site parent
                        _cacheBedsByVista[vistaId.extension].Add(b.Id.extension);
                    }
                    if (!_loadedBedsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
                        _loadedBedsCacheByVista.Add(vistaId.extension);
                    beds = null;
                }
            }
            finally { bedCacheLock.ExitWriteLock(); }
        }

        internal void CacheWard(II id)
        {            
            wardCacheLock.EnterWriteLock();
            try
            {
                Ward obj = Translator.TranslateWard(_EISWrapper.GetEntityById(new Ward() { Id = id }));                
                // ward
                if (_wards.ContainsKey(obj.Id.extension))
                    _wards[obj.Id.extension] = obj;
                else
                    _wards.Add(obj.Id.extension, obj);
                // division parents
                foreach (string key in _cacheWardsByDivision.Keys)                
                    _cacheWardsByDivision[key].Remove(obj.Id.extension);                
                if (obj.DivisionList != null && obj.DivisionList.Count > 0)
                {
                    foreach (Division d in obj.DivisionList)
                    {
                        if (!_cacheWardsByDivision.ContainsKey(d.Id.extension))
                            _cacheWardsByDivision.Add(d.Id.extension, new List<string>());
                        _cacheWardsByDivision[d.Id.extension].Add(obj.Id.extension);
                    }
                }
                // vista site parent                                           
                if (obj.VistaSite != null && obj.VistaSite.Id != null && !string.IsNullOrEmpty(obj.VistaSite.Id.extension) && !obj.VistaSite.Id.extension.Equals(Guid.Empty))
                {
                    if (!_cacheWardsByVista.ContainsKey(obj.VistaSite.Id.extension))
                        _cacheWardsByVista.Add(obj.VistaSite.Id.extension, new List<string>());
                    else                                            
                        _cacheWardsByVista[obj.VistaSite.Id.extension].Remove(obj.Id.extension);

                    _cacheWardsByVista[obj.VistaSite.Id.extension].Add(obj.Id.extension);
                }
                obj = null;
            }
            finally { wardCacheLock.ExitWriteLock(); }
        }

        internal void CacheDivision(II id)
        {            
            cacheLock.EnterWriteLock();
            try
            {
                Division obj = Translator.TranslateDivision(_EISWrapper.GetEntityById(new Division() { Id = id }));
                // division
                if (_divisions.ContainsKey(obj.Id.extension))
                    _divisions[obj.Id.extension] = obj;
                else
                    _divisions.Add(obj.Id.extension, obj);
                // facility parent
                if (obj.Facility != null && obj.Facility.Id != null && !string.IsNullOrEmpty(obj.Facility.Id.extension) && !obj.Facility.Id.extension.Equals(Guid.Empty.ToString()))
                {
                    if (!_cacheDivisionsByFacility.ContainsKey(obj.Facility.Id.extension))
                        _cacheDivisionsByFacility.Add(obj.Facility.Id.extension, new List<string>());
                    else                    
                        _cacheDivisionsByFacility[obj.Facility.Id.extension].Remove(obj.Id.extension);                    
                    _cacheDivisionsByFacility[obj.Facility.Id.extension].Add(obj.Id.extension);
                }
                obj = null;
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheFacility(II id)
        {            
            cacheLock.EnterWriteLock();
            try
            {
                Facility obj = Translator.TranslateFacility(_EISWrapper.GetEntityById(new Facility() { Id = id }));
                // facility
                if (_facilities.ContainsKey(obj.Id.extension))
                    _facilities[obj.Id.extension] = obj;
                else
                    _facilities.Add(obj.Id.extension, obj);
                // vista site parent               
                if (obj.VistaSite != null && obj.VistaSite.Id != null && !string.IsNullOrEmpty(obj.VistaSite.Id.extension) && !obj.VistaSite.Id.extension.Equals(Guid.Empty.ToString()))
                {
                    if (!_cacheFacilitiesByVista.ContainsKey(obj.VistaSite.Id.extension))
                        _cacheFacilitiesByVista.Add(obj.VistaSite.Id.extension, new List<string>());
                    else
                        _cacheFacilitiesByVista[obj.VistaSite.Id.extension].Remove(obj.Id.extension);

                    _cacheFacilitiesByVista[obj.VistaSite.Id.extension].Add(obj.Id.extension);
                }
                obj = null;
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheVistaSite(II id)
        {            
            cacheLock.EnterWriteLock();
            try
            {
                VistaSite obj = Translator.TranslateVistaSite(_EISWrapper.GetEntityById(new VistaSite() { Id = id }));
                // vista site
                if (_vistaSites.ContainsKey(obj.Id.extension))
                    _vistaSites[obj.Id.extension] = obj;
                else
                    _vistaSites.Add(obj.Id.extension, obj);
                // visn parent                
                if (obj.Visn != null && obj.Visn.Id != null && !string.IsNullOrEmpty(obj.Visn.Id.extension) && !obj.Visn.Id.extension.Equals(Guid.Empty.ToString()))
                {
                    if (!_cacheVistaSitesByVisn.ContainsKey(obj.Visn.Id.extension))
                        _cacheVistaSitesByVisn.Add(obj.Visn.Id.extension, new List<string>());
                    else
                        _cacheVistaSitesByVisn[obj.Visn.Id.extension].Remove(obj.Id.extension);

                    _cacheVistaSitesByVisn[obj.Visn.Id.extension].Add(obj.Id.extension);
                }
                obj = null;
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheVisn(II id)
        {            
            cacheLock.EnterWriteLock();
            try
            {
                Visn obj = Translator.TranslateVisn(_EISWrapper.GetEntityById(new Visn() { Id = id }));
                // visn
                if (_visns.ContainsKey(obj.Id.extension))
                    _visns[obj.Id.extension] = obj;
                else
                    _visns.Add(obj.Id.extension, obj);
                // region parent                
                if (obj.Region != null && obj.Region.Id != null && !string.IsNullOrEmpty(obj.Region.Id.extension) && !obj.Region.Id.extension.Equals(Guid.Empty.ToString()))
                {
                    if (!_cacheVisnsByRegion.ContainsKey(obj.Region.Id.extension))
                        _cacheVisnsByRegion.Add(obj.Region.Id.extension, new List<string>());
                    else
                        _cacheVisnsByRegion[obj.Region.Id.extension].Remove(obj.Id.extension);

                    _cacheVisnsByRegion[obj.Region.Id.extension].Add(obj.Id.extension);
                }
                obj = null;
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheRegion(II id)
        {            
            cacheLock.EnterWriteLock();
            try
            {
                Region obj = Translator.TranslateRegion(_EISWrapper.GetEntityById(new Region() { Id = id }));
                // region
                if (_regions.ContainsKey(obj.Id.extension))
                    _regions[obj.Id.extension] = obj;
                else
                    _regions.Add(obj.Id.extension, obj);
                obj = null;
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void CacheHospitalLocation(II id)
        {
            HospitalLocation obj = Translator.TranslateHospitalLocation(_EISWrapper.GetEntityById(new HospitalLocation() { Id = id }));
            if (!_cacheHospitalLocationsByVista.ContainsKey(obj.VistaSite.Id.extension))            
                FillHospitalLocationsByVistaCache(obj.VistaSite.Id);            
            else
            {
                hlCacheLock.EnterWriteLock();
                try
                {
                    // hospital location
                    if (_hospitalLocations.ContainsKey(obj.Id.extension))
                        _hospitalLocations[obj.Id.extension] = obj;
                    else
                        _hospitalLocations.Add(obj.Id.extension, obj);
                    // vista site parent
                    if (!_cacheHospitalLocationsByVista[obj.VistaSite.Id.extension].Contains(obj.Id.extension))
                        _cacheHospitalLocationsByVista[obj.VistaSite.Id.extension].Add(obj.Id.extension);
                }
                finally { hlCacheLock.ExitWriteLock(); }
            }
            obj = null;
        }

        private void FillHospitalLocationsByVistaCache(II vistaId)
        {
            hlCacheLock.EnterWriteLock();
            try
            {
                if (!_cacheHospitalLocationsByVista.ContainsKey(vistaId.extension))
                {
                    List<HospitalLocation> hospitalLocations = _EISWrapper.GetPlaces(new HospitalLocation(), vistaId).Cast<HospitalLocation>().ToList();

                    if (_loadedHospitalLocationsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
                        _loadedHospitalLocationsCacheByVista.Remove(vistaId.extension);
                    _cacheHospitalLocationsByVista.Add(vistaId.extension, new List<string>());

                    foreach (HospitalLocation hl in hospitalLocations)
                    {
                        // hospital location
                        if (_hospitalLocations.ContainsKey(hl.Id.extension))
                            _hospitalLocations[hl.Id.extension] = hl;
                        else
                            _hospitalLocations.Add(hl.Id.extension, hl);
                        // vista site parent
                        _cacheHospitalLocationsByVista[vistaId.extension].Add(hl.Id.extension);
                    }
                    hospitalLocations = null;
                    if (!_loadedHospitalLocationsCacheByVista.Contains(vistaId.extension, StringComparer.InvariantCultureIgnoreCase))
                        _loadedHospitalLocationsCacheByVista.Add(vistaId.extension);                    
                }
            }
            finally { hlCacheLock.ExitWriteLock(); }
        }

        internal void RemoveCacheBed(II id)
        {
            bedCacheLock.EnterWriteLock();
            try
            {
                // vista site parents
                foreach (string key in _cacheBedsByVista.Keys)
                {
                    if (_cacheBedsByVista[key].Remove(id.extension))
                        break;                    
                }
                // division parents
                foreach (string key in _cacheBedsByDivision.Keys)                
                    _cacheBedsByDivision[key].Remove(id.extension);                
                //ward parents
                foreach (string key in _cacheBedsByWard.Keys)
                    _cacheBedsByWard[key].Remove(id.extension);                
                // bed
                if (_beds.ContainsKey(id.extension))
                    _beds.Remove(id.extension);                
            }
            finally { bedCacheLock.ExitWriteLock(); }
        }

        internal void RemoveCacheWard(II id)
        {
            wardCacheLock.EnterWriteLock();
            try
            {
                // vista site parent
                foreach (string key in _cacheWardsByVista.Keys)
                {
                    if (_cacheWardsByVista[key].Remove(id.extension))                    
                        break;                    
                }
                // division parents
                foreach (string key in _cacheWardsByDivision.Keys)
                    _cacheWardsByDivision[key].Remove(id.extension);
                // beds children
                if (_cacheBedsByWard.ContainsKey(id.extension))
                    _cacheBedsByWard.Remove(id.extension);
                // ward
                if (_wards.ContainsKey(id.extension))
                    _wards.Remove(id.extension);
            }
            finally { wardCacheLock.ExitWriteLock(); }
        }

        internal void RemoveCacheDivision(II id)
        {
            cacheLock.EnterWriteLock();
            try
            {
                // facility parent
                foreach (string key in _cacheDivisionsByFacility.Keys)
                {
                    if (_cacheDivisionsByFacility[key].Remove(id.extension))
                        break;
                }
                // beds children
                if (_cacheBedsByDivision.ContainsKey(id.extension))                
                    _cacheBedsByDivision.Remove(id.extension);                
                // wards children
                if (_cacheWardsByDivision.ContainsKey(id.extension))                
                    _cacheWardsByDivision.Remove(id.extension);                
                // division
                if (_divisions.ContainsKey(id.extension))
                    _divisions.Remove(id.extension);                
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        internal void RemoveCacheHospitalLocation(II id)
        {
            hlCacheLock.EnterWriteLock();
            try
            {
                // vista site parents
                foreach (string key in _cacheHospitalLocationsByVista.Keys)
                {
                    if (_cacheHospitalLocationsByVista[key].Remove(id.extension))
                        break;
                }                
                // hospital location
                if (_hospitalLocations.ContainsKey(id.extension))
                    _hospitalLocations.Remove(id.extension);
            }
            finally { hlCacheLock.ExitWriteLock(); }
        }

        #endregion

        internal void FlushCache()
        {
            bedCacheLock.EnterWriteLock();
            try
            {
                if (_beds != null)
                    _beds.Clear();
                if (_cacheBedsByWard != null)
                    _cacheBedsByWard.Clear();
                if (_cacheBedsByDivision != null)
                    _cacheBedsByDivision.Clear();
                if (_cacheBedsByVista != null)
                    _cacheBedsByVista.Clear();
                _loadedBedsCacheByVista.Clear();
            }
            finally { bedCacheLock.ExitWriteLock(); }

            wardCacheLock.EnterWriteLock();
            try
            {
                if (_wards != null)
                    _wards.Clear();
                if (_cacheWardsByDivision != null)
                    _cacheWardsByDivision.Clear();
                if (_cacheWardsByVista != null)
                    _cacheWardsByVista.Clear();
            }
            finally { wardCacheLock.ExitWriteLock(); }

            hlCacheLock.EnterWriteLock();
            try
            {
                if (_hospitalLocations != null)
                    _hospitalLocations.Clear();
                if (_cacheHospitalLocationsByVista != null)
                    _cacheHospitalLocationsByVista.Clear();
                _loadedHospitalLocationsCacheByVista.Clear();
            }
            finally { hlCacheLock.ExitWriteLock(); }

            cacheLock.EnterWriteLock();
            try
            {
                if (_divisions != null)
                    _divisions.Clear();
                if (_facilities != null)
                    _facilities.Clear();
                if (_vistaSites != null)
                    _vistaSites.Clear();
                if (_visns != null)
                    _visns.Clear();
                if (_regions != null)
                    _regions.Clear();
                if (_cacheDomains != null)
                    _cacheDomains.Clear();

                if (_cacheDivisionsByFacility != null)
                    _cacheDivisionsByFacility.Clear();

                if (_cacheFacilitiesByVista != null)
                    _cacheFacilitiesByVista.Clear();

                if (_cacheVistaSitesByVisn != null)
                    _cacheVistaSitesByVisn.Clear();

                if (_cacheVisnsByRegion != null)
                    _cacheVisnsByRegion.Clear();
            }
            finally { cacheLock.ExitWriteLock(); }
        }
    }
}
