﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using BMS.Utils;
using FC = BMS.Facade.Data;
using SC = BMS.DataContracts;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.WF;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.VistaWorker.Exceptions;
using BMS.ServicesWrapper.BMService;

namespace BMS.VistaWorker.Writer.Concrete.WF
{
    /// <summary>
    /// The patient movement writer.
    /// </summary>
    class PatientMovementWriter : BaseWorkFlowWriter<PatientMovement>
    {
        /// <summary>
        /// Sends the event.
        /// </summary>
        /// <param name="entity">The entity.</param>
        protected override void SendEvent(PatientMovement entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                decimal transaction = 0;
                decimal.TryParse(entity.Transaction, out transaction);

                if (transaction > 0)
                {
                    string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                    IList<CD> movementTypeList = EVS.GetCodes(new ServicesWrapper.EVS.CodeFilterParameters() { VocabularyDomain = FC.Util.Vocabulary.TypeOfMovement.ToString() });                    
                    FC.Ward ward = EIS.GetWard(entity.WardLocationId, VistaSite.Id);
                    FC.Bed bed = EIS.GetBed(entity.RoomBedId, VistaSite.Id);

                    Dictionary<string, string> enteredByNames = null;
                    if (entity.EnteredBy != null)
                        enteredByNames = Utilities.SplitPersonFullName(entity.EnteredBy.Name);

                    //admission
                    if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MAX)
                    {
                        SC.AdmissionEvent wfAdmission = new SC.AdmissionEvent()
                        {
                            Id = new II(domainId, null),
                            PatientId = MakeWfPatient(entity.Patient),
                            EnteredDate = entity.EnteredOnDateTime,
                            Ien = entity.IEN,
                            BedId = (bed != null) ? bed.Id : null,
                            WardId = (ward != null) ? ward.Id : null,
                            EnteredById = null,
                            AdmissionDate = entity.DateTime,
                            MovementType = movementTypeList.Where(a => a.code == VistaSite.Code + "_" + entity.TypeOfMovementId.ToString()).FirstOrDefault(),
                            VistaSiteId = (VistaSite != null) ? VistaSite.Id : null
                        };

                        ConvertToUtc(wfAdmission);
                        BMSFactory.BedManagerOperationsClientWindows.CreateAdmissionEvent(wfAdmission);
                        string msj = string.Format("Create admission event for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);                        
                    }

                    //transfer
                    if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MAX)                        
                    {
                        SC.MovementEvent wfMovement = new SC.MovementEvent()
                        {
                            Id = new II(domainId, null),
                            PatientId = MakeWfPatient(entity.Patient),
                            EnteredDate = entity.EnteredOnDateTime,
                            Ien = entity.IEN,
                            BedId = (bed != null) ? bed.Id : null,
                            WardId = (ward != null) ? ward.Id : null,
                            EnteredById = null,
                            MovementDate = entity.DateTime,
                            MovementType = movementTypeList.Where(a => a.code == VistaSite.Code + "_" + entity.TypeOfMovementId.ToString()).FirstOrDefault(),
                            VistaSiteId = (VistaSite != null) ? VistaSite.Id : null
                        };
                        ConvertToUtc(wfMovement);
                        BMSFactory.BedManagerOperationsClientWindows.CreateMovementEvent(wfMovement);
                        string msj = string.Format("Create movement event for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);                        
                    }

                    // specialty transfer
                    if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MAX)
                    {
                        SC.SpecialtyTransfer wfSpecialtyTransfer = new SC.SpecialtyTransfer()
                        {
                            Id = new II(domainId, null),
                            Ien = entity.IEN,
                            PatientId = MakeWfPatient(entity.Patient),                                                        
                            VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                            DateTime = entity.DateTime,
                            TypeOfMovement = movementTypeList.Where(a => a.code == VistaSite.Code + "_" + entity.TypeOfMovementId.ToString()).FirstOrDefault(),
                            EnteredDate = entity.EnteredOnDateTime,
                            EnteredById = null
                        };
                        ConvertToUtc(wfSpecialtyTransfer);                        
                        BMSFactory.BedManagerOperationsClientWindows.CreateSpecialtyTransfer(wfSpecialtyTransfer);
                        string msj = string.Format("Create specialty transfer for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);                        
                    }

                    //discharge
                    if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MAX)
                    {
                        SC.DischargeEvent wfDischarge = new SC.DischargeEvent()
                        {
                            Id = new II(domainId, null),
                            PatientId = MakeWfPatient(entity.Patient),                            
                            EnteredDate = entity.EnteredOnDateTime,
                            Ien = entity.IEN,
                            BedId = (bed != null) ? bed.Id : null,
                            WardId = (ward != null) ? ward.Id : null,
                            EnteredById = null,
                            DischargedDate = entity.DateTime,
                            MovementType = movementTypeList.Where(a => a.code == VistaSite.Code + "_" + entity.TypeOfMovementId.ToString()).FirstOrDefault(),
                            VistaSiteId = (VistaSite != null) ? VistaSite.Id : null
                        };
                        ConvertToUtc(wfDischarge);                        
                        BMSFactory.BedManagerOperationsClientWindows.CreateDischargeEvent(wfDischarge);
                        string msj = string.Format("Create discharge event for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);                        
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
