﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.ServicesWrapper.EVS;

namespace BMS.VistaWorker.Writer.Concrete.EVS
{
    /// <summary>
    /// A specialized concept writer for the Specialty vocabulary.
    /// </summary>
    class SpecialtyWriter : BaseEvsWriter<Specialty>
    {
        /// <summary>
        /// Gets the code system.
        /// </summary>
        protected override string CodeSystem
        {
            get { return Constants.VistaSpecialtyCodeSystem; }
        }
        /// <summary>
        /// Gets the name of the code system.
        /// </summary>
        /// <value>
        /// The name of the code system.
        /// </value>
        protected override string CodeSystemName
        {
            get { return Util.Vocabulary.VistaSpecialty.ToString(); }
        }

        /// <summary>
        /// Gets the concept.
        /// </summary>
        /// <param name="concept">The EVS concept.</param>
        /// <returns></returns>
        protected override CD GetBmsEntity(Specialty concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<CD> cds = EVS.GetCodes(new CodeFilterParameters() { VocabularyDomain = CodeSystemName });
                string code = concept.IEN.ToString();
                return cds.FirstOrDefault(cd => cd.code == code);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the VistA entity to an existing EVS concept (for use in BMS).
        /// </summary>
        /// <param name="entity">The VistA entity.</param>
        /// <param name="concept">The EVS concept.</param>
        protected override void TranslateFromVistaToBMS(Specialty entity, CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                concept.code = entity.IEN.ToString();
                concept.codeSystem = CodeSystem;
                concept.codeSystemName = CodeSystemName;
                concept.displayName = entity.Name;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
