﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using BMS.Facade.Data;
using BMS.ServicesWrapper.BMService;
using BMS.Facade.Translators;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using InfoWorld.EVS.CTSMAPI;
using BMS.ServicesWrapper.EVS;
using BMS.ServicesWrapper.WF;

namespace BMS.VistaWorker.Writer.Concrete.EIS
{
    /// <summary>
    /// The room bed writer.
    /// </summary>
    class RoomBedWriter : BaseEisWriter<RoomBed, Bed>
    {
        RoomBed roomBedEntity = null;

        /// <summary>
        /// Gets the BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        protected override Bed GetBmsEntity(RoomBed entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EIS.GetBed(entity.IEN, VistaSite.Id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Adds the modifications for BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void TranslateFromVistaToBMS(RoomBed entity, Bed bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                roomBedEntity = entity;
                if (entity.WardsWhichCanAssign != null && entity.WardsWhichCanAssign.Count > 0)
                {
                    bmsEntity.WardList = new List<Ward>();
                    foreach (WardsWhichCanAssign ward in entity.WardsWhichCanAssign)
                        bmsEntity.WardList.Add(EIS.GetWard(ward.WardsWhichCanAssignId, VistaSite.Id));
                }
                bmsEntity.Ien = entity.IEN;
                bmsEntity.Name = entity.Name;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Inserts the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void InsertBmsEntity(Bed bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.CreateBed(bmsEntity);
                CreateBedUnavailable(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void UpdateBmsEntity(Bed bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.UpdateBed(bmsEntity);
                CreateBedUnavailable(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void DeleteBmsEntity(Bed bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.DeleteBed(bmsEntity.Id);
                roomBedEntity.CurrentlyOutOfService = false;
                CreateBedUnavailable(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CreateBedUnavailable(Bed bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (roomBedEntity != null)
                {
                    BedUnavailable bedUnavailable = null;
                    try
                    {
                        bedUnavailable = BMSFactory.BedManagerOperationsClientWindows.GetBedUnavailable(null, bmsEntity.Id).ToFacadeContract();
                    }
                    catch { }

                    if (roomBedEntity.CurrentlyOutOfService)
                    {
                        CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                        codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.UnavailableReason.ToString();
                        codeFilterParamWithProperties.Properties = new List<string>();
                        codeFilterParamWithProperties.Properties.Add("Type");
                        List<CDWithProperties> reasonList = EVS.GetCodesWithProperties(codeFilterParamWithProperties).ToList();
                        List<CD> typeList = EVS.GetCodes(new CodeFilterParameters() { VocabularyDomain = Util.Vocabulary.AdminURType.ToString() }).ToList();
                        CDWithProperties reason = reasonList.Where(a => a.Properties.Where(b => b.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text == Constants.BED_OUT_OF_SERVICE_TYPE_CODE
                                                        && a.code.Equals(Constants.NATIONAL_BED_OUT_OF_SERVICE_CODE)).FirstOrDefault();
                        if (reason == null)
                            reason = reasonList.Where(a => a.Properties.Where(b => b.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text == Constants.BED_OUT_OF_SERVICE_TYPE_CODE
                                                        && a.code.StartsWith(Constants.NATIONAL_VOCABULARY_CODE_PREFIX)).FirstOrDefault();
                        string typeCode = reason.Properties.Where(a => a.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text;

                        if (bedUnavailable == null)
                        {
                            //create                                                       
                            string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();

                            bedUnavailable = new BedUnavailable()
                            {
                                Bed = (bmsEntity != null) ? new Place() { Id = bmsEntity.Id, Name = bmsEntity.Name } : null,
                                CanceledBy = null,
                                CanceledDate = null,
                                CreatedBy = string.Empty,
                                CreationDate = roomBedEntity.OutOfServiceDate,
                                EditedBy = null,
                                EditedDate = null,
                                Id = new II(domainId, null),
                                Parent = null,
                                Patient = null,
                                Reason = new CD(reason.code, reason.codeSystem, reason.codeSystemName, null, reason.displayName, null, null, null),
                                Type = typeList.Where(a => a.code == typeCode).FirstOrDefault(),
                                WardList = bmsEntity.WardList,
                                OutOfServiceVistA = true,
                                VistaSite = VistaSite
                            };
                            if (roomBedEntity.WardsWhichCanAssign != null && roomBedEntity.WardsWhichCanAssign.Count > 0)
                            {
                                bedUnavailable.WardList = new List<Ward>();
                                foreach (WardsWhichCanAssign ward in roomBedEntity.WardsWhichCanAssign)
                                    bedUnavailable.WardList.Add(EIS.GetWard(ward.WardsWhichCanAssignId, VistaSite.Id));
                            }
                            ConvertToUtc(bedUnavailable);
                            WFFactory.BedUnavailableWorkflowClientWindows.CreateBedUnavailable(bedUnavailable);
                        }
                        else
                        {
                            //update
                            bedUnavailable.EditedBy = string.Empty;
                            bedUnavailable.EditedDate = DateTime.UtcNow;
                            bedUnavailable.Reason = new CD(reason.code, reason.codeSystem, reason.codeSystemName, null, reason.displayName, null, null, null);
                            bedUnavailable.Type = typeList.Where(a => a.code == typeCode).FirstOrDefault();
                            bedUnavailable.OutOfServiceVistA = true;
                            if (roomBedEntity.WardsWhichCanAssign != null && roomBedEntity.WardsWhichCanAssign.Count > 0)
                            {
                                bedUnavailable.WardList = new List<Ward>();
                                foreach (WardsWhichCanAssign ward in roomBedEntity.WardsWhichCanAssign)
                                    bedUnavailable.WardList.Add(EIS.GetWard(ward.WardsWhichCanAssignId, VistaSite.Id));
                            }
                            WFFactory.BedUnavailableWorkflowClientWindows.UpdateBedUnavailable(bedUnavailable);
                        }
                    }
                    else
                    {
                        if (bedUnavailable != null)
                        {
                            //cancel
                            bedUnavailable.CanceledBy = string.Empty;
                            bedUnavailable.CanceledDate = DateTime.UtcNow;
                            bedUnavailable.OutOfServiceVistA = true;
                            if (roomBedEntity.WardsWhichCanAssign != null && roomBedEntity.WardsWhichCanAssign.Count > 0)
                            {
                                bedUnavailable.WardList = new List<Ward>();
                                foreach (WardsWhichCanAssign ward in roomBedEntity.WardsWhichCanAssign)
                                    bedUnavailable.WardList.Add(EIS.GetWard(ward.WardsWhichCanAssignId, VistaSite.Id));
                            }
                            WFFactory.BedUnavailableWorkflowClientWindows.CancelBedUnavailable(bedUnavailable);
                        }
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
