﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.EVS;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.VistaWorker.Exceptions;
using BMS.VistaWorker.Writer.Abstract;
using BMS.VistaWorker.Writer.Concrete.CatchExceptions;

namespace BMS.VistaWorker.Writer.Concrete
{
    /// <summary>
    /// Abstract base writer for eis entities.
    /// </summary>
    /// <typeparam name="T">The entity</typeparam>
    /// <typeparam name="B">The bms entity.</typeparam>
    abstract class BaseEisWriter<T, B> : BaseEntityWriter<T>
        where T : IEntity
        where B : class , new()
    {
        public BaseEisWriter() : base(CatchExceptionFactory.CatchEisExceptions) { }

        protected BaseEisWriter(ICatchExceptions catchException) : base(catchException) { }

        /// <summary>
        /// Gets the BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        protected abstract B GetBmsEntity(T entity);

        /// <summary>
        /// Translates the VistA entity to an existing BMS entity.
        /// </summary>
        /// <param name="entity">The VistA entity.</param>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected abstract void TranslateFromVistaToBMS(T entity, B bmsEntity);

        /// <summary>
        /// Inserts the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected abstract void InsertBmsEntity(B bmsEntity);

        /// <summary>
        /// Updates the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected abstract void UpdateBmsEntity(B bmsEntity);

        /// <summary>
        /// Deletes the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected abstract void DeleteBmsEntity(B bmsEntity);


        protected override void OnInsert(T entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                OnUpdate(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        protected override void OnUpdate(T entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                B bmsEntity;
                bool exists = TranslateEntity(entity, out bmsEntity);
                if (exists)
                {
                    UpdateBmsEntity(bmsEntity);
                    Logger.LogInformation(string.Format("Update {0} with success", entity));
                }
                else
                {
                    InsertBmsEntity(bmsEntity);
                    Logger.LogInformation(string.Format("Insert {0} with success", entity));
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        protected override void OnDelete(T entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                B bmsEntity = GetBmsEntity(entity);
                if (bmsEntity != null)
                {
                    DeleteBmsEntity(bmsEntity);
                    Logger.LogInformation(string.Format("Delete {0} with success", entity));
                }
                else
                {
                    Logger.LogInformation(string.Format("{0} not found for delete ", entity));
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool TranslateEntity(T entity, out B bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bmsEntity = GetBmsEntity(entity);
                bool exists = bmsEntity != null;
                if (!exists)
                    bmsEntity = new B();
                TranslateFromVistaToBMS(entity, bmsEntity);
                return exists;

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


    }
}
