﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Mdws2.QuerySvcService;
using BMS.VistaIntegration.Mdws2.Implementation.EIS;
using BMS.VistaIntegration.Mdws2.Implementation.EVS;
using BMS.VistaIntegration.Mdws2.Implementation.WF;
using BMS.VistaIntegration.Mdws2.Implementation;
using System.ServiceModel;
using BMS.VistaIntegration.Exceptions;
using BMS.VistaIntegration.VistA;


namespace BMS.VistaIntegration.Mdws2
{
    public class MdwsVistASession : IVistASession
    {
        private QuerySvcSoapClient _client;
        public VistASite VistASite { get; private set; }
        private bool _isOpen = false;
        private bool _isDisposable = false;

        private void CheckIsOpen()
        {
            if (_isDisposable)
                throw new ObjectDisposedException("MdwsVistASession");
            if (!_isOpen)
                throw new InvalidOperationException("VistA session is not open");
        }

        private void OnFault(FaultTO fault)
        {

            if (fault.message == "That site id is in use")
                return;
            if (TryAgain(fault))
                throw new VistAConnectionException("Try again later.");
            throw new VistAConnectionException(fault.message);

        }

        private bool TryAgain(FaultTO fault)
        {
            return fault.message == "The remote procedure XUS INTRO MSG is not registered to the option XUS SIGNON.";
        }

        private DataSourceArray Connect(VistASite site)
        {
            var connectDto = _client.connect(VistASite.Id);
            if (connectDto.fault != null && TryAgain(connectDto.fault))
            {
                System.Threading.Thread.Sleep(TimeSpan.FromSeconds(1));
                connectDto = _client.connect(VistASite.Id);
            }
            return connectDto;
        }

        public void Open(VistAConnectionInfo vistAConnection)
        {
            if (_isOpen)
                throw new InvalidOperationException("VistA session is already open");
            if (_isDisposable)
                throw new ObjectDisposedException("MdwsVistASession");
            if (string.IsNullOrEmpty(vistAConnection.AccessCode))
                throw new VistAConnectionException("AccessCode is empty");

            if (string.IsNullOrEmpty(vistAConnection.VerifyCode))
                throw new VistAConnectionException("VerifyCode is empty");

            this.VistASite = vistAConnection.VistASite;
            VistAConnectionString connection = Utilities.MakeVistAConnection(vistAConnection.ConnectionString);
            _client = new QuerySvcSoapClient();
            _client.Endpoint.Behaviors.Add(new MdwsManagerEndpointBehavior());
            try
            {
                var addDataSourceDto = _client.addDataSource(VistASite.Id, VistASite.Name, connection.Source, connection.Port, connection.Modality, connection.Protocol, "999999999");
                if (addDataSourceDto.fault != null)
                    OnFault(addDataSourceDto.fault);
                var connectDto = Connect(VistASite);
                if (connectDto.fault != null)
                    OnFault(connectDto.fault);

                var loginDto = _client.login(vistAConnection.AccessCode, vistAConnection.VerifyCode, "");
                if (loginDto.fault != null)
                {
                    _client.disconnect();
                    throw new VistAConnectionException(loginDto.fault.message);
                }
            }

            catch (CommunicationException e) { throw new VistAConnectionException(e); }
            catch (TimeoutException e) { throw new VistAConnectionException(e); }
            _isOpen = true;
        }

        public void RunClientAction(Action<QuerySvcSoap> action)
        {
            try
            {
                CheckIsOpen();
                action(_client);
            }
            catch (CommunicationException e)
            {
                throw new VistAConnectionException(e);
            }
            catch (TimeoutException e)
            {
                throw new VistAConnectionException(e);
            }
        }

        public PatientMovement GetPatientMovementByIen(string ien)
        {
            return null;
        }

        public void Close()
        {
            if (_isDisposable)
                return;
            _isDisposable = true;
            _isOpen = false;
            if (_client == null)
                return;

            if (_client.State == CommunicationState.Faulted)
            {
                _client.Abort();
            }
            else
            {
                try
                {
                    _client.disconnect();
                    _client.Close();
                }
                catch (Exception)
                {
                    _client.Abort();
                }
            }
        }

        public IVistAQuery MakeQuery()
        {
            return new MdwsVistAQuery(this);
        }

        public void Dispose()
        {
            Close();
        }
    }
}
