﻿using BMS.MT;
using System;
using BMS.Utils;
using System.Collections.Generic;
using InfoWorld.HL7.ITS;
using BMS.FaultContracts;
using System.ServiceModel;
using BMS.ServiceImplementation.Utils;
using BMS.DataContracts;
using BMS.ServiceImplementation.Properties;

namespace BMS.Translators
{
    public static class TranslatorElement
    {
        public static void CopyTo(this DataContracts.Element element, ConfigBO configBO)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                configBO.CONFIG_KEY.Value = element.Key;
                configBO.FRIENDLY_NAME.Value = element.FriendlyName;

                if (element.Value is INT)
                {
                    configBO.CONFIG_VALUE.Value = (element.Value as INT).value;
                    configBO.VALUE_TYPE.Value = BasicType.INT.ToString();
                }
                else if (element.Value is BL)
                {
                    configBO.CONFIG_VALUE.Value = (element.Value as BL).value.ToString();
                    configBO.VALUE_TYPE.Value = BasicType.BL.ToString();
                }
                else if (element.Value is REAL)
                {
                    configBO.CONFIG_VALUE.Value = (element.Value as REAL).value;
                    configBO.VALUE_TYPE.Value = BasicType.REAL.ToString();
                }
                else if (element.Value is ST)
                {
                    configBO.CONFIG_VALUE.Value = (element.Value as ST).text;
                    configBO.VALUE_TYPE.Value = BasicType.ST.ToString();
                }
                else if (element.Value is TS)
                {
                    configBO.CONFIG_VALUE.Value = (element.Value as TS).value;
                    configBO.VALUE_TYPE.Value = BasicType.TS.ToString();
                }
                else if (element.Value is CD)
                {
                    configBO.VALUE_TYPE.Value = BasicType.EVS.ToString();

                    CD newCD = element.Value as CD;
                    ConfigEvsBO configEvs = null;
                    if (configBO.ConfigEvsList != null && configBO.ConfigEvsList.Count == 1)
                        configEvs = (ConfigEvsBO)configBO.ConfigEvsList[0];
                    else
                        configEvs = (ConfigEvsBO)configBO.ConfigEvsList.AddNew();
                    configEvs.CODE.Value = newCD.code;
                    configEvs.CODE_SYSTEM.Value = newCD.codeSystem;
                    configEvs.CODE_SYSTEM_NAME.Value = newCD.codeSystemName;
                    configEvs.DISPLAY_NAME.Value = newCD.displayName;
                }
                else if (element.Value is EntityIdentification)
                {
                    configBO.VALUE_TYPE.Value = BasicType.EIS.ToString();

                    EntityIdentification newEis = element.Value as EntityIdentification;
                    ConfigEisBO configEis = null;
                    if (configBO.ConfigEisList != null && configBO.ConfigEisList.Count == 1)
                        configEis = (ConfigEisBO)configBO.ConfigEisList[0];
                    else
                        configEis = (ConfigEisBO)configBO.ConfigEisList.AddNew();
                    configEis.ENTITY_ID.Value = newEis.EntityIdentifier;
                    configEis.ENTITY_NAME.Value = newEis.EntityName;
                    configEis.ENTITY_TYPE_ID.Value = newEis.EntityTypeIdentifier;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void CopyTo(this ConfigBO configBO, DataContracts.Element element)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                element.Key = configBO.CONFIG_KEY.Value;
                element.FriendlyName = configBO.FRIENDLY_NAME.Value;

                BasicType valType;
                try
                {
                    valType = (BasicType)Enum.Parse(typeof(BasicType), configBO.VALUE_TYPE.Value);
                }
                catch (ArgumentException)
                {
                    throw new FaultException<ConfigValueException>(
                        ExceptionFactory.CreateConfigValueException(configBO, Resources.InvalidValueTypeException));
                }

                switch (valType)
                {
                    case BasicType.BL:
                        try
                        {
                            BL boolValue = new BL();
                            boolValue.value = bool.Parse(configBO.CONFIG_VALUE.Value);
                            element.Value = boolValue;
                        }
                        catch (ArgumentException)
                        {
                            throw new FaultException<ConfigValueException>(
                                ExceptionFactory.CreateConfigValueException(configBO, Resources.InvalidBooleanValueException));
                        }
                        break;

                    case BasicType.INT:
                        INT intValue = new INT();
                        intValue.value = configBO.CONFIG_VALUE.Value;

                        element.Value = intValue;
                        break;

                    case BasicType.ST:
                        element.Value = new ST(configBO.CONFIG_VALUE.Value);
                        break;

                    case BasicType.REAL:
                        REAL real = new REAL();
                        real.value = configBO.CONFIG_VALUE.Value;

                        element.Value = real;
                        break;

                    case BasicType.TS:
                        TS dateTimeValue = new TS();
                        dateTimeValue.value = configBO.CONFIG_VALUE.Value;

                        element.Value = dateTimeValue;
                        break;

                    case BasicType.EVS:
                        if (configBO.ConfigEvsList.Count != 1)
                        {
                            throw new FaultException<ConfigValueException>(
                                ExceptionFactory.CreateConfigValueException(configBO, Resources.CDReferenceException));
                        }
                        else
                        {
                            CD code = new CD();
                            code.code = configBO.ConfigEvsList[0].CODE.Value;
                            code.codeSystem = configBO.ConfigEvsList[0].CODE_SYSTEM.Value;
                            code.codeSystemName = configBO.ConfigEvsList[0].CODE_SYSTEM_NAME.Value;
                            code.displayName = configBO.ConfigEvsList[0].DISPLAY_NAME.Value;

                            element.Value = code;
                        }
                        break;

                    case BasicType.EIS:
                        if (configBO.ConfigEisList.Count != 1)
                        {
                            throw new FaultException<ConfigValueException>(
                                ExceptionFactory.CreateConfigValueException(configBO, Resources.EntityIdentificationReferenceException));
                        }
                        else
                        {
                            EntityIdentification ei = new EntityIdentification();
                            ei.DomainIdentifier = configBO.DOMAIN_ID.Value;
                            ei.EntityIdentifier = configBO.ConfigEisList[0].ENTITY_ID.Value;
                            ei.EntityName = configBO.ConfigEisList[0].ENTITY_NAME.Value;
                            ei.EntityTypeIdentifier = configBO.ConfigEisList[0].ENTITY_TYPE_ID.Value;

                            element.Value = ei;
                        }
                        break;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
