﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using DC = BMS.DataContracts;
using BMS.ServicesWrapper.EVS;
using InfoWorld.EVS.CTSMAPI;
using BMS.Utils;
using InfoWorld.HL7.ITS;

namespace BMS.Facade.Translators
{
    public static class BMOperationsTranslator
    {
        #region From Data Contracts to Facade

        internal static Diversion ToFacadeContract(this DC.Diversion serviceContract, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Diversion facadeContract = new Diversion()
                {
                    Id = serviceContract.Id,
                    Facility = new Place { Id = facility.Id, Name = facility.Name },
                    Status = serviceContract.Active == true ? Constants.Yes : Constants.No,
                    StartDate = serviceContract.StartDate,
                    EndDate = serviceContract.EndDate,
                    EnteredBy = serviceContract.EnteredBy,
                    EnteredDate = serviceContract.EnteredDate,
                    Comments = serviceContract.Comments == null ? string.Empty : serviceContract.Comments,
                    LastEditBy = serviceContract.LastEditBy,
                    Location = serviceContract.Location
                };

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static Transfer ToFacadeContract(this DC.Transfer serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Transfer facadeContract = new Transfer()
                {
                    CanceledDate = serviceContract.CanceledDate,
                    Comment = serviceContract.Comment,
                    Contract = serviceContract.Contract,
                    CreationDate = serviceContract.CreationDate,
                    CurrentLocation = serviceContract.CurrentLocation,
                    Diagnosis = serviceContract.Diagnosis,
                    DischargeComment = serviceContract.DischargeComment,
                    Disposition = serviceContract.Disposition,
                    DispositionDate = serviceContract.DispositionDate,
                    Era = serviceContract.Era,
                    FacilityFrom = (serviceContract.FacilityId == null) ? null :
                        new Facility
                        {
                            Id = serviceContract.FacilityId,
                            VistaSite = new VistaSite()
                            {
                                Id = serviceContract.VistaSiteId,
                                Visn = new Visn() { Number = serviceContract.FacilityVISN, Region = new Region() { Number = int.Parse(serviceContract.FacilityRegion) } }
                            }
                        },
                    FacilityTo = (serviceContract.AdmissionFacilityId == null) ? null : new Facility { Id = serviceContract.AdmissionFacilityId },
                    Id = serviceContract.Id,
                    Parent = (serviceContract.Parent == null) ? null : serviceContract.Parent.ToFacadeContract(),
                    PatientId = serviceContract.PatientId,
                    Speciality = serviceContract.Speciality,
                    AdmissionDate = serviceContract.AdmissionDate,
                    EnteredBy = serviceContract.EnteredBy,
                    Edited = serviceContract.Edited,
                    LastEditBy = serviceContract.LastEditBy,
                    RequestedDate = serviceContract.RequestedDate,
                    IsNationalWaitList = serviceContract.IsNationalWaitList
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static WaitingListItem ToFacadeContract(this DC.WaitingListItem serviceContract, Facility facility, Patient patient, Bed bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingListItem facadeContract = new WaitingListItem()
                {
                    Authorization = serviceContract.Authorization,
                    CanceledDate = serviceContract.CanceledDate,
                    CommentFee = serviceContract.CommentFee,
                    Contract = serviceContract.Contract,
                    CreationDate = serviceContract.CreationDate,
                    Facility = facility,
                    FeeDisposition = serviceContract.FeeDisposition,
                    FlowId = serviceContract.FlowId,
                    Id = serviceContract.Id,
                    IEN = serviceContract.IEN,
                    Parent = serviceContract.Parent.ToFacadeContract(),
                    Patient = patient,
                    AdmittingDiagnosis = patient != null ? patient.AdmittingDiagnosis : string.Empty,
                    Problem = serviceContract.Problem,
                    ReasonFee = serviceContract.ReasonFee,
                    RemovedDate = serviceContract.RemovedDate,
                    RequestedBed = bed != null ? new Place { Id = bed.Id, Name = bed.Name } : null,
                    RequestedBedDate = serviceContract.RequestedBedDate,
                    RequestedDate = serviceContract.RequestedDate,
                    ServiceReceive = serviceContract.ServiceReceive,
                    TypeOfBedWard = serviceContract.TypeOfBedWard,
                    WaitingArea = serviceContract.WaitingArea,
                    IsInHouse = serviceContract.IsInHouse,
                    LastEditBy = serviceContract.LastEditBy,
                    Comments = serviceContract.Comments,
                    IsBedInIsolation = serviceContract.IsBedInIsolation,
                    WaitTimeAlertDate = serviceContract.WaitTimeAlertDate,
                    IsCommunityLivingCenter = serviceContract.IsCommunityLivingCenter,
                    IsEvacuationPatient = serviceContract.IsEvacuationPatient,
                    EvacuationActualDate = serviceContract.EvacuationActualDate,
                    EvacuationComments = serviceContract.EvacuationComments,
                    EvacuationDestinationAddress = serviceContract.EvacuationDestinationAddress,
                    EvacuationDestinationCityState = serviceContract.EvacuationDestinationCityState,
                    EvacuationDestinationPOCTelephone = serviceContract.EvacuationDestinationPOCTelephone,
                    EvacuationDispositionStatus = serviceContract.EvacuationDispositionStatus,
                    EvacuationEnrouteMedicalAttendant = serviceContract.EvacuationEnrouteMedicalAttendant,
                    EvacuationEnrouteNonMedicalAttendant = serviceContract.EvacuationEnrouteNonMedicalAttendant,
                    EvacuationEquipSupplyRequirements = serviceContract.EvacuationEquipSupplyRequirements,
                    EvacuationExpectedDate = serviceContract.EvacuationExpectedDate,
                    EvacuationOriginatorPOCTelephone = serviceContract.EvacuationOriginatorPOCTelephone,
                    EvacuationReasonForDelay = serviceContract.EvacuationReasonForDelay,
                    EvacuationTransferEvacuateTo = serviceContract.EvacuationTransferEvacuateTo,
                    EvacuationTransportationCareLevel = serviceContract.EvacuationTransportationCareLevel,
                    EvacuationTransportationComments = serviceContract.EvacuationTransportationComments,
                    EvacuationTransportationProvider = serviceContract.EvacuationTransportationProvider,
                    EvacuationTransportationType = serviceContract.EvacuationTransportationType,
                    EvacuationEnrouteMedicalAttendantNumber = serviceContract.EvacuationEnrouteMedicalAttendantNumber,
                    EvacuationEnrouteNonMedicalAttendantNumber = serviceContract.EvacuationEnrouteNonMedicalAttendantNumber,
                    VistaSite = (serviceContract.VistaSiteId != null) ? FacadeManager.EntityInterface.GetVistaSite(serviceContract.VistaSiteId) : null
                };

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static Act ToFacadeContract(this DC.Act serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (serviceContract == null)
                    return null;
                Act facadeContract = new Act()
                {
                    CanceledDate = serviceContract.CanceledDate,
                    CreationDate = serviceContract.CreationDate,
                    Id = serviceContract.Id,
                    Parent = (serviceContract.Parent == null) ? null : serviceContract.Parent.ToFacadeContract(),
                    IEN = serviceContract.IEN,
                    FlowId = serviceContract.FlowId
                };

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static BedClean ToFacadeContract(this DC.BedCleaningOperation bedCleaningServiceContract, BedClean bedClean)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bedClean.Id = bedCleaningServiceContract.Id;
                bedClean.Bed = new Place { Id = bedCleaningServiceContract.BedId };
                bedClean.Ward = (bedCleaningServiceContract.WardId != null) ? new Ward { Id = bedCleaningServiceContract.WardId } : null;                
                bedClean.TypeOfClean = bedCleaningServiceContract.TypeOfClean;
                bedClean.ManualRequest = bedCleaningServiceContract.ManualRequest;
                bedClean.AcceptedBy = bedCleaningServiceContract.AcceptedBy;
                bedClean.AcceptedDate = bedCleaningServiceContract.AcceptedDate;
                bedClean.CompletedBy = bedCleaningServiceContract.CompletedBy;
                bedClean.CompletedDate = bedCleaningServiceContract.CompletedDate;
                bedClean.LastEditedBy = bedCleaningServiceContract.LastEditedBy;
                bedClean.LastEditDate = bedCleaningServiceContract.LastEditDate;
                bedClean.RequestedDate = bedCleaningServiceContract.RequestedDate;
                bedClean.Division = bedCleaningServiceContract.DivisionId != null ? new Division { Id = bedCleaningServiceContract.DivisionId } : null;
                bedClean.EMSNotify = bedCleaningServiceContract.EMSNotify;
                bedClean.SpecialInstructions = bedCleaningServiceContract.SpecialInstructions;
                bedClean.Comment = bedCleaningServiceContract.Comment;
                bedClean.CommentedBy = bedCleaningServiceContract.CommentedBy;
                bedClean.CommentedDate = bedCleaningServiceContract.CommentedDate;
                bedClean.VacatedDate = bedCleaningServiceContract.VacatedDate;
                bedClean.EventType = bedCleaningServiceContract.EventType;
                bedClean.EventIen = bedCleaningServiceContract.EventIen;
                bedClean.VistaSite = (bedCleaningServiceContract.VistaSiteId != null) ? FacadeManager.EntityInterface.GetVistaSite(bedCleaningServiceContract.VistaSiteId) : null;

                return bedClean;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static BedUnavailable ToFacadeContract(this DC.BedUnavailable dataContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Bed bed = null;
                if (dataContract.BedId != null)
                    bed = FacadeManager.BedInterface.GetBed(dataContract.BedId, dataContract.VistaSiteId);
                BedUnavailable facadeContract = new BedUnavailable()
                {
                    Bed = (bed != null) ? new Place() { Id = bed.Id, Name = bed.Name } : null,
                    CanceledBy = dataContract.CanceledBy,
                    CanceledDate = dataContract.CanceledDate,
                    CreatedBy = dataContract.CreatedBy,
                    CreationDate = dataContract.CreationDate,
                    Division = (dataContract.DivisionId != null) ? FacadeManager.EntityInterface.GetDivisionById(dataContract.DivisionId) : null,
                    EditedBy = dataContract.EditedBy,
                    EditedDate = dataContract.EditedDate,
                    FlowId = dataContract.FlowId,
                    Id = dataContract.Id,
                    IEN = dataContract.IEN,
                    Parent = null,
                    Patient = null,
                    Reason = dataContract.Reason,
                    Type = dataContract.Type,
                    OutOfServiceVistA = dataContract.IsUnavailableFromVistA,
                    VistaSite = (dataContract.VistaSiteId != null) ? FacadeManager.EntityInterface.GetVistaSite(dataContract.VistaSiteId) : null,
                    ExpectedCompletedDate = dataContract.ExpectedCompletedDate
                };
                if (dataContract.WardList != null && dataContract.WardList.Count > 0)
                {
                    facadeContract.WardList = new List<Ward>();
                    foreach (II id in dataContract.WardList)
                    {
                        facadeContract.WardList.Add(new Ward() { Id = id });
                    }
                }
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static IconDetail ToFacadeContract(this DC.IconDetail serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IconDetail facadeContract = new IconDetail()
                {
                    Active = serviceContract.Active,
                    Comment = serviceContract.Comment,
                    EnteredBy = serviceContract.EnteredBy,
                    EnteredDate = serviceContract.EnteredDate,
                    Facility = (serviceContract.FacilityId != null) ? FacadeManager.EntityInterface.GetFacility(serviceContract.FacilityId) : null,
                    FacilityActive = serviceContract.FacilityActive,
                    FacilityComment = serviceContract.FacilityComment,
                    FacilityDescription = serviceContract.FacilityDescription,
                    FacilityIconName = serviceContract.FacilityIconName,
                    FacilityMouseOverText = serviceContract.FacilityMouseOverText,
                    FacilityOrderIndex = serviceContract.FacilityOrderIndex,
                    FacilityPatientOrRoomBed = serviceContract.FacilityPatientOrRoomBed,
                    Group = serviceContract.Group,
                    IconFlagId = serviceContract.IconFlagId,
                    IconName = serviceContract.IconName,
                    IconType = serviceContract.IconType,
                    Id = serviceContract.Id,
                    ImageBytes = serviceContract.ImageBytes,
                    ImageBytesInactive = serviceContract.ImageBytesInactive,
                    ImageName = serviceContract.ImageName,
                    LastEditBy = serviceContract.LastEditBy,
                    LastEditDate = serviceContract.LastEditDate,
                    LongDescription = serviceContract.LongDescription,
                    MouseOverText = serviceContract.MouseOverText,
                    OrderIndex = serviceContract.OrderIndex,
                    PatientOrRoomBed = serviceContract.PatientOrRoomBed,
                    ShortDescription = serviceContract.ShortDescription,
                    Subgroup = serviceContract.Subgroup,
                    Published = serviceContract.Published,
                    IsAssociated = serviceContract.IsAssociated
                };

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static EmsStaff ToFacadeContract(this DC.EmsStaff serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EmsStaff facadeContract = new EmsStaff()
                {
                    Id = serviceContract.Id,
                    Pin = serviceContract.Pin,
                    UserName = serviceContract.UserName,
                    Facility = (serviceContract.FacilityId != null) ? new Facility() { Id = serviceContract.FacilityId } : null
                };

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DischargeClinic ToFacadeContract(this DC.HospitalLocationAppointment serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DischargeClinic facadeContract = new DischargeClinic()
                {
                    Id = serviceContract.Id,
                    HospitalLocation = (serviceContract.HospitalLocationId != null) ? new HospitalLocation() { Id = serviceContract.HospitalLocationId } : null,
                    Facility = (serviceContract.FacilityId != null) ? new Facility() { Id = serviceContract.FacilityId } : null,
                    VistaSite = (serviceContract.VistaSiteId != null) ? new VistaSite() { Id = serviceContract.VistaSiteId } : null
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region From Facade to Data Contract

        public static DC.BedCleaningOperation ToDataContract(this BedClean facade)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.BedCleaningOperation dataContract = new DC.BedCleaningOperation
                {
                    Id = facade.Id,                    
                    EventIen = facade.EventIen,
                    BedId = facade.Bed.Id,                    
                    AcceptedBy = facade.AcceptedBy,
                    AcceptedDate = facade.AcceptedDate,
                    CompletedBy = facade.CompletedBy,
                    CompletedDate = facade.CompletedDate,
                    ManualRequest = facade.ManualRequest,
                    TypeOfClean = facade.TypeOfClean,
                    LastEditedBy = facade.LastEditedBy,
                    LastEditDate = facade.LastEditDate,
                    RequestedDate = facade.RequestedDate,
                    DivisionId = facade.Division != null ? facade.Division.Id : null,
                    VistaSiteId = facade.VistaSite.Id,
                    EMSNotify = facade.EMSNotify,
                    WardId = facade.Ward.Id,
                    SpecialInstructions = facade.SpecialInstructions,
                    Comment = facade.Comment,
                    CommentedBy = facade.CommentedBy,
                    CommentedDate = facade.CommentedDate,
                    VacatedDate = facade.VacatedDate,
                    EventType = facade.EventType
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.BedUnavailable ToDataContract(this BedUnavailable facade)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.BedUnavailable dataContract = new DC.BedUnavailable()
                {
                    BedId = (facade.Bed != null) ? facade.Bed.Id : null,
                    CanceledDate = facade.CanceledDate,
                    CanceledBy = facade.CanceledBy,
                    CreatedBy = facade.CreatedBy,
                    CreationDate = facade.CreationDate,
                    EditedBy = facade.EditedBy,
                    EditedDate = facade.EditedDate,
                    Id = facade.Id,
                    Parent = null,
                    PatientId = null,
                    Reason = facade.Reason,
                    Type = facade.Type,
                    DivisionId = (facade.Division != null) ? facade.Division.Id : null,
                    IsUnavailableFromVistA = facade.OutOfServiceVistA,
                    VistaSiteId = (facade.VistaSite != null) ? facade.VistaSite.Id : null,
                    ExpectedCompletedDate = facade.ExpectedCompletedDate
                };
                if (facade.WardList != null && facade.WardList.Count > 0)
                {
                    dataContract.WardList = new List<II>();
                    foreach (Ward ward in facade.WardList)
                        dataContract.WardList.Add(ward.Id);
                }
                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.BedUnavailableHistory ToDataContract(this BedUnavailableHistory facade)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.BedUnavailableHistory dataContract = new DC.BedUnavailableHistory()
                {
                    BedId = (facade.Bed != null) ? facade.Bed.Id : null,
                    CanceledDate = facade.CanceledDate,
                    CanceledBy = facade.CanceledBy,
                    CreatedBy = facade.CreatedBy,
                    CreationDate = facade.CreationDate,
                    EditedBy = facade.EditedBy,
                    EditedDate = facade.EditedDate,
                    Id = facade.Id,
                    Parent = null,
                    PatientId = null,
                    Reason = facade.Reason,
                    Type = facade.Type,
                    DivisionId = (facade.Division != null) ? facade.Division.Id : null,
                    VistaSiteId = (facade.VistaSite != null) ? facade.VistaSite.Id : null,
                    IsChangeOnlyComment = facade.IsChangedOnlyComment,
                    IsClearAll = facade.IsClearAll,
                    Comment = facade.Comment,
                    ExpectedCompletedDate = facade.ExpectedCompletedDate
                };
                if (facade.DivisionList != null && facade.DivisionList.Count > 0)
                {
                    dataContract.DivisionList = new List<II>();
                    foreach (Division division in facade.DivisionList)
                        dataContract.DivisionList.Add(division.Id);
                }
                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.Diversion ToDataContract(this Diversion facade)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.Diversion dataContract = new DC.Diversion()
                {
                    Id = facade.Id,
                    StartDate = facade.StartDate,
                    EndDate = facade.EndDate,
                    Comments = facade.Comments,
                    Active = facade.Status == Constants.Yes ? true : false,
                    FacilityId = (facade.Facility != null) ? facade.Facility.Id : null,
                    EnteredBy = facade.EnteredBy,
                    EnteredDate = facade.EnteredDate,
                    LastEditBy = facade.LastEditBy,
                    Location = facade.Location
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.Transfer ToDataContract(this Transfer facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.Transfer dataContract = new DC.Transfer()
                {
                    CanceledDate = facadeContract.CanceledDate,
                    Comment = facadeContract.Comment,
                    Contract = facadeContract.Contract,
                    CreationDate = facadeContract.CreationDate,
                    CurrentLocation = facadeContract.CurrentLocation,
                    Diagnosis = facadeContract.Diagnosis,
                    DischargeComment = facadeContract.DischargeComment,
                    Disposition = facadeContract.Disposition,
                    DispositionDate = facadeContract.DispositionDate,
                    Era = facadeContract.Era,
                    FacilityId = (facadeContract.FacilityFrom != null) ? facadeContract.FacilityFrom.Id : null,
                    AdmissionFacilityId = (facadeContract.FacilityTo == null) ? null : facadeContract.FacilityTo.Id,
                    Id = facadeContract.Id,
                    Parent = facadeContract.Parent.ToDataContract(),
                    PatientId = facadeContract.Patient.Id,
                    Speciality = facadeContract.Speciality,
                    AdmissionDate = facadeContract.AdmissionDate,
                    FacilityVISN = (facadeContract.FacilityFrom == null || facadeContract.FacilityFrom.VistaSite == null || facadeContract.FacilityFrom.VistaSite.Visn == null) ? string.Empty : facadeContract.FacilityFrom.VistaSite.Visn.Number,
                    FacilityRegion = (facadeContract.FacilityFrom == null || facadeContract.FacilityFrom.VistaSite == null || facadeContract.FacilityFrom.VistaSite.Visn == null || facadeContract.FacilityFrom.VistaSite.Visn.Region == null) ? string.Empty : facadeContract.FacilityFrom.VistaSite.Visn.Region.Number.ToString(),
                    EnteredBy = facadeContract.EnteredBy,
                    Edited = facadeContract.Edited,
                    LastEditBy = facadeContract.LastEditBy,
                    RequestedDate = facadeContract.RequestedDate,
                    VistaSiteId = (facadeContract.FacilityFrom != null && facadeContract.FacilityFrom.VistaSite != null) ? facadeContract.FacilityFrom.VistaSite.Id : null,
                    IsNationalWaitList = facadeContract.IsNationalWaitList
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.WaitingListItem ToDataContract(this WaitingListItem facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.WaitingListItem dataContract = new DC.WaitingListItem()
                {
                    Authorization = facadeContract.Authorization,
                    CanceledDate = facadeContract.CanceledDate,
                    CommentFee = facadeContract.CommentFee,
                    Contract = facadeContract.Contract,
                    CreationDate = facadeContract.CreationDate,
                    FacilityId = (facadeContract.Facility != null) ? facadeContract.Facility.Id : null,
                    FeeDisposition = facadeContract.FeeDisposition,
                    FlowId = facadeContract.FlowId,
                    Id = facadeContract.Id,
                    IEN = facadeContract.IEN,
                    Parent = facadeContract.Parent.ToDataContract(),
                    PatientId = facadeContract.Patient.Id,
                    Problem = facadeContract.Problem,
                    ReasonFee = facadeContract.ReasonFee,
                    RemovedDate = facadeContract.RemovedDate,
                    RequestedBedId = (facadeContract.RequestedBed != null) ? facadeContract.RequestedBed.Id : null,
                    RequestedBedDate = facadeContract.RequestedBedDate,
                    RequestedDate = facadeContract.RequestedDate,
                    ServiceReceive = facadeContract.ServiceReceive,
                    TypeOfBedWard = facadeContract.TypeOfBedWard,
                    WaitingArea = facadeContract.WaitingArea,
                    VistaSiteId = (facadeContract.VistaSite != null) ? facadeContract.VistaSite.Id : null,
                    IsInHouse = facadeContract.IsInHouse,
                    LastEditBy = facadeContract.LastEditBy,
                    Comments = facadeContract.Comments,
                    IsBedInIsolation = facadeContract.IsBedInIsolation,
                    WaitTimeAlertDate = facadeContract.WaitTimeAlertDate,
                    IsCommunityLivingCenter = facadeContract.IsCommunityLivingCenter,
                    IsEvacuationPatient = facadeContract.IsEvacuationPatient,
                    EvacuationActualDate = facadeContract.EvacuationActualDate,
                    EvacuationComments = facadeContract.EvacuationComments,
                    EvacuationDestinationAddress = facadeContract.EvacuationDestinationAddress,
                    EvacuationDestinationCityState = facadeContract.EvacuationDestinationCityState,
                    EvacuationDestinationPOCTelephone = facadeContract.EvacuationDestinationPOCTelephone,
                    EvacuationDispositionStatus = facadeContract.EvacuationDispositionStatus,
                    EvacuationEnrouteMedicalAttendant = facadeContract.EvacuationEnrouteMedicalAttendant,
                    EvacuationEnrouteNonMedicalAttendant = facadeContract.EvacuationEnrouteNonMedicalAttendant,
                    EvacuationEquipSupplyRequirements = facadeContract.EvacuationEquipSupplyRequirements,
                    EvacuationExpectedDate = facadeContract.EvacuationExpectedDate,
                    EvacuationOriginatorPOCTelephone = facadeContract.EvacuationOriginatorPOCTelephone,
                    EvacuationReasonForDelay = facadeContract.EvacuationReasonForDelay,
                    EvacuationTransferEvacuateTo = facadeContract.EvacuationTransferEvacuateTo,
                    EvacuationTransportationCareLevel = facadeContract.EvacuationTransportationCareLevel,
                    EvacuationTransportationComments = facadeContract.EvacuationTransportationComments,
                    EvacuationTransportationProvider = facadeContract.EvacuationTransportationProvider,
                    EvacuationTransportationType = facadeContract.EvacuationTransportationType,
                    EvacuationEnrouteMedicalAttendantNumber = facadeContract.EvacuationEnrouteMedicalAttendantNumber,
                    EvacuationEnrouteNonMedicalAttendantNumber = facadeContract.EvacuationEnrouteNonMedicalAttendantNumber
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.Act ToDataContract(this Act facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (facadeContract == null)
                    return null;
                DC.Act dataContract = new DC.Act()
                {
                    CanceledDate = facadeContract.CanceledDate,
                    CreationDate = facadeContract.CreationDate,
                    Id = facadeContract.Id,
                    Parent = (facadeContract.Parent == null) ? null : facadeContract.Parent.ToDataContract()
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public static DC.IconDetail ToDataContract(this IconDetail facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.IconDetail dataContract = new DC.IconDetail()
                {
                    Active = facadeContract.Active,
                    Comment = facadeContract.Comment,
                    EnteredBy = facadeContract.EnteredBy,
                    EnteredDate = facadeContract.EnteredDate,
                    FacilityId = (facadeContract.Facility != null) ? facadeContract.Facility.Id : null,
                    FacilityActive = facadeContract.FacilityActive,
                    FacilityComment = facadeContract.FacilityComment,
                    FacilityDescription = facadeContract.FacilityDescription,
                    FacilityIconName = facadeContract.FacilityIconName,
                    FacilityMouseOverText = facadeContract.FacilityMouseOverText,
                    FacilityOrderIndex = facadeContract.FacilityOrderIndex,
                    FacilityPatientOrRoomBed = facadeContract.FacilityPatientOrRoomBed,
                    Group = facadeContract.Group,
                    IconFlagId = facadeContract.IconFlagId,
                    IconName = facadeContract.IconName,
                    IconType = facadeContract.IconType,
                    Id = facadeContract.Id,
                    ImageBytes = facadeContract.ImageBytes,
                    ImageBytesInactive = facadeContract.ImageBytesInactive,
                    ImageName = facadeContract.ImageName,
                    LastEditBy = facadeContract.LastEditBy,
                    LastEditDate = facadeContract.LastEditDate,
                    LongDescription = facadeContract.LongDescription,
                    MouseOverText = facadeContract.MouseOverText,
                    OrderIndex = facadeContract.OrderIndex,
                    PatientOrRoomBed = facadeContract.PatientOrRoomBed,
                    ShortDescription = facadeContract.ShortDescription,
                    Subgroup = facadeContract.Subgroup,
                    Published = facadeContract.Published,
                    IsAssociated = facadeContract.IsAssociated
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.IconInfo ToDataContract(this IconInfo facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.IconInfo dataContract = new DC.IconInfo()
                {
                    Active = facadeContract.Active,
                    FacilityId = (facadeContract.Facility != null) ? facadeContract.Facility.Id : null,
                    Group = facadeContract.Group,
                    IconDescription = facadeContract.IconDescription,
                    IconId = facadeContract.IconId,
                    IconName = facadeContract.IconName,
                    ImageBytes = facadeContract.ImageBytes,
                    ImageBytesInactive = facadeContract.ImageBytesInactive,
                    MouseOverText = facadeContract.MouseOverText,
                    OrderIndex = facadeContract.OrderIndex,
                    PatientOrRoomBed = facadeContract.PatientOrRoomBed,
                    Subgroup = facadeContract.Subgroup,
                    IsAssociation = facadeContract.IsAssociation,
                    IconType = facadeContract.IconType,
                    Published = facadeContract.Published,
                    ImageName = facadeContract.ImageName
                };
                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.IconDetail ToIconDetailDataContract(this IconInfo facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.IconDetail dataContract = new DC.IconDetail()
                {
                    FacilityId = (facadeContract.Facility != null) ? facadeContract.Facility.Id : null,
                    FacilityActive = facadeContract.Active,
                    FacilityComment = null,
                    FacilityDescription = null,
                    FacilityIconName = null,
                    FacilityMouseOverText = null,
                    FacilityOrderIndex = facadeContract.OrderIndex,
                    FacilityPatientOrRoomBed = facadeContract.PatientOrRoomBed,
                    IconFlagId = facadeContract.IconId,
                    Id = facadeContract.IconId,
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.IconAssociation ToDataContract(this IconAssociation facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.IconAssociation dataContract = new DC.IconAssociation()
                {
                    BedId = facadeContract.Bed == null ? null : facadeContract.Bed.Id,
                    PatientId = facadeContract.Patient == null ? null : facadeContract.Patient.Id,
                    FacilityId = facadeContract.Facility == null ? null : facadeContract.Facility.Id,
                    ListIconFlagFacilityIdsOfBed = facadeContract.ListIconFlagFacilityIdsOfBed,
                    ListIconFlagFacilityIdsOfPatient = facadeContract.ListIconFlagFacilityIdsOfPatient,
                    ListIconFlagIdsOfBed = facadeContract.ListIconFlagIdsOfBed,
                    ListIconFlagIdsOfPatient = facadeContract.ListIconFlagIdsOfPatient
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.EmsStaff ToDataContract(this EmsStaff facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.EmsStaff dataContract = new DC.EmsStaff()
                {
                    Id = facadeContract.Id,
                    Pin = facadeContract.Pin,
                    UserName = facadeContract.UserName,
                    FacilityId = facadeContract.Facility == null ? null : facadeContract.Facility.Id
                };

                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DC.HospitalLocationAppointment ToDataContract(this DischargeClinic facadeContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.HospitalLocationAppointment dataContract = new DC.HospitalLocationAppointment()
                {
                    Id = facadeContract.Id,
                    HospitalLocationId = (facadeContract.HospitalLocation != null) ? facadeContract.HospitalLocation.Id : null,
                    FacilityId = (facadeContract.Facility != null) ? facadeContract.Facility.Id : null,
                    VistaSiteId = (facadeContract.VistaSite != null) ? facadeContract.VistaSite.Id : null,
                };
                return dataContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
