﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Facade.Data;
using BMS.Facade.Service;
using BMS.ServicesWrapper.EVS;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;
using BMS.Utils;


namespace BMS.Facade.Implementation
{
    public class VocabularyOperationsImplementation : IVocabularyOperations
    {
        /// <summary>
        /// Gets the corresponding vocabulary.
        /// </summary>
        /// <param name="vocabularyName">Name of the vocabulary.</param>
        /// <returns></returns>
        public CDList GetVocabulary(Util.Vocabulary vocabularyName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CDList vocabulary = new CDList();
                CDList result = new CDList();
                CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                codeFilterParam.MaxSelectedCodes = int.MaxValue;
                User user = null;
                string vistaCode = null;
                switch (vocabularyName)
                {
                    case Util.Vocabulary.VistaMedicalCenterDivision:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.VistaMedicalCenterDivision.ToString();
                        vistaCode = FacadeManager.UserInterface.GetProfile().VistaSite.Code + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode)));
                        break;                    
                    case Util.Vocabulary.OrderableItem:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.OrderableItem.ToString();
                        vistaCode = FacadeManager.UserInterface.GetProfile().VistaSite.Code;
                        codeFilterParam.MatchAlgorithm = MatchAlgorithm.StartsWithIgnoreCase;
                        codeFilterParam.MatchProperty = MatchProperty.Code;
                        codeFilterParam.MatchText = vistaCode;                        
                        vocabulary.AddRange(EVSFactory.Instance.SearchConceptsWithPaging(codeFilterParam));
                        break;                                       
                    case Util.Vocabulary.VistaTreatingSpecialty:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.VistaTreatingSpecialty.ToString();
                        vistaCode = FacadeManager.UserInterface.GetProfile().VistaSite.Code + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode)));
                        break;
                    case Util.Vocabulary.TypeOfMovement:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.TypeOfMovement.ToString();
                        vistaCode = FacadeManager.UserInterface.GetProfile().VistaSite.Code + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode)));
                        break;
                    case Util.Vocabulary.ADTOrderableItem:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.ADTOrderableItem.ToString();
                        vistaCode = FacadeManager.UserInterface.GetProfile().VistaSite.Code + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode)));
                        break;
                    case Util.Vocabulary.WaitingArea:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.WaitingArea.ToString();
                        user = FacadeManager.UserInterface.GetProfile();
                        vistaCode = user.VistaSite.Code + "_" + user.Facility.Code + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode) || a.code.StartsWith(Constants.NATIONAL_VOCABULARY_CODE_PREFIX)));
                        break;
                    case Util.Vocabulary.UnavailableReason:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.UnavailableReason.ToString();
                        user = FacadeManager.UserInterface.GetProfile();
                        vistaCode = user.VistaSite.Code + "_" + user.Facility.Code + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode) || a.code.StartsWith(Constants.NATIONAL_VOCABULARY_CODE_PREFIX)));
                        break;
                    case Util.Vocabulary.StrictDecision:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.StrictDecision.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.Decision:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Decision.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.PatientWaitingViews:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.PatientWaitingViews.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.PatientSortBy:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.PatientSortBy.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.Gender:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Gender.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.PatientWaitingTimes:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.PatientWaitingTimes.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.BedStatus:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.BedStatus.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.SchedAdmNextDays:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.SchedAdmNextDays.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.QuickDate:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.QuickDate.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.TransactionBed:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.TransactionBed.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.Reports:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Reports.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.SelectReport:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.SelectReport.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;                    
                    case Util.Vocabulary.Disposition:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Disposition.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.ServicesReceiving:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.ServicesReceiving.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.FeeReason:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.FeeReason.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.TypeOfBed:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.TypeOfBed.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.SelectBedGroupingsView:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.SelectBedGroupingsView.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.BedBoardModule:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.BedBoardModule.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.AdminURType:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.AdminURType.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;                    
                    case Util.Vocabulary.OrderableItemType:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.OrderableItemType.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;                    
                    case Util.Vocabulary.EventType:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.EventType.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.EMSBedNotification:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.EMSBedNotification.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.FeeDisposition:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.FeeDisposition.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.GenderColor:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.GenderColor.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.PTDisplay:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.PTDisplay.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.State:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.State.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.TimeZone:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.TimeZone.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.WaitListView:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.WaitListView.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.WaitOrderBy:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.WaitOrderBy.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.Era:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Era.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.NationalPatientSortBy:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.NationalPatientSortBy.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.VistaSpecialty:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;                    
                    case Util.Vocabulary.Location:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Location.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.Comments:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.Comments.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.EvacDispositionStatus:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.EvacDispositionStatus.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.TransportationProvider:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.TransportationProvider.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.TransportationType:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.TransportationType.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.NUMA:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.NUMA.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                    case Util.Vocabulary.HAvBED:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.HAvBED.ToString();
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam));
                        break;
                }
                result.AddRange(vocabulary.OrderBy(a => a.displayName).ToList());
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<CDWithProperties> GetReasons()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                User user = FacadeManager.UserInterface.GetProfile();
                string code = user.VistaSite.Code + "_" + user.Facility.Code + "_";
                CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
                codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.UnavailableReason.ToString();
                codeFilterParamWithProperties.Properties = new List<string>();
                codeFilterParamWithProperties.Properties.Add("Type");
                return EVSFactory.Instance.GetCodesWithProperties(codeFilterParamWithProperties).Where(a => a.code.StartsWith(code) || a.code.StartsWith(Constants.NATIONAL_VOCABULARY_CODE_PREFIX)).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<CDWithProperties> GetAllUnavailableReasons()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {                
                CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
                codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.UnavailableReason.ToString();
                codeFilterParamWithProperties.Properties = new List<string>();
                codeFilterParamWithProperties.Properties.Add("Type");
                return EVSFactory.Instance.GetCodesWithProperties(codeFilterParamWithProperties).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<CDWithProperties> GetAdtOrderableItems()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string vistaCode = FacadeManager.UserInterface.GetProfile().VistaSite.Code + "_";
                CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
                codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.ADTOrderableItem.ToString();
                codeFilterParamWithProperties.Properties = new List<string>();
                codeFilterParamWithProperties.Properties.Add("IEN");
                codeFilterParamWithProperties.Properties.Add("OrderableItemType");
                return EVSFactory.Instance.GetCodesWithProperties(codeFilterParamWithProperties).Where(a => a.code.StartsWith(vistaCode)).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the time zones.
        /// </summary>
        /// <returns>List of vocabulary timezones.</returns>
        public IList<CDWithProperties> GetTimeZones()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
                codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.TimeZone.ToString();
                codeFilterParamWithProperties.Properties = new List<string>();
                codeFilterParamWithProperties.Properties.Add("Value");
                return EVSFactory.Instance.GetCodesWithProperties(codeFilterParamWithProperties).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<CDWithProperties> GetVistaSpecialties()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
                codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString();
                codeFilterParamWithProperties.Properties = new List<string>();
                codeFilterParamWithProperties.Properties.Add(Constants.NUMA);
                codeFilterParamWithProperties.Properties.Add(Constants.HAvBED);
                return EVSFactory.Instance.GetCodesWithProperties(codeFilterParamWithProperties).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<CD> GetWaitingAreas()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                codeFilterParam.MaxSelectedCodes = int.MaxValue;
                codeFilterParam.VocabularyDomain = Util.Vocabulary.WaitingArea.ToString();
                return EVSFactory.Instance.GetCodes(codeFilterParam).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<CD> GetOrderableItemsDB(string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                codeFilterParam.MaxSelectedCodes = int.MaxValue;
                codeFilterParam.VocabularyDomain = Util.Vocabulary.OrderableItem.ToString();                
                codeFilterParam.MatchAlgorithm = MatchAlgorithm.StartsWithIgnoreCase;
                codeFilterParam.MatchProperty = MatchProperty.Code;
                codeFilterParam.MatchText = vistaCode;
                return EVSFactory.InstanceFromWCF.SearchConceptsWithPagingDB(codeFilterParam).OrderBy(a=>a.displayName).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Allows the insertion of a concept in EVS and updates the underlying ValueSet.
        /// </summary>
        /// <param name="concept">The CD or CDWithProperties concept to insert in EVS. It's inserted in the concept.codeSystem CodeSystem.</param>
        public void InsertConcept(CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVSFactory.Instance.InsertConcept(concept);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates an existing concept in EVS.
        /// </summary>
        /// <param name="concept">The CD or CDWithProperties concept to updaye in EVS. It's updated based on the code and codeSystem properties.</param>
        public void UpdateConcept(CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVSFactory.Instance.UpdateConcept(concept);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes an existing concept from EVS.
        /// </summary>
        /// <param name="concept">The concept to delete. Searches for the concept relies on the code and codeSystem properties.</param>
        public void DeleteConcept(CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVSFactory.Instance.DeleteConcept(concept);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void ClearCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVSFactory.Instance.FlushCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public CDList GetVocabularyByVistaCode(Util.Vocabulary vocabularyName, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CDList vocabulary = new CDList();
                CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                codeFilterParam.MaxSelectedCodes = int.MaxValue;
                switch (vocabularyName)
                {
                    case Util.Vocabulary.VistaMedicalCenterDivision:
                        codeFilterParam.VocabularyDomain = Util.Vocabulary.VistaMedicalCenterDivision.ToString();
                        vistaCode = vistaCode + "_";
                        vocabulary.AddRange(EVSFactory.Instance.GetCodes(codeFilterParam).Where(a => a.code.StartsWith(vistaCode)).OrderBy(a=>a.displayName));
                        break;
                }
                return vocabulary;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
