﻿using System;
using System.Collections.Generic;
using System.Linq;
using InfoWorld.HL7.ITS;
using BMS.Facade.Service;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.ServicesWrapper.BMService;
using DC = BMS.DataContracts;
using BMS.Facade.Translators;
using BMS.Facade.Fault;
using BMS.Utils.Properties;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.Security;

namespace BMS.Facade.Implementation
{
    public class ADTOperationsImplementation : IADTOperations
    {
        /// <summary>
        /// Gets the patients waiting list.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="filterAdmissionCreationTime">The filter admission creation time.</param>
        /// <param name="isOnlyCurrentWaitingAdmission">if set to true get all the admission where remove date is null esle get all admission where admission creation date is GTE the filter admission date.</param>        
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        public IList<AdmissionInfo> GetWaitingList(Guid facilityUid, DateTime filterAdmissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IconInfo isolationIcon = FacadeManager.IconInterface.GetIconByCode(Constants.BED_ISOLATION);
                IList<Bed> roomBedAssignedList = new List<Bed>();
                IList<AdmissionInfo> admissionInfoList = BMSFactory.BedManagerQueryClient.FilterAdmissions(facilityUid, filterAdmissionCreationTime, isOnlyCurrentWaitingAdmission, isOnlyInHouseWaitingAdmission).Select<DC.AdmissionInfo, AdmissionInfo>(dcAI => DateTimeConverter.ConvertDateFromUTC(dcAI.ToFacadeContract(isolationIcon), facilityId)).ToList();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                roomBedAssignedList = EISFactory.InstanceFromWCF.GetBeds(admissionInfoList.Select(bi => bi.RoomBedAssignedId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id);
                List<II> patientIds = (from item in admissionInfoList
                                       where !string.IsNullOrEmpty(item.RootPatientId) && !string.IsNullOrEmpty(item.ExtensionPatientId) && item.ExtensionPatientId != Guid.Empty.ToString()
                                       select new II()
                                       {
                                           root = item.RootPatientId,
                                           extension = item.ExtensionPatientId
                                       }).ToList();
                IList<Patient> patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(patientIds.Distinct().ToList(), null);
                Bed roomBedAssigned = null;
                Patient assignedPatient = null;
                List<CD> genderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Gender);
                Dictionary<string, string> genders = new Dictionary<string,string>();
                genderList.ForEach(a => genders.Add(a.code, a.displayName.Substring(0, 1)));                
                foreach (AdmissionInfo admission in admissionInfoList)
                {
                    if (admission.RoomBedAssignedId != null)
                    {
                        roomBedAssigned = roomBedAssignedList.Where(b => b.Id.extension.Equals(admission.RoomBedAssignedId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RoomBedAssignedId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (roomBedAssigned != null)
                            admission.RoomBedAssigned = roomBedAssigned.Name;
                    }

                    assignedPatient = patientList.Where(b => b.Id.extension.Equals(admission.ExtensionPatientId, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RootPatientId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (assignedPatient != null)
                    {
                        admission.RootPatientId = assignedPatient.Id.root;
                        admission.PatientName = string.Format("{0}, {1}", assignedPatient.LastName, assignedPatient.FirstName);
                        admission.Gender = (assignedPatient.Gender != null && !string.IsNullOrEmpty(assignedPatient.Gender.code)) ? genders[assignedPatient.Gender.code] : string.Empty;
                        admission.PatientLastFourDigitsOfSSN = string.Format("{0}{1}", assignedPatient.LastName.Substring(0, 1), assignedPatient.SSN.extension.Substring(assignedPatient.SSN.extension.Length - 4));
                    }
                }
                return admissionInfoList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the patients in community hospitals.
        /// </summary>
        /// <param name="facility">The facility.</param>
        /// <param name="facilityVISN">The facility VISN.</param>
        /// <returns></returns>
        public IList<TransferInfo> GetPatientsInCommunityHospitals(Facility facility, Visn facilityVISN)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Guid? facilityUid = null;
                if (facility != null && facility.Id != null && !string.IsNullOrEmpty(facility.Id.extension))
                    facilityUid = Guid.Parse(facility.Id.extension);
                List<Facility> eisFacilities = FacadeManager.EntityInterface.GetFacilitiesByVisn(facilityVISN.Id).ToList();
                List<TransferInfo> transfers = BMSFactory.BedManagerQueryClient.FilterTransfers(facilityUid, facilityVISN.Number).
                    Select<DC.TransferInfo, TransferInfo>(dcTI => DateTimeConverter.ConvertDateFromUTC(dcTI.ToFacadeContract(eisFacilities), dcTI.FacilityId == null ? null : dcTI.FacilityId)).ToList();
                IList<Patient> patientList = new List<Patient>();
                patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(transfers.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);
                foreach (TransferInfo transfer in transfers)
                {
                    Patient patient = patientList.Where(b => b.Id.extension.Equals(transfer.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(transfer.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (patient != null)
                    {
                        transfer.PatientName = string.Format("{0}, {1}", patient.LastName, patient.FirstName);
                        transfer.SCRating = patient.ServiceConnectedPercentage;
                        transfer.SSN = patient.SSN.extension;
                    }
                }
                return transfers;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<ValidationAlert> TransferRequest(Transfer transfer)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Transfer GetTransferById(II id, User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.Transfer result = BMSFactory.BedManagerOperationsClient.GetTransfer(id);
                if (result == null)
                    throw new BedManagerFault(Resources.BMS_TRANSFER_NOT_EXISTS);
                Transfer transfer = result.ToFacadeContract();
                transfer = DateTimeConverter.ConvertDateFromUTC(transfer, transfer.FacilityFrom.Id);
                if (transfer.FacilityFrom != null && transfer.FacilityFrom.Id != null)
                    transfer.FacilityFrom = FacadeManager.EntityInterface.GetFacility(transfer.FacilityFrom.Id);
                if (transfer.FacilityTo != null && transfer.FacilityTo.Id != null)
                    transfer.FacilityTo = FacadeManager.EntityInterface.GetFacility(transfer.FacilityTo.Id);
                if (transfer.PatientId != null)
                    transfer.Patient = FacadeManager.EntityInterface.GetPatientById(transfer.PatientId, null);
                return transfer;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<TransferInfo> GetPatientsInCommunityHospitalsStandardView(bool isDisplayOnlyNationalPatients, string regionCode, string VISNCode, CD specialty, bool isCurrentWaitingList, DateTime transferDateTime)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<TransferInfo> transfers = new List<TransferInfo>();
                IList<DC.PatientWaitingStandardView> patientWaitingList = BMSFactory.BedManagerQueryClient.FilterNationalPatientWaitingStandardView(isDisplayOnlyNationalPatients, regionCode, VISNCode, specialty.code, specialty.codeSystem, transferDateTime, isCurrentWaitingList);
                if (patientWaitingList != null && patientWaitingList.Count > 0)
                {
                    Dictionary<string, string> facilities = new Dictionary<string, string>(StringComparer.InvariantCultureIgnoreCase);
                    List<CD> timeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                    foreach (DC.PatientWaitingStandardView a in patientWaitingList)
                    {
                        if (a.FacilityId != null && !string.IsNullOrEmpty(a.FacilityId.extension) && !a.FacilityId.extension.Equals(Guid.Empty.ToString()) && !facilities.ContainsKey(a.FacilityId.extension))
                            facilities.Add(a.FacilityId.extension, timeZoneList.Where(x => x.displayName.Equals(FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(a.FacilityId).Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().code);
                    }

                    transfers = patientWaitingList.Select<DC.PatientWaitingStandardView, TransferInfo>(dcTI => DateTimeConverter.ConvertDateFromUTC(dcTI.ToFacadeContract(), dcTI.FacilityId == null ? null : dcTI.FacilityId)).ToList();
                    IList<Patient> patientList = new List<Patient>();
                    patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(transfers.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);
                    foreach (TransferInfo ti in transfers)
                    {
                        if (ti.FacilityFrom != null && ti.FacilityFrom.Id != null && facilities.ContainsKey(ti.FacilityFrom.Id.extension))
                            ti.FacilityTimeZoneCode = facilities[ti.FacilityFrom.Id.extension];

                        Patient patient = patientList.Where(b => b.Id.extension.Equals(ti.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(ti.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (patient != null)
                        {
                            ti.PatientName = string.Format("{0}, {1}", patient.LastName, patient.FirstName);
                            ti.SCRating = patient.ServiceConnectedPercentage;
                            ti.SSN = patient.SSN.extension.Substring(patient.SSN.extension.Length - 4);
                        }
                    }
                }
                return transfers;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<DateView> GetPatientsInCommunityHospitalsDateView(bool isDisplayOnlyNationalPatients, string regionCode, string VISNCode, CD specialty, bool isCurrentWaitingList, DateTime transferDateTime)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<DateView> datesView = new List<DateView>();
                IList<DC.PatientWaitingDateView> patientWaitingList = BMSFactory.BedManagerQueryClient.FilterNationalPatientWaitingDateView(isDisplayOnlyNationalPatients, regionCode, VISNCode, specialty.code, specialty.codeSystem, transferDateTime, isCurrentWaitingList);

                if (patientWaitingList != null && patientWaitingList.Count > 0)
                {
                    Dictionary<string, string> facilities = new Dictionary<string, string>(StringComparer.InvariantCultureIgnoreCase);
                    List<CD> timeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                    foreach (DC.PatientWaitingDateView a in patientWaitingList)
                    {
                        if (a.FacilityId != null && !string.IsNullOrEmpty(a.FacilityId.extension) && !a.FacilityId.extension.Equals(Guid.Empty.ToString()) && !facilities.ContainsKey(a.FacilityId.extension))
                            facilities.Add(a.FacilityId.extension, timeZoneList.Where(x => x.displayName.Equals(FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(a.FacilityId).Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().code);
                        if (a.AdmissionFacilityId != null && !string.IsNullOrEmpty(a.AdmissionFacilityId.extension) && !a.AdmissionFacilityId.extension.Equals(Guid.Empty.ToString()) && !facilities.ContainsKey(a.AdmissionFacilityId.extension))
                            facilities.Add(a.AdmissionFacilityId.extension, timeZoneList.Where(x => x.displayName.Equals(FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(a.AdmissionFacilityId).Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().code);
                    }

                    List<Facility> eisFacilities = FacadeManager.EntityInterface.GetFacilities().ToList();
                    datesView = patientWaitingList.Select<DC.PatientWaitingDateView, DateView>(dcDV => dcDV.ToFacadeContract(eisFacilities)).ToList();
                    List<DateView> datesViewList = new List<DateView>();
                    datesView.ToList().ForEach((item) =>
                    {
                        datesViewList.Add(new DateView() { DispositionDate = item.DispositionDate, FacilityTo = item.FacilityTo });
                    });

                    datesView = datesView.Select<DateView, DateView>(dcDV => DateTimeConverter.ConvertDateFromUTC(dcDV, (dcDV.FacilityFrom == null) ? null : dcDV.FacilityFrom.Id)).ToList();
                    datesViewList = datesViewList.Select<DateView, DateView>(dcDV => DateTimeConverter.ConvertDateFromUTC(dcDV, (dcDV.FacilityTo == null) ? null : dcDV.FacilityTo.Id)).ToList();
                    for (int i = 0; i < datesView.Count; i++)
                        datesView[i].DispositionDate = datesViewList[i].DispositionDate;
                    IList<Patient> patientList = new List<Patient>();
                    patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(datesView.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);

                    foreach (DateView dw in datesView)
                    {
                        if (dw.FacilityFrom != null && dw.FacilityFrom.Id != null && facilities.ContainsKey(dw.FacilityFrom.Id.extension))
                            dw.FacilityFromTimeZoneCode = facilities[dw.FacilityFrom.Id.extension.ToLower()];
                        if (dw.FacilityTo != null && dw.FacilityTo.Id != null && facilities.ContainsKey(dw.FacilityTo.Id.extension))
                            dw.FacilityToTimeZoneCode = facilities[dw.FacilityTo.Id.extension.ToLower()];

                        Patient patient = patientList.Where(b => b.Id.extension.Equals(dw.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(dw.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (patient != null)
                        {
                            dw.PatientName = string.Format("{0}, {1}", patient.LastName, patient.FirstName);
                            dw.PatientSSN = patient.SSN.extension;
                        }
                    }
                }
                return datesView;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public WaitingListItem GetWaitingListItem(II id, User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.WaitingListItem result = BMSFactory.BedManagerOperationsClient.GetWaitingListItem(id, null);

                if (result == null)
                    throw new BedManagerFault(Resources.BMS_ADMISSION_NOT_EXISTS);
                Facility facility = null;
                Patient patient = null;
                Bed bed = null;
                if (result.FacilityId != null)
                    facility = FacadeManager.EntityInterface.GetFacility(result.FacilityId);
                if (result.PatientId != null)
                    patient = FacadeManager.EntityInterface.GetPatientById(result.PatientId, null);
                if (result.RequestedBedId != null)
                    bed = EISFactory.InstanceFromWCF.GetBed(result.RequestedBedId, user.VistaSite.Id);
                WaitingListItem wli = result.ToFacadeContract(facility, patient, bed);   
                wli = DateTimeConverter.ConvertDateFromUTC(wli, user.Facility.Id);
                return wli;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public PatientLocation GetPatientLocation(II patientId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.PatientLocation patientLocation = BMSFactory.BedManagerOperationsClient.GetPatientLocation(patientId);
                if (patientLocation == null)
                    return null;

                return new PatientLocation()
                    {
                        CurrentAdmissionBed = (patientLocation.CurrentAdmissionBedId != null) ? EISFactory.InstanceFromWCF.GetBed(patientLocation.CurrentAdmissionBedId, vistaSiteId) : null,
                        CurrentAdmissionWard = (patientLocation.CurrentAdmissionWardId != null) ? EISFactory.InstanceFromWCF.GetWard(patientLocation.CurrentAdmissionWardId) : null
                    };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void ModifyWaitingListItem(WaitingListItem waitingListItem, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DateTimeConverter.ConvertDateToUTC(waitingListItem, facility.Id);
                DC.WaitingListItem result = waitingListItem.ToDataContract();
                BMSFactory.BedManagerOperationsClient.UpdateWaitingListItem(result);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Act GetLastAct(string patientSSN, ActType actType, string vistaSiteNumber)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string querySSN = patientSSN;
                //if the ssn is without dashes, they must be added before making the query in EIS.
                if (patientSSN.Length == 9)
                    querySSN = patientSSN.Insert(3, "-").Insert(6, "-");
                Patient patient = EISFactory.InstanceFromWCF.GetPatientBySsn(new II(Constants.SSNROOT, querySSN), null);
                if (patient == null)
                    return null;
                IList<Facility> facilities = EISFactory.InstanceFromWCF.GetFacilitiesBySiteNumber(vistaSiteNumber);
                if (facilities == null || facilities.Count == 0)
                    return null;

                return BMSFactory.BedManagerQueryClientFromWCF.GetLastAct(patient.Id, actType, facilities[0].VistaSite.Id).ToFacadeContract();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public PatientWaitingCount GetPatientWaitingCount(II facilityId, II vistaSiteId, string vistaDivisionCode, int timeZoneOffset)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.PatientWaitingCount result = BMSFactory.BedManagerQueryClient.GetPatientsWaitingCount(Guid.Parse(facilityId.extension), Guid.Parse(vistaSiteId.extension), vistaDivisionCode, timeZoneOffset);
                return new PatientWaitingCount()
                {
                    WaitingListCount = result.WaitingListCount,
                    ScheduledAdmissionsCount = result.ScheduledAdmissionsCount
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<WaitingListReport> GetWaitingListReport(II facilityId, string filterBy, string filterPatientName, string filterGender)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<WaitingListReport> result = new List<WaitingListReport>();
                List<WaitingListReport> waitingListReport = BMSFactory.BedManagerQueryClient.FilterWaitingListReport(Guid.Parse(facilityId.extension), filterBy).Select<DC.WaitingListReport, WaitingListReport>(dcAI => DateTimeConverter.ConvertDateFromUTC(dcAI.ToFacadeContract(), facilityId)).ToList();
                IList<Bed> roomBedAssignedList = new List<Bed>();
                IList<Patient> patientList = new List<Patient>();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                roomBedAssignedList = EISFactory.InstanceFromWCF.GetBeds(waitingListReport.Select(bi => bi.RoomBedAssignedId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id);
                patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(waitingListReport.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);

                string filterPatientFirstName = null, filterPatientLastName = null;
                if (filterPatientName.Contains(","))
                {
                    filterPatientLastName = filterPatientName.Substring(0,filterPatientName.IndexOf(",")).Trim();
                    filterPatientFirstName = filterPatientName.Substring(filterPatientName.IndexOf(",") + 1).Trim();
                }
                foreach (WaitingListReport item in waitingListReport)
                {
                    if (item.RoomBedAssignedId != null)
                    {
                        Bed roomBedAssigned = roomBedAssignedList.Where(b => b.Id.extension.Equals(item.RoomBedAssignedId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(item.RoomBedAssignedId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (roomBedAssigned != null)
                            item.RoomBedAssigned = roomBedAssigned.Name;
                    }

                    Patient assignedPatient = patientList.Where(b => b.Id.extension.Equals(item.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(item.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (!string.IsNullOrEmpty(filterPatientName) && assignedPatient != null)
                    {
                        if (!filterPatientName.Contains(","))
                        {
                            if (!assignedPatient.LastName.Contains(filterPatientName) && !assignedPatient.FirstName.Contains(filterPatientName))
                                continue;
                        }
                        else
                        {
                            if (!assignedPatient.LastName.Contains(filterPatientLastName) || !assignedPatient.FirstName.Contains(filterPatientFirstName))
                                continue;
                        }
                    }
                    if (!string.IsNullOrEmpty(filterGender) && assignedPatient != null && assignedPatient.Gender != null && assignedPatient.Gender.code != filterGender.Substring(0,1))
                        continue;

                    if (assignedPatient != null)
                    {
                        item.PatientGender = (assignedPatient.Gender != null) ? assignedPatient.Gender.GetDisplayName() : string.Empty;
                        item.PatientName = string.Format("{0}{1}{2}{3}", assignedPatient.LastName, ",", assignedPatient.FirstName, assignedPatient.SSN.extension.Substring(assignedPatient.SSN.extension.Length - 4));
                        result.Add(item);
                    }
                }

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<EvacuationPatient> GetEvacuationPatients(II facilityId, string wardList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                IList<EvacuationPatient> evacuationPatientList = BMSFactory.BedManagerQueryClientFromWCF.GetEvacuationPatients(wardList, facilityId).Select<DC.EvacuationPatient, EvacuationPatient>(dcEP => dcEP.ToFacadeContract()).ToList();
                IList<Patient> patientList = new List<Patient>();
                patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(evacuationPatientList.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);
                foreach (EvacuationPatient patient in evacuationPatientList)
                {
                    Patient evacuationPatient = patientList.Where(b => b.Id.extension.Equals(patient.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(patient.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (evacuationPatient != null)
                        patient.Patient = evacuationPatient;
                }
                return evacuationPatientList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public WaitingListItem GetWaitingListItem(II waitingListItemId, string patientUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.WaitingListItem result = BMSFactory.BedManagerOperationsClient.GetWaitingListItem(null, patientUid);

                if (result == null)
                    throw new BedManagerFault(Resources.BMS_ADMISSION_NOT_EXISTS);
                Facility facility = null;
                Patient patient = null;
                Bed bed = null;
                if (result.FacilityId != null)
                    facility = FacadeManager.EntityInterface.GetFacility(result.FacilityId);
                if (result.PatientId != null)
                    patient = FacadeManager.EntityInterface.GetPatientById(result.PatientId, null);
                if (result.RequestedBedId != null)
                    bed = EISFactory.InstanceFromWCF.GetBed(result.RequestedBedId, result.VistaSiteId);
                WaitingListItem wli = result.ToFacadeContract(facility, patient, bed);
                wli = DateTimeConverter.ConvertDateFromUTC(wli, FacadeManager.UserInterface.GetProfile().Facility.Id);
                return wli;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void GetHomePageBMSData(II vistaSiteId, II facilityId, string wardIdList, List<Module> flags, int timeZoneMinutesOffset, DateTime filterAdmissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission,
                                        IList<DataRetrievalMethod> retrievalMethods, IList<VistaIntegration.FacadeContracts.Admin.VistaDataType> dataTypes, IList<JobStatus> status,
                                        out List<BedOccupancyCount> bedOccupancylist, out int newEventsCount, out List<AdmissionInfo> facilityWaitingList, out DateTime? bpLastRunDate)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<DC.BedOccupancyCount> bmsOccupancyCount = new List<DC.BedOccupancyCount>();
                List<DC.NewEvent> newEvents = new List<DC.NewEvent>();
                List<DC.AdmissionInfo> admissionInfoList = new List<DC.AdmissionInfo>();                
                BMSFactory.BedManagerQueryClient.GetHomePageBMSData(Guid.Parse(facilityId.extension), vistaSiteId.extension, wardIdList, timeZoneMinutesOffset, filterAdmissionCreationTime, isOnlyCurrentWaitingAdmission, isOnlyInHouseWaitingAdmission,
                                retrievalMethods, dataTypes.Select<VistaIntegration.FacadeContracts.Admin.VistaDataType, Utils.VistaDataType>(vdt => (Utils.VistaDataType)vdt).ToList(), status,
                                out bmsOccupancyCount, out newEvents, out admissionInfoList, out bpLastRunDate);

                #region bed occupancy percent

                int totalBeds = 0;
                int totalOccupiedBeds = 0;
                int totalUnavailableBeds = 0;

                List<Division> divisions = EISFactory.InstanceFromWCF.GetDivisions(facilityId).ToList();
                List<Bed> beds = new List<Bed>();
                foreach (Division division in divisions)
                    beds.AddRange(EISFactory.InstanceFromWCF.GetBedsInDivision(division.Id, vistaSiteId));

                beds = beds.Distinct().ToList();
                totalBeds = beds.Count();
                totalOccupiedBeds = (from bed in beds
                                     where (from bedOccupacyInfo in bmsOccupancyCount where bedOccupacyInfo.IsOccupiedBed == true select bedOccupacyInfo.BedId.extension).Contains(bed.Id.extension, StringComparer.InvariantCultureIgnoreCase)
                                     select bed).Count();
                totalUnavailableBeds = (from bed in beds
                                        where (from bedOccupacyInfo in bmsOccupancyCount where bedOccupacyInfo.IsUnavailableBed == true select bedOccupacyInfo.BedId.extension).Contains(bed.Id.extension, StringComparer.InvariantCultureIgnoreCase)
                                        select bed).Count();

                bedOccupancylist = new List<BedOccupancyCount>();
                bedOccupancylist.Add(new BedOccupancyCount() { FacilityId = facilityId, OccupancyPercent = (int)((double)(totalOccupiedBeds) / (totalBeds - totalUnavailableBeds) * 100) });

                #endregion

                #region new events count

                newEventsCount = newEvents[0].TotalCount;
                foreach (Module m in flags)
                {
                    switch (m.Code)
                    {
                        case Constants.ADM_ORD:
                            if (m.CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.SignedAdmission).FirstOrDefault().EventCount;
                            break;
                        case Constants.ANTIC_DISCH:
                            if (m.CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.SignedAnticipatedDischarge).FirstOrDefault().EventCount;
                            break;
                        case Constants.DISCH_ORD:
                            if (m.CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.SignedDischarge).FirstOrDefault().EventCount;
                            break;
                        case Constants.TRSF_ORD:
                            if (m.CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.SignedTransfer).FirstOrDefault().EventCount;
                            break;
                        case Constants.DISCHARGE_APPOINTMENT:
                            if (m.CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.DischargeAppointment).FirstOrDefault().EventCount;
                            break;
                        case Constants.EMS_MODULE:
                            if (m.CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                            {
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.VacatedBed).FirstOrDefault().EventCount;
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.CurrentlyCleaningBed).FirstOrDefault().EventCount;
                                newEventsCount = newEventsCount - newEvents.Where(a => a.EventName == Utils.NewEvent.CompletedCleaningBed).FirstOrDefault().EventCount;
                            }
                            break;
                    }
                }

                #endregion

                #region admission waiting list

                IconInfo isolationIcon = FacadeManager.IconInterface.GetIconByCode(Constants.BED_ISOLATION);
                IList<Bed> roomBedAssignedList = new List<Bed>();
                facilityWaitingList = admissionInfoList.Select<DC.AdmissionInfo, AdmissionInfo>(dcAI => DateTimeConverter.ConvertDateFromUTC(dcAI.ToFacadeContract(isolationIcon), facilityId)).ToList();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                roomBedAssignedList = EISFactory.InstanceFromWCF.GetBeds(facilityWaitingList.Select(bi => bi.RoomBedAssignedId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id);
                List<II> patientIds = (from item in facilityWaitingList
                                       where !string.IsNullOrEmpty(item.RootPatientId) && !string.IsNullOrEmpty(item.ExtensionPatientId) && item.ExtensionPatientId != Guid.Empty.ToString()
                                       select new II()
                                       {
                                           root = item.RootPatientId,
                                           extension = item.ExtensionPatientId
                                       }).ToList();
                IList<Patient> patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(patientIds.Distinct().ToList(), null);
                Bed roomBedAssigned = null;
                Patient assignedPatient = null;
                List<CD> genderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Gender);
                Dictionary<string, string> genders = new Dictionary<string, string>();
                genderList.ForEach(a => genders.Add(a.code, a.displayName.Substring(0, 1)));
                foreach (AdmissionInfo admission in facilityWaitingList)
                {
                    if (admission.RoomBedAssignedId != null)
                    {
                        roomBedAssigned = roomBedAssignedList.Where(b => b.Id.extension.Equals(admission.RoomBedAssignedId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RoomBedAssignedId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (roomBedAssigned != null)
                            admission.RoomBedAssigned = roomBedAssigned.Name;
                    }

                    assignedPatient = patientList.Where(b => b.Id.extension.Equals(admission.ExtensionPatientId, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RootPatientId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (assignedPatient != null)
                    {
                        admission.RootPatientId = assignedPatient.Id.root;
                        admission.PatientName = string.Format("{0}, {1}", assignedPatient.LastName, assignedPatient.FirstName);
                        admission.Gender = (assignedPatient.Gender != null && !string.IsNullOrEmpty(assignedPatient.Gender.code)) ? genders[assignedPatient.Gender.code] : string.Empty;
                        admission.PatientLastFourDigitsOfSSN = string.Format("{0}{1}", assignedPatient.LastName.Substring(0, 1), assignedPatient.SSN.extension.Substring(assignedPatient.SSN.extension.Length - 4));
                    }
                }

                #endregion
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        private string GetFacilitiesUids(IList<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string facilitiesUids = string.Empty;
                foreach (Facility facility in facilities)
                    facilitiesUids += facility.Id.extension + ";";
                return facilitiesUids;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
