﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using BMS.Utils;
using InfoWorld.HL7.ITS;
using InfoWorld.EVS.CTSMAPI;
using BMS.ServicesWrapper.EVS;

namespace BMS.Facade
{
    public class FacadeUtil
    {
        public static IList<CDWithProperties> GetOrderableItemByType(string type, IEVSWrapper EVS) 
        {
            IList<CDWithProperties> cdList = GetOrderableItemsFromEVS(EVS);
            if (string.IsNullOrEmpty(type))
                return cdList;
            return cdList.Where(cd => cd.Properties != null && cd.Properties.Exists(property => property.PropertyValue != null && property.PropertyValue.text == type)).ToList();
        }        

        public static IList<CDWithProperties> GetOrderableItem(IEVSWrapper EVS, string vistASiteCode)
        {
            int vistaSiteCodeLength = vistASiteCode.Length;
            IList<CDWithProperties> cdList = GetOrderableItemsFromEVS(EVS);
            IList<CDWithProperties> cdpList = cdList.Where(a => (a.code.Substring(0, vistaSiteCodeLength) == vistASiteCode)).ToList();
            return cdpList;
        }

        private static IList<CDWithProperties> GetOrderableItemsFromEVS(IEVSWrapper EVS) 
        {
            //construct parameters to get ADT Orderable Items from EVS
            CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
            codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
            codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.ADTOrderableItem.ToString();
            codeFilterParamWithProperties.Properties = new List<string>();
            codeFilterParamWithProperties.Properties.Add(Constants.CONCEPT_PROPERTY_IEN);
            codeFilterParamWithProperties.Properties.Add(Constants.CONCEPT_PROPERTY_ORDERABLE_ITEM_TYPE);
            //get ADT Orderable Items from EVS
            IList<CDWithProperties> cdList = EVS.GetCodesWithProperties(codeFilterParamWithProperties);
            return cdList;
        }        

        public static int GetOccupancyPercent(IList<BedInfo> bedsInfo) 
        {
            int free = 0;
            int total = 0;
            foreach (BedInfo bedInfo in bedsInfo)
            {
                if (bedInfo.BedAvailabilityStatusCode != Constants.BED_OOS_FROM_BEDBOARD && bedInfo.BedAvailabilityStatusCode != Constants.BED_OOS_FROM_VISTA)
                {
                    total++;
                    if (string.IsNullOrEmpty(bedInfo.SSN))
                        free++;
                }
            }

            return (int)((double)(total - free) / total * 100);
        }

        /// <summary>
        /// Gets the facility time zone info.
        /// </summary>
        /// <returns>Facility TimeZoneInfo.</returns>
        public static TimeZoneInfo GetFacilityTimeZoneInfo(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetFacilityTimeZoneInfoByFacilityId(facilityId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        /// <summary>
        /// Gets the facility time zone info by facility id.
        /// </summary>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>Facility TimeZoneInfo.</returns>
        public static TimeZoneInfo GetFacilityTimeZoneInfoByFacilityId(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeZoneInfo timeZone = TimeZoneUtil.GetCachedTimeZone(facilityId);
                if (timeZone == null)
                {
                    FacilitySettings facilitySettings = FacadeManager.ConfigurationInterface.GetFacilitySettings(facilityId);
                    if (facilitySettings != null)
                    {
                        if (facilitySettings.FacilitySiteTimeZone != null)
                        {
                            timeZone = TimeZoneInfo.FindSystemTimeZoneById(facilitySettings.FacilitySiteTimeZone.displayName);
                            if (facilitySettings.LocalTimeAdjust == 0)                             
                            {
                                TimeZoneUtil.CacheTimeZoneForFacility(facilityId, timeZone);
                                return timeZone;
                            }
                            else
                            {
                                TimeSpan utcOffset = timeZone.GetUtcOffset(TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, timeZone));
                                TimeZoneInfo tzi = TimeZoneInfo.CreateCustomTimeZone("CTZ", utcOffset + TimeSpan.FromHours(facilitySettings.LocalTimeAdjust), "Custom Time Zone", "CustomTimeZone");
                                TimeZoneUtil.CacheTimeZoneForFacility(facilityId, tzi);
                                return tzi;
                            }
                        }
                        else
                            return TimeZoneInfo.Local;
                    }
                    else
                        return TimeZoneInfo.Local;
                }
                else
                    return timeZone;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Converts date from UTC.
        /// </summary>
        /// <param name="obj">The source object.</param>
        /// <returns></returns>
        public static T ConvertDateFromUTC<T>(T obj, II facilityId) where T : class
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeZoneUtil.ConvertObjectDates(obj, true, GetFacilityTimeZoneInfoByFacilityId(facilityId));
                return obj;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Converts the date to UTC.
        /// </summary>
        /// <param name="obj">The source object.</param>
        /// <returns></returns>
        public static T ConvertDateToUTC<T>(T obj, II facilityId) where T : class
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeZoneUtil.ConvertObjectDates(obj, false, GetFacilityTimeZoneInfoByFacilityId(facilityId));
                return obj;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }


}
