﻿using System;
using System.Linq.Expressions;
using System.IO;
using System.Windows;
using System.Collections.Generic;
using System.Windows.Input;
using System.ComponentModel;
using System.Collections.ObjectModel;
using BMSConfigurationWarlock.Base;
using System.Text.RegularExpressions;
using System.Xml;
using System.Text;
using System.Configuration;
using System.Linq;
using System.Collections.Specialized;

namespace BMSConfigurationWarlock
{
    public class NameOf<T>
    {
        public static string Property<TProp>(Expression<Func<T, TProp>> expression)
        {
            var body = expression.Body as MemberExpression;
            if (body == null)
                throw new ArgumentException("'expression' should be a member expression");
            return body.Member.Name;
        }
    }

    public class OverrideCursor : IDisposable
    {
        static Stack<Cursor> cursorStack = new Stack<Cursor>();

        public OverrideCursor(Cursor changeToCursor)
        {
            cursorStack.Push(changeToCursor);

            if (Mouse.OverrideCursor != changeToCursor)
                Mouse.OverrideCursor = changeToCursor;
        }

        public void Dispose()
        {
            cursorStack.Pop();

            Cursor cursor = cursorStack.Count > 0 ? cursorStack.Peek() : null;

            if (cursor != Mouse.OverrideCursor)
                Mouse.OverrideCursor = cursor;
        }
    }

    public class IndentXmlWriter : XmlTextWriter
    {

        public IndentXmlWriter(string file)
            : base(file, Encoding.UTF8)
        {
            this.Formatting = Formatting.Indented;
        }
        public override void WriteFullEndElement()
        {
            base.WriteEndElement();
        }
    }

    public class NotifyOnElementChangedObservableCollection<T> : ObservableCollection<T> where T : INotifyPropertyChanged
    {
        public event PropertyChangedEventHandler ElementPropertyChanged;

        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs e) 
        {
            if(e.NewItems != null)
                foreach (T item in e.NewItems)
                    item.PropertyChanged += new PropertyChangedEventHandler(itemPropertyChanged);
            if(e.OldItems != null)
                foreach (T item in e.OldItems)
                    item.PropertyChanged -= new PropertyChangedEventHandler(itemPropertyChanged);
        }

        void itemPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (ElementPropertyChanged != null)
                ElementPropertyChanged(sender, e);
        }

        public NotifyOnElementChangedObservableCollection (IEnumerable<T> collection) : base(collection)
        {
            if(collection != null)
                foreach(T item in collection)
                    item.PropertyChanged += new PropertyChangedEventHandler(itemPropertyChanged);
        }
    }

    public static class Utils
    {
        public static string GetUserPasswordDomain(string user, string password, string domain) 
        {
            return String.Format(Constants.UserPasswordDomainFormat, user, password, domain);
        }

        public static string GetValidLocations(string locations)
        {
            string result = string.Empty;

            string[] locationList = locations.Split(new char[] { ';' });
            foreach (string location in locationList)
                if (Directory.Exists(location))
                    result += location + ";";

            return result;
        }

        public static ObservableCollection<DatabaseConfiguration> GetDatabaseConfigurations(string configurations) 
        {
            ObservableCollection<DatabaseConfiguration> result = new ObservableCollection<DatabaseConfiguration>();

            string[] configurationPairs = configurations.Split(new char[] { ';' });
            foreach(string configurationPair in configurationPairs)
            {
                string[] pairElements = configurationPair.Split(new char[] {','});
                result.Add(new DatabaseConfiguration() { DatabaseKey = pairElements[0], DatabaseName = pairElements[1] });
            }
            
            return result;
        }

        public static void ShowSuccessMessage(this Window window, string message)
        {
            MessageBox.Show(window, message, (string)window.FindResource(Constants.SuccessWindow), MessageBoxButton.OK, MessageBoxImage.Information);
            ((IModelWindow)window).MessageModel.Messages += message + "\r";
        }

        public static void ShowErrorMessage(this Window window, string message) 
        {
            MessageBox.Show(window, message, (string)window.FindResource(Constants.ErrorWindow), MessageBoxButton.OK, MessageBoxImage.Error);
            ((IModelWindow)window).MessageModel.Messages += message + "\r";
        }

        public static string GetConnectionString(string DatabaseName, string DatabaseServer) 
        {
            return String.Format(Constants.SqlConnectionString, DatabaseServer, DatabaseName);
        }

        internal static string RegExp(System.Xml.XmlNode node, string servicesServer, string servicesPort, int httpsPortOffset)
        {
            string nText = node.InnerText;

            Match matchHttp = Constants.httpAddressRegex.Match(nText);

            //if matches found
            if (matchHttp.Length > 0)
            {
                //replace the http endpoint with the specified parameters
                nText = (matchHttp.Groups[2].Success) ? nText.Replace(matchHttp.Groups[2].ToString(), servicesServer) : node.InnerText;
                nText = (matchHttp.Groups[3].Success) ? nText.Replace(matchHttp.Groups[3].ToString(), servicesPort) : node.InnerText;
            }

            Match matchHttps = Constants.httpsAddressRegex.Match(nText);

            string servicesHttpsPort = (int.Parse(servicesPort) + httpsPortOffset).ToString();


            //if matches found
            if (matchHttps.Length > 0)
            {
                //replace the https endpoint with the specified parameters
                nText = (matchHttps.Groups[2].Success) ? nText.Replace(matchHttps.Groups[2].ToString(), servicesServer) : node.InnerText;
                nText = (matchHttps.Groups[3].Success) ? nText.Replace(matchHttps.Groups[3].ToString(), servicesHttpsPort) : node.InnerText;
            }

            return nText;
        }

        public static void ModifyConfigFile(string configurationPath, MainWindowModel model) 
        {
            //replace service addresses by using regular expressions
            XmlDocument doc = new XmlDocument();
            doc.Load(configurationPath);
            foreach (string attrPath in Constants.AttributePaths)
            {
                XmlNodeList nodes = doc.SelectNodes(attrPath);
                if (nodes.Count != 0)
                    foreach (XmlNode node in nodes)
                        //run the matching procedure
                        node.InnerText = Utils.RegExp(node, model.GeneralConfiguration.ServicesServer, model.GeneralConfiguration.ServicesPort, model.HttpsPortOffset);
            }
            XmlNodeList list = doc.SelectNodes(Constants.AppSettingsXPath);
            foreach (XmlNode node in list)
            {
                if (node.Attributes == null || node.Attributes[Constants.Key] == null) continue;

                string key = node.Attributes[Constants.Key].Value;
                //Audit
                if (string.IsNullOrEmpty(key)) continue;
                if (key.ToLower().Equals(Constants.AuditServer, StringComparison.InvariantCultureIgnoreCase))
                    node.Attributes[Constants.Value].Value = model.GeneralConfiguration.ServicesServer;
                else if (key.ToLower().Equals(Constants.AuditPort, StringComparison.InvariantCultureIgnoreCase))
                    node.Attributes[Constants.Value].Value = model.GeneralConfiguration.ServicesPort;
                //Key = Port
                else if (key.Equals(Constants.Port, StringComparison.InvariantCultureIgnoreCase))
                    node.Attributes[Constants.Value].Value = model.GeneralConfiguration.ServicesPort;
                //ReportingServicesURL
                else if (key.Equals(Constants.ReportingServicesUrl, StringComparison.InvariantCultureIgnoreCase))
                    node.Attributes[Constants.Value].Value = model.ReportingServicesUrl;
                //ReportingServices user_pass_dom
                else if (key.Equals(Constants.ReportingServicesUserPasswordDomain, StringComparison.InvariantCultureIgnoreCase))
                    node.Attributes[Constants.Value].Value = Utils.GetUserPasswordDomain(model.ReportsUserName, model.ReportsUserPassword, model.ReportsUserDomain);
                //super users key in the STS
                else if (key.Equals(Constants.SuperUsers, StringComparison.InvariantCultureIgnoreCase))
                    node.Attributes[Constants.Value].Value = model.SuperUsers;
                //domains key in the STS
                else 
                if (key.Equals(Constants.Domains, StringComparison.InvariantCultureIgnoreCase))
                {
                    string[] domains = model.ApplicationDomain.Split(new char[]{','}, StringSplitOptions.RemoveEmptyEntries);
                    string domainsList = string.Empty;
                    foreach(string domain in domains)
                        domainsList = domainsList + "," + domain + "|" + domain;
                    domainsList = domainsList.Substring(1);
                    node.Attributes[Constants.Value].Value = domainsList;
                }
            }

            XmlNodeList userNodeList = doc.SelectNodes(Constants.UserPrincipalNameXPath);
            foreach (XmlNode node in userNodeList)
            {
                if (node.Attributes == null || node.Attributes[Constants.Value] == null)
                    continue;
                node.Attributes[Constants.Value].Value = model.ReportsUserName + "@" + model.ReportsUserDomain;
            }
            //save the modified file and move on to the next one
            IndentXmlWriter xw = new IndentXmlWriter(configurationPath);
            doc.Save(xw);
            xw.Close();
        }

        internal static ObservableCollection<SqlScript> GetSqlScriptsCollection()
        {
            string[] sqlFiles = ConfigurationManager.AppSettings[Constants.SqlScriptsToRun].Split(new char[] { ',' });
            IEnumerable<string> sqlScripts = Directory.GetFiles(Constants.SqlScriptsFolder).Where(file => sqlFiles.Contains(Path.GetFileName(file), StringComparer.InvariantCultureIgnoreCase));

            return new ObservableCollection<SqlScript>(sqlScripts.Select(sqlScript => new SqlScript() { Name = sqlScript, Run = false, Variables = Utils.GetScriptVariables(sqlScript) }));
        }

        private static List<string> GetScriptVariables(string sqlScript)
        {
            MatchCollection matchCollection = Constants.sqlScriptVariableRegEx.Matches(File.ReadAllText(sqlScript));
            List<string> variables = new List<string>();
            foreach (Match match in matchCollection)
                if (!variables.Contains(match.Groups[1].Value, StringComparer.InvariantCultureIgnoreCase))
                    variables.Add(match.Groups[1].Value);
            return variables;
        }

        internal static string GetVariables(SqlScript sqlScript, MainWindowModel model)
        {
            string result = " ";

            foreach (string variable in sqlScript.Variables)
            {
                DatabaseConfiguration config;
                //variable in database config collection
                config = model.DatabaseConfigurationCollection.FirstOrDefault(item => item.DatabaseScriptVariable.Equals(variable, StringComparison.InvariantCultureIgnoreCase));
                if (config != null)
                    result += config.DatabaseScriptVariable + "=\"" + config.DatabaseName + "\" ";
                else
                {
                    //variable in variables collection
                    config = model.Variables.FirstOrDefault(item => item.DatabaseScriptVariable.Equals(variable, StringComparison.InvariantCultureIgnoreCase));
                    result += config.DatabaseScriptVariable + "=\"" + config.ScriptVariableValue + "\" ";
                }
            }

            return result;
        }
    }
}
