﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.ComponentModel.DataAnnotations;
using System.Web.Mvc;

namespace BMS.Web.Controllers.Shared
{
    /// <summary>
    /// Checks if to moments in time are equal or different to each other. Use only on the property that represents the hour part.
    /// </summary>
    public class CheckTimeEqualToIfAttribute : ValidationAttribute
    {
        public string DependentProperty { get; set; }
        public object TargetValue { get; set; }
        public string MinuteProperty { get; set; }
        public string CheckHourProperty { get; set; }
        public string CheckMinuteProperty { get; set; }
        public bool CheckEqual { get; set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="CheckEqualToAttribute"/> class.
        /// </summary>
        /// <param name="minuteProperty">The minute property associated to the current hour property.</param>
        /// <param name="checkHourProperty">The hour property of the time to check against.</param>
        /// <param name="checkMinuteProperty">The minute property of the time to check against.</param>
        /// <param name="checkEqual">if set to <c>true</c> the property is checked against dependentProperty for equality. If set to <c>false</c>,
        /// the property has to be different from dependentPropery.</param>
        /// <param name="dependentProperty">The dependent property.</param>
        /// <param name="targetValue">The target value for the dependent property.</param>
        public CheckTimeEqualToIfAttribute(string minuteProperty, string checkHourProperty, string checkMinuteProperty, bool checkEqual, string dependentProperty, object targetValue)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this.MinuteProperty = minuteProperty;
                this.CheckHourProperty = checkHourProperty;
                this.CheckMinuteProperty = checkMinuteProperty;
                this.CheckEqual = checkEqual;
                this.DependentProperty = dependentProperty;
                this.TargetValue = targetValue;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected override ValidationResult IsValid(object value, ValidationContext validationContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // get a reference to the property this validation depends upon
                var containerType = validationContext.ObjectInstance.GetType();
                var minuteField = containerType.GetProperty(this.MinuteProperty);
                var checkHourField = containerType.GetProperty(this.CheckHourProperty);
                var checkMinuteField = containerType.GetProperty(this.CheckMinuteProperty);
                var dependentField = containerType.GetProperty(this.DependentProperty);

                if (minuteField != null && checkHourField != null && checkMinuteField != null && dependentField != null)
                {
                    // get the value of the dependent property
                    var minuteValue = minuteField.GetValue(validationContext.ObjectInstance, null);
                    var checkHourValue = checkHourField.GetValue(validationContext.ObjectInstance, null);
                    var checkMinuteValue = checkMinuteField.GetValue(validationContext.ObjectInstance, null);
                    var dependentvalue = dependentField.GetValue(validationContext.ObjectInstance, null);

                    // compare the value against the target value
                    if ((dependentvalue == null && this.TargetValue == null) || (dependentvalue != null && dependentvalue.Equals(this.TargetValue)))
                        if (minuteValue != null && checkHourValue != null && checkMinuteValue != null)
                            if (((string)checkHourValue) + ((string)checkMinuteValue) == ((string)value) + ((string)minuteValue))
                                if (CheckEqual)
                                    return ValidationResult.Success;
                                else
                                {
                                    //member name is null here because Microsoft commented out the line of code that would have filled this field. Too bad, so here's the hack.
                                    var memberName = validationContext.MemberName ??
                                        validationContext.ObjectType.GetProperties().Where(p => p.GetCustomAttributes(false).OfType<DisplayAttribute>().
                                            Any(a => a.Name == validationContext.DisplayName)).Select(p => p.Name).FirstOrDefault() ??
                                        validationContext.DisplayName;
                                    //apparently Microsoft didn't bother to implement the constructor bellow, so it defaults to the constructor which doesn't have
                                    //member names specified.
                                    return new ValidationResult(this.ErrorMessage, new[] { memberName, MinuteProperty, CheckHourProperty, CheckMinuteProperty });
                                }
                }

                return ValidationResult.Success;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}