﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using Microsoft.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade.Data;
using BMS.Facade;
using BMS.ServicesWrapper.EIS;
using System.Web.Security;
using InfoWorld.HL7.ITS;
using BMS.Authentication;
using BMS.Utils;
using BMS.Facade.Fault;
using BMS.ServicesWrapper.BMService;
using BMS.Facade.Translators;
using BMS.Web.Controllers.Shared;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class EMSMobileLogonController : Controller
    {
        public ActionResult Index(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EMSMobileViewModel model = new EMSMobileViewModel();
                model.FacilityCode = code;
                if (string.IsNullOrEmpty(code))
                    model.DisplayFacilityMissingMessage = true;
                else
                {
                    model.Facility = EISFactory.InstanceWindows.GetFacilityByCode(code);
                    if (model.Facility == null || model.Facility.Id == null)
                        model.DisplayFacilityNotFoundMessage = true;
                    else
                    {
                        model.EMSUserList = new List<EmsStaff>();
                        List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(model.Facility.Id.root, model.Facility.Id).ToList();
                        if (flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                            model.DisplayEMSModuleInactive = true;
                        else
                        {
                            FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(model.Facility.Id);
                            if (string.IsNullOrEmpty(settings.EMSDefaultUserName) || string.IsNullOrEmpty(settings.EMSDefaultPassword))
                                model.DisplayFacilityUserNotFoundMessage = true;
                            else
                                model.LogonErrorMessage = TryLogon(settings, model.Facility.Id);
                            if (string.IsNullOrEmpty(model.LogonErrorMessage))
                            {
                                model.EMSUserList = FacadeManager.UserInterface.FilterEmsStaff(model.Facility);
                                model.EMSUserList = (from a in model.EMSUserList
                                                     orderby a.UserName
                                                     select a).ToList();
                            }
                        }
                    }
                }                                
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]        
        public ActionResult Index(EMSMobileViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TempData.Put<string>("SelectedUserName", input.SelectedUserName);
                TempData.Put<string>("SelectedUserId", input.SelectedUserId);
                User user = FacadeManager.UserInterface.GetProfile();
                return this.RedirectToAction<EMSMobileController>(act => act.Users(BaseController.EncryptQueryString(new string[] { "code" }, new string[] { input.FacilityCode }, user.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        private string TryLogon(FacilitySettings settings, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (HttpContext.User.Identity.IsAuthenticated)
                {
                    System.Web.SessionState.HttpSessionState currentSession = System.Web.HttpContext.Current.Session;
                    MvcApplication.SessionEnd(currentSession);
                    currentSession.Clear();
                    FormsAuthentication.SignOut();
                }
                try
                {
                    if (Membership.ValidateUser(settings.EMSDefaultUserName, Encryptor.Decrypt(settings.EMSDefaultPassword)))
                    {
                        User user = FacadeManager.UserInterface.GetProfile();
                        Facility facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                        FacadeManager.UserInterface.InsertLoggedUser(ProxyManager.GetCurrentSessionID2(), user.UserName, facility, facility.VistaSite.Visn);
                        if (user.EMSUser.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                            return "The user does not have the EMS Role associated!";
                        if (!FacadeManager.UserInterface.CheckBMSReadPermission(user, facilityId) || !FacadeManager.UserInterface.CheckBMSWritePermission(user, facilityId))
                            return "The user does not have permissions for the selected facility!";
                        else
                        {
                            FormsAuthentication.SetAuthCookie(settings.EMSDefaultUserName, false);
                            user.Facility = user.DefaultFacility = facility;
                            user.VistaSite = facility.VistaSite;
                            user.Visn = facility.VistaSite.Visn;
                            return null;
                        }
                    }
                    else
                        return "The user name or password provided is incorrect!";
                }
                catch (UserHasNoProfileException)
                {
                    return "The user does not have a profile. Please contact the administrator!";
                }
                catch (UserHasNoRoleException)
                {
                    return "The user does not have a role associated. Please contact the administrator!";
                }
                catch (Exception e)
                {
                    return "There was an error when trying to login: " + e.Message;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }

    [ValidateInput(false)]
    public class EMSMobileController : BaseController
    {                
        [ReadPermissionAuthFilter]
        public ActionResult Users(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {                
                EMSMobileViewModel model = new EMSMobileViewModel();
                model.SelectedUserName = TempData.Get<string>("SelectedUserName");
                model.SelectedUserId = TempData.Get<string>("SelectedUserId");
                if (string.IsNullOrEmpty(model.SelectedUserName) || string.IsNullOrEmpty(model.SelectedUserId))
                    return this.RedirectToAction<EMSMobileLogonController>(act => act.Index(model.FacilityCode));
                model.FacilityCode = QueryStrings["code"];
                model.Facility = EISFactory.Instance.GetFacilityByCode(model.FacilityCode);
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult Users(EMSMobileViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(input.SelectedUserName) || string.IsNullOrEmpty(input.SelectedUserId))
                    return this.RedirectToAction<EMSMobileLogonController>(act => act.Index(input.FacilityCode));                
                EmsStaff ems = FacadeManager.UserInterface.GetEmsStaff(new II(this.loggedUser.Domain, input.SelectedUserId));
                input.LoggedUser = loggedUser;
                if (string.IsNullOrEmpty(input.PIN))
                {
                    input.Facility = FacadeManager.EntityInterface.GetFacilityByCode(input.FacilityCode);
                    ModelState.AddModelError("PIN", "Please enter a PIN!");
                    return View(input);
                }
                if (!input.PIN.Equals(ems.Pin, StringComparison.InvariantCultureIgnoreCase))
                {
                    input.Facility = FacadeManager.EntityInterface.GetFacilityByCode(input.FacilityCode);
                    ModelState.AddModelError("PIN", "The supplied PIN is not correct!");
                    return View(input);
                }
                TempData.Put<string>("SelectedUserName", input.SelectedUserName);
                TempData.Put<string>("SelectedUserId", input.SelectedUserId);
                return this.RedirectToAction<EMSMobileController>(act => act.EMSList(EncryptQueryString(new string[] { "code" }, new string[] { input.FacilityCode }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilter]
        public ActionResult EMSList(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EMSMobileMainViewModel model = new EMSMobileMainViewModel();
                model.FacilityCode = QueryStrings["code"];
                model.UserName = TempData.Get<string>("SelectedUserName");
                model.UserId = TempData.Get<string>("SelectedUserId");
                if (string.IsNullOrEmpty(model.UserName) || string.IsNullOrEmpty(model.UserId))
                    return this.RedirectToAction<EMSMobileLogonController>(act => act.Index(model.FacilityCode));
                model.Facility = EISFactory.Instance.GetFacilityByCode(model.FacilityCode);
                List<Ward> wardList = FacadeManager.EntityInterface.GetWardsByFacility(model.Facility.Id).ToList();
                AntiXssEncoder xss = new AntiXssEncoder();
                List<BedCleanInfo> list = FacadeManager.BedInterface.FilterBedCleanAssignedAndPending(wardList, xss.Encode(model.UserName), model.Facility.Id);
                model.AssignedBedCleanList = list.Where(a => !string.IsNullOrEmpty(a.AcceptedBy) && a.AcceptedBy.Equals(xss.Encode(model.UserName), StringComparison.InvariantCultureIgnoreCase)).ToList();
                model.PendingBedCleanList = list.Where(a => string.IsNullOrEmpty(a.AcceptedBy)).ToList();
                model.AssignedBedCleanList = (from a in model.AssignedBedCleanList orderby a.WardName, a.Bed.Name select a).ToList();
                model.PendingBedCleanList = (from a in model.PendingBedCleanList orderby a.WardName, a.Bed.Name select a).ToList();
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult EMSList(EMSMobileMainViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(input.UserName) || string.IsNullOrEmpty(input.UserId))
                    return this.RedirectToAction<EMSMobileLogonController>(act => act.Index(input.FacilityCode));
                input.Facility = FacadeManager.EntityInterface.GetFacilityByCode(input.FacilityCode);
                DateTime dateNowOnFacility = TimeZoneInfo.ConvertTime(DateTime.UtcNow, FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(input.Facility.Id));
                AntiXssEncoder xss = new AntiXssEncoder();
                if (!string.IsNullOrEmpty(input.ButtonAcceptBedClean))
                {
                    BedClean bedClean = FacadeManager.BedInterface.GetBedCleanById(new II(input.Facility.Id.root, input.SelectedBedCleanId), input.Facility.Id);
                    bedClean.AcceptedBy = xss.Encode(input.UserName);
                    bedClean.LastEditedBy = this.loggedUser.UserName;
                    bedClean.LastEditDate = bedClean.AcceptedDate = dateNowOnFacility;
                    DateTimeConverter.ConvertDateToUTC(bedClean, input.Facility.Id);
                    BMSFactory.BedManagerOperationsClient.UpdateBedCleaningOperation(bedClean.ToDataContract());
                }
                else if (!string.IsNullOrEmpty(input.ButtonCompleteBedClean))
                {
                    BedClean bedClean = FacadeManager.BedInterface.GetBedCleanById(new II(input.Facility.Id.root, input.SelectedBedCleanId), input.Facility.Id);
                    bedClean.CompletedBy = xss.Encode(input.UserName);
                    bedClean.LastEditedBy = this.loggedUser.UserName;
                    bedClean.LastEditDate = bedClean.CompletedDate = dateNowOnFacility;
                    DateTimeConverter.ConvertDateToUTC(bedClean, input.Facility.Id);
                    BMSFactory.BedManagerOperationsClient.UpdateBedCleaningOperation(bedClean.ToDataContract());
                    FacadeManager.BedInterface.SendCleanDirtyBedNotifications(bedClean);
                }

                TempData.Put<string>("SelectedUserName", input.UserName);
                TempData.Put<string>("SelectedUserId", input.UserId);
                return this.RedirectToAction<EMSMobileController>(act => act.EMSList(EncryptQueryString(new string[] { "code" }, new string[] { input.FacilityCode }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        
    }
}
