﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using BMS.Facade.Data;
using BMS.Facade;
using InfoWorld.HL7.ITS;
using BMS.Web.App_GlobalResource;
using BMS.Utils;
using System.ComponentModel.DataAnnotations;
using BMS.Web.Controllers.Shared;

namespace BMS.Web.Models
{
    /// <summary>
    /// Facility Diversion View Model class.
    /// </summary>
    public class FacilityDiversionViewModel : BaseViewModel
    {
        #region Public Members
        /// <summary>
        /// Gets or sets the location list.
        /// </summary>
        /// <value>
        /// The location list.
        /// </value>
        public IList<CD> LocationList { get; set; }

        /// <summary>
        /// Gets or sets the location.
        /// </summary>
        /// <value>
        /// The location.
        /// </value>
        public string Location { get; set; }

        /// <summary>
        /// Gets or sets the location object.
        /// </summary>
        /// <value>
        /// The location object.
        /// </value>
        public CD DiversionLocation { get; set; }

        /// <summary>
        /// Gets or sets the facility.
        /// </summary>
        /// <value>
        /// The facility.
        /// </value>
        public Place Facility { get; set; }

        /// <summary>
        /// Gets or sets the status.
        /// </summary>
        /// <value>
        /// The status.
        /// </value>
        public CD Status { get; set; }

        /// <summary>
        /// Gets or sets the start date of diversion.
        /// </summary>
        /// <value>
        /// The start date.
        /// </value>
        public DateTime? StartDate { get; set; }

        /// <summary>
        /// Gets or sets the start date message error.
        /// </summary>
        /// <value>
        /// The start date message error.
        /// </value>
        public string StartDateMessageError { get; set; }

        /// <summary>
        /// Gets or sets the start date string.
        /// </summary>
        /// <value>
        /// The start date string.
        /// </value>
        public string StartDateString
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (StartDate == null)
                        return string.Empty;
                    else
                        return StartDate.Value.ToString(Strings.USDate);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (value == null)
                    {
                        StartDate = null;
                        StartDateMessageError = Strings.StartDateIsRequired;
                    }
                    else
                    {
                        try
                        {
                            StartDate = DateTime.Parse(value);
                            StartDateMessageError = string.Empty;
                        }
                        catch
                        {
                            StartDateMessageError = Strings.StartDateIsNotInCorrectFormat;
                        }
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the hour of the diversion start date.
        /// </summary>
        /// <value>
        /// The hour of the diversion start date.
        /// </value>
        public string HourStartDateSelected { get; set; }

        /// <summary>
        /// Gets or sets the minutes of the diversion start date.
        /// </summary>
        /// <value>
        /// The minutes of the diversion start date.
        /// </value>
        public string MinuteStartDateSelected { get; set; }

        /// <summary>
        /// Gets or sets the end date of diversion.
        /// </summary>
        /// <value>
        /// The end date.
        /// </value>
        public DateTime? EndDate { get; set; }

        /// <summary>
        /// Gets or sets the end date message error.
        /// </summary>
        /// <value>
        /// The end date message error.
        /// </value>
        public string EndDateMessageError { get; set; }

        public string EndDateString
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (EndDate == null)
                        return string.Empty;
                    else
                        return EndDate.Value.ToString(Strings.USDate);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (value == null)
                    {
                        EndDate = null;
                        EndDateMessageError = Strings.EndDateIsRequired;
                    }
                    else
                    {
                        try
                        {
                            EndDate = DateTime.Parse(value);
                            EndDateMessageError = string.Empty;
                        }
                        catch
                        {
                            EndDateMessageError = Strings.EndDateIsNotInCorrectFormat;
                        }
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the hour of diversion end date selected.
        /// </summary>
        /// <value>
        /// The hour of diversion end date selected.
        /// </value>
        public string HourEndDateSelected { get; set; }

        /// <summary>
        /// Gets or sets the minutes of diversion end date selected.
        /// </summary>
        /// <value>
        /// The minutes of diversion end date selected.
        /// </value>
        public string MinuteEndDateSelected { get; set; }

        /// <summary>
        /// Gets or sets the diversion comments.
        /// </summary>
        /// <value>
        /// Diversion comments.
        /// </value>
        [StringLength(Constants.COMMENTS_MAX_LENGTH, ErrorMessageResourceType = typeof(Strings), ErrorMessageResourceName = "FacilityDiversionErrorCommentsBad")]
        [ValidateXss]
        public string Comments { get; set; }

        public string WardGroup { get; set; }

        public string TreatingSpecialty { get; set; }

        public IEnumerable<string> WardGroups { get; set; }
        public IEnumerable<string> TreatingSpecialties { get; set; }

        public bool IsFromVisn { get; set; }

        /// <summary>
        /// Gets or sets the diversion list.
        /// </summary>
        /// <value>
        /// The diversion list.
        /// </value>
        public IList<DiversionViewModel> DiversionList { get; set; }

        /// <summary>
        /// Gets or sets the flags list.
        /// </summary>
        /// <value>
        /// The flags list.
        /// </value>
        public CDList FlagsList { get; set; }

        /// <summary>
        /// Gets or sets the current button.
        /// </summary>
        /// <value>
        /// The current button.
        /// </value>
        public string CurrentButton { get; set; }

        /// <summary>
        /// Gets or sets the id of diversion.
        /// </summary>
        /// <value>
        /// The id of diversion.
        /// </value>
        public string Id { get; set; }

        /// <summary>
        /// Gets or sets the facility already exists.
        /// </summary>
        /// <value>
        /// The facility already exists.
        /// </value>
        public string FacilityAlreadyExists { get; set; }

        /// <summary>
        /// Gets or sets the facility add display.
        /// </summary>
        /// <value>
        /// The facility add display.
        /// </value>
        public string FacilityAddDisplay { get; set; }

        /// <summary>
        /// Gets or sets the existing facility extension.
        /// </summary>
        /// <value>
        /// The existing facility extension.
        /// </value>
        public string ExistingFacilityExtension { get; set; }

        /// <summary>
        /// Gets or sets the existing facility root.
        /// </summary>
        /// <value>
        /// The existing facility root.
        /// </value>
        public string ExistingFacilityRoot { get; set; }

        /// <summary>
        /// Gets or sets the name of the existing facility.
        /// </summary>
        /// <value>
        /// The name of the existing facility.
        /// </value>
        public string ExistingFacilityName { get; set; }

        /// <summary>
        /// Gets or sets the button all diversions.
        /// </summary>
        /// <value>
        /// The button all diversions.
        /// </value>
        public string ButtonAllDiversions { get; set; }

        /// <summary>
        /// Gets or sets the button show report.
        /// </summary>
        /// <value>
        /// The button show report.
        /// </value>
        public string ButtonShowReport { get; set; }

        /// <summary>
        /// Gets or sets the button current diversions.
        /// </summary>
        /// <value>
        /// The button current diversions.
        /// </value>
        public string ButtonCurrentDiversions { get; set; }

        /// <summary>
        /// Gets or sets the button add.
        /// </summary>
        /// <value>
        /// The button add.
        /// </value>
        public string ButtonAdd { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is add confirmation.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is add confirmation; otherwise, <c>false</c>.
        /// </value>
        public bool IsAddConfirmation { get; set; }

        #endregion
    }

    /// <summary>
    /// Diversion view model class. Contains information about a diversion, used in facility diversion view.
    /// </summary>
    public class DiversionViewModel
    {
        #region Public Properties

        /// <summary>
        /// Gets or sets the id.
        /// </summary>
        /// <value>
        /// The id.
        /// </value>
        public string Id { get; set; }

        /// <summary>
        /// Gets or sets the location.
        /// </summary>
        /// <value>
        /// The location.
        /// </value>
        public string Location { get; set; }

        /// <summary>
        /// Gets or sets the start date.
        /// </summary>
        /// <value>
        /// The start date.
        /// </value>
        public DateTime StartDateTime { get; set; }

        /// <summary>
        /// Gets or sets the start date.
        /// </summary>
        /// <value>
        /// The start date.
        /// </value>
        public string StartDate { get; set; }

        /// <summary>
        /// Gets or sets the start time.
        /// </summary>
        /// <value>
        /// The start time.
        /// </value>
        public string StartTime { get; set; }

        /// <summary>
        /// Gets or sets the end date.
        /// </summary>
        /// <value>
        /// The end date.
        /// </value>
        public string EndDate { get; set; }

        /// <summary>
        /// Gets or sets the end time.
        /// </summary>
        /// <value>
        /// The end time.
        /// </value>
        public string EndTime { get; set; }

        /// <summary>
        /// Gets or sets the status.
        /// </summary>
        /// <value>
        /// The status.
        /// </value>
        public string Status { get; set; }

        /// <summary>
        /// Gets or sets the entered by.
        /// </summary>
        /// <value>
        /// The entered by.
        /// </value>
        public string EnteredBy { get; set; }

        /// <summary>
        /// Gets or sets the entered date.
        /// </summary>
        /// <value>
        /// The entered date.
        /// </value>
        public string EnteredDate { get; set; }

        /// <summary>
        /// Gets or sets the last edit by.
        /// </summary>
        /// <value>
        /// The last edit by.
        /// </value>
        public string LastEditBy { get; set; }

        /// <summary>
        /// Gets or sets the duration.
        /// </summary>
        /// <value>
        /// The duration.
        /// </value>
        public TimeSpan Duration { get; set; }

        /// <summary>
        /// Gets or sets the duration message.
        /// </summary>
        /// <value>
        /// The duration message.
        /// </value>
        public string DurationMessage { get; set; }

        public string ForegroundDuration { get; set; }

        #endregion

        public static DiversionViewModel GetViewModel(Diversion model, DateTime dateTimeNowOnCurrentFacility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (model == null)
                    return null;
                var res = new DiversionViewModel();

                res.Id = model.Id.extension;
                res.Location = string.Format("{0}|{1}|{2}-{3}", model.Location.displayName, String.IsNullOrEmpty(model.WardGroup) ? "All" : model.WardGroup,
                    String.IsNullOrEmpty(model.TreatingSpecialty) ? "All" : model.TreatingSpecialty, model.Comments);

                res.Status = model.Status;
                res.StartDateTime = model.StartDate;
                res.StartDate = string.Format("{0:MM/dd/yyyy}", model.StartDate);
                res.StartTime = string.Format("{0:HH:mm}", model.StartDate);
                res.EndDate = string.Format("{0:MM/dd/yyyy}", model.EndDate);
                res.EndTime = string.Format("{0:HH:mm}", model.EndDate);
                res.EnteredBy = model.EnteredBy;
                res.EnteredDate = string.Format("{0:MM/dd/yy} - {1:HH:mm}", model.EnteredDate, model.EnteredDate);
                res.LastEditBy = model.LastEditBy;
                //set the values for properties used in view

                if (model.Status == Constants.Yes)
                    res.Duration = dateTimeNowOnCurrentFacility.Subtract(model.StartDate);
                else
                    res.Duration = model.EndDate.Subtract(model.StartDate);

                if (res.Duration.Minutes > -1)
                    res.DurationMessage = string.Format("{0:00}:{1:00}:{2:00}", (int)(res.Duration.TotalDays), (int)(res.Duration.Hours), (int)(res.Duration.Minutes));
                else
                    res.DurationMessage = Strings.Error;
                if (model.Status == Constants.Yes && res.DurationMessage != Strings.Error)
                    res.ForegroundDuration = "Red";
                else
                    res.ForegroundDuration = "Black";
                return res;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}