﻿using System;
using System.Web.Mvc;
using BMS.Web.Models;
using System.Xml.Serialization;
using System.IO;
using System.Text;
using System.Linq;
using System.Collections.Generic;
using System.Web.Script.Serialization;
using BMS.Facade;
using VI = BMS.VistaIntegration.FacadeContracts;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using System.Text.RegularExpressions;
using Microsoft.Web.Mvc;
using System.ServiceModel;
using BMS.Web.App_GlobalResource;
using BMS.Utils;
using BMS.VistaIntegration.FacadeContracts.Admin;
using BMS.Web.Models.Shared;
using BMS.Web.Controllers.Shared;
using System.Configuration;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class VistaIntegrationController : BaseController
    {
        public const string DeleteMessage = "DeleteMessage";

        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return this.RedirectToAction<VistaIntegrationController>(act => act.VistASites((string)null));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region VistaSite

        [ReadPermissionAuthFilterAttribute]
        public ActionResult VistASites(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ViewBag.VistASitesSelectedBackgroundColor = Strings.AnchorBackgroundColor;

                VI.VistASite vistaSite = GetVistaSite(string.IsNullOrEmpty(p) ? null : QueryStrings["vistaSiteId"]);
                VistaSitesViewModel model = VistaSiteToModel(vistaSite);
                ViewBag.StartIndex = 0;
                ViewBag.EndIndex = model.VistaSites.Count - 1;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult TestMDWSConnection(string model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                JavaScriptSerializer jss = new JavaScriptSerializer();
                VistaSitesViewModel viewModel = jss.Deserialize<VistaSitesViewModel>(model);
                string result = "Please try again";
                result = FacadeManager.VistAIntegrationInterface.TestConnection(ModelToVistaSite(viewModel), DataRetrievalMethod.MDWS);

                if (string.IsNullOrWhiteSpace(result))
                    result = Strings.VistaMDWSConnectionSuccessfullyTested;

                return Json(result);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult TestODBCConnection(string model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                JavaScriptSerializer jss = new JavaScriptSerializer();
                VistaSitesViewModel viewModel = jss.Deserialize<VistaSitesViewModel>(model);
                string result = "Please try again";
                result = FacadeManager.VistAIntegrationInterface.TestConnection(ModelToVistaSite(viewModel), DataRetrievalMethod.ODBC);

                if (string.IsNullOrWhiteSpace(result))
                    result = Strings.VistaODBCConnectionSuccessfullyTested;

                return Json(result);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [UpdatePermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult VistASites([Bind(Exclude = "RememberMe")] VistaSitesViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VI.VistASite vistaSite = ModelToVistaSite(model);
                if (ModelState.IsValid)
                {
                    AntiXssEncoder xss = new AntiXssEncoder();
                    vistaSite.OdbcConnectionString = xss.Encode(vistaSite.OdbcConnectionString);
                    FacadeManager.ConfigurationInterface.SaveVistASite(vistaSite, loggedUser.Domain);
                    return this.RedirectToAction<VistaIntegrationController>(act => act.VistASites(EncryptQueryString(new string[] { "vistaSiteId" }, new string[] { vistaSite.Id }, loggedUser.Salt)));
                }
                return View(FillSitesModel(model));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private VistaSitesViewModel FillSitesModel(VistaSitesViewModel oldModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VistaSitesViewModel newModel = oldModel;
                if (oldModel == null)
                    newModel = new VistaSitesViewModel();

                if (newModel.TestResultMDWSConnection == null)
                    newModel.TestResultMDWSConnection = string.Empty;

                newModel.LoggedUser = this.loggedUser;
                //Get vista sites from EIS
                newModel.VistaEntities = FacadeManager.EntityInterface.GetVistaSites().Select<VistaSite, VistaSite>(a => new VistaSite() { Id = a.Id, Name = a.Name + " - " + a.Number, Code = a.Code, Number = a.Number, Visn = a.Visn }).OrderBy(vistaSite => vistaSite.Name).ToList();

                newModel.VistaSites = FacadeManager.ConfigurationInterface.GetVistaSites(loggedUser.Domain).OrderBy(vistaSite => vistaSite.Name).ToList();
                VistaSite site = null;
                foreach (VI.VistASite vs in newModel.VistaSites)
                {
                    site = newModel.VistaEntities.Where(a => a.Id.extension.Equals(vs.Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    vs.Name = vs.Name + " - " + site.Number;
                }
                newModel.TimeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);

                List<string> mdwsUrlKeys = ConfigurationManager.AppSettings.AllKeys.Where(a => a.StartsWith(Constants.Mdws_Endpoint_Url)).ToList();
                newModel.MdwsEndpointConfigKeyList = new Dictionary<string, string>();
                foreach (string key in mdwsUrlKeys)
                    newModel.MdwsEndpointConfigKeyList.Add(key, key + " - " + ConfigurationManager.AppSettings[key]);

                return newModel;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private VI.VistASite ModelToVistaSite(VistaSitesViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeZoneInfo info = TimeZoneInfo.Local;
                try
                {
                    info = TimeZoneInfo.FindSystemTimeZoneById(FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone).Find(tz => tz.code == model.VistaSiteTimeZone).displayName);
                }
                catch { }
                VistaSite vs = FacadeManager.EntityInterface.GetVistaSite(new II(this.loggedUser.Domain, model.VistaSiteId));
                AntiXssEncoder xss = new AntiXssEncoder();
                VI.VistASite result = new VI.VistASite()
                {
                    Number = vs.Number,
                    Name = vs.Name,
                    Id = model.VistaSiteId,
                    OdbcConnectionString = xss.Decode(model.VistaOdbcConnectionString),
                    OdbcPassword = model.VistaOdbcPassword,
                    OdbcUser = model.VistaOdbcUser,
                    TimeZone = info,
                    MdwsEndpointConfigKey = model.MdwsEndpointConfigKey
                };

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private VistaSitesViewModel VistaSiteToModel(VI.VistASite site)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VistaSitesViewModel model = FillSitesModel(null);
                model.VistASiteNumber = site.Number;
                model.VistaSiteName = site.Name;
                model.VistaSiteId = site.Id;
                model.MdwsEndpointConfigKey = site.MdwsEndpointConfigKey;
                AntiXssEncoder xss = new AntiXssEncoder();
                model.VistaOdbcConnectionString = xss.Decode(site.OdbcConnectionString);
                model.VistaOdbcPassword = site.OdbcPassword;
                model.VistaOdbcUser = site.OdbcUser;
                model.VistaSiteTimeZone = site.TimeZone != null ? FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone).Find(tz => tz.displayName == site.TimeZone.Id).code : null;

                return model;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Schedulers

        [ReadPermissionAuthFilterAttribute]
        public ActionResult Schedulers(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ViewBag.SchedulersSelectedBackgroundColor = Strings.AnchorBackgroundColor;
                VI.Admin.JobCalendar calendar = GetScheduler(string.IsNullOrEmpty(p) ? null : QueryStrings["schedulerId"]);
                SchedulersViewModel model = SchedulerToModel(calendar);
                if (TempData.ContainsKey(DeleteMessage))
                    ViewData[DeleteMessage] = TempData[DeleteMessage];
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [UpdatePermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult Schedulers([Bind(Exclude = "RememberMe")] SchedulersViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VI.Admin.JobCalendar calendar = ModelToScheduler(model);
                if (ModelState.IsValid)
                {
                    FacadeManager.ConfigurationInterface.SaveCalendar(calendar, loggedUser.Domain);
                    return this.RedirectToAction<VistaIntegrationController>(act => act.Schedulers(EncryptQueryString(new string[] { "schedulerId" }, new string[] { calendar.Id }, loggedUser.Salt)));
                }
                //model has errors
                SchedulersViewModel newModel = FillSchedulersModel(model);
                return View(newModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool IsSchedulerUsedByVistAIntegration(string schedulerId)
        {
            var sites = FacadeManager.ConfigurationInterface.GetVistaSites(loggedUser.Domain);
            foreach (var site in sites)
            {
                if (site.DataRetrievalDetails.Values.FirstOrDefault(s => s.JobCalendar.Id == schedulerId) != null)
                    return true;
            }
            return false;
        }

        private bool IsSchedulerUsedByNumi(string schedulerId)
        {
            var numiData = FacadeManager.ConfigurationInterface.GetNumiEntries(loggedUser.Domain);
            foreach (var data in numiData)
            {
                if (data.Calendar.Id == schedulerId)
                    return true;
            }
            return false;
        }

        private bool IsSchedulerUsedByWhiteboardReport(string schedulerId)
        {
            var entries = FacadeManager.ConfigurationInterface.GetWhiteboardReportEntries(this.loggedUser.Domain);
            foreach (var entry in entries)
            {
                if (entry.Calendar.Id == schedulerId)
                    return true;
            }
            return false;
        }

        private string GetDeleteMessage(string schedulerId)
        {
            string format = "Cannot delete scheduler beceause is used by {0}.";
            if (IsSchedulerUsedByVistAIntegration(schedulerId))
                return string.Format(format, "VistA Integration");
            if (IsSchedulerUsedByNumi(schedulerId))
                return string.Format(format, "Numi");
            if (IsSchedulerUsedByWhiteboardReport(schedulerId))
                return string.Format(format, "Whiteboard Report");

            return null;
        }

        [ReadPermissionAuthFilter]
        public ActionResult DeleteScheduler(string p)
        {
            string message = GetDeleteMessage(QueryStrings["schedulerId"]);
            if (message != null)
            {
                TempData[DeleteMessage] = message;
                return this.RedirectToAction<VistaIntegrationController>(act => act.Schedulers(p));
            }

            SchedulersViewModel model = new SchedulersViewModel()
            {
                CalendarId = QueryStrings["schedulerId"],
                LoggedUser = loggedUser
            };
            VI.Admin.JobCalendar calendar = GetScheduler(string.IsNullOrEmpty(p) ? null : model.CalendarId);
            model.CalendarName = calendar.Name;
            return View(model);
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult DeleteScheduler([Bind(Exclude = "RememberMe")] SchedulersViewModel model)
        {
            if (IsSchedulerUsedByVistAIntegration(model.CalendarId) ||
                IsSchedulerUsedByNumi(model.CalendarId) ||
                IsSchedulerUsedByWhiteboardReport(model.CalendarId))
                throw new InvalidOperationException();

            FacadeManager.ConfigurationInterface.DeleteCalendar(model.CalendarId, loggedUser.Domain);
            return this.RedirectToAction(s => s.Schedulers((string)null));
        }


        private VI.Admin.JobCalendar ModelToScheduler(SchedulersViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VI.Admin.JobCalendar calendar = new VI.Admin.JobCalendar();

                calendar.Name = model.CalendarName;
                calendar.Id = model.CalendarId;
                calendar.Occurs = (OccurrenceInterval)Enum.Parse(typeof(OccurrenceInterval), model.OccurrenceInterval);
                calendar.Recurs = model.Recurs;
                calendar.StartTime = new TimeSpan(Convert.ToInt32(model.StartHour), Convert.ToInt32(model.StartMinute), 0);
                if (model.OccursOnce)
                    calendar.EndTime = calendar.StartTime;
                else
                {
                    calendar.EndTime = new TimeSpan(Convert.ToInt32(model.EndHour), Convert.ToInt32(model.EndMinute), 0);
                    calendar.RunInterval = model.RunInterval;
                    calendar.RunIntervalUnit = (RunInterval)Enum.Parse(typeof(RunInterval), model.RunIntervalUnit);
                }

                if (calendar.Occurs == OccurrenceInterval.Month)
                {
                    VI.Admin.RunOnDay runOnDay = new VI.Admin.RunOnDay();
                    if (model.SpecificDayOfSpecificWeek)
                    {
                        runOnDay.IsDayOfWeekIndexSpecified = true;
                        runOnDay.DayIndex = model.IndexOfWeekday;
                        runOnDay.Day = (DayOfWeek)Enum.Parse(typeof(DayOfWeek), model.Weekday);
                    }
                    else
                        runOnDay.DayIndex = model.IndexOfDayInMonth;
                    calendar.RunOnDays = new List<VI.Admin.RunOnDay>() { runOnDay };
                }
                if (calendar.Occurs == OccurrenceInterval.Week)
                {
                    calendar.RunOnDays = new List<VI.Admin.RunOnDay>();
                    if (model.MondaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Monday });
                    if (model.TuesdaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Tuesday });
                    if (model.WednesdaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Wednesday });
                    if (model.ThursdaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Thursday });
                    if (model.FridaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Friday });
                    if (model.SaturdaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Saturday });
                    if (model.SundaySelected)
                        calendar.RunOnDays.Add(new VI.Admin.RunOnDay() { Day = DayOfWeek.Sunday });
                }
                calendar.TimeZoneId = TimeZoneInfo.FindSystemTimeZoneById(FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone).Find(tz => tz.code == model.TimeZone).displayName).Id;
                return calendar;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private SchedulersViewModel SchedulerToModel(VI.Admin.JobCalendar calendar)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SchedulersViewModel model = FillSchedulersModel(null);
                model.CalendarName = calendar.Name;
                model.CalendarId = calendar.Id;
                model.OccurrenceInterval = calendar.Occurs.ToString("G");
                model.Recurs = calendar.Recurs;
                model.StartHour = calendar.StartTime.Hours.ToString("D2");
                model.StartMinute = calendar.StartTime.Minutes.ToString("D2");
                model.EndHour = calendar.EndTime.Hours.ToString("D2");
                model.EndMinute = calendar.EndTime.Minutes.ToString("D2");
                if (calendar.StartTime == calendar.EndTime)
                    model.OccursOnce = true;
                else
                {
                    model.RunInterval = calendar.RunInterval;
                    model.RunIntervalUnit = calendar.RunIntervalUnit.ToString("G");
                }
                if (calendar.Occurs == OccurrenceInterval.Month && calendar.RunOnDays != null && calendar.RunOnDays.Count > 0)
                {
                    VI.Admin.RunOnDay runOnDay = calendar.RunOnDays[0];
                    if (runOnDay.IsDayOfWeekIndexSpecified)
                    {
                        model.IndexOfWeekday = runOnDay.DayIndex;
                        model.Weekday = runOnDay.Day.ToString("G");
                        model.SpecificDayOfSpecificWeek = true;
                    }
                    else
                        model.IndexOfDayInMonth = runOnDay.DayIndex;

                }
                if (calendar.Occurs == OccurrenceInterval.Week && calendar.RunOnDays != null)
                    foreach (VI.Admin.RunOnDay runOnDay in calendar.RunOnDays)
                    {
                        if (runOnDay.Day == DayOfWeek.Monday)
                            model.MondaySelected = true;
                        if (runOnDay.Day == DayOfWeek.Tuesday)
                            model.TuesdaySelected = true;
                        if (runOnDay.Day == DayOfWeek.Wednesday)
                            model.WednesdaySelected = true;
                        if (runOnDay.Day == DayOfWeek.Thursday)
                            model.ThursdaySelected = true;
                        if (runOnDay.Day == DayOfWeek.Friday)
                            model.FridaySelected = true;
                        if (runOnDay.Day == DayOfWeek.Saturday)
                            model.SaturdaySelected = true;
                        if (runOnDay.Day == DayOfWeek.Sunday)
                            model.SundaySelected = true;
                    }
                model.TimeZone = calendar.TimeZoneId != null ? FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone).Find(tz => tz.displayName == calendar.TimeZoneId).code : null;
                return model;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private SchedulersViewModel FillSchedulersModel(SchedulersViewModel oldModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SchedulersViewModel newModel = oldModel;
                if (oldModel == null)
                    newModel = new SchedulersViewModel();

                newModel.LoggedUser = this.loggedUser;
                AntiXssEncoder xss = new AntiXssEncoder();
                newModel.Calendars = FacadeManager.ConfigurationInterface.GetDefinedCalendars(loggedUser.Domain).OrderBy(jobCal => jobCal.Name).ToList();
                newModel.Calendars.ForEach(a => { a.Name = xss.Decode(a.Name); });

                ViewBag.StartIndex = 0;
                ViewBag.EndIndex = newModel.Calendars.Count - 1;
                newModel.TimeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                newModel.DaysOfWeek = new List<string>() 
            { 
                DayOfWeek.Monday.ToString("G"), 
                DayOfWeek.Tuesday.ToString("G"),
                DayOfWeek.Wednesday.ToString("G"),
                DayOfWeek.Thursday.ToString("G"),
                DayOfWeek.Friday.ToString("G"),
                DayOfWeek.Saturday.ToString("G"),
                DayOfWeek.Sunday.ToString("G") 
            };
                newModel.OccurrenceIntervals = new List<string>()
            {
                OccurrenceInterval.Day.ToString("G"),
                OccurrenceInterval.Week.ToString("G"),
                OccurrenceInterval.Month.ToString("G")
            };
                newModel.RunIntervals = new List<string>() 
            {
                RunInterval.Hour.ToString("G"),
                RunInterval.Minute.ToString("G")
            };
                newModel.RetrievalMethods = new List<string>() 
            {
                DataRetrievalMethod.MDWS.ToString("G"),
                DataRetrievalMethod.ODBC.ToString("G")
            };

                //default values for validators
                if (newModel.IndexOfDayInMonth == 0)
                    newModel.IndexOfDayInMonth = 1;
                if (newModel.IndexOfWeekday == 0)
                    newModel.IndexOfWeekday = 1;
                if (newModel.RunInterval == 0)
                    newModel.RunInterval = 1;

                return newModel;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Categories

        [ReadPermissionAuthFilterAttribute]
        public ActionResult Categories(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return View(GetCategoriesModel(string.IsNullOrEmpty(p) ? null : QueryStrings["vistaSiteId"], string.IsNullOrEmpty(p) ? VI.Admin.VistaDataType.None.ToString("G") : SplitCamelCase(QueryStrings["dataType"])));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [UpdatePermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult Categories(CategoriesViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (model.SelectedDataType == null)
                    model.SelectedDataType = VI.Admin.VistaDataType.None.ToString("G");

                CategoriesViewModel newModel = new CategoriesViewModel();
                if (model.InvisibleButton != null)
                {
                    //newModel = GetCategoriesModel(model.VistaSiteId, model.SelectedDataType);
                    return this.RedirectToAction<VistaIntegrationController>(act => act.Categories(EncryptQueryString(new string[] { "vistaSiteId", "dataType" }, new string[] { model.VistaSiteId, ((VI.Admin.VistaDataType)Enum.Parse(typeof(VI.Admin.VistaDataType), model.SelectedDataType.Replace(" ", ""))).ToString("G") }, loggedUser.Salt)));
                }

                newModel = FillCategoriesModel(model);

                if (model.SaveButton != null || model.RemoveButton != null)
                {
                    VI.VistASite vistaSite = GetVistaSite(model.VistaSiteId);
                    VI.Admin.VistaDataType dataType = VI.Admin.VistaDataType.None;
                    bool success = Enum.TryParse<VI.Admin.VistaDataType>(model.SelectedDataType.Replace(" ", ""), out dataType);
                    if (!success)
                        return View(newModel);

                    if (dataType != VI.Admin.VistaDataType.None)
                    {
                        if (model.SaveButton != null)
                        {
                            //save
                            VI.Admin.DataRetrieval dataRetrieval = null;
                            if (vistaSite.DataRetrievalDetails.ContainsKey(dataType))
                                dataRetrieval = vistaSite.DataRetrievalDetails[dataType];
                            else
                            {
                                dataRetrieval = new VI.Admin.DataRetrieval();
                                vistaSite.DataRetrievalDetails.Add(dataType, dataRetrieval);
                            }
                            dataRetrieval.DataRetrievalMethod = (DataRetrievalMethod)Enum.Parse(typeof(DataRetrievalMethod), model.RetrievalMethod);
                            if (dataType == VI.Admin.VistaDataType.ADT) dataRetrieval.IsHL7Enabled = model.HL7Checked;
                            if (!string.IsNullOrEmpty(model.SelectedSchedulerId))
                                dataRetrieval.JobCalendar = GetScheduler(model.SelectedSchedulerId);
                        }
                        else if (model.RemoveButton != null)
                        {
                            vistaSite.DataRetrievalDetails.Remove(dataType);
                        }

                        FacadeManager.ConfigurationInterface.SaveVistASite(vistaSite, loggedUser.Domain);
                        return this.RedirectToAction<VistaIntegrationController>(act => act.Categories(EncryptQueryString(new string[] { "vistaSiteId", "dataType" }, new string[] { model.VistaSiteId, dataType.ToString("G") }, loggedUser.Salt)));
                    }
                }

                if (model.RunButton != null)
                {
                    try
                    {
                        if (ModelState.IsValid)
                        {
                            DataRetrievalMethod? retrievalMethod = null;
                            if (model.RetrievalMethodRun.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                                retrievalMethod = DataRetrievalMethod.MDWS;
                            else if (model.RetrievalMethodRun.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                                retrievalMethod = DataRetrievalMethod.ODBC;

                            VI.VistASite vistaSite = GetVistaSite(model.VistaSiteId);
                            VI.Admin.VistaDataType vistaDataTypes = GetCheckedDataTypes(model);
                            DateTime startTime = model.StartDateTime + new TimeSpan(Convert.ToInt32(model.StartHour), Convert.ToInt32(model.StartMinute), 0);
                            DateTime endTime = model.EndDateTime + new TimeSpan(Convert.ToInt32(model.EndHour), Convert.ToInt32(model.EndMinute), 0);

                            FacadeManager.VistAIntegrationInterface.RunOnDemand(vistaSite, vistaDataTypes, startTime, endTime, retrievalMethod.Value);

                            newModel.RunOnDemandResult = String.Format("Fetch operation started successfully at: {0} for VistA Site: {1} - {2}, Data Types: {3}, Start Time: {4}, End Time: {5}. Go to the Audit page to watch the progress."
                                                                       , DateTime.Now
                                                                       , (!string.IsNullOrEmpty(vistaSite.Name) ? vistaSite.Name : string.Empty)
                                                                       , (!string.IsNullOrEmpty(vistaSite.Number) ? vistaSite.Number : string.Empty)
                                                                       , vistaDataTypes
                                                                       , startTime
                                                                       , endTime);

                        }
                    }
                    catch (FaultException<BMS.FaultContracts.VistAException> e)
                    {
                        newModel.RunOnDemandResult = e.Detail.ErrorMessage;
                    }
                }
                return View(newModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public ActionResult CategoriesRun([Bind(Exclude = "RememberMe")] CategoriesViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return this.RedirectToAction<VistaIntegrationController>(act => act.Categories(EncryptQueryString(new string[] { "vistaSiteId", "dataType" }, new string[] { model.VistaSiteId, GetDataType(model.SelectedDataType).ToString("G") }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private VI.Admin.VistaDataType GetCheckedDataTypes(CategoriesViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                int dataTypes = 0;
                if (model.RetrieveHospitalLocation)
                    dataTypes |= (int)VI.Admin.VistaDataType.HospitalLocation;
                if (model.RetrievePatient)
                    dataTypes |= (int)VI.Admin.VistaDataType.Patient;
                if (model.RetrieveRoomBed)
                    dataTypes |= (int)VI.Admin.VistaDataType.RoomBed;
                if (model.RetrieveWardLocation)
                    dataTypes |= (int)VI.Admin.VistaDataType.WardLocation;
                if (model.RetrieveOrderableItem)
                    dataTypes |= (int)VI.Admin.VistaDataType.OrderableItem;
                if (model.RetrieveSpecialty)
                    dataTypes |= (int)VI.Admin.VistaDataType.Specialty;
                if (model.RetrieveTreatingSpecialty)
                    dataTypes |= (int)VI.Admin.VistaDataType.TreatingSpecialty;
                if (model.RetrieveFacilityMovementType)
                    dataTypes |= (int)VI.Admin.VistaDataType.FacilityMovementType;
                if (model.RetrieveADT)
                    dataTypes |= (int)VI.Admin.VistaDataType.ADT;
                if (model.RetrieveMedicalCenterDivision)
                    dataTypes |= (int)VI.Admin.VistaDataType.MedicalCenterDivision;
                if (model.RetrieveWaitingList)
                    dataTypes |= (int)VI.Admin.VistaDataType.PatientsPendingBedPlacementList;

                return (VI.Admin.VistaDataType)dataTypes;

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CategoriesViewModel GetCategoriesModel(string vistaSiteId, string dataTypeString)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetCategoriesModel(vistaSiteId, dataTypeString, null);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CategoriesViewModel GetCategoriesModel(string vistaSiteId, string dataTypeString, CategoriesViewModel oldModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CategoriesViewModel newModel = FillCategoriesModel(oldModel);
                VI.Admin.VistaDataType dataType = (VI.Admin.VistaDataType)Enum.Parse(typeof(VI.Admin.VistaDataType), dataTypeString.Replace(" ", ""));

                if (vistaSiteId == null)
                {
                    VI.VistASite firstVistaSite = newModel.VistaSites.FirstOrDefault();
                    vistaSiteId = firstVistaSite != null ? firstVistaSite.Id : null;
                }

                VI.VistASite vistaSite = GetVistaSite(vistaSiteId);
                newModel.VistaSiteId = vistaSiteId;
                newModel.SelectedVistaSite = vistaSite;
                if (dataType == VI.Admin.VistaDataType.None)
                    newModel.SelectedDataType = string.Empty;
                else
                {
                    newModel.SelectedDataType = SplitCamelCase(dataType.ToString("G"));

                    //load the settings for the selected dataType in the model
                    if (vistaSite.DataRetrievalDetails != null &&
                        vistaSite.DataRetrievalDetails.ContainsKey(dataType) &&
                        vistaSite.DataRetrievalDetails[dataType].DataRetrievalMethod != (DataRetrievalMethod)0)
                    {
                        newModel.RetrievalMethod = vistaSite.DataRetrievalDetails[dataType].DataRetrievalMethod.ToString("G");
                        newModel.HL7Checked = vistaSite.DataRetrievalDetails[dataType].IsHL7Enabled;
                        if (vistaSite.DataRetrievalDetails[dataType].JobCalendar != null &&
                             vistaSite.DataRetrievalDetails[dataType].JobCalendar.Id != null)
                            newModel.SelectedSchedulerId = vistaSite.DataRetrievalDetails[dataType].JobCalendar.Id;
                    }
                }

                newModel.StartDateTime = DateTime.Now;
                newModel.EndDateTime = DateTime.Now;

                ViewBag.CategoriesSelectedBackgroundColor = Strings.AnchorBackgroundColor;

                return newModel;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CategoriesViewModel FillCategoriesModel(CategoriesViewModel oldModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CategoriesViewModel newModel = oldModel;
                if (oldModel == null)
                    newModel = new CategoriesViewModel();

                if (newModel.RunOnDemandResult == null)
                    newModel.RunOnDemandResult = string.Empty;

                newModel.LoggedUser = this.loggedUser;
                List<VistaSite> sites = FacadeManager.EntityInterface.GetVistaSites().ToList();
                newModel.VistaSites = FacadeManager.ConfigurationInterface.GetVistaSites(loggedUser.Domain).OrderBy(vistaSite => vistaSite.Name).ToList();
                VistaSite site = null;
                foreach (VI.VistASite vs in newModel.VistaSites)
                {
                    site = sites.Where(a => a.Id.extension.Equals(vs.Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    vs.Name = vs.Name + " - " + site.Number;
                }
                newModel.Calendars = FacadeManager.ConfigurationInterface.GetDefinedCalendars(loggedUser.Domain).OrderBy(jobCal => jobCal.Name).ToList();

                newModel.RetrievalMethods = new List<DataRetrievalMethod>((DataRetrievalMethod[])Enum.GetValues(typeof(DataRetrievalMethod))).Select(drm => new SelectableEnum<DataRetrievalMethod>(drm, false)).ToList();
                newModel.OnDemandRetrievalMethods = new List<string>() { DataRetrievalMethod.MDWS.ToString("G"), DataRetrievalMethod.ODBC.ToString("G") };
                if (oldModel != null && oldModel.SelectedVistaSite != null)
                {
                    newModel.SelectedVistaSite = newModel.VistaSites.First(s => s.Id == oldModel.SelectedVistaSite.Id);
                }
                ViewBag.CategoriesSelectedBackgroundColor = Strings.AnchorBackgroundColor;
                return newModel;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }



        private VI.Admin.VistaDataType GetDataType(string dataType)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VI.Admin.VistaDataType result = VI.Admin.VistaDataType.None;

                Enum.TryParse<VI.Admin.VistaDataType>(dataType.Replace(" ", ""), out result);

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Audit

        [ReadPermissionAuthFilterAttribute]
        public ActionResult Audit()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return View(FillAuditModel(null));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilter]
        [HttpPost]
        public ActionResult Audit([Bind(Exclude = "RememberMe")] AuditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AuditViewModel newModel = FillAuditModel(model);
                Dictionary<string, string> vistaTimeZones = new Dictionary<string, string>();
                DateTime sDate, eDate;
                if (model.FilterButton != null)
                {
                    newModel.CurrentPage = 1;

                    if (model.SelectedPeriod != Strings.SpecifyPeriod)
                    {
                        model.EndDateTime = DateTime.UtcNow.Date.AddDays(1);
                        if (model.SelectedPeriod == Strings.Today)
                            model.StartDateTime = DateTime.UtcNow.Date;
                        else
                            if (model.SelectedPeriod == Strings.LastWeek)
                                model.StartDateTime = DateTime.UtcNow.Date - TimeSpan.FromDays(7);
                            else
                                if (model.SelectedPeriod == Strings.LastMonth)
                                    model.StartDateTime = DateTime.UtcNow.Date.AddMonths(-1);
                        sDate = model.StartDateTime;
                        eDate = model.EndDateTime;
                    }
                    else
                    {
                        sDate = GetDate(newModel.StartDateTime, newModel.HourStartDateSelected, newModel.MinuteStartDateSelected, newModel.SecondStartDateSelected);
                        eDate = GetDate(newModel.EndDateTime, newModel.HourEndDateSelected, newModel.MinuteEndDateSelected, newModel.SecondEndDateSelected);
                    }

                    ModelState.Remove("PreviousFilterVistaSiteId");
                    ModelState.Remove("PreviousFilterRetrievalMethod");
                    ModelState.Remove("PreviousFilterDataType");
                    ModelState.Remove("PreviousFilterSelectedStatus");
                    ModelState.Remove("PreviousFilterStartDateTimeString");
                    ModelState.Remove("PreviousFilterEndDateTimeString");
                    ModelState.Remove("PreviousFilterSelectedPeriod");

                    List<string> vistaSitesIds = model.VistaSites.FindAll(site => site.IsSelected).Select<SelectableSite, string>(site => site.Id).ToList();
                    newModel.PreviousFilterVistaSiteId = vistaSitesIds.Aggregate((ids, next) => ids + "," + next);
                    List<SelectableEnum<DataRetrievalMethod>> retrievalMethods = model.RetrievalMethods.FindAll(method => method.IsSelected);
                    newModel.PreviousFilterRetrievalMethod = retrievalMethods.Aggregate<SelectableEnum<DataRetrievalMethod>, string>("", (methods, next) => methods + "," + next.EntityName);
                    List<SelectableEnum<VI.Admin.VistaDataType>> dataTypes = model.DataTypes.FindAll(dt => dt.IsSelected);
                    newModel.PreviousFilterDataType = dataTypes.Aggregate<SelectableEnum<VI.Admin.VistaDataType>, string>("", (types, next) => types + "," + next.EntityName);

                    newModel.PreviousFilterStartDateTimeString = String.Format("{0:G}", sDate);
                    newModel.PreviousFilterEndDateTimeString = String.Format("{0:G}", eDate);
                    newModel.PreviousFilterSelectedPeriod = model.SelectedPeriod;

                    vistaTimeZones = GetVistaTimeZone(vistaSitesIds);
                    newModel.LogEntries = GetLogEntries(vistaSitesIds.ToList(), retrievalMethods.Select<SelectableEnum<DataRetrievalMethod>, DataRetrievalMethod>(sel => sel.EnumValue).ToList(),
                                                        dataTypes.Select<SelectableEnum<VI.Admin.VistaDataType>, VI.Admin.VistaDataType>(sel => sel.EnumValue).ToList(), sDate, eDate,
                                                        model.Status.Where(s => s.IsSelected).Select(s => s.EnumValue).ToList(), new PagingContext() { PageNumber = 1, PageSize = model.PageSize }, model.SelectedPeriod == Strings.LastOccurrence ? true : false, vistaTimeZones);
                }
                if (model.OtherPageButton != null)
                {
                    List<string> siteIds = new List<string>();
                    List<DataRetrievalMethod> methods = new List<DataRetrievalMethod>();
                    List<VI.Admin.VistaDataType> types = new List<VI.Admin.VistaDataType>();

                    if (!string.IsNullOrEmpty(newModel.PreviousFilterVistaSiteId))
                        foreach (string siteId in newModel.PreviousFilterVistaSiteId.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries))
                        {
                            newModel.VistaSites.Find(site => site.Id == siteId).IsSelected = true;
                            siteIds.Add(siteId);
                        }
                    if (!string.IsNullOrEmpty(newModel.PreviousFilterRetrievalMethod))
                        foreach (string rm in newModel.PreviousFilterRetrievalMethod.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries))
                        {
                            newModel.RetrievalMethods.Find(method => method.EntityName == rm).IsSelected = true;
                            methods.Add((DataRetrievalMethod)Enum.Parse(typeof(DataRetrievalMethod), rm));
                        }
                    if (!string.IsNullOrEmpty(newModel.PreviousFilterDataType))
                        foreach (string dt in newModel.PreviousFilterDataType.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries))
                        {
                            newModel.DataTypes.Find(type => type.EntityName == dt).IsSelected = true;
                            types.Add((VI.Admin.VistaDataType)Enum.Parse(typeof(VI.Admin.VistaDataType), dt));
                        }

                    vistaTimeZones = GetVistaTimeZone(siteIds);
                    newModel.StartDateTime = DateTime.Parse(model.PreviousFilterStartDateTimeString);
                    newModel.EndDateTime = DateTime.Parse(model.PreviousFilterEndDateTimeString);
                    newModel.SelectedPeriod = model.PreviousFilterSelectedPeriod;
                    newModel.LogEntries = GetLogEntries(siteIds, methods, types, newModel.StartDateTime, newModel.EndDateTime, model.Status.Where(s => s.IsSelected).Select(s => s.EnumValue).ToList(),
                                                new PagingContext() { PageNumber = model.CurrentPage, PageSize = model.PageSize }, model.SelectedPeriod == Strings.LastOccurrence ? true : false, vistaTimeZones);
                }
                return View(newModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the date.
        /// </summary>
        /// <param name="dateTime">The date time.</param>
        /// <param name="hour">The hour.</param>
        /// <param name="minutes">The minutes.</param>
        /// <param name="seconds">The seconds.</param>
        /// <returns>DateTime</returns>
        private DateTime GetDate(DateTime dateTime, string hour, string minutes, string seconds)
        {
            return new DateTime(dateTime.Year, dateTime.Month, dateTime.Day, Convert.ToInt32(hour), Convert.ToInt32(minutes), Convert.ToInt32(seconds));
        }

        private Dictionary<string, string> GetVistaTimeZone(List<string> siteIds)
        {
            Dictionary<string, string> vistaTimeZones = new Dictionary<string, string>();
            List<CD> timezoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
            IList<VI.VistASite> vistASites = FacadeManager.ConfigurationInterface.GetVistaSites(loggedUser.Domain);
            if (vistASites == null || vistASites.Count == 0)
                return vistaTimeZones;
            VI.VistASite vs = null;
            CD timezoneSelected = null;

            foreach (string id in siteIds)
            {
                if (!vistaTimeZones.ContainsKey(id.ToLower()))
                {
                    vs = vistASites.Where(a => a.Id == id).FirstOrDefault();
                    if (vs != null && vs.TimeZone != null)
                    {
                        timezoneSelected = timezoneList.Where(a => a.displayName.Equals(vs.TimeZone.Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (timezoneSelected != null)
                            vistaTimeZones.Add(id.ToLower(), timezoneSelected.code);
                    }
                }
            }
            return vistaTimeZones;
        }

        private IList<JobLogModel> GetLogEntries(IList<string> vistaIds, IList<BMS.Utils.DataRetrievalMethod> retrievalMethods, IList<VI.Admin.VistaDataType> dataTypes, DateTime startDate, DateTime? endDate, IList<BMS.Utils.JobStatus> status, PagingContext pagingContext, bool isLastOccurrence, Dictionary<string, string> vistaTimeZones)
        {
            IList<JobLogModel> result = FacadeManager.VistAIntegrationInterface.FilterLogData(vistaIds, retrievalMethods, dataTypes, startDate, endDate, status, pagingContext, isLastOccurrence).Select<JobLogInfo, JobLogModel>(info => GetLogModel(info, vistaTimeZones)).ToList();
            if (isLastOccurrence)
                //each method that has status "running", set end date = null
                result.Where(a => a.Status == JobStatus.Running.ToString()).ForEach(c => { c.EndDate = string.Empty; });

            return result;
        }

        private AuditViewModel FillAuditModel(AuditViewModel oldModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AuditViewModel newModel = oldModel;

                if (oldModel == null)
                {
                    newModel = new AuditViewModel();
                    newModel.VistaSites = FacadeManager.ConfigurationInterface.GetVistaSites(loggedUser.Domain).Select<VI.VistASite, SelectableSite>(vs => new SelectableSite(vs, false)).OrderBy(ss => ss.EntityName).ToList();
                    List<VistaSite> sites = FacadeManager.EntityInterface.GetVistaSites().ToList();
                    VistaSite site = null;
                    foreach (SelectableSite vs in newModel.VistaSites)
                    {
                        site = sites.Where(a => a.Id.extension.Equals(vs.Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        vs.EntityName = vs.EntityName + " - " + site.Number;
                    }
                    newModel.RetrievalMethods = GetRetrievalMethods();
                    newModel.DataTypes = GetDataTypesList();
                    newModel.SelectedPeriod = Strings.LastOccurrence;
                    newModel.LogEntries = new List<JobLogModel>();
                }
                if (newModel.SelectedPeriod != Strings.SpecifyPeriod)
                {
                    newModel.StartDateTime = DateTime.Now.Date;
                    newModel.EndDateTime = DateTime.Now.Date.AddDays(1);
                }

                newModel.LoggedUser = this.loggedUser;
                if (newModel.Status == null)
                    newModel.Status = GetJobStatus();
                newModel.PageSize = FacadeManager.ConfigurationInterface.GetRecordsNumberPerPage(newModel.LoggedUser.Domain);
                if (newModel.CurrentPage == 0)
                    newModel.CurrentPage = 1;

                ViewBag.AuditSelectedBackgroundColor = Strings.AnchorBackgroundColor;

                return newModel;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
