﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Facade;
using BMS.Web.App_GlobalResource;
using BMS.Facade.Data;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Check if the user is authorized to perform the operation.
    /// </summary>
    public class PermissionAuthFilterAttribute : ActionFilterAttribute
    {
        /// <summary>
        /// Gets or sets the operation.
        /// </summary>
        /// <value>
        /// The operation.
        /// </value>
        public string Operation { get; protected set; }

        protected PermissionAuthFilterAttribute() { }

        /// <summary>
        /// Initializes a new instance of the <see cref="PermissionAuthFilterAttribute"/> class.
        /// </summary>
        /// <param name="operation">The operation.</param>
        public PermissionAuthFilterAttribute(string operation)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(operation))
                    throw new ArgumentNullException("operation");

                Operation = operation;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected void CheckPermission(ActionExecutingContext filterContext, string operation)
        {
            bool result = Facade.FacadeManager.UserInterface.CheckPermission(operation);
            if (!result)
            {
                filterContext.Result = BaseController.UnauthorizedUrlRequestResult(filterContext.HttpContext);
            }
            else
            {
                if (operation.EndsWith("Update", StringComparison.InvariantCultureIgnoreCase) && !CheckIsOperationException(operation))
                {
                    User user = FacadeManager.UserInterface.GetProfile();

                    bool hasBMSWritePremission = FacadeManager.UserInterface.CheckBMSWritePermission(user, user.Facility.Id);

                    if (hasBMSWritePremission)
                        base.OnActionExecuting(filterContext);
                    else if (operation.StartsWith("WardWhiteboard") && FacadeManager.UserInterface.CheckBMSWritePermissionWhiteboard(user, user.Facility.Id))
                        base.OnActionExecuting(filterContext);
                    else
                        filterContext.Result = BaseController.UnauthorizedWriteRequestResult(user.Facility.Name);
                        
                }
                else
                    base.OnActionExecuting(filterContext);
            }
        }

        public override void OnActionExecuting(ActionExecutingContext filterContext)
        {
            CheckPermission(filterContext, Operation);
        }

        /// <summary>
        /// Checks the is operation exception.
        /// If the action is Save on the VISN Add/Edit Transfer page, ignore the check for write permission on the user facility. 
        /// The write permission must be checked on the facility selected from transfer page. This check is made in the VISN Transfer page.
        /// </summary>
        /// <param name="operation">The operation.</param>
        /// <returns></returns>
        private bool CheckIsOperationException(string operation)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (operation.Equals(BMS.Utils.Constants.VISN_Add_Transfer_Operation_Permission, StringComparison.InvariantCultureIgnoreCase)
                    || operation.Equals(BMS.Utils.Constants.VISN_Edit_Transfer_Operation_Permission, StringComparison.InvariantCultureIgnoreCase)
                    || operation.Equals(BMS.Utils.Constants.VISN_Finalize_Transfer_Operation_Permission, StringComparison.InvariantCultureIgnoreCase))
                    return true;
                else
                    return false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}