using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using Microsoft.Web.Mvc;
using BMS.Facade;
using BMS.Facade.Service;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Web.Models;
using BMS.Web.App_GlobalResource;
using BMS.Facade.Implementation;
using Microsoft.Reporting.WebForms;
using System.Text;
using BMS.Utils;
using BMS.Facade.Fault;
using VI = BMS.VistaIntegration.FacadeContracts;
using BMS.Web.Models.Shared;
using BMS.VistaIntegration.FacadeContracts.Admin;
using BMS.Web.Controllers.Shared;
using BMS.ServicesWrapper.BMService;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Reports Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class ReportsController : BaseController
    {
        /// <summary>
        /// Get action for the Reports controller
        /// </summary>
        /// <param name="input">The input (home model).</param>
        /// <returns>Returns the view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                var model = new ReportsViewModel();
                FillViewModel(model);
                return PartialView("Index", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Reports controller - used for the reports buttons.
        /// </summary>
        /// <param name="buttonSubmitWard">The button submit ward.</param>
        /// <param name="buttonSubmitBedGroup">The button submit bed group.</param>
        /// <param name="buttonSubmitAdmission">The button submit admission.</param>
        /// <param name="buttonSubmitMovement">The button submit movement.</param>
        /// <param name="buttonSubmitOther">The button submit other.</param>
        /// <param name="input">The input (home model).</param>
        /// <returns>Returns the view result.</returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(ReportsViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CD bedStatus = GetBedStatusSelected(input);
                Division division = GetWardGroupSelected(input);
                TempData.Put<CD>(Constants.TempData_BedStatus, bedStatus);
                TempData.Put<Division>(Constants.TempData_WardGroup, division);

                input.ReportsList = FacadeManager.ReportsInterface.GetReports();

                input.ReportsListSecond = FacadeManager.ReportsInterface.GetOtherReports();

                if (input.WardOperation != null)
                    return this.RedirectToAction<WardOccupancyController>(action => action.Index(EncryptQueryString(new string[] { "divisionId", "bedStatus", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { division.Id.extension, bedStatus.code, "false", "", "", string.Empty }, loggedUser.Salt)));

                IReportInfo report = null;
                string reportName = null;

                //Bed Specialty Roster Report
                if (input.BedGroupOperation != null)
                {
                    //Get the Bed Specialty Roster Report (from OtherReports path)                
                    if (input.ReportsListSecond.Count > 0)
                    {
                        if (input.View.ToUpper() == Strings.Report.ToUpper())
                            reportName = Constants.BedSpecialtyReport;
                        else
                            reportName = Constants.BedSpecialtyRoster;

                        report = input.ReportsListSecond.Where(r => r.Name == reportName).FirstOrDefault();
                        if (report != null)
                        {
                            report = FillParameters(report, input.TypeGroup);
                            report.Parameters.Add(Constants.REP_BED_SPECIALTY_ROSTER_TYPEGROUP,
                                                    input.TypeGroup);
                        }
                    }
                }

                //Scheduled Admissions Report
                if (input.AdmissionOperation != null)
                {
                    input.DaysList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.SchedAdmNextDays);

                    if (input.ReportsListSecond.Count > 0)
                    {
                        reportName = Constants.ScheduledAdmissions;
                        report = input.ReportsListSecond.Where(r => r.Name == reportName).FirstOrDefault();
                        if (report != null)
                        {
                            report = FillParameters(report);
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSIONS_REPORT_NEXT_DAYS,
                                                input.DaysList.Where(a => a.code == input.Days).
                                                    Select(a => a.displayName).FirstOrDefault());

                        }
                    }
                }
                
                //Facility Diversion Report
                if (input.FacilityDiversionOperation != null)
                {
                    input.DaysList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.SchedAdmNextDays);

                    if (input.ReportsListSecond.Count > 0)
                    {
                        reportName = Constants.FacilityDiversion;
                        report = input.ReportsListSecond.Where(r => r.Name == reportName).FirstOrDefault();
                        if (report != null)
                        {
                            report = FillParameters(report);
                        }
                    }
                }

                //Patient Movement Report
                if (input.MovementOperation != null)
                {
                    input.TransactionBedList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.TransactionBed).OrderBy(a => a.displayName).ToList();

                    reportName = Constants.PatientMovement;
                    report = input.ReportsListSecond.Where(r => r.Name == reportName).FirstOrDefault();


                    DateTime dt = DateTime.Today;
                    switch (input.QuickDate)
                    {
                        case Constants.QuickDate1D:
                            dt = DateTime.Today.AddDays(-1);
                            break;
                        case Constants.QuickDate2D:
                            dt = DateTime.Today.AddDays(-2);
                            break;
                        case Constants.QuickDate3D:
                            dt = DateTime.Today.AddDays(-3);
                            break;
                        case Constants.QuickDate4D:
                            dt = DateTime.Today.AddDays(-4);
                            break;
                        case Constants.QuickDate5D:
                            dt = DateTime.Today.AddDays(-5);
                            break;
                        case Constants.QuickDate6D:
                            dt = DateTime.Today.AddDays(-6);
                            break;
                        case Constants.QuickDate7D:
                            dt = DateTime.Today.AddDays(-7);
                            break;
                    }

                    if (report != null)
                    {
                        report = FillParameters(report);
                        report.Parameters.Add(Constants.REP_PATIENT_MOVEMENT_REPORT_DATE,
                                                dt.ToString());
                        report.Parameters.Add(Constants.REP_PATIENT_MOVEMENT_REPORT_TRANSATION,
                                                input.TransactionBedList.Where(a => a.code == input.TransactionBed).Select(a => a.displayName).FirstOrDefault());
                    }

                }
                //Selected a report from the dropdown
                if (input.OtherReportsOperation != null)
                {
                    if (input.SelectedReportName == Constants.WAIT_LIST_STATUS_REPORT)
                        return this.RedirectToAction<BedStatusReportController>(action => action.Index());
                    if (input.SelectedReportName == Constants.EMSBedStatus)
                        return this.RedirectToAction<EmsBedStatusAdminController>(action => action.Index(EncryptQueryString(new string[] { "division", "startDate", "endDate", "returnLink", "divisionFromReport" }, new string[] { string.Empty, string.Empty, string.Empty, Constants.HOME, string.Empty }, loggedUser.Salt)));
                    reportName = input.SelectedReportName;
                    report = input.ReportsList.Where(r => r.Name == reportName).FirstOrDefault();

                    if (report != null)
                        report = FillParameters(report);
                }

                //Redirect to the report
                if (report == null)
                    throw new ReportRenderException(reportName, Strings.ERR_ReportNotAvailable);
                else
                {
                    if (report.Name != Constants.EMERGENCY_MANAGEMENT_REPORT)
                    {
                        report.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION, input.LoggedUser.Facility.Id.extension);
                        report.Parameters.Add(Constants.REP_TIME_ZONE_MINUTES_OFFSET, RepTimeZoneMinutesOffset);
                    }
                    return Redirect(report.Url);
                }

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the parameters.
        /// </summary>
        /// <param name="report">The report.</param>
        /// <param name="specialty">The specialty.</param>
        /// <returns></returns>
        private IReportInfo FillParameters(IReportInfo report, string specialty)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                report = report.Clone();
                string reportTitle = string.Empty;
                specialty = specialty == Strings.All ? Strings.AllSpecialties : specialty;
                if (report.Parameters == null)
                    report.Parameters = new Dictionary<String, String>();
                report.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, Url.Action("Index", "Reports"));
                report.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToReports);
                if (report.Name == Constants.BedSpecialtyReport)
                    reportTitle = Strings.BedSpecialtyReportTitle + " - " + specialty;
                else
                    reportTitle = Strings.BedSpecialtyRosterTitle + " - " + specialty;
                report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);

                return report;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the parameters.
        /// </summary>
        /// <param name="report">The report.</param>
        private IReportInfo FillParameters(IReportInfo report)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string reportTitle = string.Empty;
                report = report.Clone();
                if (report.Parameters == null)
                    report.Parameters = new Dictionary<String, String>();
                report.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, Url.Action("Index", "Reports"));
                report.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToReports);
                FacilitySettings settings;

                switch (report.Name)
                {
                    case Constants.SignedAdmission:
                        reportTitle = Strings.ActiveAdmissionOrdersReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.SignedDischarge:
                        reportTitle = Strings.ActiveDischargeOrdersReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.AnticipatedDischarge:
                        reportTitle = Strings.AnticDischargeOrdersReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.BedOutOfService:
                        report.Parameters.Add(Constants.REPORT_TITLE, Strings.BedsOutOfServiceReportAllTitle);
                        break;
                    case Constants.BedsOutOfService:
                        reportTitle = Strings.BedsOutOfServiceReportByDateTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.DISCHARGES_IN_PROGRESS_REPORT:
                        reportTitle = Strings.DischargesInProgressReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.DISCHARGE_ORDER_DIFF_REPORT:
                        reportTitle = Strings.DischargeOrderDiffReport + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.PatientMovementByDate:
                        reportTitle = Strings.PatientMovementsReportByDateTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.DischargeAppointment1:
                        report.Parameters.Add(Constants.REPORT_TITLE, Strings.PatientsWDischargeAppointmentsReportTitle);
                        break;
                    case Constants.BED_SUMMARY_REPORT:
                        reportTitle = Strings.BedSummaryReportTitle + ": " + this.loggedUser.Facility.Name;
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);

                        settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);

                        if (!string.IsNullOrEmpty(settings.MedicalCenterID) && settings.IntegratedFacility != null &&
                            settings.IntegratedFacility.code.ToUpper() == "YES")
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, settings.MedicalCenterID);
                        else
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, "");
                        report.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);
                        break;
                    case Constants.ScheduledAdmissionsByDate:
                        reportTitle = Strings.ScheduledAdmissionsReportByDateTitle + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);

                        settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);

                        if (!string.IsNullOrEmpty(settings.MedicalCenterID) && settings.IntegratedFacility != null &&
                            settings.IntegratedFacility.code.ToUpper() == "YES")
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, settings.MedicalCenterID);
                        else
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, "");

                        break;
                    case Constants.ScheduledAdmissions:
                        reportTitle = Strings.ScheduledAdmissionsReportTitle + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);

                        settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);

                        if (!string.IsNullOrEmpty(settings.MedicalCenterID) && settings.IntegratedFacility != null &&
                            settings.IntegratedFacility.code.ToUpper() == "YES")
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, settings.MedicalCenterID);
                        else
                            report.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, "");
                        break;
                    case Constants.PatientMovement:
                        reportTitle = Strings.PatientMovementsReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.BED_TURNAROUND_TIME_REPORT:
                        reportTitle = Strings.BedTurnaroundTimeReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.EMSBedStatus:
                        reportTitle = this.loggedUser.Facility.Code + " - " + Strings.EMSBedStatusReportTitle;
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        report.Parameters.Add(Constants.REP_GENERAL_WARDGROUP_ID, "-1");
                        break;
                    case Constants.EMERGENCY_MANAGEMENT_REPORT:
                        report.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2} {3}", this.loggedUser.Facility.Code, Constants.ROSTER, Strings.ReportFor, Strings.AllWards));
                        report.Parameters.Add(Constants.REP_PAGE_TITLE, string.Format("{0} {1}", Strings.EmergencyManagementEvacuation, Strings.Report));
                        report.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION_TO_UPPER, this.loggedUser.Facility.Id.extension);
                        report.Parameters.Add(Constants.REP_GENERAL_FACILITY_CODE, this.loggedUser.Facility.Code);
                        report.Parameters.Add(Constants.REP_GENERAL_IS_ROSTER_REPORT, "True");
                        report.Parameters.Add(Constants.REP_GENERAL_IS_REGULATE_REPORT, "False");
                        report.Parameters.Add(Constants.REP_GENERAL_IS_MANIFEST_REPORT, "False");
                        report.Parameters.Add(Constants.REP_GENERAL_IS_TRANSFER_REPORT, "False");
                        report.Parameters.Add(Constants.REP_GENERAL_REPORT_TYPE, Constants.ROSTER);
                        report.Parameters.Add(Constants.REP_GENERAL_DEFAULT_DIVISION_ID, "-1");
                        report.Parameters.Add(Constants.REP_GENERAL_DEFAULT_DIVISION_NAME, Strings.AllWards);
                        report.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);
                        break;
                    case Constants.ACTIVE_TRANSFER_REPORT:
                        reportTitle = Strings.ActiveTransferOrdersReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.BED_SWITCH_REPORT:
                        reportTitle = Strings.BedSwitchReportTitle + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        break;
                    case Constants.FacilityDiversion:
                        reportTitle = Strings.FacilityDiversion + " - " + this.loggedUser.Facility.Name + " (" + this.loggedUser.Facility.Code + ")";
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        report.Parameters.Add(Constants.REP_GENERAL_DEFAULT_DIVISION_NAME, Strings.AllWards);
                        break;
                }

                return report;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillViewModel(ReportsViewModel hm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                hm.LoggedUser = this.loggedUser;

                hm.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                hm.BedStatusList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.BedStatus);
                hm.TypeGroupList = FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).OrderBy(a => a.TypeGroup).Select(a => a.TypeGroup).Distinct().ToList();
                hm.ViewList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.SelectBedGroupingsView);
                hm.DaysList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.SchedAdmNextDays);
                hm.QuickDateList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.QuickDate);
                hm.TransactionBedList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.TransactionBed).OrderBy(a => a.displayName).ToList();
                hm.TransactionBed = Strings.DefaultSelectedTransactionBedCode;
                IList<IReportInfo> reports = new List<IReportInfo>();
                IReportInfo report;
                reports = FacadeManager.ReportsInterface.GetReports();

                IReportInfo rpt = reports.Where(r => r.Name == Constants.DischargeAppointment).FirstOrDefault();
                if (rpt != null)
                    rpt.Name = Constants.DischargeAppointment1;

                if (hm.ReportsList == null)
                    hm.ReportsList = new List<IReportInfo>();

                //get summary report;
                report = reports.Where(a => a.Name == Constants.BED_SUMMARY_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                // Bed Switch Report
                report = reports.Where(a => a.Name == Constants.BED_SWITCH_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get BED TURNAROUND TIME_REPORT
                report = reports.Where(a => a.Name == Constants.BED_TURNAROUND_TIME_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get EMS Bed Status Report (Admin)
                report = reports.Where(a => a.Name == Constants.EMSBedStatus).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Beds Out of Service Report (By Date)
                report = reports.Where(a => a.Name == Constants.BedsOutOfService).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Beds Out of Service Report (All)
                report = reports.Where(a => a.Name == Constants.BedOutOfService).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Scheduled Admissions by Date
                report = reports.Where(a => a.Name == Constants.ScheduledAdmissionsByDate).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Active Admission Orders Report
                report = reports.Where(a => a.Name == Constants.SignedAdmission).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Patient Movements by Date
                report = reports.Where(a => a.Name == Constants.PatientMovementByDate).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Patients w/ Discharge Appointments
                report = reports.Where(a => a.Name == Constants.DischargeAppointment1).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Antic Discharge Orders Report
                report = reports.Where(a => a.Name == Constants.AnticipatedDischarge).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Active Discharge Orders Report
                report = reports.Where(a => a.Name == Constants.SignedDischarge).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Discharges in Progress
                report = reports.Where(a => a.Name == Constants.DISCHARGES_IN_PROGRESS_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                //get Discharge Order Diff
                report = reports.Where(a => a.Name == Constants.DISCHARGE_ORDER_DIFF_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);

                //get Emergency Management Report
                report = reports.Where(a => a.Name == Constants.EMERGENCY_MANAGEMENT_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);

                // add Wait List Status Report
                if (FacadeManager.UserInterface.CheckPermission(Constants.WAIT_LIST_STATUS_REPORT_PERMISSION))
                {
                    report = new ReportInfo();
                    report.Name = Constants.WAIT_LIST_STATUS_REPORT;
                    hm.ReportsList.Add(report);
                }

                //get Active Transfer Report
                report = reports.Where(a => a.Name == Constants.ACTIVE_TRANSFER_REPORT).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);
                
                //get facility diversion report;
                report = reports.Where(a => a.Name == Constants.FacilityDiversion).FirstOrDefault();
                if (report != null)
                    hm.ReportsList.Add(report);

                hm.SelectedReportName = Constants.BED_SUMMARY_REPORT;

                //Insert defaul value for drop down list (TypeGroup)
                hm.WardGroupList.Insert(0, new Division() { Id = new II() { root = hm.LoggedUser.Domain, extension = (default(Guid)).ToString() }, Name = Strings.All });
                hm.TypeGroupList.Insert(0, Strings.All);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the ward group selected.
        /// </summary>
        /// <param name="hm">The home model.</param>
        /// <returns>The Ward Group object selected.</returns>
        private Division GetWardGroupSelected(ReportsViewModel hm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (hm.WardGroupList == null)
                {
                    if (hm.LoggedUser == null)
                        hm.LoggedUser = FacadeManager.UserInterface.GetProfile();
                    hm.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id);
                }
                if (hm.WardGroup == (default(Guid)).ToString())
                    return new Division() { Id = new II() { root = this.loggedUser.Domain, extension = (default(Guid)).ToString() }, Name = Strings.All };
                foreach (Division obj in hm.WardGroupList)
                {
                    if (obj.Id.extension.Equals(hm.WardGroup, StringComparison.InvariantCultureIgnoreCase))                    
                        return obj;                    
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the bed status selected.
        /// </summary>
        /// <param name="hm">The home model.</param>
        /// <returns>the Bed Status object selected.</returns>
        private CD GetBedStatusSelected(ReportsViewModel hm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (hm.BedStatusList == null)
                    hm.BedStatusList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.BedStatus);
                foreach (CD obj in hm.BedStatusList)
                {
                    if (obj.code == hm.BedStatus)
                        return obj;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

    }
}
