﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Facade;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using BMS.Facade.Service;
using BMS.Web.Models;
using Microsoft.Web.Mvc;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.ServicesWrapper.WF;
using System.Text.RegularExpressions;
using System.Web.UI;
using System.ServiceModel;
using BMS.Facade.Fault;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class PatientController : BaseController
    {
        /// <summary>
        /// Selects a pacient.
        /// </summary>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Select()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                PatientSelectViewModel vm = new PatientSelectViewModel()
                {
                    LoggedUser = this.loggedUser,
                    PatientList = null
                };
                return View(vm);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// POST for select a pacient.
        /// </summary>
        /// <param name="input">The input.</param>
        /// <returns></returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Select([Bind(Exclude = "RememberMe")] PatientSelectViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                input.LoggedUser = loggedUser;
                string vistaCode = this.loggedUser.Facility.VistaSite.Code;
                if (!string.IsNullOrEmpty(input.SelectedPatientExtension))
                {
                    Patient patient = FacadeManager.EntityInterface.GetPatientById(new II(input.LoggedUser.Facility.Id.root, input.SelectedPatientExtension), vistaCode);
                    PatientViewModel patientVm = FillModel(DateTimeNowOnCurrentFacility, null);
                    patientVm.PatientFullName = GetPatientFullName(patient);
                    patientVm.PatientSSN = string.Format("{0}{1}", "xxx-xx-", patient.SSN.extension.Substring(patient.SSN.extension.Length - 4));
                    patientVm.PatientExtension = patient.Id.extension;
                    patientVm.PatientRoot = patient.Id.root;
                    patientVm.PatientGender = (patient.Gender != null) ? patient.Gender.displayName : string.Empty;
                    patientVm.BedAssignmentDisplay = Strings.StyleDisplayNone;
                    patientVm.BedList = new List<Bed>();
                    patientVm.AdmittingDiagnosis = (patient.AdmittingDiagnosis != null) ? patient.AdmittingDiagnosis : string.Empty;
                    PatientLocation location = FacadeManager.ADTInterface.GetPatientLocation(patient.Id, loggedUser.VistaSite.Id);
                    if (location != null)
                    {
                        patientVm.CurrentBed = (location.CurrentAdmissionBed != null) ? location.CurrentAdmissionBed.Name : string.Empty;
                        patientVm.CurrentWard = (location.CurrentAdmissionWard != null) ? location.CurrentAdmissionWard.Name : string.Empty;
                    }
                    else
                    {
                        patientVm.CurrentBed = string.Empty;
                        patientVm.CurrentWard = string.Empty;
                    }
                    return View("Admission", patientVm);
                }

                Regex regexSSN = new Regex(@"^\d{3}-\d{2}-\d{4}$");
                Regex regexSSNWithoutDashes = new Regex(@"^\d{3}\d{2}\d{4}$");
                Regex regexFirstLastNameFourSSN = new Regex(@"^([A-Za-z]){1}\d{4}$");
                Regex regexLastName = new Regex(@"^[a-zA-Z]+$");
                input.PatientSearchText = input.PatientSearchText.Trim();
                IList<Patient> patientList = null;

                if (input.PatientSearchText.Length > 2)
                {
                    if (!regexSSN.IsMatch(input.PatientSearchText) && !regexSSNWithoutDashes.IsMatch(input.PatientSearchText))
                    {
                        if (!regexFirstLastNameFourSSN.IsMatch(input.PatientSearchText))
                        {
                            if (!regexLastName.IsMatch(input.PatientSearchText))
                            {
                                ModelState.AddModelError("PatientSearchText", Strings.SSNWasNotInAValidFormat);
                                return View(input);
                            }
                            else
                                patientList = FacadeManager.EntityInterface.GetPatientsByLastName(input.PatientSearchText, vistaCode);
                        }
                        else
                            patientList = FacadeManager.EntityInterface.GetPatientsByLastNameInitialAndLastFourSSN(input.PatientSearchText, vistaCode);
                    }
                    else
                    {
                        string ssn = input.PatientSearchText;
                        if (!input.PatientSearchText.Contains('-'))
                            ssn = string.Format("{0}-{1}-{2}", ssn.Substring(0, 3), ssn.Substring(3, 2), ssn.Substring(5, 4));
                        Patient patient = FacadeManager.EntityInterface.GetPatientBySsn(new II(Constants.SSNROOT, ssn), vistaCode);
                        if (patient != null)
                        {
                            patientList = new List<Patient>();
                            patientList.Add(patient);
                        }
                    }
                }

                if (patientList != null && patientList.Count > 0)
                    patientList = patientList.Where(a => a.IsVistaValidated).ToList();

                if (patientList != null && patientList.Count > 0)
                {
                    if (patientList.Count == 1)
                    {
                        PatientViewModel patientVm = FillModel(DateTimeNowOnCurrentFacility, null);
                        patientVm.PatientFullName = GetPatientFullName(patientList[0]);
                        patientVm.PatientSSN = string.Format("{0}{1}", "xxx-xx-", patientList[0].SSN.extension.Substring(patientList[0].SSN.extension.Length - 4));
                        patientVm.PatientExtension = patientList[0].Id.extension;
                        patientVm.PatientRoot = patientList[0].Id.root;
                        patientVm.PatientGender = (patientList[0].Gender != null) ? patientList[0].Gender.displayName : string.Empty;
                        patientVm.BedAssignmentDisplay = Strings.StyleDisplayNone;
                        patientVm.BedList = new List<Bed>();
                        patientVm.AdmittingDiagnosis = (patientList[0].AdmittingDiagnosis != null) ? patientList[0].AdmittingDiagnosis : string.Empty;
                        PatientLocation location = FacadeManager.ADTInterface.GetPatientLocation(patientList[0].Id, loggedUser.VistaSite.Id);
                        if (location != null)
                        {
                            patientVm.CurrentBed = (location.CurrentAdmissionBed != null) ? location.CurrentAdmissionBed.Name : string.Empty;
                            patientVm.CurrentWard = (location.CurrentAdmissionWard != null) ? location.CurrentAdmissionWard.Name : string.Empty;
                        }
                        else
                        {
                            patientVm.CurrentBed = string.Empty;
                            patientVm.CurrentWard = string.Empty;
                        }
                        return View("Admission", patientVm);
                    }
                    else
                    {
                        input.PatientList = patientList.OrderBy(patient => patient.LastName).ThenBy(patient => patient.FirstName).ThenBy(patient => patient.SSN.extension).ToList();
                        return View(input);
                    }
                }
                else
                {
                    if (input.PatientSearchText.Length > 2)
                    {
                        ModelState.AddModelError("PatientNotFound", Strings.SSNNotFound);
                        input.PatientList = null;
                        input.SelectedPatientExtension = null;
                        return View(input);
                    }
                    else
                    {
                        ModelState.AddModelError("PatientSearchText", Strings.SearchStringToShort);
                        input.PatientList = null;
                        input.SelectedPatientExtension = null;
                        return View(input);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Edits the admission.
        /// </summary>
        /// <param name="idAdmission">The id admission.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Admission(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingListItem waitingListItem = FacadeManager.ADTInterface.GetWaitingListItem(new II(this.loggedUser.Domain, QueryStrings["idAdmission"]), loggedUser);
                PatientViewModel vm = FillModelForEdit(waitingListItem);
                vm.IsFromEvacuationReport = bool.Parse(QueryStrings["isFromEvacuationReport"]);
                if (vm.IsFromEvacuationReport)
                {
                    if (!QueryStrings.ContainsKey("evacuationReportUrl"))
                        vm.EvacuationReportUrl = CreateEvacuationReportUrl(QueryStrings["selectedDivisionName"], QueryStrings["selectedDivisionId"], bool.Parse(QueryStrings["isRosterReport"]), bool.Parse(QueryStrings["isRegulateReport"]), bool.Parse(QueryStrings["isManifestReport"]), bool.Parse(QueryStrings["isTransferReport"]));
                    else
                        vm.EvacuationReportUrl = QueryStrings["evacuationReportUrl"];
                }
                else
                    vm.IsFromHomePage = true;

                return View("Admission", vm);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private string CreateEvacuationReportUrl(string selectedDivisionName, string selectedDivisionId, bool isRosterReport, bool isRegulateReport, bool isManifestReport, bool isTransferReport)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IReportInfo evacuationReport = FacadeManager.ReportsInterface.GetReports().Where(a => a.Name == Constants.EMERGENCY_MANAGEMENT_REPORT).FirstOrDefault();

                if (evacuationReport != null)
                {
                    //create paramets for report
                    if (evacuationReport.Parameters == null)
                        evacuationReport.Parameters = new Dictionary<String, String>();

                    evacuationReport.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToBedBoardHomePage);
                    evacuationReport.Parameters.Add(Constants.REP_PAGE_TITLE, string.Format("{0} {1}", Strings.EmergencyManagementEvacuation, Strings.Report));
                    evacuationReport.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, string.Format("{0}://{1}", this.Request.Url.Scheme, this.Request.Headers["host"]));
                    evacuationReport.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION_TO_UPPER, this.loggedUser.Facility.Id.extension);
                    evacuationReport.Parameters.Add(Constants.REP_GENERAL_FACILITY_CODE, this.loggedUser.Facility.Code);

                    if (isRosterReport)
                    {
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_ROSTER_REPORT, "True");
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_REPORT_TYPE, Constants.ROSTER);
                        evacuationReport.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2} {3}", this.loggedUser.Facility.Code, Constants.ROSTER, Strings.ReportFor, (selectedDivisionId == "-1") ? Strings.AllWards : selectedDivisionName));
                    }
                    else
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_ROSTER_REPORT, "False");

                    if (isRegulateReport)
                    {
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_REGULATE_REPORT, "True");
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_REPORT_TYPE, Constants.REGULATE);
                        evacuationReport.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2} {3}", this.loggedUser.Facility.Code, Constants.REGULATE, Strings.ReportFor, (selectedDivisionId == "-1") ? Strings.AllWards : selectedDivisionName));
                    }
                    else
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_REGULATE_REPORT, "False");
                    if (isManifestReport)
                    {
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_MANIFEST_REPORT, "True");
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_REPORT_TYPE, Constants.MANIFEST);
                        evacuationReport.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2} {3}", this.loggedUser.Facility.Code, Constants.MANIFEST, Strings.ReportFor, (selectedDivisionId == "-1") ? Strings.AllWards : selectedDivisionName));
                    }
                    else
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_MANIFEST_REPORT, "False");
                    if (isTransferReport)
                    {
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_TRANSFER_REPORT, "True");
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_REPORT_TYPE, Constants.TRANSFER);
                        evacuationReport.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2} {3}", this.loggedUser.Facility.Code, Constants.TRANSFER, Strings.ReportFor, (selectedDivisionId == "-1") ? Strings.AllWards : selectedDivisionName));
                    }
                    else
                        evacuationReport.Parameters.Add(Constants.REP_GENERAL_IS_TRANSFER_REPORT, "False");

                    evacuationReport.Parameters.Add(Constants.REP_GENERAL_DEFAULT_DIVISION_ID, selectedDivisionId);
                    evacuationReport.Parameters.Add(Constants.REP_GENERAL_DEFAULT_DIVISION_NAME, selectedDivisionName);
                    evacuationReport.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);

                    return evacuationReport.Url;
                }
                else
                    throw new ReportRenderException(string.Format("{0} {1} {2}", Strings.EmergencyManagementEvacuation, isRosterReport == true ? Constants.ROSTER : isRegulateReport ? Constants.REGULATE : Constants.MANIFEST, Strings.Report), Strings.ERR_ReportNotAvailable);


            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }


        }

        /// <summary>
        /// POST for edit the specified model.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Admission([Bind(Exclude = "RememberMe")] PatientViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.LoggedUser = loggedUser;
                //check to see if a division was selected, if so refresh the page
                if (!string.IsNullOrEmpty(model.RefreshPage))
                {
                    return RefreshAdmission(model);
                }

                if (!string.IsNullOrEmpty(model.ReasonsUsingFeeCode) && model.ReasonsUsingFeeCode.ToLower().Equals("other"))
                {
                    if (string.IsNullOrEmpty(model.FeeComments))
                        ModelState.AddModelError("FeeComments", "Fee Comments field is required if the selected Reason Using Fee field is Other(requires comment).");
                }
                if (!string.IsNullOrEmpty(model.SelectedBedId) && (string.IsNullOrEmpty(model.DateTimeBedAssignedString) || string.IsNullOrEmpty(model.AssignmentHourSelected) || string.IsNullOrEmpty(model.AssignmentMinuteSelected)))
                {
                    ModelState.AddModelError("DateTimeBedAssigned", "Assignment Date Time required.");
                }
                if (ModelState.IsValid)
                {
                    WaitingListItem waitingListItem;
                    bool isWaitingAreaAvailable = true;
                    GetWaitingListItemForSave(model, out waitingListItem, out isWaitingAreaAvailable);
                    model.IsWaitingAreaAvailable = isWaitingAreaAvailable;
                    if (isWaitingAreaAvailable)
                    {
                        if (waitingListItem.Id != null && !string.IsNullOrEmpty(waitingListItem.Id.extension))
                            if (waitingListItem.RemovedDate.HasValue)
                                FacadeManager.ADTInterface.ModifyWaitingListItem(waitingListItem, loggedUser.Facility);
                            else
                                FacadeManager.WorkflowInterface.ModifyWaitingListItem(waitingListItem, loggedUser.Facility);
                        else
                            FacadeManager.WorkflowInterface.CreateWaitingListItem(waitingListItem, loggedUser.Facility);
                    }

                    return View("Confirmation", model);
                }
                else
                {
                    model.WaitingAreaList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea);
                    model.DispositionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeDisposition);
                    model.DispositionList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                    model.ContractList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                    model.ContractList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                    model.AuthorizationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                    model.AuthorizationList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                    model.VistaSpecialitiesList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaSpecialty);
                    model.VistaSpecialitiesList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                    model.FeeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeReason);
                    model.FeeList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                    model.EvacDispositionStatusList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.EvacDispositionStatus);
                    model.EvacTransportationTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationType);
                    model.EvacTransportationProviderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationProvider);
                    model.FacilitySiteCodeTimeZone = GetFacilityTimeZone();

                    #region Division
                    model.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();

                    if (!string.IsNullOrEmpty(model.SelectedWardGroup))
                    {
                        foreach (var item in model.WardGroupList)
                        {
                            if (item.Id.extension.Equals(model.SelectedWardGroup, StringComparison.InvariantCultureIgnoreCase))
                            {
                                model.SelectedDivision = item;
                            }
                        }
                    }
                    #endregion //Division

                    #region Bed
                    if (model.SelectedDivision != null)
                    {
                        model.BedList = FacadeManager.EntityInterface.GetBedsInDivision(model.SelectedDivision.Id, loggedUser.VistaSite.Id)
                                                                     .OrderBy(bed => bed.Name)
                                                                     .ToList();
                        model.BedListDisplay = Strings.StyleDisplayTableRow;
                    }
                    else
                    {
                        model.BedList = new List<Bed>();
                        model.BedListDisplay = Strings.StyleDisplayNone;
                    }
                    #endregion //Bed

                    if (!string.IsNullOrEmpty(model.WaitingAreaCode) && model.WaitingAreaList.Where(a => a.code == model.WaitingAreaCode).FirstOrDefault() == null)
                    {
                        model.WaitingAreaList.Add(new CD() { code = model.WaitingAreaCode, displayName = model.WaitingAreaDisplayName });
                    }

                    return View(model);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Refill model after selecting a division from drop-down and display Admission View.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns>Admission view.</returns>
        [HttpPost]
        public ActionResult RefreshAdmission([Bind(Exclude = "RememberMe")] PatientViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.LoggedUser = loggedUser;
                model.WaitingAreaList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea);
                model.DispositionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeDisposition);
                model.DispositionList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                model.ContractList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                model.ContractList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                model.AuthorizationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                model.AuthorizationList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                model.VistaSpecialitiesList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaSpecialty);
                model.VistaSpecialitiesList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                model.FeeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeReason);
                model.FeeList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                model.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                model.WardGroupList.Insert(0, new Division() { Id = new II() { root = this.loggedUser.Domain, extension = (default(Guid)).ToString() }, Name = string.Empty });
                model.BedList = new List<Bed>();
                model.EvacDispositionStatusList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.EvacDispositionStatus);
                model.EvacTransportationTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationType);
                model.EvacTransportationProviderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationProvider);
                model.FacilitySiteCodeTimeZone = GetFacilityTimeZone();

                if (!string.IsNullOrEmpty(model.SelectedWardGroup) && (model.SelectedWardGroup != (default(Guid)).ToString()))
                {
                    model.BedList = FacadeManager.EntityInterface.GetBedsInDivision(new II(this.loggedUser.Domain, model.SelectedWardGroup), loggedUser.VistaSite.Id)
                                                                 .OrderBy(bed => bed.Name)
                                                                 .ToList();
                    model.DateTimeBedAssigned = DateTimeNowOnCurrentFacility;
                    if (model.DateTimeBedAssigned != null)
                    {
                        model.AssignmentHourSelected = (model.DateTimeBedAssigned.Value.Hour < 10) ? "0" + model.DateTimeBedAssigned.Value.Hour.ToString() : model.DateTimeBedAssigned.Value.Hour.ToString();
                        model.AssignmentMinuteSelected = (model.DateTimeBedAssigned.Value.Minute < 10) ? "0" + model.DateTimeBedAssigned.Value.Minute.ToString() : model.DateTimeBedAssigned.Value.Minute.ToString();
                    }
                    else
                    {
                        model.AssignmentHourSelected = "00";
                        model.AssignmentMinuteSelected = "00";
                    }
                }
                else
                {
                    model.DateTimeBedAssigned = null;
                    model.DateTimeBedAssignedString = string.Empty;
                    model.AssignmentHourSelected = "00";
                    model.AssignmentMinuteSelected = "00";
                    model.BedListDisplay = Strings.StyleDisplayNone;
                }

                if (!string.IsNullOrEmpty(model.WaitingAreaCode) && model.WaitingAreaList.Where(a => a.code == model.WaitingAreaCode).FirstOrDefault() == null)
                    model.WaitingAreaList.Add(new CD() { code = model.WaitingAreaCode, displayName = model.WaitingAreaDisplayName });

                ModelState.Clear();

                return View("Admission", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        #region Private methods

        private PatientViewModel FillModelForEdit(WaitingListItem waitingListItem)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DateTime requestedDate = GetNotNullableDateTime(waitingListItem.RequestedDate);
                DateTime? assignmentDate = waitingListItem.RequestedBedDate;
                PatientViewModel vm = FillModel(requestedDate, assignmentDate);
                vm.IdAct = (waitingListItem.Id == null) ? null : waitingListItem.Id.extension;
                vm.PatientFullName = GetPatientFullName(waitingListItem.Patient);
                vm.PatientSSN = string.Format("{0}{1}", "xxx-xx-", waitingListItem.Patient.SSN.extension.Substring(waitingListItem.Patient.SSN.extension.Length - 4));
                vm.PatientRoot = waitingListItem.Patient.Id.root;
                vm.PatientExtension = waitingListItem.Patient.Id.extension;
                vm.PatientGender = (waitingListItem.Patient.Gender != null) ? waitingListItem.Patient.Gender.displayName : string.Empty;
                vm.EntryDateTime = GetNotNullableDateTime(waitingListItem.CreationDate);
                vm.DateTimeBedAssigned = assignmentDate;
                vm.DateTimeBedRequested = requestedDate;
                vm.TypeOfBed = waitingListItem.TypeOfBedWard;
                vm.WaitingAreaCode = (waitingListItem.WaitingArea == null) ? string.Empty : waitingListItem.WaitingArea.code;
                vm.DispositionCode = (waitingListItem.FeeDisposition == null) ? string.Empty : waitingListItem.FeeDisposition.code;
                vm.ContractCode = (waitingListItem.Contract == null) ? string.Empty : waitingListItem.Contract.code;
                vm.AuthorizedCode = (waitingListItem.Authorization == null) ? string.Empty : waitingListItem.Authorization.code;
                vm.VistaSpecialtyCode = (waitingListItem.ServiceReceive == null) ? string.Empty : waitingListItem.ServiceReceive.code;
                vm.ReasonsUsingFeeCode = (waitingListItem.ReasonFee == null) ? string.Empty : waitingListItem.ReasonFee.code;
                vm.FeeAccepted = waitingListItem.FeeAccepted;
                vm.FeeAcceptingMD = waitingListItem.FeeAcceptingMD;
                vm.FeeComments = waitingListItem.CommentFee;
                vm.PresentProblem = waitingListItem.Problem;
                vm.BedAssignmentDisplay = Strings.StyleDisplayBlock;
                vm.RemovedDate = waitingListItem.RemovedDate;
                vm.IsInHouse = waitingListItem.IsInHouse;
                vm.Comments = waitingListItem.Comments;
                vm.IsBedInIsolation = waitingListItem.IsBedInIsolation;
                vm.WaitTimeAlertDate = waitingListItem.WaitTimeAlertDate;
                vm.IsCLCPatient = waitingListItem.IsCommunityLivingCenter;
                vm.IsEvacuationPatient = waitingListItem.IsEvacuationPatient;
                vm.IsMentalHealthPatient = waitingListItem.IsMentalHealthPatient;
                vm.EvacActualDate = waitingListItem.EvacuationActualDate;
                vm.AdmittingDiagnosis = (waitingListItem.AdmittingDiagnosis != null) ? waitingListItem.AdmittingDiagnosis : string.Empty;
                PatientLocation location = FacadeManager.ADTInterface.GetPatientLocation(waitingListItem.Patient.Id, loggedUser.VistaSite.Id);
                if (location != null)
                {
                    vm.CurrentBed = (location.CurrentAdmissionBed != null) ? location.CurrentAdmissionBed.Name : string.Empty;
                    vm.CurrentWard = (location.CurrentAdmissionWard != null) ? location.CurrentAdmissionWard.Name : string.Empty;
                }
                else
                {
                    vm.CurrentBed = string.Empty;
                    vm.CurrentWard = string.Empty;
                }
                FillModelEvacuationData(waitingListItem, vm);

                if (waitingListItem.IsMentalHealthPatient)
                {
                    vm.DateOfScreening = waitingListItem.DateOfScreening;
                    vm.ContactPhoneNumber = waitingListItem.ContactPhoneNumber;
                    vm.VeteranPreference = waitingListItem.VeteranPreference;
                    vm.MentalHealthComments = waitingListItem.MentalHealthComment;

                    if (waitingListItem.DateOfScreening != null)
                    {
                        vm.DateOfScreeningHourSelected = (vm.DateOfScreening.Value.Hour < 10) ? "0" + vm.DateOfScreening.Value.Hour.ToString() : vm.DateOfScreening.Value.Hour.ToString();
                        vm.DateOfScreeningMinuteSelected = (vm.DateOfScreening.Value.Minute < 10) ? "0" + vm.DateOfScreening.Value.Minute.ToString() : vm.DateOfScreening.Value.Minute.ToString();
                    }
                    else
                    {
                        vm.DateOfScreeningHourSelected = "00";
                        vm.DateOfScreeningMinuteSelected = "00";
                    }
                }

                vm.FacilitySiteCodeTimeZone = GetFacilityTimeZone();

                #region Division
                //load divisions
                vm.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                vm.WardGroupList.Insert(0, new Division() { Id = new II() { root = vm.LoggedUser.Domain, extension = (default(Guid)).ToString() }, Name = string.Empty });

                //select division based on bed and facility
                if (waitingListItem.RequestedBed != null && !string.IsNullOrEmpty(waitingListItem.RequestedBed.Id.extension))
                    vm.SelectedDivision = FacadeManager.EntityInterface.GetDivisionsByBedAndFacility(waitingListItem.RequestedBed.Id, this.loggedUser.Facility.Id)
                                                                       .First();
                //set division
                vm.SelectedWardGroup = (vm.SelectedDivision != null) ? vm.SelectedDivision.Id.extension : (default(Guid)).ToString();

                #endregion //Division

                #region Bed
                //load beds for the selected division
                if (vm.SelectedDivision != null)
                {
                    vm.BedList = FacadeManager.EntityInterface.GetBedsInDivision(vm.SelectedDivision.Id, loggedUser.VistaSite.Id)
                                                              .OrderBy(bed => bed.Name)
                                                              .ToList();
                    vm.BedListDisplay = Strings.StyleDisplayTableRow;
                }
                else
                    vm.BedListDisplay = Strings.StyleDisplayNone;

                //set bed
                vm.SelectedBedId = (waitingListItem.RequestedBed != null) ? waitingListItem.RequestedBed.Id.extension : string.Empty;

                #endregion //Bed

                if (!string.IsNullOrEmpty(vm.WaitingAreaCode) && vm.WaitingAreaList.Where(a => a.code == vm.WaitingAreaCode).FirstOrDefault() == null)
                {
                    vm.WaitingAreaDisplayName = waitingListItem.WaitingArea.displayName;
                    vm.WaitingAreaList.Add(waitingListItem.WaitingArea);
                }

                return vm;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillModelEvacuationData(WaitingListItem waitingListItem, PatientViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (waitingListItem.IsEvacuationPatient)
                {
                    model.EvacExpectedDate = waitingListItem.EvacuationExpectedDate;
                    model.EvacActualDate = waitingListItem.EvacuationActualDate;
                    if (waitingListItem.EvacuationActualDate != null)
                    {
                        model.EvacActualDateHourSelected = (model.EvacActualDate.Value.Hour < 10) ? "0" + model.EvacActualDate.Value.Hour.ToString() : model.EvacActualDate.Value.Hour.ToString();
                        model.EvacActualDateMinuteSelected = (model.EvacActualDate.Value.Minute < 10) ? "0" + model.EvacActualDate.Value.Minute.ToString() : model.EvacActualDate.Value.Minute.ToString();
                    }
                    else
                    {
                        model.EvacActualDateHourSelected = "00";
                        model.EvacActualDateMinuteSelected = "00";
                    }

                    if (waitingListItem.EvacuationExpectedDate != null)
                    {
                        model.EvacExpectedDateHourSelected = (model.EvacExpectedDate.Value.Hour < 10) ? "0" + model.EvacExpectedDate.Value.Hour.ToString() : model.EvacExpectedDate.Value.Hour.ToString();
                        model.EvacExpectedDateMinuteSelected = (model.EvacExpectedDate.Value.Minute < 10) ? "0" + model.EvacExpectedDate.Value.Minute.ToString() : model.EvacExpectedDate.Value.Minute.ToString();
                    }
                    else
                    {
                        model.EvacExpectedDateHourSelected = "00";
                        model.EvacExpectedDateMinuteSelected = "00";
                    }

                    model.EvacComments = waitingListItem.EvacuationComments;
                    model.EvacDestinationAddress = waitingListItem.EvacuationDestinationAddress;
                    model.EvacDestinationCityState = waitingListItem.EvacuationDestinationCityState;
                    model.EvacDestinationPOCTelephone = waitingListItem.EvacuationDestinationPOCTelephone;
                    model.EvacEnrouteMedicalAttendant = waitingListItem.EvacuationEnrouteMedicalAttendant;
                    model.EvacEnrouteNonMedicalAttendant = waitingListItem.EvacuationEnrouteNonMedicalAttendant;
                    model.EvacEquipSupplyRequirements = waitingListItem.EvacuationEquipSupplyRequirements;
                    model.EvacOriginatorPOCTelephone = waitingListItem.EvacuationOriginatorPOCTelephone;
                    model.EvacReasonForDelay = waitingListItem.EvacuationReasonForDelay;
                    model.EvacTransferEvacuateTo = waitingListItem.EvacuationTransferEvacuateTo;
                    model.EvacTransportationCareLevel = waitingListItem.EvacuationTransportationCareLevel;
                    model.EvacTransportationComments = waitingListItem.EvacuationTransportationComments;
                    model.SelectedEvacDispositionStatusCode = waitingListItem.EvacuationDispositionStatus == null ? string.Empty : waitingListItem.EvacuationDispositionStatus.code;
                    model.SelectedEvacTransportationProviderCode = waitingListItem.EvacuationTransportationProvider == null ? string.Empty : waitingListItem.EvacuationTransportationProvider.code;
                    model.SelectedEvacTransportationTypeCode = waitingListItem.EvacuationTransportationType == null ? string.Empty : waitingListItem.EvacuationTransportationType.code;
                    model.EvacEnrouteMedicalAttendantNumber = waitingListItem.EvacuationEnrouteMedicalAttendantNumber;
                    model.EvacEnrouteNonMedicalAttendantNumber = waitingListItem.EvacuationEnrouteNonMedicalAttendantNumber;
                    
                    model.EvacArrivalTime = waitingListItem.EvacuationArrivalTime;
                    if (waitingListItem.EvacuationArrivalTime != null)
                    {
                        model.EvacArrivalTimeHourSelected = (model.EvacArrivalTime.Value.Hour < 10) ? "0" + model.EvacArrivalTime.Value.Hour.ToString() : model.EvacArrivalTime.Value.Hour.ToString();
                        model.EvacArrivalTimeMinuteSelected = (model.EvacArrivalTime.Value.Minute < 10) ? "0" + model.EvacArrivalTime.Value.Minute.ToString() : model.EvacArrivalTime.Value.Minute.ToString();
                    }
                    else
                    {
                        model.EvacArrivalTimeHourSelected = "00";
                        model.EvacArrivalTimeMinuteSelected = "00";
                    }

                    model.EvacVehicleID = waitingListItem.EvacuationVehicleID;
                    model.EvacVehicleSeatPosition = waitingListItem.EvacuationVehicleSeatPosition;
                    model.EvacMATelephone = waitingListItem.EvacuationMATelephone;
                    model.EvacNMATelephone = waitingListItem.EvacuationNMATelephone;
                    model.EvacOriginatingFacilityName = waitingListItem.EvacuationOriginatingFacilityName;
                    model.EvacOriginatingAddress = waitingListItem.EvacuationOriginatingAddress;
                    model.EvacOriginatingCityState = waitingListItem.EvacuationOriginatingCityState;
                    model.EvacVehicleOperatorName = waitingListItem.EvacuationVehicleOperatorName;
                    model.EvacVehicleOperatorTelephone = waitingListItem.EvacuationVehicleOperatorTelephone;
                }
                else
                {
                    model.EvacActualDateHourSelected = "00";
                    model.EvacActualDateMinuteSelected = "00";
                    model.EvacExpectedDateHourSelected = "00";
                    model.EvacExpectedDateMinuteSelected = "00";
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private PatientViewModel FillModel(DateTime requestedDate, DateTime? assignmentDate)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                PatientViewModel vm = new PatientViewModel()
                {
                    EntryDateTime = DateTimeNowOnCurrentFacility,
                    DateTimeBedRequested = requestedDate,
                    DateTimeBedAssigned = assignmentDate,
                    WaitingAreaList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea),
                    EvacDispositionStatusList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.EvacDispositionStatus),
                    EvacTransportationTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationType),
                    EvacTransportationProviderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationProvider),
                    WardGroupList = new List<Division>(),
                    BedList = new List<Bed>(),
                    LoggedUser = this.loggedUser
                };

                vm.DispositionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeDisposition);
                vm.DispositionList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                vm.ContractList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                vm.ContractList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                vm.AuthorizationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                vm.AuthorizationList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                vm.VistaSpecialitiesList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaSpecialty);
                vm.VistaSpecialitiesList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                vm.FeeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeReason);
                vm.FeeList.Insert(0, new CD { code = string.Empty, displayName = string.Empty });
                vm.RequestHourSelected = (requestedDate.Hour < 10) ? "0" + requestedDate.Hour.ToString() : requestedDate.Hour.ToString();
                vm.RequestMinuteSelected = (requestedDate.Minute < 10) ? "0" + requestedDate.Minute.ToString() : requestedDate.Minute.ToString();
                if (assignmentDate != null)
                {
                    vm.AssignmentHourSelected = (assignmentDate.Value.Hour < 10) ? "0" + assignmentDate.Value.Hour.ToString() : assignmentDate.Value.Hour.ToString();
                    vm.AssignmentMinuteSelected = (assignmentDate.Value.Minute < 10) ? "0" + assignmentDate.Value.Minute.ToString() : assignmentDate.Value.Minute.ToString();
                }
                else
                {
                    vm.AssignmentHourSelected = "00";
                    vm.AssignmentMinuteSelected = "00";
                }

                vm.IsInHouse = false;
                vm.IsBedInIsolation = false;
                vm.EvacExpectedDateHourSelected = "00";
                vm.EvacExpectedDateMinuteSelected = "00";
                vm.EvacActualDateHourSelected = "00";
                vm.EvacActualDateMinuteSelected = "00";
                vm.EvacArrivalTimeHourSelected = "00";
                vm.EvacArrivalTimeMinuteSelected = "00";
                vm.DateOfScreeningHourSelected = "00";
                vm.DateOfScreeningMinuteSelected = "00";

                return vm;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void GetWaitingListItemForSave(PatientViewModel model, out WaitingListItem waitingListItem, out bool isWaitingAreaAvailable)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                waitingListItem = null;
                isWaitingAreaAvailable = true;
                double value = 0;
                if (double.TryParse(model.RequestHourSelected, out value))
                    model.DateTimeBedRequested = model.DateTimeBedRequested.AddHours(value);
                if (double.TryParse(model.RequestMinuteSelected, out value))
                    model.DateTimeBedRequested = model.DateTimeBedRequested.AddMinutes(value);

                if (model.DateTimeBedAssigned != null && double.TryParse(model.AssignmentHourSelected, out value))
                    model.DateTimeBedAssigned = model.DateTimeBedAssigned.Value.AddHours(value);
                if (model.DateTimeBedAssigned != null && double.TryParse(model.AssignmentMinuteSelected, out value))
                    model.DateTimeBedAssigned = model.DateTimeBedAssigned.Value.AddMinutes(value);

                var waitingAreaList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea);
                var dispositionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeDisposition);
                var contractList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                var authorizationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                var vistaSpecialitiesList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaSpecialty);
                var reasonUsingFeeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.FeeReason);
                var CLCList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                var evacDispositionStatusList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.EvacDispositionStatus);
                var evacTransportationTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationType);
                var evacTransportationProviderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationProvider);

                Bed bed = string.IsNullOrEmpty(model.SelectedBedId) ? null : FacadeManager.BedInterface.GetBed(new II(loggedUser.Domain, model.SelectedBedId), loggedUser.VistaSite.Id);

                if (!string.IsNullOrEmpty(model.IdAct))
                    waitingListItem = FacadeManager.ADTInterface.GetWaitingListItem(new II(this.loggedUser.Domain, model.IdAct), loggedUser);
                else
                    waitingListItem = new WaitingListItem()
                    {
                        Id = new II(this.loggedUser.Domain, model.IdAct),
                        Patient = FacadeManager.EntityInterface.GetPatientById(new II(model.PatientRoot, model.PatientExtension), null),
                        CreationDate = model.EntryDateTime,
                        Facility = this.loggedUser.Facility,
                        VistaSite = FacadeManager.EntityInterface.GetVistaSite(this.loggedUser.Facility.VistaSite.Id)
                    };

                waitingListItem.RequestedBedDate = model.DateTimeBedAssigned;
                waitingListItem.RequestedDate = model.DateTimeBedRequested;
                waitingListItem.Problem = model.PresentProblem;
                waitingListItem.TypeOfBedWard = model.TypeOfBed;
                waitingListItem.CommentFee = model.FeeComments;
                waitingListItem.RequestedBed = (bed != null) ? new Place() { Id = bed.Id, Name = bed.Name } : null;
                waitingListItem.IsInHouse = model.IsInHouse;
                waitingListItem.LastEditBy = this.loggedUser.UserName;
                waitingListItem.Comments = model.Comments;
                waitingListItem.IsBedInIsolation = model.IsBedInIsolation;
                waitingListItem.WaitTimeAlertDate = model.WaitTimeAlertDate;
                waitingListItem.IsEvacuationPatient = model.IsEvacuationPatient;
                waitingListItem.IsMentalHealthPatient = model.IsMentalHealthPatient;

                if (model.WaitingAreaCode != null)
                {
                    waitingListItem.WaitingArea = waitingAreaList.SingleOrDefault(w => w.code == model.WaitingAreaCode);
                    if (waitingListItem.WaitingArea == null)
                        isWaitingAreaAvailable = false;
                }
                if (model.DispositionCode != null)
                    waitingListItem.FeeDisposition = dispositionList.SingleOrDefault(d => d.code == model.DispositionCode);
                if (model.ContractCode != null)
                    waitingListItem.Contract = contractList.SingleOrDefault(c => c.code == model.ContractCode);
                if (model.AuthorizedCode != null)
                    waitingListItem.Authorization = authorizationList.SingleOrDefault(a => a.code == model.AuthorizedCode);
                if (model.VistaSpecialtyCode != null)
                    waitingListItem.ServiceReceive = vistaSpecialitiesList.SingleOrDefault(s => s.code == model.VistaSpecialtyCode);
                if (model.ReasonsUsingFeeCode != null)
                    waitingListItem.ReasonFee = reasonUsingFeeList.SingleOrDefault(s => s.code == model.ReasonsUsingFeeCode);
                waitingListItem.IsCommunityLivingCenter = model.IsCLCPatient;

                waitingListItem.FeeAccepted = model.FeeAccepted;
                waitingListItem.FeeAcceptingMD = model.FeeAcceptingMD;

                if (model.IsEvacuationPatient)
                {
                    if (!string.IsNullOrEmpty(model.SelectedEvacTransportationTypeCode))
                        waitingListItem.EvacuationTransportationType = evacTransportationTypeList.SingleOrDefault(s => s.code == model.SelectedEvacTransportationTypeCode);
                    else
                        waitingListItem.EvacuationTransportationType = null;

                    if (!string.IsNullOrEmpty(model.SelectedEvacTransportationProviderCode))
                        waitingListItem.EvacuationTransportationProvider = evacTransportationProviderList.SingleOrDefault(s => s.code == model.SelectedEvacTransportationProviderCode);
                    else
                        waitingListItem.EvacuationTransportationProvider = null;

                    if (!string.IsNullOrEmpty(model.SelectedEvacDispositionStatusCode))
                        waitingListItem.EvacuationDispositionStatus = evacDispositionStatusList.SingleOrDefault(s => s.code == model.SelectedEvacDispositionStatusCode);
                    else
                        waitingListItem.EvacuationDispositionStatus = null;

                    waitingListItem.EvacuationExpectedDate = CreateDateTime(model.EvacExpectedDate, model.EvacExpectedDateHourSelected, model.EvacExpectedDateMinuteSelected);
                    waitingListItem.EvacuationActualDate = CreateDateTime(model.EvacActualDate, model.EvacActualDateHourSelected, model.EvacActualDateMinuteSelected);
                    waitingListItem.EvacuationEnrouteMedicalAttendant = model.EvacEnrouteMedicalAttendant;
                    waitingListItem.EvacuationEnrouteNonMedicalAttendant = model.EvacEnrouteNonMedicalAttendant;
                    waitingListItem.EvacuationEquipSupplyRequirements = model.EvacEquipSupplyRequirements;

                    waitingListItem.EvacuationOriginatorPOCTelephone = model.EvacOriginatorPOCTelephone;
                    waitingListItem.EvacuationReasonForDelay = model.EvacReasonForDelay;
                    waitingListItem.EvacuationTransferEvacuateTo = model.EvacTransferEvacuateTo;
                    waitingListItem.EvacuationTransportationCareLevel = model.EvacTransportationCareLevel;
                    waitingListItem.EvacuationTransportationComments = model.EvacTransportationComments;

                    waitingListItem.EvacuationComments = model.EvacComments;
                    waitingListItem.EvacuationDestinationAddress = model.EvacDestinationAddress;
                    waitingListItem.EvacuationDestinationCityState = model.EvacDestinationCityState;
                    waitingListItem.EvacuationDestinationPOCTelephone = model.EvacDestinationPOCTelephone;
                    waitingListItem.EvacuationEnrouteMedicalAttendantNumber = model.EvacEnrouteMedicalAttendantNumber;
                    waitingListItem.EvacuationEnrouteNonMedicalAttendantNumber = model.EvacEnrouteNonMedicalAttendantNumber;
                    waitingListItem.EvacuationArrivalTime = CreateDateTime(model.EvacArrivalTime, model.EvacArrivalTimeHourSelected, model.EvacArrivalTimeMinuteSelected); ;
                    waitingListItem.EvacuationVehicleID = model.EvacVehicleID;
                    waitingListItem.EvacuationVehicleSeatPosition = model.EvacVehicleSeatPosition;

                    waitingListItem.EvacuationMATelephone = model.EvacMATelephone;
                    waitingListItem.EvacuationNMATelephone = model.EvacNMATelephone;
                    waitingListItem.EvacuationOriginatingFacilityName = model.EvacOriginatingFacilityName;
                    waitingListItem.EvacuationOriginatingAddress = model.EvacOriginatingAddress;
                    waitingListItem.EvacuationOriginatingCityState = model.EvacOriginatingCityState;
                    waitingListItem.EvacuationVehicleOperatorName = model.EvacVehicleOperatorName;
                    waitingListItem.EvacuationVehicleOperatorTelephone = model.EvacVehicleOperatorTelephone;
                }
                else
                {
                    waitingListItem.EvacuationTransportationType = null;
                    waitingListItem.EvacuationTransportationProvider = null;
                    waitingListItem.EvacuationEnrouteMedicalAttendant = null;
                    waitingListItem.EvacuationEnrouteNonMedicalAttendant = null;
                    waitingListItem.EvacuationEquipSupplyRequirements = null;
                    waitingListItem.EvacuationExpectedDate = null;
                    waitingListItem.EvacuationOriginatorPOCTelephone = null;
                    waitingListItem.EvacuationReasonForDelay = null;
                    waitingListItem.EvacuationTransferEvacuateTo = null;
                    waitingListItem.EvacuationTransportationCareLevel = null;
                    waitingListItem.EvacuationTransportationComments = null;
                    waitingListItem.EvacuationDispositionStatus = null;
                    waitingListItem.EvacuationActualDate = null;
                    waitingListItem.EvacuationComments = null;
                    waitingListItem.EvacuationDestinationAddress = null;
                    waitingListItem.EvacuationDestinationCityState = null;
                    waitingListItem.EvacuationDestinationPOCTelephone = null;
                    waitingListItem.EvacuationEnrouteMedicalAttendantNumber = null;
                    waitingListItem.EvacuationEnrouteNonMedicalAttendantNumber = null;
                    waitingListItem.EvacuationArrivalTime = null;
                    waitingListItem.EvacuationVehicleID = null;
                    waitingListItem.EvacuationVehicleSeatPosition = null;
                    waitingListItem.EvacuationMATelephone = null;
                    waitingListItem.EvacuationNMATelephone = null;
                    waitingListItem.EvacuationOriginatingFacilityName = null;
                    waitingListItem.EvacuationOriginatingAddress = null;
                    waitingListItem.EvacuationOriginatingCityState = null;
                    waitingListItem.EvacuationVehicleOperatorName = null;
                    waitingListItem.EvacuationVehicleOperatorTelephone = null;
                }

                if (waitingListItem.IsMentalHealthPatient)
                {
                    waitingListItem.DateOfScreening = CreateDateTime(model.DateOfScreening, model.DateOfScreeningHourSelected, model.DateOfScreeningMinuteSelected);
                    waitingListItem.ContactPhoneNumber = model.ContactPhoneNumber;
                    waitingListItem.VeteranPreference = model.VeteranPreference;
                    waitingListItem.MentalHealthComment = model.MentalHealthComments;
                }
                else
                {
                    waitingListItem.DateOfScreening = null;
                    waitingListItem.ContactPhoneNumber = null;
                    waitingListItem.VeteranPreference = null;
                    waitingListItem.MentalHealthComment = null;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private DateTime? CreateDateTime(DateTime? date, string hours, string minutes)
        {
            if (date == null || date == DateTime.MinValue)
                return null;
            else
            {
                DateTime returnDate = date.Value;
                double value = 0;
                if (double.TryParse(hours, out value))
                    returnDate = returnDate.AddHours(value);
                if (double.TryParse(minutes, out value))
                    returnDate = returnDate.AddMinutes(value);
                return returnDate;
            }
        }

        private string GetPatientFullName(Patient patient)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (patient == null)
                    return null;
                return string.Format("{0}, {1} {2}", patient.LastName, patient.FirstName, patient.MiddleName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private DateTime GetNotNullableDateTime(DateTime? time)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return time.HasValue ? time.Value : DateTimeNowOnCurrentFacility;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
