﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using Microsoft.Web.Mvc;
using InfoWorld.HL7.ITS;
using BMS.Web.App_GlobalResource;
using BMS.Utils;
using BMS.Facade.Data;
using System.Text;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Facility Diversion Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class FacilityDiversionController : BaseController
    {
        /// <summary>
        /// Default action for Facility Diversion Controller.
        /// </summary>
        /// <returns>Returns Index view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FacilityDiversionViewModel mainViewModel = new FacilityDiversionViewModel();
                mainViewModel.IsFromVisn = QueryStrings.ContainsKey("isFromVisn") && QueryStrings["isFromVisn"].ToString() == "True";
                mainViewModel.CurrentButton = Strings.CurrentDiversions;
                FillViewModel(mainViewModel);
                return View(mainViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for Add or Edit View.
        /// </summary>
        /// <param name="mainViewModel">The main view model.</param>
        /// <returns>Returns AddConfirmation or EditConfirmation view result.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(FacilityDiversionViewModel mainViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                mainViewModel.Facility = GetFacility();
                mainViewModel.IsFromVisn = QueryStrings.ContainsKey("isFromVisn") && QueryStrings["isFromVisn"].ToString() == "True"; ;

                if (!string.IsNullOrEmpty(mainViewModel.ButtonAdd))
                    return this.RedirectToAction<FacilityDiversionController>(act => act.Add(EncryptQueryString(new string[] { "location", "facilityId", "isFromVisn" }, new string[] { mainViewModel.Location, mainViewModel.Facility.Id.extension, mainViewModel.IsFromVisn.ToString() }, loggedUser.Salt)));
                
                if (!string.IsNullOrEmpty(mainViewModel.ButtonAllDiversions))
                {
                    mainViewModel.CurrentButton = Strings.AllDiversions;                    
                    FillViewModel(mainViewModel);
                    return View(mainViewModel);
                }
                
                if (!string.IsNullOrEmpty(mainViewModel.ButtonCurrentDiversions))
                {
                    mainViewModel.CurrentButton = Strings.CurrentDiversions;
                    FillViewModel(mainViewModel);
                    return View(mainViewModel);
                }
                
                if (!string.IsNullOrEmpty(mainViewModel.ButtonShowReport))
                {
                    StringBuilder diversionsReportUrl = new StringBuilder();
                    diversionsReportUrl.Append("currentButton=");
                    diversionsReportUrl.Append(mainViewModel.CurrentButton);
                    return Redirect("/Reporting/LocalReportViewer.aspx?" + CustomEncryption.Encrypt(diversionsReportUrl.ToString(), loggedUser.Salt) + "*fd");
                }
                return null;                                                    
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        public ActionResult Add(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FacilityDiversionViewModel mainViewModel = new FacilityDiversionViewModel();
                mainViewModel.Location = QueryStrings["location"];
                mainViewModel.LoggedUser = this.loggedUser;
                mainViewModel.IsFromVisn = QueryStrings.ContainsKey("isFromVisn") && QueryStrings["isFromVisn"].ToString() == "True"; ;
                mainViewModel.StartDate = DateTimeNowOnCurrentFacility;
                mainViewModel.HourStartDateSelected = (mainViewModel.StartDate.Value.Hour < 10) ? "0" + mainViewModel.StartDate.Value.Hour.ToString() : mainViewModel.StartDate.Value.Hour.ToString();
                mainViewModel.MinuteStartDateSelected = (mainViewModel.StartDate.Value.Minute < 10) ? "0" + mainViewModel.StartDate.Value.Minute.ToString() : mainViewModel.StartDate.Value.Minute.ToString();
                mainViewModel.Comments = string.Empty;                
                mainViewModel.FacilityAddDisplay = string.Empty;
                mainViewModel.FacilityAlreadyExists = Strings.StyleDisplayNone;
                mainViewModel.Facility = GetFacility();

                if (mainViewModel.Location == "Facility")
                {
                    IEnumerable<string> wardGroups;
                    Dictionary<string, List<string>> wardGroupSpecialties;
                    LoadWardGroupsAndSpecialties(mainViewModel.Facility as Facility, out wardGroups, out wardGroupSpecialties);

                    mainViewModel.WardGroups = wardGroups;
                    mainViewModel.WardGroup = "All";

                    if (wardGroupSpecialties.ContainsKey(mainViewModel.WardGroup))
                        mainViewModel.TreatingSpecialties = wardGroupSpecialties[mainViewModel.WardGroup];
                }
                else
                {
                    mainViewModel.WardGroup = "All";
                    mainViewModel.TreatingSpecialty = "All";
                }

                return View(mainViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show Add view.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns>Returns Add view result.</returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Add(FacilityDiversionViewModel mainViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                mainViewModel.LocationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Location);
                mainViewModel.LoggedUser = this.loggedUser;
                mainViewModel.IsFromVisn = QueryStrings.ContainsKey("isFromVisn") && QueryStrings["isFromVisn"].ToString() == "True"; ;

                if (!string.IsNullOrWhiteSpace(mainViewModel.StartDateMessageError))
                {
                    ModelState.AddModelError("StartDate", mainViewModel.StartDateMessageError);                    
                    mainViewModel.FacilityAddDisplay = string.Empty;
                    mainViewModel.FacilityAlreadyExists = Strings.StyleDisplayNone;

                    return View("Add", mainViewModel);
                }

                mainViewModel.Facility = GetFacility();

                Diversion diversion = new Diversion();
                diversion.Facility = mainViewModel.Facility;
                if (mainViewModel.LocationList != null && mainViewModel.LocationList.Count > 0)
                    diversion.Location = mainViewModel.LocationList
                                                      .First(loc => loc.displayName == mainViewModel.Location);
                diversion.StartDate = GetDate(mainViewModel.StartDate.Value, mainViewModel.HourStartDateSelected, mainViewModel.MinuteStartDateSelected);
                /*
                 * HHG: INC000001050969 Replace diversion.EndDate with DateTimeNowOnCurrentFacility                 
                 */
                diversion.EndDate = DateTimeNowOnCurrentFacility;
                diversion.Comments = mainViewModel.Comments;

                if (mainViewModel.Location == "Facility")
                {
                    mainViewModel.WardGroup = String.IsNullOrEmpty(mainViewModel.WardGroup) ? "All" : mainViewModel.WardGroup;
                    mainViewModel.TreatingSpecialty = String.IsNullOrEmpty(mainViewModel.TreatingSpecialty) ? "All" : mainViewModel.TreatingSpecialty;
                }
                else
                {
                    mainViewModel.WardGroup = "All";
                    mainViewModel.TreatingSpecialty = "All";
                }

                diversion.WardGroup = mainViewModel.WardGroup;
                diversion.TreatingSpecialty = mainViewModel.TreatingSpecialty;
                diversion.Status = Constants.Yes;
                diversion.EnteredBy = this.loggedUser.UserName;
                diversion.EnteredDate = DateTimeNowOnCurrentFacility;
                diversion.LastEditBy = string.Empty;
                diversion.Id = new II { root = this.loggedUser.Domain, extension = null };

                //save in DB
                FacadeManager.DiversionInterface.SaveDiversion(diversion, true, mainViewModel.Facility.Id);

                //display Confirmation view                
                mainViewModel.IsAddConfirmation = true;
                return View("Confirmation", mainViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show edit view.
        /// </summary>
        /// <param name="id">The id.</param>
        /// <returns>Returns Edit view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Edit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FacilityDiversionViewModel mainViewModel = new FacilityDiversionViewModel();
                mainViewModel.Id = QueryStrings["id"];
                mainViewModel.LoggedUser = this.loggedUser;
                mainViewModel.IsFromVisn = QueryStrings.ContainsKey("isFromVisn") && QueryStrings["isFromVisn"].ToString() == "True";
                mainViewModel.Facility = GetFacility();
                Diversion diversion = FacadeManager.DiversionInterface.FindDiversionById(this.loggedUser.Domain, QueryStrings["id"], mainViewModel.Facility.Id);

                mainViewModel.LocationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Location).OrderByDescending(loc => loc.displayName).ToList();
                if (mainViewModel.LocationList != null && mainViewModel.LocationList.Count > 0)
                    mainViewModel.DiversionLocation = mainViewModel.LocationList.First(loc => loc.displayName == diversion.Location.displayName);
                mainViewModel.Location = diversion.Location.displayName;
                mainViewModel.StartDate = diversion.StartDate;
                mainViewModel.HourStartDateSelected = (diversion.StartDate.Hour < 10) ? "0" + diversion.StartDate.Hour.ToString() : diversion.StartDate.Hour.ToString();
                mainViewModel.MinuteStartDateSelected = (diversion.StartDate.Minute < 10) ? "0" + diversion.StartDate.Minute.ToString() : diversion.StartDate.Minute.ToString();
                /*
                 * HHG: INC000001050969 Replace diversion.EndDate with DateTimeNowOnCurrentFacility                 
                 */
                mainViewModel.EndDate = DateTimeNowOnCurrentFacility;               
                mainViewModel.HourEndDateSelected = (DateTimeNowOnCurrentFacility.Hour < 10) ? "0" + DateTimeNowOnCurrentFacility.Hour.ToString() : DateTimeNowOnCurrentFacility.Hour.ToString();
                mainViewModel.MinuteEndDateSelected = (DateTimeNowOnCurrentFacility.Minute < 10) ? "0" + DateTimeNowOnCurrentFacility.ToString() : DateTimeNowOnCurrentFacility.Minute.ToString();

                mainViewModel.Comments = diversion.Comments;

                if (mainViewModel.Location == "Facility")
                {
                    mainViewModel.WardGroup = String.IsNullOrEmpty(diversion.WardGroup) ? "All" : diversion.WardGroup;
                    mainViewModel.TreatingSpecialty = String.IsNullOrEmpty(diversion.TreatingSpecialty) ? "All" : diversion.TreatingSpecialty;
                }
                else
                {
                    mainViewModel.WardGroup = "All";
                    mainViewModel.TreatingSpecialty = "All";
                }

                IEnumerable<string> wardGroups;
                Dictionary<string, List<string>> wardGroupSpecialties;
                LoadWardGroupsAndSpecialties(mainViewModel.Facility as Facility, out wardGroups, out wardGroupSpecialties);

                mainViewModel.WardGroups = wardGroups;

                if (mainViewModel.WardGroup != null && wardGroupSpecialties.ContainsKey(mainViewModel.WardGroup))
                    mainViewModel.TreatingSpecialties = wardGroupSpecialties[mainViewModel.WardGroup];

                mainViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                mainViewModel.Status = mainViewModel.FlagsList.Find(cd => cd.code == diversion.Status);

                mainViewModel.ExistingFacilityRoot = diversion.Facility.Id.root;
                mainViewModel.ExistingFacilityExtension = diversion.Facility.Id.extension;
                mainViewModel.ExistingFacilityName = diversion.Facility.Name;

                return View(mainViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult Edit(FacilityDiversionViewModel mainViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrWhiteSpace(mainViewModel.StartDateMessageError))
                    AddEditModelError("StartDate", mainViewModel.StartDateMessageError, mainViewModel);
                if (!string.IsNullOrWhiteSpace(mainViewModel.EndDateMessageError))
                    AddEditModelError("EndDate", mainViewModel.EndDateMessageError, mainViewModel);
                if (!ModelState.IsValid)
                    return View("Edit", mainViewModel);
                DateTime startDate = GetDate(mainViewModel.StartDate.Value, mainViewModel.HourStartDateSelected, mainViewModel.MinuteStartDateSelected);
                DateTime endDate = GetDate(mainViewModel.EndDate.Value, mainViewModel.HourEndDateSelected, mainViewModel.MinuteEndDateSelected);
                //check to see if end date is lower than start date
                if (endDate < startDate)
                {
                    AddEditModelError("EndDate", Strings.EndDateTimeMustBeGreaterThanStartDate, mainViewModel);
                    return View("Edit", mainViewModel);
                }

                mainViewModel.IsFromVisn = QueryStrings.ContainsKey("isFromVisn") && QueryStrings["isFromVisn"].ToString() == "True";
                mainViewModel.Facility = GetFacility();

                Diversion diversion = new Diversion();

                diversion.Facility = new Place { Id = new II { root = mainViewModel.ExistingFacilityRoot, extension = mainViewModel.ExistingFacilityExtension }, Name = mainViewModel.ExistingFacilityName };
                mainViewModel.LocationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Location);
                if (mainViewModel.LocationList != null && mainViewModel.LocationList.Count > 0)
                    diversion.Location = mainViewModel.LocationList
                                                        .First(loc => loc.displayName == mainViewModel.DiversionLocation.displayName);
                diversion.StartDate = startDate;
                diversion.EndDate = endDate;
                diversion.Comments = mainViewModel.Comments;
                diversion.WardGroup = mainViewModel.WardGroup;
                diversion.TreatingSpecialty = mainViewModel.TreatingSpecialty;
                diversion.Status = mainViewModel.Status.code;
                diversion.LastEditBy = this.loggedUser.UserName;
                diversion.Id = new II { extension = mainViewModel.Id, root = this.loggedUser.Domain };

                //save in DB
                diversion.Comments = mainViewModel.Comments;
                FacadeManager.DiversionInterface.SaveDiversion(diversion, false, mainViewModel.Facility.Id);

                //display Confirmation view    
                mainViewModel.LoggedUser = loggedUser;
                mainViewModel.IsAddConfirmation = false;
                mainViewModel.Location = mainViewModel.DiversionLocation.displayName;
                return View("Confirmation", mainViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        #region Private Methods

        private void FillViewModel(FacilityDiversionViewModel mainViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                mainViewModel.Facility = GetFacility();

                mainViewModel.LoggedUser = this.loggedUser;
                mainViewModel.LocationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Location).OrderByDescending(loc => loc.displayName).ToList();
                mainViewModel.DiversionList = FacadeManager.DiversionInterface.GetDiversions(mainViewModel.Facility.Id).Select(d => DiversionViewModel.GetViewModel(d, DateTimeNowOnCurrentFacility)).ToList();
                if (mainViewModel.CurrentButton.Equals(Strings.AllDiversions))
                    mainViewModel.DiversionList = mainViewModel.DiversionList.OrderByDescending(fdvm => fdvm.Status).ThenByDescending(fdvm => fdvm.StartDateTime).ToList();
                else
                    mainViewModel.DiversionList = mainViewModel.DiversionList.Where(fdvm => fdvm.Status == Constants.Yes && fdvm.StartDateTime != null).OrderBy(fdvm => fdvm.StartDateTime).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Checks if a diversion is unique.
        /// </summary>
        /// <param name="mainViewModel">Facility Diversion View Model.</param>
        /// <returns>True - diversion unique/ False - not unique</returns>
        private bool CheckUnique(FacilityDiversionViewModel mainViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = GetFacility();
                mainViewModel.LoggedUser = this.loggedUser;
                mainViewModel.Comments = mainViewModel.Comments ?? string.Empty;
                Diversion diversion = FacadeManager.DiversionInterface.GetDiversions(mainViewModel.Facility.Id)
                                                                      .Where(d => d.Facility.Id.extension.Equals(facility.Id.extension, StringComparison.InvariantCultureIgnoreCase) && d.Location.displayName == mainViewModel.Location && d.Comments.ToLower() == mainViewModel.Comments.ToLower())
                                                                      .FirstOrDefault();

                if (diversion != null)
                {
                    mainViewModel.FacilityAddDisplay = Strings.StyleDisplayNone;
                    mainViewModel.FacilityAlreadyExists = string.Empty;
                    return true;
                }
                else
                {
                    mainViewModel.FacilityAddDisplay = string.Empty;
                    mainViewModel.FacilityAlreadyExists = Strings.StyleDisplayNone;
                    return false;
                }

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the date.
        /// </summary>
        /// <param name="dateTime">The date time.</param>
        /// <param name="hour">The hour.</param>
        /// <param name="minutes">The minutes.</param>
        /// <returns></returns>
        private DateTime GetDate(DateTime dateTime, string hour, string minutes)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new DateTime(dateTime.Year, dateTime.Month,
                                                   dateTime.Day,
                                                   Convert.ToInt32(hour),
                                                   Convert.ToInt32(minutes), 0);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void AddEditModelError(string propertyName, string errorMessage, FacilityDiversionViewModel mainViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ModelState.AddModelError(propertyName, errorMessage);
                mainViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                mainViewModel.LocationList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Location);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void LoadWardGroupsAndSpecialties(Facility facility, out IEnumerable<string> wardGroupList, out Dictionary<string, List<string>> wardGroupSpecialties)
        {
            Dictionary<string, string> wardGroups = new Dictionary<string, string>();
            wardGroups.Add("All", "All");

            wardGroupSpecialties = new Dictionary<string, List<string>>();
            wardGroupSpecialties.Add("All", new List<string> { "All" });

            IList<Ward> wards = FacadeManager.EntityInterface.GetWardsByFacility(facility.Id);
            foreach (Ward ward in wards)
            {
                if (ward.IsVirtual || ward.Division == null)
                    continue;

                string wardGroup = ward.Division.Name.Trim();
                string treatingSpecialty = ward.Specialty.displayName.Trim();

                if (!wardGroups.ContainsKey(wardGroup))
                {
                    wardGroups.Add(wardGroup, wardGroup);
                    wardGroupSpecialties.Add(wardGroup, new List<string> { "All" });
                }

                if (ward.Specialty != null && !wardGroupSpecialties[wardGroup].Contains(treatingSpecialty))
                {
                    wardGroupSpecialties[wardGroup].Add(treatingSpecialty);

                    if (!wardGroupSpecialties["All"].Contains(treatingSpecialty))
                        wardGroupSpecialties["All"].Add(treatingSpecialty);
                }
            }

            wardGroupList = wardGroups.Values;
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute(OperationOverride="FacilityDiversion, Index")]
        public JsonResult GetTreatingSpecialties(string facilityId, string wardGroupName)
        {
            QueryStrings.Add("facilityId", facilityId);
            IEnumerable<string> wardGroups;
            Dictionary<string, List<string>> wardGroupSpecialties;

            Facility facility = GetFacility();
            LoadWardGroupsAndSpecialties(facility, out wardGroups, out wardGroupSpecialties);

            if (!wardGroupSpecialties.ContainsKey(wardGroupName))
                return null;

            return Json(wardGroupSpecialties[wardGroupName]);
        }

        private Facility GetFacility()
        {
            string facilityExtension = QueryStrings["facilityId"];
            II id = new II(this.loggedUser.Domain, facilityExtension);
            Facility facility = FacadeManager.EntityInterface.GetFacility(id);

            return facility;
        }

        #endregion
    }
}