﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Web.App_GlobalResource;
using BMS.Facade.Data;
using BMS.Facade;
using InfoWorld.HL7.ITS;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Discharge Clinic Controller
    /// </summary>
    public class DischargeClinicController : BaseController
    {
        /// <summary>
        /// Default action for the discharge clinic controller.
        /// </summary>
        /// <returns>The Index view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DischargeClinicViewModel model = new DischargeClinicViewModel();
                FillViewModel(model, false);
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Delete facility
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DischargeClinicViewModel model = new DischargeClinicViewModel();
                HospitalLocation hospitalLocation = FacadeManager.EntityInterface.GetHospitalLocation(new II(this.loggedUser.Domain, QueryStrings["identifier"]), this.loggedUser.VistaSite.Id);
                model.Ien = hospitalLocation.Ien;
                model.Name = hospitalLocation.Name;
                model.FirstMessage = Strings.YouHaveSuccessfullyDeletedTheFollowingInformation;
                model.SecondMessage = Strings.FromTheDischageClinicConfiguration;
                FacadeManager.EntityInterface.DeleteDischargeClinic(new II(this.loggedUser.Domain, QueryStrings["id"]));

                return View("MessageConfirmation", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Discharge Clinic Controller - used to save clinic.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index([Bind(Exclude = "RememberMe")] DischargeClinicViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(model.ButtonLoadHL))
                {
                    FillViewModel(model, true);
                    ModelState.Clear();
                    return View(model);
                }
                else
                {
                    HospitalLocation hospitalLocation = FacadeManager.EntityInterface.GetHospitalLocation(new II(this.loggedUser.Domain, model.DischargeClinicExtension), this.loggedUser.VistaSite.Id);
                    model.DischargeClinics = FacadeManager.EntityInterface.GetDischargeClinics(this.loggedUser.Facility, null);
                    if (model.DischargeClinics.Where(a => a.HospitalLocation.Id.extension.Equals(model.DischargeClinicExtension, StringComparison.InvariantCultureIgnoreCase) && a.Facility.Id.extension.Equals(this.loggedUser.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                    {
                        model.Ien = hospitalLocation.Ien;
                        model.Name = hospitalLocation.Name;
                        model.FirstMessage = Strings.ThisDischargeClinicAlreadyExistsInTheDatabase;
                        model.SecondMessage = Strings.ASecondEntryIsNotAllowed;
                    }
                    else
                    {
                        DischargeClinic clinic = new DischargeClinic()
                        {
                            Id = new II(this.loggedUser.Domain, null),
                            HospitalLocation = hospitalLocation,
                            Facility = this.loggedUser.Facility,
                            VistaSite = this.loggedUser.Facility.VistaSite
                        };
                        model.Ien = hospitalLocation.Ien;
                        model.Name = hospitalLocation.Name;
                        model.FirstMessage = Strings.YouHaveSuccessfullyAddedTheFollowingData;
                        model.SecondMessage = Strings.ToTheDischargeClinicConfiguration;
                        try
                        {
                            FacadeManager.EntityInterface.SaveDischargeClinic(clinic);
                        }
                        catch (Exception exception)
                        {
                            model.FirstMessage = exception.Message;
                            model.SecondMessage = exception.InnerException.Message;
                        }
                    }
                    return View("MessageConfirmation", model);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }                

        private void FillViewModel(DischargeClinicViewModel model, bool loadHL)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.LoggedUser = loggedUser;
                model.HospitalLocations = new List<HospitalLocation>();                
                model.DischargeClinics = FacadeManager.EntityInterface.GetDischargeClinics(this.loggedUser.Facility, null).OrderBy(dc => dc.HospitalLocation.Name).ToList();
                if (loadHL)
                {
                    List<HospitalLocation> hlList = FacadeManager.EntityInterface.GetHospitalLocations(this.loggedUser.VistaSite.Id).OrderBy(hl => hl.Name).ThenBy(hl => hl.Ien).ToList<HospitalLocation>();
                    foreach (HospitalLocation hl in hlList)
                    {
                        if (!hl.Name.StartsWith("ZZ"))
                            model.HospitalLocations.Add(new HospitalLocation() { Id = hl.Id, Name = hl.Name + "{" + Strings.IEN + ":" + hl.Ien + "}", Ien = hl.Ien });
                    }
                }
                model.HospitalLocations.Insert(0, new HospitalLocation() { Id = new II(this.loggedUser.Domain, string.Empty), Name = Strings.SelectaDischargeClinicLocation });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
