﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.Utils;
using BMS.VistaIntegration.Data.WF;

namespace BMS.VistaIntegration.VistA
{
    /// <summary>
    /// Operations for querying VistA data.
    /// </summary>
    public interface IVistAQuery
    {
        /// <summary>
        /// Gets or sets the maximum number of entitites that are brought from VistA ​​in a method call.
        /// </summary>
        /// <value>
        /// The bulk count.
        /// </value>
        int BulkCount { get; set; }

        #region Query Methods for EIS data

        /// <summary>
        /// Gets the hospital locations.
        /// </summary>
        /// <returns></returns>
        IList<HospitalLocation> GetHospitalLocations();

        /// <summary>
        /// Gets the patients with field date_entered_into_file within a period of time.
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <returns></returns>
        IList<Patient> GetPatients(DateTime? startDate = null, DateTime? endDate = null, string vistaPatientIen = null);

        /// <summary>
        /// Gets the patients.
        /// </summary>
        /// <param name="ssns">The SSNS.</param>
        /// <returns></returns>
        IList<Patient> GetPatients(IEnumerable<string> ssns);

        /// <summary>
        /// Gets the room beds.
        /// </summary>
        /// <returns></returns>
        IList<RoomBed> GetRoomBeds();

        /// <summary>
        /// Gets the ward locations.
        /// </summary>
        /// <returns></returns>
        IList<WardLocation> GetWardLocations();

        /// <summary>
        /// Gets the new persons with field date_entered within a period of time..
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <returns></returns>
        IList<NewPerson> GetNewPersons(DateTime? startDate = null, DateTime? endDate = null);

        /// <summary>
        /// Gets the admitted patients for update.
        /// </summary>
        /// <returns></returns>
        IList<Patient> GetAdmittedPatientsForUpdate();

        #endregion

        #region Query Methods for EVS data


        /// <summary>
        /// Gets the facility treating specialties.
        /// </summary>
        /// <returns></returns>
        IList<FacilityTreatingSpecialty> GetFacilityTreatingSpecialties();

        /// <summary>
        /// Gets the facility movement types.
        /// </summary>
        /// <returns></returns>
        IList<FacilityMovementType> GetFacilityMovementTypes();

        /// <summary>
        /// Gets the orderable items.
        /// </summary>
        /// <returns></returns>
        IList<OrderableItem> GetOrderableItems();

        /// <summary>
        /// Gets the specialties.
        /// </summary>
        /// <returns></returns>
        IList<Specialty> GetSpecialties();

        /// <summary>
        /// Gets the order statuses.
        /// </summary>
        /// <returns></returns>
        IList<OrderStatus> GetOrderStatuses();

        /// <summary>
        /// Gets the medical center divisions.
        /// </summary>
        /// <returns></returns>
        IList<MedicalCenterDivision> GetMedicalCenterDivisions();

        #endregion

        #region Query Methods for Acts data


        /// <summary>
        /// Gets the order actions with field date_time_ordered within a period of time.
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="vistASite">The vistA site.</param>
        /// <returns></returns>
        IList<OrderAction> GetOrderActions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> orderableItemsIen = null);

        /// <summary>
        /// Gets the patient appointments with field appointment_date_time within a period of time.
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="vistASite">The vistA site.</param>
        /// <returns></returns>
        IList<PatientAppointment> GetPatientAppointments(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> clinicIens = null);

        /// <summary>
        /// Gets the patient movements with field date_time within a period of time..
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="vistASite">The vistA site.</param>
        /// <returns></returns>
        IList<PatientMovement> GetPatientMovements(DateTime? startDate = null, DateTime? endDate = null);

        /// <summary>
        /// Gets the patient movements for admission.
        /// </summary>
        /// <param name="admissionIen">The admission ien.</param>
        /// <returns></returns>
        IList<PatientMovement> GetPatientMovementsForAdmission(string admissionIen);

        /// <summary>
        /// Gets the admitted patients.
        /// </summary>
        /// <param name="lastPatientMovementIen">The last patient movement ien.</param>
        /// <returns></returns>
        IList<AdmittedPatient> GetAdmittedPatients(string lastPatientMovementIen);

        /// <summary>
        /// Gets the scheduled admissions with field reservation_date_time within a period of time..
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="vistASite">The vistA site.</param>
        /// <returns></returns>
        IList<ScheduledAdmission> GetScheduledAdmissions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null);

        /// <summary>
        /// Gets the patient movement.
        /// </summary>
        /// <param name="patientIen">The patient ien.</param>
        /// <param name="dateTime">The date time.</param>
        /// <param name="movementType">Type of the movement.</param>
        /// <returns></returns>
        PatientMovement GetPatientMovement(string patientIen, DateTime dateTime, MovementTransactionType movementType);

        /// <summary>
        /// Gets the patient movement by ien.
        /// </summary>
        /// <param name="patientMovementIen">The patient movement ien.</param>
        /// <returns></returns>
        PatientMovement GetPatientMovementByIen(string patientMovementIen);

        /// <summary>
        /// Gets the patient movement iens.
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <returns></returns>
        IList<PatientMovementIen> GetPatientMovementIens(DateTime startDate, DateTime endDate);

        /// <summary>
        /// Gets the patient by ien.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        Patient GetPatientByIen(string ien);

        /// <summary>
        /// Gets the beds switch.
        /// </summary>
        /// <param name="iens">The iens.</param>
        /// <returns></returns>
        IList<BedSwitch> GetBedsSwitch(IEnumerable<string> iens);

        /// <summary>
        /// Gets the canceled orders.
        /// </summary>
        /// <param name="iens">The iens.</param>
        /// <returns></returns>
        IList<string> GetCanceledOrders(IEnumerable<string> iens);

        #endregion
    }
}
