﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.ServiceModel;
using System.Reflection;

namespace BMS.Utils
{
    /// <summary>
    /// Utility class used to trace exceptions in custom format.
    /// </summary>
    public static class Tracer
    {
        static object _treadSync = new object();

        /// <summary>
        /// Writes the exception and all the inner exceptions using a custom format.
        /// </summary>
        /// <param name="exception">Main exception</param>
        public static void TraceException(Exception exception)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                lock (_treadSync)
                {
                    AppendStartLine();
                    Trace.WriteLine(exception.Message);
                    Type exceptionType = exception.GetType();
                    bool parseRecursive = true;

                    //try to trace FaultException<T>
                    if (exceptionType.IsGenericType && exception is FaultException)
                    {
                        PropertyInfo detail = exceptionType.GetProperties().FirstOrDefault(pi => pi.Name == "Detail");
                        if (detail != null)
                        {
                            object detailException = detail.GetValue(exception, null);
                            if (detailException != null)
                            {
                                parseRecursive = false;
                                Trace.WriteLine(detailException.GetType().ToString()
                                    + ": " + detailException.ToString());
                            }
                        }
                    }
                    while (parseRecursive == true && exception.InnerException != null)
                    {
                        exception = exception.InnerException;
                        Trace.WriteLine(exception.Message);
                    }
                    Trace.WriteLine(exception.StackTrace);
                    AppendEndLine();
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Traces a service start event.
        /// </summary>
        /// <param name="service">Service host.</param>
        public static void TraceServiceStart(ServiceHostBase service)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraceServiceEvent(service, "started");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Traces a service stop event.
        /// </summary>
        /// <param name="service">Service host.</param>
        public static void TraceServiceStop(ServiceHostBase service)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraceServiceEvent(service, "stopped");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Traces a custom string message.
        /// </summary>
        /// <param name="customMessage">Custom message.</param>
        public static void TraceMessage(string customMessage)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                lock (_treadSync)
                {
                    AppendStartLine();
                    Trace.WriteLine(customMessage);
                    AppendEndLine();
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        private static void TraceServiceEvent(ServiceHostBase service, string serviceEvent)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                lock (_treadSync)
                {
                    if (service == null || service.BaseAddresses == null || service.BaseAddresses.Count == 0)
                        return;
                    AppendStartLine();
                    Trace.WriteLine("Service " + serviceEvent + ": " + service.BaseAddresses[0].AbsoluteUri);
                    AppendEndLine();
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        public const string START_LINE = "**************************** {0:dd-MM-yyyy HH:mm:ss} ***************************";
        public const string END_LINE = "----------------------------------------------------------------------------";

        static void AppendStartLine()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Trace.WriteLine(String.Format(START_LINE, DateTime.Now));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        static void AppendEndLine()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Trace.WriteLine(END_LINE);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
