﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using InfoWorld.HL7.ITS;
using InfoWorld.EVS.CTSMAPI;

namespace BMS.ServicesWrapper.EVS
{
    /// <summary>
    /// EVS Wrapper Interface. Exposes EVS accessing methods.
    /// </summary>
    public interface IEVSWrapper
    {
        /// <summary>
        /// Gets all the concepts from a vocabulary.
        /// </summary>
        /// <param name="parameters">Parameter class used to pass the vocabulary query.</param>
        /// <returns></returns>
        IList<CD> GetCodes(CodeFilterParameters parameters);

        /// <summary>
        /// Gets all the concepts from a vocabulary including properties set in the parameter.
        /// </summary>
        /// <param name="parameters">Parameter class used to pass the vocabulary query. Includes the list of properties to return.</param>
        /// <returns></returns>
        IList<CDWithProperties> GetCodesWithProperties(CodeFilterParametersWithProperty parameters);

        /// <summary>
        /// Gets the codes DB.
        /// </summary>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        List<CD> GetCodesDB(CodeFilterParameters parameters);

        /// <summary>
        /// Gets the codes with properties DB.
        /// </summary>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        List<CDWithProperties> GetCodesWithPropertiesDB(CodeFilterParametersWithProperty parameters);

        /// <summary>
        /// Searches the concepts with paging.
        /// </summary>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        List<CD> SearchConceptsWithPaging(CodeFilterParameters parameters);

        /// <summary>
        /// Searches the concepts DB.
        /// </summary>
        /// <param name="parameters">The parameters.</param>
        /// <param name="limit">The limit.</param>
        /// <returns></returns>
        List<CD> SearchConceptsDB(CodeFilterParameters parameters, CTSConstant limit = null);

        /// <summary>
        /// Searches the concepts with properties DB.
        /// </summary>
        /// <param name="parameters">The parameters.</param>
        /// <param name="limit">The limit.</param>
        /// <returns></returns>
        List<CDWithProperties> SearchConceptsWithPropertiesDB(CodeFilterParametersWithProperty parameters, CTSConstant limit = null);

        /// <summary>
        /// Searches the concepts DB with paging.
        /// </summary>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        List<CD> SearchConceptsWithPagingDB(CodeFilterParameters parameters);

        /// <summary>
        /// Allows the insertion of a concept in EVS and updates the underlying ValueSet.
        /// </summary>
        /// <param name="concept">The CD or CDWithProperties concept to insert in EVS. It's inserted in the concept.codeSystem CodeSystem.</param>
        void InsertConcept(CD concept);

        /// <summary>
        /// Updates an existing concept in EVS>
        /// </summary>
        /// <param name="concept">The CD or CDWithProperties concept to updaye in EVS. It's updated based on the code and codeSystem properties.</param>
        void UpdateConcept(CD concept);

        /// <summary>
        /// Deletes an existing concept from EVS.
        /// </summary>
        /// <param name="concept">The concept to delete. Searches for the concept relies on the code and codeSystem properties.</param>
        void DeleteConcept(CD concept);

        /// <summary>
        /// Clears the internal cache.
        /// </summary>
        void FlushCache();

        /// <summary>
        /// Fills the cache.
        /// </summary>
        void FillCache();

        /// <summary>
        /// Caches the specified vocabulary domain.
        /// </summary>
        /// <param name="vocabularyDomain">The vocabulary domain.</param>
        /// <param name="code">The code.</param>
        void CacheCD(string vocabularyDomain, string code);

        /// <summary>
        /// Removes the cache CD.
        /// </summary>
        /// <param name="vocabularyDomain">The vocabulary domain.</param>
        /// <param name="code">The code.</param>
        void RemoveCacheCD(string vocabularyDomain, string code);

        /// <summary>
        /// Concepts the code exists.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <param name="codeSystem">The code system.</param>
        /// <returns></returns>
        bool ConceptCodeExists(string code, string codeSystem);
    }
}

