﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;
using System.ServiceModel;
using BMS.Utils;
using System.Collections;
using System.Threading;
using BMS.Facade.Data;
using System.Configuration;


namespace BMS.ServicesWrapper.EVS
{
    class EVSWrapper : IEVSWrapper
    {
        private static readonly ReaderWriterLockSlim evsLock = new ReaderWriterLockSlim();
        EVSCache _cache;
        bool _isWindows = false;
        Semaphore evsSemaphore;
        Semaphore evsCacheSemaphore;
        
        private Proxy.EVS.MAPI.BrowserOperationsClient CtsMessageBrowseClient
        {
            get
            {
                if (_isWindows)
                    return EVSFactory.CTSMessageBrowseWindows;
                return EVSFactory.CTSMessageBrowse;
            }
        }
        private Proxy.EVS.VAPI.RuntimeOperationsClient CtsVocabularyRuntimeClient
        {
            get
            {
                if (_isWindows)
                    return EVSFactory.CTSVocabularyRuntimeWindows;
                return EVSFactory.CTSVocabularyRuntime;
            }
        }
        private Proxy.EVS.MAPI.RuntimeOperationsClient CtsMessageRuntimeClient
        {
            get
            {
                if (_isWindows)
                    return EVSFactory.CTSMessageRuntimeWindows;
                return EVSFactory.CTSMessageRuntime;
            }
        }
        private Proxy.EVS.Edit.CTSEditVocabularyClient CtsEditVocabularyClient
        {
            get
            {
                if (_isWindows)
                    return EVSFactory.CTSEditVocabularyWindows;
                return EVSFactory.CTSEditVocabulary;
            }
        }
        private Proxy.EVS.Edit.MessageEditClient MessageEditClient
        {
            get
            {
                if (_isWindows)
                    return EVSFactory.MessageEditWindows;
                return EVSFactory.MessageEdit;
            }
        }
        private Proxy.EVS.VAPI.BrowserOperationsClient CtsVocabularyBrowseClient
        {
            get
            {
                if (_isWindows)
                    return EVSFactory.CTSVocabularyBrowseWindows;
                return EVSFactory.CTSVocabularyBrowse;
            }
        }

        public EVSWrapper(bool isWindows)
        {
            this._isWindows = isWindows;
            _cache = new EVSCache(this);
            int _evsLongRunningMethodsMaxConcurrentCalls = 0;
            try
            {
                _evsLongRunningMethodsMaxConcurrentCalls = int.Parse(ConfigurationManager.AppSettings["BMS_EVS_LongRunningMethodsMaxConcurrentCalls"]);
            }
            catch { _evsLongRunningMethodsMaxConcurrentCalls = 30; }
            evsSemaphore = new Semaphore(_evsLongRunningMethodsMaxConcurrentCalls, _evsLongRunningMethodsMaxConcurrentCalls);
            
            int _evsCacheMethodsMaxConcurrentCalls = 0;
            try
            {
                _evsCacheMethodsMaxConcurrentCalls = int.Parse(ConfigurationManager.AppSettings["BMS_Cache_MaxConcurrentCalls"]);
            }
            catch { _evsCacheMethodsMaxConcurrentCalls = 100; }
            evsCacheSemaphore = new Semaphore(_evsCacheMethodsMaxConcurrentCalls, _evsCacheMethodsMaxConcurrentCalls);
        }

        #region IEVSWrapper Members

        public List<CD> SearchConceptsWithPaging(CodeFilterParameters parameters)
        {
            List<CD> result = null;
            if (parameters.VocabularyDomain.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase))
            {
                result = _cache.GetOrderableItemCache(parameters.MatchText);
                if (result != null && result.Count > 0)
                    return result;
                else                
                    return _cache.FillOrderableItemCache(parameters.MatchText);
            }
            else
            {
                result = _cache.GetCodesCache(parameters);
                if (result != null && result.Count > 0)
                    return result;
                return SearchConceptsWithPagingDB(parameters);
            }
        }

        public IList<CD> GetCodes(CodeFilterParameters parameters)
        {
            List<CD> result = _cache.GetCodesCache(parameters);
            if (result != null && result.Count > 0)
                return result;

            return GetCodesDB(parameters);
        }

        public IList<CDWithProperties> GetCodesWithProperties(CodeFilterParametersWithProperty parameters)
        {
            List<CDWithProperties> result = _cache.GetCodesWithPropertiesCache(parameters);
            if (result != null && result.Count > 0)
                return result;

            return GetCodesWithPropertiesDB(parameters);            
        }

        public List<CD> GetCodesDB(CodeFilterParameters parameters)
        {
            try
            {
                evsSemaphore.WaitOne();
                byte[] nextPageContext;
                List<CD> nextCodes;
                int _pageCount = int.Parse(ConfigurationManager.AppSettings["BMS_EVS_GET_CONCEPT_PAGE_SIZE"]);
                List<CD> codes = GetAllCodesPaged(parameters, _pageCount, out nextPageContext);
                while (nextPageContext != null && nextPageContext.Length > 0)
                {
                    nextCodes = GetAllCodesPaged(_pageCount, ref nextPageContext);
                    if (nextCodes != null && nextCodes.Count > 0)
                        codes.AddRange(nextCodes);
                }
                return codes;
            }
            finally 
            { 
                evsSemaphore.Release();
            }
        }

        public List<CDWithProperties> GetCodesWithPropertiesDB(CodeFilterParametersWithProperty parameters)
        {
            try
            {
                evsSemaphore.WaitOne();                
                Util.VocabularyCodes vocabularyCode = (Util.VocabularyCodes)Enum.Parse(typeof(Util.VocabularyCodes), parameters.VocabularyDomain);

                List<CDWithProperties> concepts = new List<CDWithProperties>();
                List<ST> properties = new List<ST>();
                if (parameters.Properties == null || parameters.Properties.Count < 0)
                    properties = new List<ST>();
                else
                    foreach (string prop in parameters.Properties)
                        properties.Add(new ST(prop));
                if (parameters.MatchText == null || parameters.MatchText == "")
                {
                    ListOfValueSetExpansionWithProperties vsExpansion;

                    vsExpansion = CtsMessageRuntimeClient.expandValueSetWithProperties(new ST(vocabularyCode.ToDescription()), new ST(parameters.Language), new BL(true), 0, parameters.MaxSelectedCodes == 1 ? int.MaxValue : parameters.MaxSelectedCodes, properties == null ? null : properties.ToArray());

                    foreach (ValueSetExpansionWithProperties vse in vsExpansion)
                    {
                        CDWithProperties concept = new CDWithProperties();
                        if (vse.NodeType_code.text != "A")
                        {
                            concept.code = vse.Concept_id.Concept_code.text;
                            concept.codeSystem = vse.Concept_id.CodeSystem_id.V;
                            concept.codeSystemName = _cache.GetSupportedCodeSystem(concept.codeSystem).CodeSystem_name.text;
                            concept.displayName = vse.DisplayName.text;
                            concept.Properties = vse.Properties;
                            concepts.Add(concept);
                        }
                    }
                }
                else
                {
                    CDWithProperties[] con = null;
                    con = CtsMessageBrowseClient.lookupConceptCodesByDesignationInValueSetWithProperties(vocabularyCode.ToDescription(), parameters.MatchText, parameters.MatchAlgorithm.ToString(), Utilities.GetBaseLanguage(parameters.Language), true, 0, parameters.MaxSelectedCodes, parameters.Properties.ToArray());
                    foreach (CDWithProperties c in con)
                        concepts.Add(c);
                }
                return concepts;
            }
            finally 
            { 
                evsSemaphore.Release();
            }
        }

        public List<CD> SearchConceptsWithPagingDB(CodeFilterParameters parameters)
        {
            try
            {
                evsSemaphore.WaitOne();
                ValueSetDescriptor valueSet;
                valueSet = CtsMessageBrowseClient.lookupValueSetForDomain(new ST(parameters.VocabularyDomain), null);
                List<CD> concepts = null;

                if (!String.IsNullOrEmpty(parameters.MatchText))
                {
                    InfoWorld.EVS.CTSVAPI.CodeSystemInfo cs = null;
                    FullValueSetDescription vs = null;

                    vs = CtsMessageBrowseClient.lookupValueSetInfo(valueSet.ValueSet_id, valueSet.ValueSet_name);

                    cs = CtsVocabularyRuntimeClient.lookupCodeSystemInfo(vs.Description.BasedOnCodeSystem.CodeSystem_id.V, null);
                    if (cs != null && cs.SupportedLanguages != null && cs.SupportedLanguages.Count > 0)
                        for (int i = 0; i < cs.SupportedLanguages.Count; i++)
                            cs.SupportedLanguages[i] = cs.SupportedLanguages[i].ToLower();

                    if (cs != null && cs.SupportedLanguages.Count > 0)
                    {
                        string languageCode = null;
                        if (cs.SupportedLanguages.Contains(Utilities.GetBaseLanguage(parameters.Language).ToLower()))
                            languageCode = Utilities.GetBaseLanguage(parameters.Language);
                        else
                            languageCode = cs.SupportedLanguages[0];

                        int lastPk = 0, minPk = 0;
                        int _pageCount = int.Parse(ConfigurationManager.AppSettings["BMS_EVS_GET_CONCEPT_PAGE_SIZE"]);
                        concepts = CtsMessageBrowseClient.lookupConceptCodesByCodeInValueSetWithPaging(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), languageCode, true, 0, _pageCount, minPk, out lastPk).ToList();
                        int count = concepts.Count;
                        while (count == _pageCount)
                        {
                            minPk = lastPk;
                            count = concepts.Count;
                            concepts.AddRange(CtsMessageBrowseClient.lookupConceptCodesByCodeInValueSetWithPaging(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), languageCode, true, 0, _pageCount, minPk, out lastPk).ToList());
                            count = concepts.Count - count;
                        }
                    }
                    else
                        throw new FaultException<InfoWorld.EVS.CTSMAPI.UnknownLanguage>(new UnknownLanguage(new ST(parameters.Language)));
                }
                return concepts;
            }
            finally 
            { 
                evsSemaphore.Release();
            }
        }

        public List<CD> SearchConceptsDB(CodeFilterParameters parameters, CTSConstant limit = null)
        {
            ValueSetDescriptor valueSet;
            if (limit == null)
                limit = new CTSConstant();

            valueSet = CtsMessageBrowseClient.lookupValueSetForDomain(new ST(parameters.VocabularyDomain), null);
            CD[] concepts = null;

            if (!String.IsNullOrEmpty(parameters.MatchText))
            {
                InfoWorld.EVS.CTSVAPI.CodeSystemInfo cs = null;
                FullValueSetDescription vs = null;

                vs = CtsMessageBrowseClient.lookupValueSetInfo(valueSet.ValueSet_id, valueSet.ValueSet_name);
                
                cs = CtsVocabularyRuntimeClient.lookupCodeSystemInfo(vs.Description.BasedOnCodeSystem.CodeSystem_id.V, null);
                if (cs != null && cs.SupportedLanguages != null && cs.SupportedLanguages.Count > 0)
                    for (int i = 0; i < cs.SupportedLanguages.Count; i++)
                        cs.SupportedLanguages[i] = cs.SupportedLanguages[i].ToLower();

                if (cs != null && cs.SupportedLanguages.Count > 0)
                    if (cs.SupportedLanguages.Contains(Utilities.GetBaseLanguage(parameters.Language).ToLower()))
                        if (parameters.MatchProperty == MatchProperty.Code)
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByCodeInValueSet(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), Utilities.GetBaseLanguage(parameters.Language), true, 0, limit.MaxExpansionSize);
                        else
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByDesignationInValueSet(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), Utilities.GetBaseLanguage(parameters.Language), true, 0, limit.MaxSearchSize);
                    else
                        if (parameters.MatchProperty == MatchProperty.Code)
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByCodeInValueSet(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), cs.SupportedLanguages[0], true, 0, limit.MaxSearchSize);
                        else
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByDesignationInValueSet(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), cs.SupportedLanguages[0], true, 0, limit.MaxSearchSize);
                else
                    throw new FaultException<InfoWorld.EVS.CTSMAPI.UnknownLanguage>(new UnknownLanguage(new ST(parameters.Language)));
            }
            return new List<CD>(concepts);
        }

        public List<CDWithProperties> SearchConceptsWithPropertiesDB(CodeFilterParametersWithProperty parameters, CTSConstant limit = null)
        {
            ValueSetDescriptor valueSet;
            if (limit == null)
                limit = new CTSConstant();

            CDWithProperties[] concepts = null;
            valueSet = CtsMessageBrowseClient.lookupValueSetForDomain(new ST(parameters.VocabularyDomain), null);

            if (!String.IsNullOrEmpty(parameters.MatchText))
            {
                InfoWorld.EVS.CTSVAPI.CodeSystemInfo cs = null;
                FullValueSetDescription vs = null;

                vs = CtsMessageBrowseClient.lookupValueSetInfo(valueSet.ValueSet_id, valueSet.ValueSet_name);
                cs = CtsVocabularyRuntimeClient.lookupCodeSystemInfo(vs.Description.BasedOnCodeSystem.CodeSystem_id.V, null);

                if (cs != null && cs.SupportedLanguages.Count > 0)
                    if (cs.SupportedLanguages.Contains(Utilities.GetBaseLanguage(parameters.Language).ToLower()))
                        if (parameters.MatchProperty == MatchProperty.Code)
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByCodeInValueSetWithProperties(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), Utilities.GetBaseLanguage(parameters.Language), true, 0, limit.MaxExpansionSize, parameters.Properties == null ? null : parameters.Properties.ToArray());
                        else
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByDesignationInValueSetWithProperties(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), Utilities.GetBaseLanguage(parameters.Language), true, 0, limit.MaxSearchSize, parameters.Properties == null ? null : parameters.Properties.ToArray());
                    else
                        if (parameters.MatchProperty == MatchProperty.Code)
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByCodeInValueSetWithProperties(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), cs.SupportedLanguages[0], true, 0, limit.MaxSearchSize, parameters.Properties == null ? null : parameters.Properties.ToArray());
                        else
                            concepts = CtsMessageBrowseClient.lookupConceptCodesByDesignationInValueSetWithProperties(valueSet.ValueSet_id.V, parameters.MatchText, parameters.MatchAlgorithm.ToString(), cs.SupportedLanguages[0], true, 0, limit.MaxSearchSize, parameters.Properties == null ? null : parameters.Properties.ToArray());
                else
                    throw new FaultException<InfoWorld.EVS.CTSMAPI.UnknownLanguage>(new UnknownLanguage(new ST(parameters.Language)));
            }
            return new List<CDWithProperties>(concepts);            
        }

        public void InitSupportedCodeSystems()
        {
            IList<CodeSystemDescriptor> supportedCodeSystems = GetSupportedCodeSystems();
            _cache.InitSupportedCodeSystems(supportedCodeSystems);
        }
        
        public void InsertConcept(CD concept)
        {
            InsertConceptMethod(concept);                                  
            _cache.CacheCD(concept.codeSystemName, concept.code);
            if (concept.codeSystemName.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase))
                return;
            try
            {
                evsCacheSemaphore.WaitOne();
                ThreadPool.QueueUserWorkItem((o) =>
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEVSCache(concept.code, concept.codeSystemName);
                    }
                    catch (CommunicationObjectAbortedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEVSCache(concept.code, concept.codeSystemName);
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EVS InsertConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEVSCache(concept.code, concept.codeSystemName);
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EVS InsertConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EVS InsertConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                        Tracer.TraceException(ex);
                    }
                });
            }
            finally { evsCacheSemaphore.Release(); }            
        }

        public void UpdateConcept(CD concept)
        {
            DeleteConceptMethod(concept);
            InsertConceptMethod(concept);              
            _cache.CacheCD(concept.codeSystemName, concept.code);
            if (concept.codeSystemName.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase))
                return;
            try
            {
                evsCacheSemaphore.WaitOne();
                ThreadPool.QueueUserWorkItem((o) =>
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEVSCache(concept.code, concept.codeSystemName);
                    }
                    catch (CommunicationObjectAbortedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEVSCache(concept.code, concept.codeSystemName);
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EVS UpdateConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEVSCache(concept.code, concept.codeSystemName);
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EVS UpdateConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EVS UpdateConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                        Tracer.TraceException(ex);
                    }
                });
            }
            finally { evsCacheSemaphore.Release(); }            
        }

        public void DeleteConcept(CD concept)
        {
            DeleteConceptMethod(concept);                                   
            _cache.RemoveCacheCD(concept.codeSystemName, concept.code);
            if (concept.codeSystemName.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase))
                return;
            ThreadPool.QueueUserWorkItem((o) =>
            {
                try
                {
                    BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEVSCache(concept.code, concept.codeSystemName);
                }
                catch (CommunicationObjectAbortedException)
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEVSCache(concept.code, concept.codeSystemName);
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EVS DeleteConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                        Tracer.TraceException(ex);
                    }
                }
                catch (ObjectDisposedException)
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEVSCache(concept.code, concept.codeSystemName);
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EVS DeleteConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                        Tracer.TraceException(ex);
                    }
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("EVS DeleteConcept: domain - " + concept.codeSystemName + " code - " + concept.code);
                    Tracer.TraceException(ex);
                }
            });            
        }

        public bool ConceptCodeExists(string code, string codeSystem)
        {
            return CtsVocabularyRuntimeClient.isConceptIdValid(new InfoWorld.EVS.CTSVAPI.ConceptId(codeSystem, code), false);
        }

        public void FlushCache()
        {
            _cache.FlushCache();
        }

        public void FillCache()
        {
            _cache.FillCache();
        }        

        public void CacheCD(string vocabularyDomain, string code)
        {
            _cache.CacheCD(vocabularyDomain, code);
        }

        public void RemoveCacheCD(string vocabularyDomain, string code)
        {
            _cache.RemoveCacheCD(vocabularyDomain, code);
        }

        #endregion

        /// <summary>
        /// Loads the supported codeSystems from EVS.
        /// </summary>
        /// <returns></returns>
        IList<CodeSystemDescriptor> GetSupportedCodeSystems()
        {
            return new List<CodeSystemDescriptor>(CtsMessageBrowseClient.getSupportedCodeSystems(null, null, 0, 0));
        }

        /// <summary>
        /// Insert a concept in EVS.
        /// </summary>
        /// <param name="concept">Concept to insert.</param>
        private void InsertConceptMethod(CD concept)
        {
            InfoWorld.EVS.CTSVAPI.CompleteCodedConceptDescription codeToInsert = new InfoWorld.EVS.CTSVAPI.CompleteCodedConceptDescription();
            codeToInsert.Concept_id = new InfoWorld.EVS.CTSVAPI.ConceptId();
            codeToInsert.Concept_id.Concept_code = concept.code;
            codeToInsert.Concept_id.CodeSystem_id = concept.codeSystem;

            codeToInsert.DesignatedBy = new InfoWorld.EVS.CTSVAPI.ListOfConceptDesignation();

            InfoWorld.EVS.CTSVAPI.ConceptDesignation conceptDesignation = new InfoWorld.EVS.CTSVAPI.ConceptDesignation();
            conceptDesignation.Designation = concept.displayName;
            conceptDesignation.PreferredForLanguage = true;
            conceptDesignation.Language_code = Constants.EVS_Language_code;

            codeToInsert.DesignatedBy.Add(conceptDesignation);
            if (concept is CDWithProperties)
            {
                CDWithProperties conceptWithProperties = (CDWithProperties)concept;
                codeToInsert.HasProperties = new InfoWorld.EVS.CTSVAPI.ListOfConceptProperty();

                foreach (InfoWorld.EVS.CTSMAPI.ConceptProperty prop in conceptWithProperties.Properties)
                {
                    InfoWorld.EVS.CTSVAPI.ConceptProperty newProp = new InfoWorld.EVS.CTSVAPI.ConceptProperty();
                    codeToInsert.HasProperties.Add(newProp);

                    newProp.Language_code = prop.Language == null ? null : prop.Language.text;
                    newProp.MimeType_code = prop.MimeType == null ? null : prop.MimeType.text;
                    newProp.Property_code = prop.PropertyName == null ? null : prop.PropertyName.text;
                    newProp.PropertyValue = prop.PropertyValue == null ? null : prop.PropertyValue.text;
                }
            }
            CtsEditVocabularyClient.InsertConceptWithExpand(new InfoWorld.EVS.CTSVAPI.CompleteCodedConceptDescription[] { codeToInsert }, new bool[] { true });
        }

        /// <summary>
        /// Deletes an existing concepts.
        /// </summary>
        /// <param name="concept">The concept to delete.</param>
        private void DeleteConceptMethod(CD concept)
        {
            InfoWorld.EVS.CTSVAPI.ConceptId codeToDelete = new InfoWorld.EVS.CTSVAPI.ConceptId();
            codeToDelete.Concept_code = concept.code;
            codeToDelete.CodeSystem_id = concept.codeSystem;

            CtsEditVocabularyClient.DeleteConceptWithExpand(new InfoWorld.EVS.CTSVAPI.ConceptId[] { codeToDelete });
        }        

        private List<CD> GetAllCodesPaged(CodeFilterParameters parameters, int pageCount, out byte[] nextPageContext)
        {
            if (parameters.MaxSelectedCodes <= 0)
            {
                nextPageContext = null;
                return new List<CD>(0);
            }                        
            try
            {
                ValueSetExpansionPage vsPage = CtsMessageRuntimeClient.lookupValueSetExansion_WP(new ST(parameters.VocabularyDomain), null, new ST(parameters.Language), 0, pageCount);
                ListOfValueSetExpansion vsExpansion = vsPage.ValueSetExpansion;
                nextPageContext = vsPage.PageContext;
                return ConvertVSEtoCDs(vsPage.ValueSetExpansion);                
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private List<CD> GetAllCodesPaged(int pageCount, ref byte[] nextPageContext)
        {
            if (nextPageContext == null) return null;
            if (pageCount <= 0) return new List<CD>(0);                        
            try
            {
                ValueSetExpansionPage vsPage = CtsMessageRuntimeClient.expandValueSetExpansionContext_WP(nextPageContext, pageCount);
                ListOfValueSetExpansion vsExpansion = vsPage.ValueSetExpansion;
                nextPageContext = vsPage.PageContext;
                return ConvertVSEtoCDs(vsPage.ValueSetExpansion);                
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private List<CD> ConvertVSEtoCDs(ListOfValueSetExpansion vsExpansion)
        {
            List<CD> concepts = new List<CD>();
            CD concept = null;
            foreach (ValueSetExpansion vse in vsExpansion)
            {
                concept = new CD();
                if (vse.NodeType_code.text != "A")
                {
                    concept.code = vse.Concept_id.Concept_code.text;
                    concept.codeSystem = vse.Concept_id.CodeSystem_id.V;
                    concept.codeSystemName = _cache.GetSupportedCodeSystem(concept.codeSystem).CodeSystem_name.text;
                    concept.displayName = vse.DisplayName.text;
                    concepts.Add(concept);
                }
            }
            return concepts;
        }
    }
}
