﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using InfoWorld.HL7.ITS;
using InfoWorld.EVS.CTSMAPI;
using System.Collections;
using BMS.Utils;
using System.Configuration;
using System.Threading;
using BMS.Facade.Data;

namespace BMS.ServicesWrapper.EVS
{
    /// <summary>
    /// Internal EVS caching mechanism.
    /// </summary>
    internal class EVSCache
    {
        IEVSWrapper _EVSWrapper;
        internal EVSCache(IEVSWrapper EVSWrapper)
        {
            _EVSWrapper = EVSWrapper;
        }

        /// <summary>
        /// The static constructor is used to read the dictionary size from application configuration file
        /// or use a default 1024 value.
        /// </summary>
        static EVSCache()
        {
        }

        private static readonly ReaderWriterLockSlim cacheLock = new ReaderWriterLockSlim();

        private static readonly ReaderWriterLockSlim orderableItemCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim specialtyCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim treatingSpecialtyCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim typeOfMovementCacheLock = new ReaderWriterLockSlim();
        private static readonly ReaderWriterLockSlim medicalCenterCacheLock = new ReaderWriterLockSlim();
        /// <summary>
        /// Used to cache the concepts from vocabularies.
        /// </summary>
        static Dictionary<CodeFilterParameters, IEnumerable> _cacheCD = new Dictionary<CodeFilterParameters, IEnumerable>(new CodeFilterParametersComparer());
        
        static Dictionary<string, List<CD>> _orderableItemCacheCD = new Dictionary<string, List<CD>>(StringComparer.InvariantCultureIgnoreCase);
        static List<CD> _specialtyCacheCD = new List<CD>();
        static List<CDWithProperties> _specialtyCacheCDWithProperties = new List<CDWithProperties>();
        static List<CD> _treatingSpecialtyCacheCD = new List<CD>();
        static List<CD> _typeOfMovementCacheCD = new List<CD>();
        static List<CD> _medicalCenterCacheCD = new List<CD>();

        /// <summary>
        /// Contains the list of all supported code systems required to match a codeSystemName with a codeSystem.
        /// </summary>
        static Dictionary<String, CodeSystemDescriptor> _supportedCodeSystems = new Dictionary<String, CodeSystemDescriptor>();

        BmsLogger bmsLog = new BmsLogger("EVS Cache: ");        

        #region FillCache

        internal void FlushCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                orderableItemCacheLock.EnterWriteLock();
                try
                {
                    _orderableItemCacheCD.Clear();
                }
                finally { orderableItemCacheLock.ExitWriteLock(); }
                specialtyCacheLock.EnterWriteLock();
                try
                {
                    _specialtyCacheCD.Clear();
                    _specialtyCacheCDWithProperties.Clear();
                }
                finally { specialtyCacheLock.ExitWriteLock(); }
                treatingSpecialtyCacheLock.EnterWriteLock();
                try
                {
                    _treatingSpecialtyCacheCD.Clear();
                }
                finally { treatingSpecialtyCacheLock.ExitWriteLock(); }
                typeOfMovementCacheLock.EnterWriteLock();
                try
                {
                    _typeOfMovementCacheCD.Clear();
                }
                finally { typeOfMovementCacheLock.ExitWriteLock(); }
                medicalCenterCacheLock.EnterWriteLock();
                try
                {
                    _medicalCenterCacheCD.Clear();
                }
                finally { medicalCenterCacheLock.ExitWriteLock(); }
                cacheLock.EnterWriteLock();
                try
                {
                    _cacheCD.Clear();
                }
                finally { cacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal void FillCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FlushCache();
                bmsLog.LogInformation("Started FillCache EVS");
                CodeFilterParameters param = null;
                CodeFilterParametersWithProperty paramWithProperties = null;
                cacheLock.EnterWriteLock();
                try
                {                                 
                    foreach (string name in Enum.GetNames(typeof(Util.Vocabulary)))
                    {
                        if (!name.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase) &&
                            !name.Equals(Util.Vocabulary.VistaSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase) &&
                            !name.Equals(Util.Vocabulary.VistaTreatingSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase) &&
                            !name.Equals(Util.Vocabulary.TypeOfMovement.ToString(), StringComparison.InvariantCultureIgnoreCase) &&
                            !name.Equals(Util.Vocabulary.VistaMedicalCenterDivision.ToString(), StringComparison.InvariantCultureIgnoreCase))
                        {
                            param = new CodeFilterParameters();
                            param.MaxSelectedCodes = int.MaxValue;
                            param.VocabularyDomain = name;
                            _cacheCD.Add(param, _EVSWrapper.GetCodesDB(param));
                            bmsLog.LogInformation("FillCache - Loaded " + name + " vocab in cache");
                        }
                    }
                    paramWithProperties = new CodeFilterParametersWithProperty();
                    paramWithProperties.MaxSelectedCodes = int.MaxValue;
                    paramWithProperties.VocabularyDomain = Util.Vocabulary.UnavailableReason.ToString();
                    paramWithProperties.Properties = new List<string>();
                    paramWithProperties.Properties.Add("Type");
                    _cacheCD.Add(paramWithProperties, _EVSWrapper.GetCodesWithPropertiesDB(paramWithProperties));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.UnavailableReason.ToString() + " vocab properties in cache");

                    paramWithProperties = new CodeFilterParametersWithProperty();
                    paramWithProperties.MaxSelectedCodes = int.MaxValue;
                    paramWithProperties.VocabularyDomain = Util.Vocabulary.ADTOrderableItem.ToString();
                    paramWithProperties.Properties = new List<string>();
                    paramWithProperties.Properties.Add("IEN");
                    paramWithProperties.Properties.Add("OrderableItemType");
                    _cacheCD.Add(paramWithProperties, _EVSWrapper.GetCodesWithPropertiesDB(paramWithProperties));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.ADTOrderableItem.ToString() + " vocab properties in cache");

                    paramWithProperties = new CodeFilterParametersWithProperty();
                    paramWithProperties.MaxSelectedCodes = int.MaxValue;
                    paramWithProperties.VocabularyDomain = Util.Vocabulary.TimeZone.ToString();
                    paramWithProperties.Properties = new List<string>();
                    paramWithProperties.Properties.Add("Value");
                    _cacheCD.Add(paramWithProperties, _EVSWrapper.GetCodesWithPropertiesDB(paramWithProperties));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.TimeZone.ToString() + " vocab properties in cache");                                        
                }
                finally { cacheLock.ExitWriteLock(); }

                FillSpecialtyCache();
                FillTreatingSpecialtyCache();
                FillTypeOfMovementCache();
                FillMedicalCenterDivisionCache();

                param = null;
                paramWithProperties = null;

                bmsLog.LogInformation("Finished FillCache");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillSpecialtyCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                specialtyCacheLock.EnterWriteLock();
                try
                {
                    CodeFilterParameters param = new CodeFilterParameters();
                    param.MaxSelectedCodes = int.MaxValue;
                    param.VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString();
                    _specialtyCacheCD.AddRange(_EVSWrapper.GetCodesDB(param));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.VistaSpecialty.ToString() + " vocab in cache");

                    CodeFilterParametersWithProperty paramWithProperties = new CodeFilterParametersWithProperty();
                    paramWithProperties.MaxSelectedCodes = int.MaxValue;
                    paramWithProperties.VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString();
                    paramWithProperties.Properties = new List<string>();
                    paramWithProperties.Properties.Add(Constants.NUMA);
                    paramWithProperties.Properties.Add(Constants.HAvBED);
                    paramWithProperties.Properties.Add(Constants.Hidden);
                    _specialtyCacheCDWithProperties.AddRange(_EVSWrapper.GetCodesWithPropertiesDB(paramWithProperties));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.VistaSpecialty.ToString() + " vocab properties in cache");

                    param = null;
                    paramWithProperties = null;
                }
                finally { specialtyCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillTreatingSpecialtyCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                treatingSpecialtyCacheLock.EnterWriteLock();
                try
                {
                    CodeFilterParameters param = new CodeFilterParameters();
                    param.MaxSelectedCodes = int.MaxValue;
                    param.VocabularyDomain = Util.Vocabulary.VistaTreatingSpecialty.ToString();
                    _treatingSpecialtyCacheCD.AddRange(_EVSWrapper.GetCodesDB(param));                    
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.VistaTreatingSpecialty.ToString() + " vocab in cache");
                    param = null;
                }
                finally { treatingSpecialtyCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillTypeOfMovementCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                typeOfMovementCacheLock.EnterWriteLock();
                try
                {
                    CodeFilterParameters param = new CodeFilterParameters();
                    param.MaxSelectedCodes = int.MaxValue;
                    param.VocabularyDomain = Util.Vocabulary.TypeOfMovement.ToString();
                    _typeOfMovementCacheCD.AddRange(_EVSWrapper.GetCodesDB(param));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.TypeOfMovement.ToString() + " vocab in cache");
                    param = null;
                }
                finally { typeOfMovementCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        private void FillMedicalCenterDivisionCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                medicalCenterCacheLock.EnterWriteLock();
                try
                {
                    CodeFilterParameters param = new CodeFilterParameters();
                    param.MaxSelectedCodes = int.MaxValue;
                    param.VocabularyDomain = Util.Vocabulary.VistaMedicalCenterDivision.ToString();
                    _medicalCenterCacheCD.AddRange(_EVSWrapper.GetCodesDB(param));
                    bmsLog.LogInformation("FillCache - Loaded " + Util.Vocabulary.VistaMedicalCenterDivision.ToString() + " vocab in cache");
                    param = null;
                }
                finally { medicalCenterCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal List<CD> FillOrderableItemCache(string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string _vistaCode = vistaCode;
                // vistaCode can be an Orderable Item code(vistaCode + "_" + ien) or vistaCode
                string[] items = vistaCode.Split('_');
                if (items != null && items.Length == 2 && !string.IsNullOrEmpty(items[1]))
                    _vistaCode = items[0];
                CodeFilterParameters param = new CodeFilterParameters();
                param.MaxSelectedCodes = int.MaxValue;
                param.VocabularyDomain = Util.Vocabulary.OrderableItem.ToString();
                param.MatchAlgorithm = MatchAlgorithm.StartsWithIgnoreCase;
                param.MatchProperty = MatchProperty.Code;
                param.MatchText = _vistaCode;
                List<CD> list = _EVSWrapper.SearchConceptsWithPagingDB(param);
                orderableItemCacheLock.EnterWriteLock();
                try
                {
                    if (!_orderableItemCacheCD.ContainsKey(_vistaCode))                    
                        _orderableItemCacheCD.Add(_vistaCode, list);                    
                }
                finally { orderableItemCacheLock.ExitWriteLock(); }                
                param = null;
                if (items != null && items.Length == 2 && !string.IsNullOrEmpty(items[1]))
                {
                    CD cd = list.Where(a => a.code.Equals(vistaCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (cd != null)
                        return new List<CD>() { cd };
                    else
                        return null;
                }
                return list;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region CacheCD

        internal void CacheCD(string vocabularyDomain, string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (vocabularyDomain.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    CacheCDOrderableItem(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.VistaSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    CacheCDSpecialty(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.VistaTreatingSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    CacheCDTreatingSpecialty(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.TypeOfMovement.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    CacheCDTypeOfMovement(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.VistaMedicalCenterDivision.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    CacheCDMedicalCenterDivision(code);
                else
                {
                    CD cd = null;
                    CDWithProperties cdp = null;
                    cacheLock.EnterWriteLock();
                    try
                    {
                        foreach (CodeFilterParameters cfp in _cacheCD.Keys)
                        {
                            if (cfp.VocabularyDomain.Equals(vocabularyDomain, StringComparison.InvariantCultureIgnoreCase))
                            {
                                if ((cfp as CodeFilterParametersWithProperty) == null)
                                {
                                    cd = _EVSWrapper.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = vocabularyDomain, MatchProperty = MatchProperty.Code }).FirstOrDefault();
                                    if (cd != null && !string.IsNullOrEmpty(cd.code))
                                    {
                                        (_cacheCD[cfp] as List<CD>).RemoveAll(a => a.code.Equals(cd.code, StringComparison.InvariantCultureIgnoreCase));
                                        (_cacheCD[cfp] as List<CD>).Add(cd);
                                    }
                                }
                                else
                                {
                                    CodeFilterParametersWithProperty x = null;
                                    x = cfp as CodeFilterParametersWithProperty;
                                    if (x != null)
                                    {
                                        cdp = _EVSWrapper.SearchConceptsWithPropertiesDB(new CodeFilterParametersWithProperty() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = vocabularyDomain, Properties = x.Properties, MatchProperty = MatchProperty.Code }).FirstOrDefault();
                                        if (cdp != null && !string.IsNullOrEmpty(cdp.code))
                                        {
                                            (_cacheCD[cfp] as List<CDWithProperties>).RemoveAll(a => a.code.Equals(cdp.code, StringComparison.InvariantCultureIgnoreCase));
                                            (_cacheCD[cfp] as List<CDWithProperties>).Add(cdp);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    finally { cacheLock.ExitWriteLock(); }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CacheCDSpecialty(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CD cd = _EVSWrapper.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString(), MatchProperty = MatchProperty.Code }).FirstOrDefault();
                CDWithProperties cdp = _EVSWrapper.SearchConceptsWithPropertiesDB(new CodeFilterParametersWithProperty() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString(), Properties = new List<string>() { Constants.NUMA, Constants.HAvBED, Constants.Hidden }, MatchProperty = MatchProperty.Code }).FirstOrDefault();
                if ((cd != null && !string.IsNullOrEmpty(cd.code)) || (cdp != null && !string.IsNullOrEmpty(cdp.code)))
                {
                    specialtyCacheLock.EnterWriteLock();
                    try
                    {
                        if (cd != null && !string.IsNullOrEmpty(cd.code))
                        {
                            _specialtyCacheCD.RemoveAll(a => a.code.Equals(cd.code, StringComparison.InvariantCultureIgnoreCase));
                            _specialtyCacheCD.Add(cd);
                        }
                        if (cdp != null && !string.IsNullOrEmpty(cdp.code))
                        {
                            _specialtyCacheCDWithProperties.RemoveAll(a => a.code.Equals(cdp.code, StringComparison.InvariantCultureIgnoreCase));
                            _specialtyCacheCDWithProperties.Add(cdp);
                        }
                    }
                    finally { specialtyCacheLock.ExitWriteLock(); }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CacheCDTreatingSpecialty(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CD cd = _EVSWrapper.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = Util.Vocabulary.VistaTreatingSpecialty.ToString(), MatchProperty = MatchProperty.Code }).FirstOrDefault();
                if (cd != null && !string.IsNullOrEmpty(cd.code))
                {
                    treatingSpecialtyCacheLock.EnterWriteLock();
                    try
                    {
                        _treatingSpecialtyCacheCD.RemoveAll(a => a.code.Equals(cd.code, StringComparison.InvariantCultureIgnoreCase));
                        _treatingSpecialtyCacheCD.Add(cd);
                    }
                    finally { treatingSpecialtyCacheLock.ExitWriteLock(); }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CacheCDTypeOfMovement(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CD cd = _EVSWrapper.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = Util.Vocabulary.TypeOfMovement.ToString(), MatchProperty = MatchProperty.Code }).FirstOrDefault();
                if (cd != null && !string.IsNullOrEmpty(cd.code))
                {
                    typeOfMovementCacheLock.EnterWriteLock();
                    try
                    {
                        _typeOfMovementCacheCD.RemoveAll(a => a.code.Equals(cd.code, StringComparison.InvariantCultureIgnoreCase));
                        _typeOfMovementCacheCD.Add(cd);
                    }
                    finally { typeOfMovementCacheLock.ExitWriteLock(); }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CacheCDMedicalCenterDivision(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CD cd = _EVSWrapper.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = Util.Vocabulary.VistaMedicalCenterDivision.ToString(), MatchProperty = MatchProperty.Code }).FirstOrDefault();
                if (cd != null && !string.IsNullOrEmpty(cd.code))
                {
                    medicalCenterCacheLock.EnterWriteLock();
                    try
                    {
                        _medicalCenterCacheCD.RemoveAll(a => a.code.Equals(cd.code, StringComparison.InvariantCultureIgnoreCase));
                        _medicalCenterCacheCD.Add(cd);
                    }
                    finally { medicalCenterCacheLock.ExitWriteLock(); }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CacheCDOrderableItem(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string vistaCode = code.Substring(0, code.IndexOf("_"));
                orderableItemCacheLock.EnterWriteLock();
                try
                {
                    if (!_orderableItemCacheCD.ContainsKey(vistaCode))
                    {
                        CodeFilterParameters param = new CodeFilterParameters() { VocabularyDomain = Util.Vocabulary.OrderableItem.ToString(), MaxSelectedCodes = int.MaxValue, MatchAlgorithm = MatchAlgorithm.StartsWithIgnoreCase, MatchProperty = MatchProperty.Code, MatchText = vistaCode };
                        _orderableItemCacheCD[vistaCode] = _EVSWrapper.SearchConceptsWithPagingDB(param);
                    }
                    else
                    {
                        CD cd = _EVSWrapper.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = code, VocabularyDomain = Util.Vocabulary.OrderableItem.ToString(), MatchProperty = MatchProperty.Code }).FirstOrDefault();
                        if (cd != null && !string.IsNullOrEmpty(cd.code))
                        {
                            _orderableItemCacheCD[vistaCode].RemoveAll(a => a.code.Equals(cd.code, StringComparison.InvariantCultureIgnoreCase));
                            _orderableItemCacheCD[vistaCode].Add(cd);
                        }
                    }                    
                }
                finally { orderableItemCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region RemoveCacheCD

        internal void RemoveCacheCD(string vocabularyDomain, string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (vocabularyDomain.Equals(Util.Vocabulary.OrderableItem.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    RemoveCacheCDOrderableItem(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.VistaSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    RemoveCacheCDSpecialty(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.VistaTreatingSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    RemoveCacheCDTreatingSpecialty(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.TypeOfMovement.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    RemoveCacheCDTypeOfMovement(code);
                else if (vocabularyDomain.Equals(Util.Vocabulary.VistaMedicalCenterDivision.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    RemoveCacheCDMedicalCenterDivision(code);
                else
                {
                    cacheLock.EnterWriteLock();
                    try
                    {
                        foreach (CodeFilterParameters cfp in _cacheCD.Keys)
                        {
                            if (cfp.VocabularyDomain.Equals(vocabularyDomain, StringComparison.InvariantCultureIgnoreCase))
                            {
                                if ((cfp as CodeFilterParametersWithProperty) == null)
                                    (_cacheCD[cfp] as List<CD>).RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));
                                else
                                    (_cacheCD[cfp] as List<CDWithProperties>).RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));
                            }
                        }
                    }
                    finally { cacheLock.ExitWriteLock(); }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RemoveCacheCDSpecialty(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                specialtyCacheLock.EnterWriteLock();
                try
                {                    
                    _specialtyCacheCD.RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));                            
                    _specialtyCacheCDWithProperties.RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));                        
                }
                finally { specialtyCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RemoveCacheCDTreatingSpecialty(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                treatingSpecialtyCacheLock.EnterWriteLock();
                try
                {                    
                    _treatingSpecialtyCacheCD.RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));                    
                }
                finally { treatingSpecialtyCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RemoveCacheCDTypeOfMovement(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                typeOfMovementCacheLock.EnterWriteLock();
                try
                {
                    _typeOfMovementCacheCD.RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));                    
                }
                finally { typeOfMovementCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RemoveCacheCDMedicalCenterDivision(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                medicalCenterCacheLock.EnterWriteLock();
                try
                {
                    _medicalCenterCacheCD.RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));                    
                }
                finally { medicalCenterCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RemoveCacheCDOrderableItem(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string vistaCode = code.Substring(0, code.IndexOf("_"));
                orderableItemCacheLock.EnterWriteLock();
                try
                {
                    if (_orderableItemCacheCD.ContainsKey(vistaCode))
                        _orderableItemCacheCD[vistaCode].RemoveAll(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase));
                }
                finally { orderableItemCacheLock.ExitWriteLock(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region GetVocabularyCache

        internal List<CD> GetOrderableItemCache(string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // vistaCode can be an Orderable Item code(vistaCode + "_" + ien) or vistaCode
                string[] items = vistaCode.Split('_');
                if (orderableItemCacheLock.TryEnterReadLock(0))
                {
                    try
                    {
                        if (items != null && items.Length == 2 && !string.IsNullOrEmpty(items[1]))
                        {
                            if (_orderableItemCacheCD.ContainsKey(items[0]))
                            {
                                CD cd = _orderableItemCacheCD[items[0]].Where(a => a.code.Equals(vistaCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                if (cd != null)
                                    return new List<CD>() { cd };
                            }
                        }
                        else
                        {
                            if (_orderableItemCacheCD.ContainsKey(vistaCode))
                                return _orderableItemCacheCD[vistaCode];
                        }
                    }
                    finally { orderableItemCacheLock.ExitReadLock(); }
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal List<CD> GetCodesCache(CodeFilterParameters param)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<CD> result = new List<CD>();
                if (param.VocabularyDomain.Equals(Util.Vocabulary.VistaSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                {
                    if (specialtyCacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            result.AddRange(_specialtyCacheCD);
                        }
                        finally { specialtyCacheLock.ExitReadLock(); }
                    }
                    else result = null;
                }
                else if (param.VocabularyDomain.Equals(Util.Vocabulary.VistaTreatingSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                {
                    if (treatingSpecialtyCacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            result.AddRange(_treatingSpecialtyCacheCD);
                        }
                        finally { treatingSpecialtyCacheLock.ExitReadLock(); }
                    }
                    else result = null;
                }
                else if (param.VocabularyDomain.Equals(Util.Vocabulary.TypeOfMovement.ToString(), StringComparison.InvariantCultureIgnoreCase))
                {
                    if (typeOfMovementCacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            result.AddRange(_typeOfMovementCacheCD);
                        }
                        finally { typeOfMovementCacheLock.ExitReadLock(); }
                    }
                    else result = null;
                }
                else if (param.VocabularyDomain.Equals(Util.Vocabulary.VistaMedicalCenterDivision.ToString(), StringComparison.InvariantCultureIgnoreCase))
                {
                    if (medicalCenterCacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            result.AddRange(_medicalCenterCacheCD);
                        }
                        finally { medicalCenterCacheLock.ExitReadLock(); }
                    }
                    else result = null;                    
                }
                else
                {
                    if (cacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            IEnumerable tempResult = null;
                            _cacheCD.TryGetValue(param, out tempResult);
                            if (tempResult != null)
                                result.AddRange(tempResult as List<CD>);
                            else
                                result = null;
                        }
                        finally { cacheLock.ExitReadLock(); }
                    }
                    else result = null;                     
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal List<CDWithProperties> GetCodesWithPropertiesCache(CodeFilterParametersWithProperty param)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<CDWithProperties> result = new List<CDWithProperties>();
                if (param.VocabularyDomain.Equals(Util.Vocabulary.VistaSpecialty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                {
                    if (specialtyCacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            result.AddRange(_specialtyCacheCDWithProperties);
                        }
                        finally { specialtyCacheLock.ExitReadLock(); }
                    }
                    else result = null;
                }
                else
                {
                    if (cacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            IEnumerable tempResult = null;
                            _cacheCD.TryGetValue(param, out tempResult);
                            if (tempResult != null)
                                result.AddRange(tempResult as List<CDWithProperties>);
                            else
                                result = null;
                        }
                        finally { cacheLock.ExitReadLock(); }
                    }
                    else result = null;  
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        #endregion

        /// <summary>
        /// Returns the SupportedCodeSystemDescriptor for CodeSystem key by EVS cached.
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        internal CodeSystemDescriptor GetSupportedCodeSystem(string key)
        {
            return _supportedCodeSystems[key];
        }

        internal void InitSupportedCodeSystems(IEnumerable<CodeSystemDescriptor> values)
        {
            foreach (CodeSystemDescriptor item in values)
                _supportedCodeSystems.Add(item.CodeSystem_id.V, item);
        }        
    }
    /// <summary>
    /// IEqualityComparer used for the first CD cache dictionary.
    /// </summary>
    class CodeFilterParametersComparer : IEqualityComparer<CodeFilterParameters>
    {
        /// <summary>
        /// Checks to see whether two CodeFilterParameters are the same by comparing references
        /// and VocabularyDomain, MatchText, MaxSelectedCodes, Language and MatchAlgorithm propeties.
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <returns></returns>
        public bool Equals(CodeFilterParameters x, CodeFilterParameters y)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (Object.ReferenceEquals(x, y))
                    return true;

                if (Object.ReferenceEquals(x, null) || Object.ReferenceEquals(y, null))
                    return false;

                if (x is CodeFilterParametersWithProperty && !(y is CodeFilterParametersWithProperty))
                    return false;

                if (y is CodeFilterParametersWithProperty && !(x is CodeFilterParametersWithProperty))
                    return false;

                bool baseCheck = (
                    x.VocabularyDomain == y.VocabularyDomain &&
                    x.MatchText == y.MatchText &&
                    x.MaxSelectedCodes == y.MaxSelectedCodes &&
                    x.Language == y.Language &&
                    x.MatchAlgorithm == y.MatchAlgorithm);

                if (x is CodeFilterParametersWithProperty)
                {
                    if (baseCheck)
                    {
                        CodeFilterParametersWithProperty xP = (CodeFilterParametersWithProperty)x;
                        CodeFilterParametersWithProperty yP = (CodeFilterParametersWithProperty)y;

                        if (Object.ReferenceEquals(xP.Properties, yP.Properties))
                            return true;

                        if (Object.ReferenceEquals(xP.Properties, null) || Object.ReferenceEquals(yP.Properties, null))
                            return false;

                        if (xP.Properties.Count != yP.Properties.Count)
                            return false;
                        for (int i = 0; i < xP.Properties.Count; i++)
                            if (xP.Properties[i] != yP.Properties[i])
                                return false;

                        return true;
                    }
                    return false;
                }
                else
                    return baseCheck;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Computes the hashcode for a CodeFilterParameters used within the dictionary based on the 
        /// same properties as the Equals method using XOR operator.
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public int GetHashCode(CodeFilterParameters obj)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (Object.ReferenceEquals(obj, null))
                    return 0;

                int hashVocabularyDomain = obj.VocabularyDomain == null ? 0 : obj.VocabularyDomain.GetHashCode();
                int hashMatchTextn = obj.MatchText == null ? 0 : obj.MatchText.GetHashCode();
                int hashMaxSelectedCodes = obj.MaxSelectedCodes.GetHashCode();
                int hashLanguage = obj.Language == null ? 0 : obj.Language.GetHashCode();
                int hashMatchAlgorithm = obj.MatchAlgorithm.GetHashCode();
                int hashProperties = 0;
                if (obj is CodeFilterParametersWithProperty)
                {
                    CodeFilterParametersWithProperty cfpwp = (CodeFilterParametersWithProperty)obj;
                    if (cfpwp.Properties != null)
                    {
                        foreach (String property in cfpwp.Properties)
                            hashProperties ^= property.GetHashCode();
                    }
                }
                return hashVocabularyDomain ^ hashMatchTextn ^ hashMaxSelectedCodes ^ hashLanguage ^ hashMatchAlgorithm ^ hashProperties;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
    /// <summary>
    /// IEqualityComparer used for the second dictionary cache (CDMatchText).
    /// Uses the same tehniques as CodeFilterParametersComparer but takes the MatchText in account as well.
    /// </summary>
    class CodeFilterParametersComparerWithMatchText : IEqualityComparer<CodeFilterParameters>
    {
        /// <summary>
        /// CodeFilterParametersComparer used for Equals and GetHashCode methods as a base IEqualityComparer
        /// </summary>
        CodeFilterParametersComparer _baseComparer = new CodeFilterParametersComparer();
        /// <summary>
        /// Checks to see whether two CodeFilterParameters are the same using CodeFilterParametersComparer and checking for MatchText as well.
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <returns></returns>
        public bool Equals(CodeFilterParameters x, CodeFilterParameters y)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool eqBase = _baseComparer.Equals(x, y);
                if (!eqBase)
                    return false;

                if (x.MatchText == y.MatchText)
                    return true;
                return false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Computes the hashcode for a CodeFilterParameters used within the dictionary using the base CodeFilterParametersComparer and 
        /// XOR operator for MatchText.
        /// </summary>
        /// <param name="obj">CodeFilterParameters object to compute the hash for.</param>
        /// <returns></returns>
        public int GetHashCode(CodeFilterParameters obj)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                int hashBase = _baseComparer.GetHashCode(obj);
                if (!String.IsNullOrEmpty(obj.MatchText))
                    return hashBase ^ obj.MatchText.GetHashCode();
                return hashBase;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
