﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using BMS.Utils;
using InfoWorld.HL7.ITS;
using InfoWorld.HL7.ITS.Extension.EIS;
using BMS.ServicesWrapper.Security;
using System.Collections;
using System.Threading;
using System.Configuration;
using System.ServiceModel;


namespace BMS.ServicesWrapper.EIS
{
    class EISWrapper : IEISWrapper
    {
        EISCache _cache;
        bool _isWindows = false;
        int _pageCount = int.Parse(ConfigurationManager.AppSettings["BMS_EIS_GET_ENTITY_FILTR_PAGE_SIZE"]);
        Semaphore eisSemaphore;
        Semaphore eisCacheSemaphore;

        private Proxy.EIS.QueryFunctionsClient QueryClient
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (_isWindows)
                        return EISFactory.QueryFunctionsClientWindows;
                    return EISFactory.QueryFunctionsClient;
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }
        public Proxy.EIS.EntityManagementClient EntityClient
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (_isWindows)
                        return EISFactory.EntityManagementClientWindows;
                    return EISFactory.EntityManagementClient;
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }
        private ISecurityWrapper SecurityWrapper
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (_isWindows)
                        return SecurityFactory.InstanceWindows;
                    return SecurityFactory.Instance;
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        public EISWrapper(bool isWindows)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this._isWindows = isWindows;
                _cache = new EISCache(this);
                int _eisLongRunningMethodsMaxConcurrentCalls = 0;
                try
                {
                    _eisLongRunningMethodsMaxConcurrentCalls = int.Parse(ConfigurationManager.AppSettings["BMS_EIS_LongRunningMethodsMaxConcurrentCalls"]);
                }
                catch { _eisLongRunningMethodsMaxConcurrentCalls = 30; }
                eisSemaphore = new Semaphore(_eisLongRunningMethodsMaxConcurrentCalls, _eisLongRunningMethodsMaxConcurrentCalls);
                
                int _eisCacheMethodsMaxConcurrentCalls = 0;
                try
                {
                    _eisCacheMethodsMaxConcurrentCalls = int.Parse(ConfigurationManager.AppSettings["BMS_Cache_MaxConcurrentCalls"]);
                }
                catch { _eisCacheMethodsMaxConcurrentCalls = 100; }
                eisCacheSemaphore = new Semaphore(_eisCacheMethodsMaxConcurrentCalls, _eisCacheMethodsMaxConcurrentCalls);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region IEISWrapper Members

        #region Ward Operations

        public IList<Ward> GetWards(II divisionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Ward> result = _cache.GetWardsByDivision(divisionId.extension);
                if (result == null)
                    result = GetPlaces(new Ward(), divisionId).Cast<Ward>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Ward> GetWards(List<II> wardIds)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Ward> result = new List<Ward>();
                if (wardIds != null && wardIds.Count > 0)
                {
                    //get all the wards that are in cache
                    result.AddRange(_cache.GetWardsByIds(wardIds));

                    //get all the wards that are not in cache
                    List<II> ids = new List<II>();
                    foreach (II id in wardIds)
                    {
                        if (result.Where(a => a.Id.extension.Equals(id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                            ids.Add(id);
                    }
                    if (ids != null && ids.Count > 0)
                    {
                        List<Entity> entities = new List<Entity>(ids.Count());
                        IEnumerable<string> domains = ids.Select(ii => ii.root).Distinct();
                        foreach (string domain in domains)
                        {
                            List<string> identifiers = ids.Where(ii => ii.root.Equals(domain)).Select(ii => ii.extension).ToList();
                            for (int i = 0; i < Math.Ceiling(((double)identifiers.Count) / EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES); i++)
                            {
                                int min = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * i;
                                int max = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * (i + 1);
                                List<string> parameters = identifiers.Where((id, index) => min <= index && index < max).ToList();
                                try
                                {
                                    eisSemaphore.WaitOne();
                                    entities.AddRange(QueryClient.FindEntitiesById(parameters, EISConstants.EIS_PLACE_TYPE, domain, EISExtensions.WARD_OUTPUT_TRAITS));
                                }
                                finally { eisSemaphore.Release(); }
                            }
                        }
                        result.AddRange(entities.Select(entity => (new Ward()).TranslateFromEntity(entity)).ToList().Cast<Ward>().ToList());
                    }
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Ward> GetWardsByVistaSiteId(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Ward> result = _cache.GetWardsByVistaSite(vistaSiteId.extension);
                if (result == null)
                {
                    TraitList input = new TraitList() { TraitManager.CreateTraitPair(EISConstants.OWN, new R_OwnedEntity() { Scoper = new VistaSite() { Id = vistaSiteId }.GetEntityInRole() }) };
                    result = GetPlaces(new Ward(), input).Cast<Ward>().ToList();
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Ward GetWard(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Ward w = _cache.GetWardById(id.extension);
                if (w == null)
                {
                    Entity entity = GetEntityById(new Ward() { Id = id });
                    w = Translator.TranslateWard(entity);
                }
                return w;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Ward GetWard(string ien, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ien == null)
                    return null;
                Ward result = _cache.GetWardByIenAndVista(ien, vistaSiteId);
                if (result != null)
                    return result;
                
                TraitList input = new TraitList()
                { 
                    TraitManager.CreateTraitPairForIen(ien),
                    TraitManager.CreateTraitPair(EISConstants.OWN, new R_OwnedEntity() { Scoper = new VistaSite() { Id = vistaSiteId }.GetEntityInRole() })
                };
                List<Entity> entities = null;
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTrait(input, EISExtensions.WARD_OUTPUT_TRAITS, EISConstants.EIS_DOMAIN_ID_LIST, new List<string>() { EISConstants.PLACE });
                }
                finally { eisSemaphore.Release(); }

                if (entities != null && entities.Count > 0)
                    return Translator.TranslateWard(entities[0]);

                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CreateWard(Ward ward)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CreatePlace(ward);
                _cache.CacheWard(ward.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS CreateWard - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS CreateWard - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS CreateWard - " + ward.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateWard(Ward ward)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UpdatePlace(ward);
                _cache.CacheWard(ward.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateWard - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateWard - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateWard - " + ward.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateAssociationBetweenWardAndDivision(Ward ward, Division oldDivision, Division newDivision)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = GetEntityById(ward);
                Entity newEntity = CopyEntityId(entity);
                // create an entity with all required traits
                ward.TranslateToEntity(newEntity);

                II oldParentId = oldDivision == null ? null : oldDivision.Id == null ? null : oldDivision.Id;
                II newParentId = newDivision == null ? null : newDivision.Id == null ? null : newDivision.Id;

                if (newParentId != null)
                {
                    if (!TraitManager.Equals(oldParentId, newParentId))
                    {
                        if (newParentId != null)
                        {
                            IdentifierValuePair ivp = new IdentifierValuePair()
                            {
                                TraitIdentifier = ward.GetParentRoleType(),
                                TraitValue = ward.GetParentRole(newParentId)
                            };

                            newEntity.TraitList.Add(ivp);
                        }

                        // remove the previous association
                        if (oldParentId != null)
                        {
                            IdentifierValuePair ivp = new IdentifierValuePair()
                            {
                                TraitIdentifier = ward.GetParentRoleType(),
                                TraitValue = ward.GetParentRole(oldParentId)
                            };

                            ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                            newEntity.TraitList.Add(ivp);
                        }
                    }
                }

                // finally, update the entity in EIS
                EntityClient.UpdateTraitValues(newEntity);

                _cache.CacheWard(ward.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateAssociationBetweenWardAndDivision - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateAssociationBetweenWardAndDivision - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateAssociationBetweenWardAndDivision - " + ward.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteAssociationBetweenWardAndDivision(Ward ward, Division division)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = GetEntityById(ward);
                // create an entity with all required traits
                Entity newEntity = CopyEntityId(entity);
                ward.TranslateToEntity(newEntity);

                II oldParentId = division == null ? null : division.Id == null ? null : division.Id;
                // remove the previous association
                if (oldParentId != null)
                {
                    IdentifierValuePair ivp = new IdentifierValuePair()
                    {
                        TraitIdentifier = ward.GetParentRoleType(),
                        TraitValue = ward.GetParentRole(oldParentId)
                    };

                    ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                    newEntity.TraitList.Add(ivp);
                }
                // finally, update the entity in EIS
                EntityClient.UpdateTraitValues(newEntity);
                _cache.CacheWard(ward.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS DeleteAssociationBetweenWardAndDivision - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(ward.Id, Util.EISEntities.Ward.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS DeleteAssociationBetweenWardAndDivision - " + ward.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteAssociationBetweenWardAndDivision - " + ward.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteWard(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                InactivatePlace(new Ward() { Id = id });
                _cache.RemoveCacheWard(id);
                ThreadPool.QueueUserWorkItem((o) =>
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Ward.ToString());
                    }
                    catch (CommunicationObjectAbortedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Ward.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteWard - " + id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Ward.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteWard - " + id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EIS DeleteWard - " + id.extension);
                        Tracer.TraceException(ex);
                    }
                });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Division Operations

        public Division GetDivision(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Division division = _cache.GetDivisionById(id.extension);
                if (division == null)
                {
                    Entity entity = GetEntityById(new Division() { Id = id });
                    division = Translator.TranslateDivision(entity);
                }
                return division;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Division> GetDivisions(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Division> result = _cache.GetDivisionsByFacility(facilityId.extension);
                if (result == null)
                    result = GetPlaces(new Division(), facilityId).Cast<Division>().ToList();
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Division FindDivision(II facilityId, string name)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Division result = null;
                List<Division> divisions = _cache.GetDivisionsByFacility(facilityId.extension);
                if (divisions != null)
                    result = divisions.Where(a => a.Name.Equals(name, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();

                if (result == null)
                {
                    divisions = GetPlaces(new Division(), facilityId, new List<TraitPair>() { TraitManager.CreateTraitPairForName(name) }).Cast<Division>().ToList();
                    result = divisions.Count == 0 ? null : divisions[0];
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Division> GetDivisionsForBed(II bedId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Bed bed = _cache.GetBedById(bedId.extension, vistaSiteId);
                List<string> divisionsIds = new List<string>();
                List<Division> result = new List<Division>();
                Division d = null;
                if (bed != null && bed.CommentList != null && bed.CommentList.Count > 0)
                {
                    foreach (BedComment bc in bed.CommentList)
                    {
                        d = _cache.GetDivisionById(bc.Division.Id.extension);
                        if (d != null)
                            result.Add(d);
                        else
                            divisionsIds.Add(bc.Division.Id.extension);
                    }
                }
                else
                {
                    Entity e = GetEntityById(new Bed() { Id = bedId });
                    List<II> divisions = TraitManager.FindScopers(e, EISConstants.MNT);
                    if (divisions != null)
                    {
                        foreach (II id in divisions)
                            divisionsIds.Add(id.extension);
                    }
                }
                if (divisionsIds != null && divisionsIds.Count > 0)
                {                                        
                    try
                    {
                        eisSemaphore.WaitOne();
                        List<Entity> entities = QueryClient.FindEntitiesById(divisionsIds, EISConstants.ORGANIZATON, bedId.root, EISExtensions.DIVISION_OUTPUT_TRAITS);
                        result.AddRange(entities.Select(entity => (new Division()).TranslateFromEntity(entity)).ToList().Cast<Division>().ToList());
                    }
                    finally { eisSemaphore.Release(); }                    
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CreateDivision(Division division)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CreatePlace(division);
                _cache.CacheDivision(division.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(division.Id, Util.EISEntities.Division.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(division.Id, Util.EISEntities.Division.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS CreateDivision - " + division.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(division.Id, Util.EISEntities.Division.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS CreateDivision - " + division.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS CreateDivision - " + division.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateDivision(Division division)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UpdatePlace(division);
                _cache.CacheDivision(division.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(division.Id, Util.EISEntities.Division.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(division.Id, Util.EISEntities.Division.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateDivision - " + division.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(division.Id, Util.EISEntities.Division.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateDivision - " + division.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateDivision - " + division.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteDivision(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Division division = GetDivision(id);
                // create an entity with all required traits
                Entity newEntity = new Entity() { EntityTypeIdentifier = EISConstants.ORGANIZATON, DomainIdentifier = id.root, Identifier = id.extension };
                division.TranslateToEntity(newEntity);

                // remove the facility parent role
                if (division.Facility != null && division.Facility.Id != null)
                {
                    IdentifierValuePair ivp = new IdentifierValuePair()
                    {
                        TraitIdentifier = division.GetParentRoleType(),
                        TraitValue = division.GetParentRole(division.Facility.Id)
                    };

                    ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                    newEntity.TraitList.Add(ivp);
                    // finally, update the entity in EIS
                    EntityClient.UpdateTraitValues(newEntity);
                }

                EntityClient.InactivateEntity(id.root, EISConstants.ORGANIZATON, id.extension);
                _cache.RemoveCacheDivision(id);
                ThreadPool.QueueUserWorkItem((o) =>
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Division.ToString());
                    }
                    catch (CommunicationObjectAbortedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Division.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteDivision - " + id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Division.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteDivision - " + id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EIS DeleteDivision - " + id.extension);
                        Tracer.TraceException(ex);
                    }
                });
            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Facility Operations

        public Facility GetFacilityByCode(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility result = null;
                List<Facility> facilities = _cache.GetFacilities();
                if (facilities != null && facilities.Count > 0)
                    result = facilities.Where(a => a.Code.Equals(code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (result != null)
                    return result;

                Entity entity = GetEntityByCode(new Facility(), code);
                return Translator.TranslateFacility(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilitiesBySiteNumber(string siteNumber)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList traits = new TraitList();
                traits.Add(TraitManager.CreateTraitPair(
                            EISConstants.SITE_NUMBER,
                            new ST(siteNumber)));

                return GetPlaces(new Facility(), traits).Cast<Facility>().ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilities()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Facility> result = _cache.GetFacilities();
                if (result == null)
                    result = GetPlaces(new Facility()).Cast<Facility>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilitiesUsingWindowsAuthentication()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Facility> result = _cache.GetFacilities();
                if (result == null)
                    result = GetPlacesUsingWindowsAuthentication(new Facility()).Cast<Facility>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilities(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Facility> result = _cache.GetFacilitiesByVistaSite(vistaSiteId.extension);
                if (result == null)
                    result = GetPlaces(new Facility(), vistaSiteId).Cast<Facility>().ToList();
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Facility GetFacility(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility result = _cache.GetFacilityById(id.extension);
                if (result == null)
                {
                    Entity entity = GetEntityById(new Facility() { Id = id });
                    result = Translator.TranslateFacility(entity);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilityIdsForWard(II wardId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<string> divisionsIds = new List<string>();
                List<Facility> facilities = new List<Facility>();
                Ward ward = _cache.GetWardById(wardId.extension);
                if (ward != null)
                {
                    if (ward.DivisionList != null && ward.DivisionList.Count > 0)
                    {
                        Division division = null;
                        Facility facility = null;
                        foreach (Division d in ward.DivisionList)
                        {
                            division = _cache.GetDivisionById(d.Id.extension);
                            if (division != null && division.Facility != null)
                            {
                                facility = _cache.GetFacilityById(division.Facility.Id.extension);
                                if (facility != null && facilities.Where(a => a.Id.extension.Equals(facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                                    facilities.Add(facility);
                                else
                                    divisionsIds.Add(d.Id.extension);
                            }
                            else
                                divisionsIds.Add(d.Id.extension);
                        }
                    }
                }
                else
                {
                    Entity e = GetEntityById(new Ward() { Id = wardId });
                    List<II> divisions = TraitManager.FindScopers(e, EISConstants.SDLOC);
                    if (divisions != null)
                    {
                        foreach (II id in divisions)
                            divisionsIds.Add(id.extension);
                    }
                }
                if (divisionsIds != null && divisionsIds.Count > 0)
                {
                    List<Division> divisionList = new List<Division>();                    
                    try
                    {
                        eisSemaphore.WaitOne();
                        List<Entity> entities = QueryClient.FindEntitiesById(divisionsIds, EISConstants.ORGANIZATON, wardId.root, EISExtensions.DIVISION_OUTPUT_TRAITS);
                        divisionList = entities.Select(entity => (new Division()).TranslateFromEntity(entity)).ToList().Cast<Division>().ToList();
                    }
                    finally { eisSemaphore.Release(); }
                    foreach (Division div in divisionList)
                    {
                        if (facilities.Where(a => a.Id.extension.Equals(div.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                            facilities.Add(div.Facility);
                    }
                }
                return facilities;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateFacility(Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UpdatePlace(facility);
                _cache.CacheFacility(facility.Id);
                ThreadPool.QueueUserWorkItem((o) =>
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(facility.Id, Util.EISEntities.Facility.ToString());
                    }
                    catch (CommunicationObjectAbortedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(facility.Id, Util.EISEntities.Facility.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateFacility - " + facility.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(facility.Id, Util.EISEntities.Facility.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateFacility - " + facility.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EIS UpdateFacility - " + facility.Id.extension);
                        Tracer.TraceException(ex);
                    }
                });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Facility GetFacilityByDivisionId(II divisionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility result = null;
                II facilityId = null;
                Division division = _cache.GetDivisionById(divisionId.extension);
                if (division != null && division.Facility != null && division.Facility.Id != null)
                {
                    facilityId = division.Facility.Id;
                }
                else
                {
                    Entity e = GetEntityById(new Division() { Id = divisionId });
                    facilityId = TraitManager.FindScopers(e, EISConstants.MBR).FirstOrDefault();
                }
                if (facilityId != null)
                {
                    result = _cache.GetFacilityById(facilityId.extension);
                    if (result != null)
                    {
                        result = GetFacility(facilityId);
                    }
                }

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Visn Operations

        public IList<Visn> GetVisnsUsingWindowsAuthentication()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Visn> result = _cache.GetVisns();
                if (result == null)
                    result = GetPlacesUsingWindowsAuthentication(new Visn()).Cast<Visn>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Visn> GetVisns(II regionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Visn> result = _cache.GetVisnsByRegion(regionId.extension);
                if (result == null)
                    result = GetPlaces(new Visn(), regionId).Cast<Visn>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Visn GetVisn(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Visn result = _cache.GetVisnById(id.extension);
                if (result == null)
                {
                    Entity entity = GetEntityById(new Visn() { Id = id });
                    result = Translator.TranslateVisn(entity);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Visn GetVisnByNumber(string number)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Visn result = null;
                List<Visn> visns = _cache.GetVisns();
                if (visns != null && visns.Count > 0)
                    result = visns.Where(a => a.Number.Equals(number, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (result != null)
                    return result;

                TraitList input = new TraitList();
                Visn place = new Visn();
                input.Add(TraitManager.CreateTraitPairForNumber(number));
                input.Add(TraitManager.CreateTraitPairForPlaceType(place.GetPlaceType()));
                List<Entity> entities = QueryClient.FindEntitiesByTrait(input, place.GetOutputTraits(), EISConstants.EIS_DOMAIN_ID_LIST, new List<string>() { place.GetEntityType() });

                Entity entity = entities.Count == 0 ? null : entities[0];
                return Translator.TranslateVisn(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Vista Site Operations

        public IList<VistaSite> GetVistaSites(II visnId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<VistaSite> result = _cache.GetVistaSitesByVisn(visnId.extension);
                if (result == null)
                    result = GetPlaces(new VistaSite(), visnId).Cast<VistaSite>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<VistaSite> GetVistaSitesUsingWindowsAuthentication()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<VistaSite> result = _cache.GetVistaSites();
                if (result == null)
                    result = GetPlacesUsingWindowsAuthentication(new VistaSite()).Cast<VistaSite>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public VistaSite GetVistaSite(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VistaSite result = _cache.GetVistaSiteById(id.extension);
                if (result == null)
                {
                    Entity entity = GetEntityById(new VistaSite() { Id = id });
                    result = Translator.TranslateVistaSite(entity);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public VistaSite GetVistaSiteByCode(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VistaSite result = null;
                List<VistaSite> vistaSites = _cache.GetVistaSites();
                if (vistaSites != null && vistaSites.Count > 0)
                    result = vistaSites.Where(a => a.Code.Equals(code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (result != null)
                    return result;

                Entity entity = GetEntityByCode(new VistaSite(), code);
                return Translator.TranslateVistaSite(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Hospital Location Operations

        public IList<HospitalLocation> GetHospitalLocations(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<HospitalLocation> result = _cache.GetHospitalLocationsByVista(vistaSiteId);
                if (result == null)
                {
                    try
                    {
                        eisSemaphore.WaitOne();
                        result = GetPlaces(new HospitalLocation(), vistaSiteId).Cast<HospitalLocation>().ToList();
                    }
                    finally { eisSemaphore.Release(); }
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<HospitalLocation> GetHospitalLocationsByVistaDB(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetPlaces(new HospitalLocation(), vistaSiteId).Cast<HospitalLocation>().ToList();                    
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<HospitalLocation> GetHospitalLocationsByIds(List<II> ids, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ids != null && ids.Count > 0)
                {
                    List<HospitalLocation> result = new List<HospitalLocation>();                    
                    //get all the hospital locations that are in cache
                    result.AddRange(_cache.GetHospitalLocationsByIds(ids, vistaSiteId));                    
                    //get all the hospital locations that are not in cache
                    List<II> _missingIds = new List<II>();
                    foreach (II id in ids)
                    {
                        if (result.Where(a => a.Id.extension.Equals(id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                            _missingIds.Add(id);
                    }                        
                    if (_missingIds != null && _missingIds.Count > 0)
                    {
                        List<Entity> entities = new List<Entity>(_missingIds.Count);
                        IEnumerable<string> domains = _missingIds.Select(ii => ii.root).Distinct();
                        foreach (string domain in domains)
                        {
                            List<string> identifiers = _missingIds.Where(ii => ii.root.Equals(domain)).Select(ii => ii.extension).ToList();
                            for (int i = 0; i < Math.Ceiling(((double)identifiers.Count) / EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES); i++)
                            {
                                int min = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * i;
                                int max = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * (i + 1);
                                List<string> parameters = identifiers.Where((id, index) => min <= index && index < max).ToList();
                                try
                                {
                                    eisSemaphore.WaitOne();
                                    entities.AddRange(QueryClient.FindEntitiesById(parameters, EISConstants.EIS_ORGANIZATION_TYPE, domain, EISExtensions.HOSPITAL_LOCATION_OUTPUT_TRAITS));                                    
                                }
                                finally { eisSemaphore.Release(); }
                            }
                        }
                        result.AddRange(entities.Select<Entity, HospitalLocation>(e => Translator.TranslateHospitalLocation(e)).ToList());
                    }                    
                    return result;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<HospitalLocation> GetHospitalLocationsByIdsDB(List<II> ids)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ids != null && ids.Count > 0)
                {
                    List<HospitalLocation> result = new List<HospitalLocation>();                    
                    List<Entity> entities = new List<Entity>(ids.Count);
                    IEnumerable<string> domains = ids.Select(ii => ii.root).Distinct();
                    foreach (string domain in domains)
                    {
                        List<string> identifiers = ids.Where(ii => ii.root.Equals(domain)).Select(ii => ii.extension).ToList();
                        for (int i = 0; i < Math.Ceiling(((double)identifiers.Count) / EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES); i++)
                        {
                            int min = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * i;
                            int max = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * (i + 1);
                            List<string> parameters = identifiers.Where((id, index) => min <= index && index < max).ToList();                                                            
                            entities.AddRange(QueryClient.FindEntitiesById(parameters, EISConstants.EIS_ORGANIZATION_TYPE, domain, EISExtensions.HOSPITAL_LOCATION_OUTPUT_TRAITS));                            
                        }
                    }
                    result.AddRange(entities.Select<Entity, HospitalLocation>(e => Translator.TranslateHospitalLocation(e)).ToList());                
                    return result;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public HospitalLocation GetHospitalLocation(II id, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                HospitalLocation result = _cache.GetHospitalLocationById(id.extension, vistaSiteId);
                if (result == null)
                {
                    Entity entity = GetEntityById(new HospitalLocation() { Id = id });
                    result = Translator.TranslateHospitalLocation(entity);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public HospitalLocation GetHospitalLocationDB(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {                
                Entity entity = GetEntityById(new HospitalLocation() { Id = id });
                return Translator.TranslateHospitalLocation(entity);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public HospitalLocation GetHospitalLocation(string ien, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ien == null)
                    return null;
                HospitalLocation result = _cache.GetHospitalLocationByIenAndVista(ien, vistaSiteId);
                if (result != null)
                    return result;
                TraitList input = new TraitList();
                input.Add(TraitManager.CreateTraitPairForIen(ien));

                IList<HospitalLocation> hospitals = GetPlaces(new HospitalLocation(), vistaSiteId, input).Cast<HospitalLocation>().ToList();
                if (hospitals != null && hospitals.Count > 0)
                    return hospitals[0];
                return null;                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CreateHospitalLocation(HospitalLocation hospitalLocation)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CreatePlace(hospitalLocation);
                _cache.CacheHospitalLocation(hospitalLocation.Id);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateHospitalLocation(HospitalLocation hospitalLocation)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UpdatePlace(hospitalLocation);
                _cache.CacheHospitalLocation(hospitalLocation.Id);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteHospitalLocation(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EntityClient.InactivateEntity(id.root, EISConstants.ORGANIZATON, id.extension);
                _cache.RemoveCacheHospitalLocation(id);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Region Operation

        public IList<Region> GetRegions()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Region> result = _cache.GetRegions();
                if (result == null)
                    result = GetPlaces(new Region()).Cast<Region>().ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Region GetRegion(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Region result = _cache.GetRegionById(id.extension);
                if (result == null)
                {
                    Entity entity = GetEntityById(new Region() { Id = id });
                    result = Translator.TranslateRegion(entity);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Bed Operations

        /// <summary>
        /// Gets the beds located in the given ward.
        /// </summary>
        /// <returns>a list of beds</returns>
        public IList<Bed> GetBedsInWard(II wardId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (wardId == null)
                    return null;
                List<Bed> result = _cache.GetBedsByWard(wardId.extension, vistaSiteId);
                if (result == null)
                    result = GetPlaces(new Bed(), wardId).Cast<Bed>().ToList();
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the beds located in the given division.
        /// </summary>
        /// <returns>a list of beds</returns>
        /// 
        public IList<Bed> GetBedsInDivision(II divisionId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (divisionId == null)
                    return null;
                List<Bed> result = _cache.GetBedsByDivision(divisionId.extension, vistaSiteId);
                if (result == null)
                {
                    TraitList traits = new TraitList();
                    //filter on grand-father, i.e. the division
                    if (divisionId != null)
                    {
                        traits.Add(TraitManager.CreateTraitPair(
                                    EISConstants.MNT,
                                    new R_MaintainedEntity() { Scoper = new Division() { Id = divisionId }.GetEntityInRole() }));
                    }
                    result = GetPlaces(new Bed(), traits).Cast<Bed>().ToList();
                }
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the beds by vista.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        public List<Bed> GetBedsByVista(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (vistaSiteId == null)
                    return null;
                List<Bed> result = _cache.GetBedsByVista(vistaSiteId);
                if (result == null)                
                    result = GetBedsByVistaTrait(vistaSiteId);
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<Bed> GetBedsByVistaTrait(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList input = new TraitList();
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = new II() { root = EISConstants.SITE, extension = vistaSiteId.extension }, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.IdenticalIgnoreCase } });
                input.Add(TraitManager.CreateTraitPairForPlaceType(EISConstants.IDENTIFIER_FOR_BED_CODE));
                return GetPlaces(new Bed(), input).Cast<Bed>().ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Bed GetBed(II id, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (id == null) return null;
                Bed result = _cache.GetBedById(id.extension, vistaSiteId);
                if (result == null)
                {
                    Entity entity = GetEntityById(new Bed() { Id = id });
                    result = Translator.TranslateBed(entity);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Bed GetBed(string ien, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ien == null)
                    return null;
                Bed result = _cache.GetBedByIenAndVista(ien, vistaSiteId);
                if (result != null)
                    return result;
                TraitList input = new TraitList();
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = new II() { root = EISConstants.SITE, extension = vistaSiteId.extension }, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.IdenticalIgnoreCase } });
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = new II() { root = EISConstants.IEN, extension = ien }, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.IdenticalIgnoreCase } });
                input.Add(TraitManager.CreateTraitPairForPlaceType(EISConstants.IDENTIFIER_FOR_BED_CODE));
                List<Entity> entities = null;
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTrait(input, EISExtensions.BED_OUTPUT_TRAITS, EISConstants.EIS_DOMAIN_ID_LIST, new List<string>() { EISConstants.PLACE });
                }
                finally {eisSemaphore.Release();}
                if (entities != null && entities.Count > 0)
                    return Translator.TranslateBed(entities[0]);
                    
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public void CreateBed(Bed bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CreatePlace(bed);
                _cache.CacheBed(bed.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS CreateBed - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS CreateBed - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS CreateBed - " + bed.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateBed(Bed bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UpdatePlace(bed);
                _cache.CacheBed(bed.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateBed - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateBed - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateBed - " + bed.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteBed(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                InactivatePlace(new Bed { Id = id });
                _cache.RemoveCacheBed(id);
                ThreadPool.QueueUserWorkItem((o) =>
                {
                    try
                    {
                        BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Bed.ToString());
                    }
                    catch (CommunicationObjectAbortedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Bed.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteBed - " + id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.RemoveEISCache(id, Util.EISEntities.Bed.ToString());
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteBed - " + id.extension);
                            Tracer.TraceException(ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceMessage("EIS DeleteBed - " + id.extension);
                        Tracer.TraceException(ex);
                    }
                });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteAssociationBetweenBedAndDivision(Bed bed, Division division)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DeleteParentRole(bed, division);
                _cache.CacheBed(bed.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS DeleteAssociationBetweenBedAndDivision - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS DeleteAssociationBetweenBedAndDivision - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS DeleteAssociationBetweenBedAndDivision - " + bed.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateAssociationBetweenBedAndDivision(Bed bed, Division oldDivision, Division newDivision)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (bed.WardList != null && bed.WardList.Count > 1)
                {
                    if (oldDivision != null && oldDivision.Id != null && !string.IsNullOrEmpty(oldDivision.Id.extension) && !oldDivision.Id.extension.Equals(Guid.Empty.ToString()))
                    {
                        List<Ward> wardsByOldDivision = GetWards(oldDivision.Id).ToList();                    
                        int count = 0;
                        foreach (Ward w in bed.WardList)
                        {
                            if (wardsByOldDivision.Where(a => a.Id.extension.Equals(w.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                                count = count + 1;
                        }
                        if (count > 1)
                            UpdateParentRole(bed, null, newDivision);
                        else
                            UpdateParentRole(bed, oldDivision, newDivision);
                    }
                    else
                        UpdateParentRole(bed, oldDivision, newDivision);
                }
                else
                    UpdateParentRole(bed, oldDivision, newDivision);
                _cache.CacheBed(bed.Id);
                try
                {
                    eisCacheSemaphore.WaitOne();
                    ThreadPool.QueueUserWorkItem((o) =>
                    {
                        try
                        {
                            BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                        }
                        catch (CommunicationObjectAbortedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateAssociationBetweenBedAndDivision - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            try
                            {
                                BMService.BMSFactory.BedManagerCacheClientWindows.UpdateEISCache(bed.Id, Util.EISEntities.Bed.ToString());
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("EIS UpdateAssociationBetweenBedAndDivision - " + bed.Id.extension);
                                Tracer.TraceException(ex);
                            }
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("EIS UpdateAssociationBetweenBedAndDivision - " + bed.Id.extension);
                            Tracer.TraceException(ex);
                        }
                    });
                }
                finally { eisCacheSemaphore.Release(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Bed> GetBeds(List<II> bedIds, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Bed> result = new List<Bed>();

                if (bedIds != null && bedIds.Count > 0)
                {
                    //get all the beds that are in cache
                    result.AddRange(_cache.GetBedsByIds(bedIds, vistaSiteId));

                    //get all the beds that are not in cache
                    List<II> ids = new List<II>();
                    foreach (II id in bedIds)
                    {
                        if (result.Where(a => a.Id.extension.Equals(id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                            ids.Add(id);
                    }
                    if (ids != null && ids.Count > 0)
                    {                        
                        List<Entity> entities = new List<Entity>(ids.Count());
                        IEnumerable<string> domains = ids.Select(ii => ii.root).Distinct();
                        foreach (string domain in domains)
                        {
                            List<string> identifiers = ids.Where(ii => ii.root.Equals(domain)).Select(ii => ii.extension).ToList();
                            for (int i = 0; i < Math.Ceiling(((double)identifiers.Count) / EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES); i++)
                            {
                                int min = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * i;
                                int max = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * (i + 1);
                                List<string> parameters = identifiers.Where((id, index) => min <= index && index < max).ToList();                                
                                try
                                {
                                    eisSemaphore.WaitOne();
                                    entities.AddRange(QueryClient.FindEntitiesById(parameters, EISConstants.EIS_PLACE_TYPE, domain, EISExtensions.BED_OUTPUT_TRAITS));
                                }
                                finally { eisSemaphore.Release(); }
                            }
                        }
                        result.AddRange(entities.Select(entity => (new Bed()).TranslateFromEntity(entity)).ToList().Cast<Bed>().ToList());
                    }
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Patient Operations

        public Patient GetPatientById(II id, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = GetEntityById(new Person() { Id = id });
                Patient patient = Translator.TranslatePatient(entity);
                if (!string.IsNullOrEmpty(vistaCode))
                {
                    if (patient != null && patient.IENList != null)
                        return patient.IENList.Where(a => a.root == vistaCode).Count() == 0 ? null : patient;
                    else
                        return null;
                }

                return patient;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Patient> GetPatientsByIds(IList<II> ids, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ids != null && ids.Count > 0)
                {
                    List<Entity> entities = new List<Entity>(ids.Count());
                    IEnumerable<string> domains = ids.Select(ii => ii.root).Distinct();
                    foreach (string domain in domains)
                    {
                        List<string> identifiers = ids.Where(ii => ii.root.Equals(domain)).Select(ii => ii.extension).ToList();
                        for (int i = 0; i < Math.Ceiling(((double)identifiers.Count) / EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES); i++)
                        {
                            int min = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * i;
                            int max = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * (i + 1);
                            List<string> parameters = identifiers.Where((id, index) => min <= index && index < max).ToList();
                            try
                            {
                                eisSemaphore.WaitOne();
                                entities.AddRange(QueryClient.FindEntitiesById(parameters, EISConstants.EIS_PERSON_TYPE, domain, EISExtensions.PATIENT_OUTPUT_TRAITS));
                            }
                            finally 
                            { 
                                eisSemaphore.Release();
                            }
                        }
                    }

                    List<Patient> patientsList = entities.Select<Entity, Patient>(e => Translator.TranslatePatient(e)).ToList();
                    if (!string.IsNullOrEmpty(vistaCode))
                    {
                        patientsList = patientsList.Where(a => a.IENList != null).ToList();
                        patientsList.RemoveAll(r => r.IENList.Where(a => a.root.Equals(vistaCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null);
                    }
                    return patientsList;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Patient GetPatientBySsn(II ssn, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ssn == null || string.IsNullOrEmpty(ssn.extension))
                    return null;
                TraitList input = new TraitList();
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = ssn, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.IdenticalIgnoreCase } });
                if (!string.IsNullOrEmpty(vistaCode))
                    input.Add(new TraitPair() { Identifier = EISConstants.IEN_LIST, Value = new ST(vistaCode), MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.ContainsPhraseIgnoreCase } });

                List<Entity> entities = new List<Entity>();
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTrait(input, EISExtensions.PATIENT_OUTPUT_TRAITS, EISConstants.EIS_DOMAIN_ID_LIST, EISConstants.EIS_PERSON_TYPE_LIST);
                }
                finally { eisSemaphore.Release(); }
                if (entities.Count == 0)
                    return null;
                else
                {
                    foreach (Entity ent in entities)
                    {
                        IdentifierValuePair ivp = ent.TraitList.Where(a => a.TraitIdentifier == EISConstants.PAT).FirstOrDefault();
                        if (ivp != null && ivp.TraitValue != null)
                        {
                            if (ivp.TraitValue as R_Patient != null)
                                return Translator.TranslatePatient(ent);
                        }
                    }
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Patient> GetPatientsByLastName(string lastName, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList input = new TraitList();
                input.Add(new TraitPair() { Identifier = EISConstants.NAME_FAMILY, Value = new ST(lastName), MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.BeginsWithIgnoreCase } });
                if (!string.IsNullOrEmpty(vistaCode))
                    input.Add(new TraitPair() { Identifier = EISConstants.IEN_LIST, Value = new ST(vistaCode), MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.ContainsPhraseIgnoreCase } });

                List<Entity> entities = QueryClient.FindEntitiesByTrait(input,
                 EISExtensions.PATIENT_OUTPUT_TRAITS, EISConstants.EIS_DOMAIN_ID_LIST, EISConstants.EIS_PERSON_TYPE_LIST);
                if (entities.Count == 0)
                    return null;
                else
                {
                    IList<Patient> result = new List<Patient>();
                    foreach (Entity ent in entities)
                    {
                        IdentifierValuePair ivp = ent.TraitList.Where(a => a.TraitIdentifier == EISConstants.PAT).FirstOrDefault();
                        if (ivp != null && ivp.TraitValue != null)
                        {
                            if (ivp.TraitValue as R_Patient != null)
                                result.Add(Translator.TranslatePatient(ent));
                        }
                    }
                    return result;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Patient> GetPatientsByLastNameInitialAndLastFourSSN(string filter, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList input = new TraitList();
                input.Add(new TraitPair() { Identifier = EISConstants.NAME_FAMILY, Value = new ST(filter.Substring(0, 1)), MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.BeginsWithIgnoreCase } });
                II ssn = new II(EISConstants.SSN, filter.Substring(1));
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = ssn, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.EndsWithIgnoreCase } });
                if (!string.IsNullOrEmpty(vistaCode))
                    input.Add(new TraitPair() { Identifier = EISConstants.IEN_LIST, Value = new ST(vistaCode), MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.ContainsPhraseIgnoreCase } });

                List<Entity> entities = QueryClient.FindEntitiesByTrait(input,
                 EISExtensions.PATIENT_OUTPUT_TRAITS, EISConstants.EIS_DOMAIN_ID_LIST, EISConstants.EIS_PERSON_TYPE_LIST);
                if (entities.Count == 0)
                    return null;
                else
                {
                    IList<Patient> result = new List<Patient>();
                    foreach (Entity ent in entities)
                    {
                        IdentifierValuePair ivp = ent.TraitList.Where(a => a.TraitIdentifier == EISConstants.PAT).FirstOrDefault();
                        if (ivp != null && ivp.TraitValue != null)
                        {
                            if (ivp.TraitValue as R_Patient != null)
                                result.Add(Translator.TranslatePatient(ent));
                        }
                    }
                    return result;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CreatePatient(Patient patient)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = new Entity()
                {
                    DomainIdentifier = SecurityWrapper.GetCurrentDomain(),
                    EntityTypeIdentifier = EISConstants.PERSON
                };
                Translator.TranslateToEntity(patient, entity);
                entity.Identifier = EntityClient.CreateEntity(entity);
                patient.Id = Translator.ExtractIdentifier(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdatePatient(Patient patient)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (patient.Id == null)
                {
                    // retrieve the existing entity to obtain the internal id
                    Entity oldEntity = GetEntityById(patient);
                    if (patient.Id == null)
                        patient.Id = Translator.ExtractIdentifier(oldEntity);
                }

                // create an entity with all required traits 
                Entity newEntity = new Entity()
                {
                    DomainIdentifier = patient.Id.root,
                    EntityTypeIdentifier = EISConstants.PERSON,
                    Identifier = patient.Id.extension
                };
                Translator.TranslateToEntity(patient, newEntity);
                EntityClient.UpdateTraitValues(newEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeletePatient(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DeletePerson(id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Medical Person Operations

        public Person GetMedicalPerson(string ien, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList input = new TraitList();
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = new II() { root = EISConstants.SITE, extension = vistaSiteId.extension }, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.IdenticalIgnoreCase } });
                input.Add(new TraitPair() { Identifier = EISConstants.UID, Value = new II() { root = EISConstants.IEN, extension = ien }, MatchingAlgorithm = new MatchingAlgorithmString() { value = MatchingAlgorithmsString.IdenticalIgnoreCase } });


                List<Entity> entities = new List<Entity>();
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTrait(input, EISExtensions.MEDICAL_OUTPUT_TRAITS, EISConstants.EIS_DOMAIN_ID_LIST, EISConstants.EIS_PERSON_TYPE_LIST);
                }
                finally { eisSemaphore.Release(); }
                if (entities.Count == 0)
                    return null;
                else
                {
                    foreach (Entity ent in entities)
                    {
                        IdentifierValuePair ivp = ent.TraitList.Where(a => a.TraitIdentifier == EISConstants.PROV).FirstOrDefault();
                        if (ivp != null && ivp.TraitValue != null)
                        {
                            if (ivp.TraitValue as R_HealthCareProvider != null)
                                return Translator.TranslatePerson(ent);
                        }
                    }
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public void CreateMedicalPerson(Person person)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = new Entity()
                {
                    DomainIdentifier = SecurityWrapper.GetCurrentDomain(),
                    EntityTypeIdentifier = EISConstants.PERSON
                };
                Translator.TranslateToEntity(person, entity);
                entity.Identifier = EntityClient.CreateEntity(entity);
                person.Id = Translator.ExtractIdentifier(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateMedicalPerson(Person person)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (person.Id == null)
                {
                    // retrieve the existing entity to obtain the internal id
                    Entity oldEntity = GetEntityById(person);
                    if (person.Id == null)
                        person.Id = Translator.ExtractIdentifier(oldEntity);
                }

                // create an entity with all required traits 
                Entity newEntity = new Entity()
                {
                    DomainIdentifier = person.Id.root,
                    EntityTypeIdentifier = EISConstants.PERSON,
                    Identifier = person.Id.extension
                };
                Translator.TranslateToEntity(person, newEntity);
                EntityClient.UpdateTraitValues(newEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Person> GetMedicalPersonsByIds(IList<II> ids)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ids != null && ids.Count > 0)
                {
                    List<Entity> entities = new List<Entity>(ids.Count());
                    IEnumerable<string> domains = ids.Select(ii => ii.root).Distinct();
                    foreach (string domain in domains)
                    {
                        List<string> identifiers = ids.Where(ii => ii.root.Equals(domain)).Select(ii => ii.extension).ToList();
                        for (int i = 0; i < Math.Ceiling(((double)identifiers.Count) / EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES); i++)
                        {
                            int min = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * i;
                            int max = EISConstants.GET_ENTITY_FILTR_MAX_ENTITIES * (i + 1);
                            List<string> parameters = identifiers.Where((id, index) => min <= index && index < max).ToList();
                            try
                            {
                                eisSemaphore.WaitOne();
                                entities.AddRange(QueryClient.FindEntitiesById(parameters, EISConstants.EIS_PERSON_TYPE, domain, EISExtensions.MEDICAL_OUTPUT_TRAITS));
                            }
                            finally 
                            { 
                                eisSemaphore.Release();
                            }
                        }
                    }
                    return entities.Select<Entity, Person>(e => Translator.TranslatePerson(e)).ToList();
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteMedicalPerson(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DeletePerson(id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Domains

        public IList<string> GetDomains()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<SupportDomain> eisDomains = QueryClient.ListSupportedDomains();
                IList<string> result = eisDomains.Where(IsActivDomain).Select(sd => sd.domain.Identifier).ToList().AsReadOnly();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool IsActivDomain(SupportDomain sd)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return sd.domain != null && sd.domain.Status != null &&
                            sd.domain.Status.code != null &&
                            sd.domain.Status.code.ToLower().Contains("activ");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #endregion

        #region Person private methods

        private void DeletePerson(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EntityClient.InactivateEntity(id.root, EISConstants.PERSON, id.extension);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Cache

        public void ClearCache()
        {
            _cache.FlushCache();
        }

        public void FillCache()
        {
            _cache.FillCache();
        }

        public void CacheBed(II id)
        {
            _cache.CacheBed(id);
        }

        public void CacheWard(II id)
        {
            _cache.CacheWard(id);
        }

        public void CacheDivision(II id)
        {
            _cache.CacheDivision(id);
        }

        public void CacheFacility(II id)
        {
            _cache.CacheFacility(id);
        }

        public void CacheVistaSite(II id)
        {
            _cache.CacheVistaSite(id);
        }

        public void CacheVisn(II id)
        {
            _cache.CacheVisn(id);
        }

        public void CacheRegion(II id)
        {
            _cache.CacheRegion(id);
        }

        public void CacheHospitalLocation(II id)
        {
            _cache.CacheHospitalLocation(id);
        }

        public void RemoveCacheBed(II id)
        {
            _cache.RemoveCacheBed(id);
        }

        public void RemoveCacheWard(II id)
        {
            _cache.RemoveCacheWard(id);
        }

        public void RemoveCacheDivision(II id)
        {
            _cache.RemoveCacheDivision(id);
        }

        public void RemoveCacheHospitalLocation(II id)
        {
            _cache.RemoveCacheHospitalLocation(id);
        }

        #endregion

        #region Common functions

        private void CreatePlace(Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = new Entity()
                {
                    DomainIdentifier = SecurityWrapper.GetCurrentDomain(),
                    EntityTypeIdentifier = place.GetEntityType()
                };
                place.TranslateToEntity(entity);
                entity.Identifier = EntityClient.CreateEntity(entity);
                place.Id = Translator.ExtractIdentifier(entity);

                if (place is Bed)
                {
                    Bed bed = place as Bed;
                    if (bed != null && bed.WardList != null && bed.WardList.Count > 0)
                    {
                        foreach (Ward ward in bed.WardList)
                        {
                            if (ward != null && ward.Id != null)
                            {
                                entity.TraitList.Add(new IdentifierValuePair()
                                {
                                    TraitIdentifier = place.GetParentRoleType(),
                                    TraitValue = place.GetParentRole(ward.Id)
                                });

                                //check if child has grand father
                                Entity addedEntityParent = GetEntityById(new Ward() { Id = ward.Id });

                                List<II> addGrandFatherIds = TraitManager.FindScopers(addedEntityParent, ward.GetParentRoleType());

                                if (addGrandFatherIds != null)
                                {
                                    foreach (II addGrandFatherId in addGrandFatherIds)
                                    {
                                        Place addGrandFather = GetDivision(addGrandFatherId);
                                        if (addGrandFather != null)
                                        {
                                            IdentifierValuePair ivpAdd = CreateParentRole(place, addGrandFather);
                                            if (ivpAdd != null && addGrandFatherId != null)
                                            {
                                                if (entity.TraitList.Where(a => (a.TraitValue as R_Base) != null && (a.TraitValue as R_Base).Scoper.Identifier == addGrandFatherId.extension).FirstOrDefault() == null)
                                                    entity.TraitList.Add(ivpAdd);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    EntityClient.UpdateTraitValues(entity);
                }
                else
                {
                    if (place.GetParent() != null && place.GetParent().Id != null)
                    {
                        entity.TraitList.Add(new IdentifierValuePair()
                        {
                            TraitIdentifier = place.GetParentRoleType(),
                            TraitValue = place.GetParentRole(place.GetParent().Id)
                        });
                        EntityClient.UpdateTraitValues(entity);
                    }
                    // associate the ward to the vista site
                    if (place is Ward)
                    {
                        Ward ward = place as Ward;
                        if (ward != null && ward.VistaSite != null && ward.VistaSite.Id != null)
                        {
                            entity.TraitList.Add(CreateParentRole(place, ward.VistaSite));
                            EntityClient.UpdateTraitValues(entity);
                        }
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IdentifierValuePair CreateParentRole(Place child, Place parent)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair ivp = new IdentifierValuePair();
                R_Base role = GetRole(child);
                role.Scoper = new EntityInRole()
                {
                    EntityTypeIdentifier = parent.GetParentEntityType(),
                    DomainIdentifier = parent.Id.root,
                    Identifier = parent.Id.extension
                };

                role.Player = new EntityInRole()
                {
                    EntityTypeIdentifier = child.GetEntityType(),
                    DomainIdentifier = child.Id.root,
                    Identifier = child.Id.extension
                };
                ivp.TraitIdentifier = GetIdentifierForRole(child);
                ivp.TraitValue = role;
                return ivp;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IdentifierValuePair CreateParentRole(Place place, VistaSite vistaSite)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair ivp = new IdentifierValuePair();
                R_Base role = GetRole(place);

                role.Scoper = new EntityInRole()
                {
                    EntityTypeIdentifier = vistaSite.GetParentEntityType(),
                    DomainIdentifier = vistaSite.Id.root,
                    Identifier = vistaSite.Id.extension
                };

                role.Player = new EntityInRole()
                {
                    EntityTypeIdentifier = place.GetEntityType(),
                    DomainIdentifier = place.Id.root,
                    Identifier = place.Id.extension
                };
                ivp.TraitIdentifier = GetIdentifierForRole(place);
                ivp.TraitValue = role;
                return ivp;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static R_Base GetRole(Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Bed)
                    return new R_MaintainedEntity() { };
                if (place is Ward)
                    return new R_OwnedEntity() { };

                return new R_ServiceDeliveryLocation() { Code = TraitManager.CreateRoleCode(EISConstants.WARD_GROUP_CODE, EISConstants.WARD_GROUP_NAME) };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static string GetIdentifierForRole(Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Bed)
                    return EISConstants.MNT;
                if (place is Ward)
                    return EISConstants.OWN;

                return EISConstants.SDLOC;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void InactivatePlace(Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = new Entity()
                {
                    Identifier = place.Id.extension,
                    DomainIdentifier = place.Id.root,
                    EntityTypeIdentifier = place.GetEntityType()
                };
                SetEntityStatus(entity, false);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Place> GetPlaces(Place metadata, TraitList traits)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string lastIndexes = null, lastIndexesResult = null;
                List<Entity> entities = new List<Entity>();
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTraitWithPaging(
                        traits,
                        metadata.GetOutputTraits(),
                        EISConstants.EIS_DOMAIN_ID_LIST,
                        new List<string>() { metadata.GetEntityType() }, lastIndexes, _pageCount, out lastIndexesResult);
                }
                finally { eisSemaphore.Release(); }
                if (entities.Count >= _pageCount)
                {
                    while (lastIndexesResult != null && !lastIndexesResult.Equals("0,0") && entities.Count % _pageCount == 0)
                    {
                        lastIndexes = lastIndexesResult;
                        try
                        {
                            eisSemaphore.WaitOne();
                            entities.AddRange(QueryClient.FindEntitiesByTraitWithPaging(
                                traits,
                                metadata.GetOutputTraits(),
                                EISConstants.EIS_DOMAIN_ID_LIST,
                                new List<string>() { metadata.GetEntityType() }, lastIndexes, _pageCount, out lastIndexesResult));
                        }
                        finally { eisSemaphore.Release(); }
                    }
                }
                return entities.Select(entity => metadata.TranslateFromEntity(entity)).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Place> GetPlaces(Place metadata, II parentId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetPlaces(metadata, parentId, null);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<Place> GetPlaces(Place metadata, II parentId, IList<TraitPair> traits)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList input = new TraitList()
                { 
                    TraitManager.CreateTraitPair(metadata.GetParentRoleType(), metadata.GetParentRole(parentId) ),
                    TraitManager.CreateTraitPairForPlaceType(metadata.GetPlaceType())
                };
                if (traits != null)
                    input.AddRange(traits);

                string lastIndexes = null, lastIndexesResult = null;
                List<Entity> entities = new List<Entity>();
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTraitWithPaging(
                        input,
                        metadata.GetOutputTraits(),
                        EISConstants.EIS_DOMAIN_ID_LIST,
                        new List<string>() { metadata.GetEntityType() }, lastIndexes, _pageCount, out lastIndexesResult);
                }
                finally { eisSemaphore.Release();}
                if (entities.Count >= _pageCount)
                {
                    while (lastIndexesResult != null && !lastIndexesResult.Equals("0,0") && entities.Count % _pageCount == 0)
                    {
                        lastIndexes = lastIndexesResult;
                        try
                        {
                            eisSemaphore.WaitOne();
                            entities.AddRange(QueryClient.FindEntitiesByTraitWithPaging(
                                input,
                                metadata.GetOutputTraits(),
                                EISConstants.EIS_DOMAIN_ID_LIST,
                                new List<string>() { metadata.GetEntityType() }, lastIndexes, _pageCount, out lastIndexesResult));
                        }
                        finally { eisSemaphore.Release(); }
                    }
                }
                return entities.Select(entity => metadata.TranslateFromEntity(entity)).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Place> GetPlaces(Place metadata)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetBasePlaces(metadata, QueryClient);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<Place> GetPlacesUsingWindowsAuthentication(Place metadata)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetBasePlaces(metadata, EISFactory.QueryFunctionsClientWindows);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<Place> GetBasePlaces(Place metadata, BMS.ServicesWrapper.Proxy.EIS.QueryFunctionsClient proxy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                //if (proxy == null)
                //    throw new ArgumentNullException("proxy");

                TraitList input = new TraitList()
                { 
                    TraitManager.CreateTraitPairForPlaceType(metadata.GetPlaceType())
                };
                string lastIndexes = null, lastIndexesResult = null;
                List<Entity> entities = new List<Entity>();
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTraitWithPaging(
                        input,
                        metadata.GetOutputTraits(),
                        EISConstants.EIS_DOMAIN_ID_LIST,
                        new List<string>() { metadata.GetEntityType() }, lastIndexes, _pageCount, out lastIndexesResult);
                }
                finally { eisSemaphore.Release(); }
                if (entities.Count >= _pageCount)
                {
                    while (lastIndexesResult != null && !lastIndexesResult.Equals("0,0") && entities.Count % _pageCount == 0)
                    {
                        lastIndexes = lastIndexesResult;
                        try
                        {
                            eisSemaphore.WaitOne();
                            entities.AddRange(QueryClient.FindEntitiesByTraitWithPaging(
                                input,
                                metadata.GetOutputTraits(),
                                EISConstants.EIS_DOMAIN_ID_LIST,
                                new List<string>() { metadata.GetEntityType() }, lastIndexes, _pageCount, out lastIndexesResult));
                        }
                        finally { eisSemaphore.Release(); }
                    }
                }
                return entities.Select(entity => metadata.TranslateFromEntity(entity)).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        private void DeleteParentRole(Place child, Place parent)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // retrieve the existing entity
                Entity entity = GetEntityById(child);
                // create an entity with all required traits
                Entity newEntity = CopyEntityId(entity);
                child.TranslateToEntity(newEntity);

                // remove the previous association
                if (parent != null && parent.Id != null)
                {

                    IdentifierValuePair ivp = CreateParentRole(child, parent);
                    ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                    newEntity.TraitList.Add(ivp);
                    // finally, update the entity in EIS
                    EntityClient.UpdateTraitValues(newEntity);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void UpdateParentRole(Place child, Place oldParent, Place newParent)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Entity entity = GetEntityById(child);
                if (child.Id == null)
                    child.Id = Translator.ExtractIdentifier(entity);

                II oldParentId = oldParent == null ? null : oldParent.Id == null ? null : oldParent.Id;
                II newParentId = newParent.Id == null ? null : newParent.Id;

                if (!TraitManager.Equals(oldParentId, newParentId))
                {
                    Entity newEntity = CopyEntityId(entity);
                    // create an entity with all required traits
                    child.TranslateToEntity(newEntity);

                    if (newParentId != null)
                    {
                        // associate the child to the parent
                        newEntity.TraitList.Add(CreateParentRole(child, newParent));
                    }

                    if (oldParentId != null)
                    {
                        IdentifierValuePair ivp = CreateParentRole(child, oldParent);
                        ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                        newEntity.TraitList.Add(ivp);
                    }

                    // finally, update the entity in EIS
                    EntityClient.UpdateTraitValues(newEntity);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void UpdatePlace(Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                // retrieve the existing entity
                Entity oldEntity = GetEntityById(place);
                if (place.Id == null)
                    place.Id = Translator.ExtractIdentifier(oldEntity);

                // create an entity with all required traits
                Entity newEntity = CopyEntityId(oldEntity);
                place.TranslateToEntity(newEntity);

                if (place is Bed)
                {
                    // keep the previous parents
                    List<II> oldParentsId = TraitManager.FindScopers(oldEntity, place.GetParentRoleType());
                    List<II> oldGrandFathersId = new List<II>();
                    IdentifierValuePair roleMntIvp = null;
                    R_MaintainedEntity roleMnt = null;
                    if (oldParentsId != null)
                    {
                        foreach (II id in oldParentsId)
                        {
                            Ward ward = GetWard(id);
                            if (ward != null && ward.DivisionList != null)
                                oldGrandFathersId.AddRange(ward.DivisionList.Select(a => a.Id));
                        }
                    }

                    //get grandfathers(divisions) for updating comments(role NameFamily field)
                    Bed bed = place as Bed;
                    if (bed != null) { 
                        if (bed.CommentList != null)
                        {
                            foreach (BedComment bc in bed.CommentList)
                            {
                                roleMnt = new R_MaintainedEntity();
                                roleMnt.Player = new EntityInRole() { EntityTypeIdentifier = EISConstants.PLACE, DomainIdentifier = place.Id.root, Identifier = place.Id.extension };
                                roleMnt.Scoper = new EntityInRole() { EntityTypeIdentifier = EISConstants.ORGANIZATON, DomainIdentifier = bc.Division.Id.root, Identifier = bc.Division.Id.extension };
                                roleMnt.NameFamily = new ST(bc.Comment);
                                roleMntIvp = new IdentifierValuePair()
                                {
                                    TraitIdentifier = EISConstants.MNT,
                                    TraitValue = roleMnt
                                };
                                roleMntIvp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.UPDATE;
                                newEntity.TraitList.Add(roleMntIvp);
                            }
                        }
                    
                        //get new parents
                        List<Place> parents = bed.WardList.Cast<Place>().ToList();
                        if (parents != null && parents.Count > 0)
                        {
                            IEnumerable<Place> newParents = null;
                            if (oldParentsId != null)
                                newParents = (from p in parents
                                              where !(from id in oldParentsId
                                                      select id.extension).Contains(p.Id.extension)
                                              select p);
                            else
                                newParents = parents;
                            foreach (Place newParent in newParents)
                            {
                                // associate the child to the parent
                                if (newParent.Id != null)
                                {
                                    IdentifierValuePair ivp = new IdentifierValuePair()
                                    {
                                        TraitIdentifier = place.GetParentRoleType(),
                                        TraitValue = place.GetParentRole(newParent.Id)
                                    };
                                    newEntity.TraitList.Add(ivp);
                                }
                                //check if child has grand father
                                Entity addedEntityParent = GetEntityById(new Ward() { Id = newParent.Id });
                                List<II> addGrandFatherIds = TraitManager.FindScopers(addedEntityParent, newParent.GetParentRoleType());
                                if (addGrandFatherIds != null)
                                {
                                    foreach (II addGrandFatherId in addGrandFatherIds)
                                    {
                                        Place newGrandFather = GetDivision(addGrandFatherId);
                                        // associate the child to the grand father
                                        IdentifierValuePair ivp = CreateParentRole(place, newGrandFather);
                                        //add comment to the bed-division role as NameFamily field
                                        R_MaintainedEntity role = ivp.TraitValue as R_MaintainedEntity;
                                        BedComment bc = bed.CommentList.Where(a => a.Division.Id.extension.Equals(addGrandFatherId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                        if (role != null)
                                        {
                                            role.NameFamily = (bc != null) ? new ST(bc.Comment) : null;
                                        }
                                        ivp.TraitValue = role;
                                        if (newEntity.TraitList.Where(a => (a.TraitValue as R_Base) != null && (a.TraitValue as R_Base).Scoper.Identifier == addGrandFatherId.extension).FirstOrDefault() == null)
                                            {
                                                IdentifierValuePair ivpp = null;
                                                ivpp = CreateParentRole(place, newGrandFather);
                                                if (ivpp != null)
                                                {
                                                    newEntity.TraitList.Add(ivpp);
                                                }
                                            }
                                        
                                    }
                                }
                            }
                        }


                        IEnumerable<II> deletedParents = null;
                        if (oldParentsId != null && oldParentsId.Count > 0)
                        {
                            if (parents != null && parents.Count > 0)
                                deletedParents = (from p in oldParentsId
                                                  where !(from id in parents
                                                          select id.Id.extension).Contains(p.extension)
                                                  select p);
                            else
                                deletedParents = oldParentsId;
                        }
                        if (deletedParents != null)
                        {
                            List<string> addGrandFatherIds = new List<string>();
                            List<II> temp = null;
                            foreach (II deletedId in deletedParents)
                            {
                                // remove the association between child and parent          
                                IdentifierValuePair ivp = new IdentifierValuePair()
                                {
                                    TraitIdentifier = place.GetParentRoleType(),
                                    TraitValue = place.GetParentRole(deletedId)
                                };

                                ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                                newEntity.TraitList.Add(ivp);

                                //check if child has grand father
                                Entity deletedEntityParent = GetEntityById(new Ward() { Id = deletedId });
                                Place deletedParent = GetWard(deletedId);

                                temp = TraitManager.FindScopers(deletedEntityParent, deletedParent.GetParentRoleType());
                                if (temp != null)
                                    addGrandFatherIds.AddRange(temp.Select(a => a.extension));
                            }
                            if (addGrandFatherIds != null && addGrandFatherIds.Count > 0)
                            {
                                addGrandFatherIds = addGrandFatherIds.Distinct(StringComparer.InvariantCultureIgnoreCase).ToList();
                                foreach (string addGrandFatherId in addGrandFatherIds)
                                {
                                    Place deletedGrandFather = GetDivision(new II(place.Id.root, addGrandFatherId));
                                    if (oldGrandFathersId.Where(a => a.extension.Equals(addGrandFatherId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                                    {
                                        IdentifierValuePair ivpDeleted = CreateParentRole(place, deletedGrandFather);
                                        ivpDeleted.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                                        newEntity.TraitList.Add(ivpDeleted);
                                    }
                                }
                            }
                        }
                    }
                }
                else
                {
                    if (!(place is Ward))
                    {
                        // keep the previous parent
                        II oldParentId = TraitManager.FindScoper(oldEntity, place.GetParentRoleType());


                        // check if the parent has changed. 
                        // if so, remove the previous association and create the new one
                        Place newParent = place.GetParent();
                        II newParentId = newParent == null ? null : newParent.Id;
                        if (!TraitManager.Equals(oldParentId, newParentId))
                        {
                            // associate the child to the parent
                            if (newParentId != null)
                            {
                                IdentifierValuePair ivp = new IdentifierValuePair()
                                {
                                    TraitIdentifier = place.GetParentRoleType(),
                                    TraitValue = place.GetParentRole(newParentId)
                                };

                                newEntity.TraitList.Add(ivp);
                            }

                            // remove the previous association
                            if (oldParentId != null)
                            {
                                IdentifierValuePair ivp = new IdentifierValuePair()
                                {
                                    TraitIdentifier = place.GetParentRoleType(),
                                    TraitValue = place.GetParentRole(oldParentId)
                                };

                                ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                                newEntity.TraitList.Add(ivp);
                            }
                        }
                    }
                }

                // finally, update the entity in EIS
                EntityClient.UpdateTraitValues(newEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private Entity CopyEntityId(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new Entity() { EntityTypeIdentifier = entity.EntityTypeIdentifier, DomainIdentifier = entity.DomainIdentifier, Identifier = entity.Identifier };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private Entity GetEntityByCode(Place place, string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList input = new TraitList();
                input.Add(TraitManager.CreateTraitPairForCode(code));
                input.Add(TraitManager.CreateTraitPairForPlaceType(place.GetPlaceType()));
                List<Entity> entities = new List<Entity>();
                try
                {
                    eisSemaphore.WaitOne();
                    entities = QueryClient.FindEntitiesByTrait(input, place.GetOutputTraits(), EISConstants.EIS_DOMAIN_ID_LIST, new List<string>() { place.GetEntityType() });
                }
                finally { eisSemaphore.Release(); }
                return entities.Count == 0 ? null : entities[0];
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Entity GetEntityById(Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return QueryClient.GetInformationForEntity(place.Id.root, place.GetEntityType(), place.Id.extension);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private Entity GetEntityById(Person person)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return QueryClient.GetInformationForEntity(person.Id.root, EISConstants.EIS_PERSON_TYPE, person.Id.extension);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Activates/inactivates the entity from facility.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <param name="active">The new status.</param>
        private void SetEntityStatus(Entity entity, bool active)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                entity.TraitList.Clear();
                entity.TraitList.Add(new IdentifierValuePair()
                {
                    TraitIdentifier = EISConstants.IS_ACTIVE,
                    TraitValue = new ST() { text = active ? EISConstants.TRUE : EISConstants.FALSE }
                });

                EntityClient.UpdateTraitValues(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }

    static class EISExtensions
    {
        public static readonly List<string> MEDICAL_OUTPUT_TRAITS = new List<string>() { EISConstants.UID, EISConstants.GENDER, EISConstants.PROV, EISConstants.NAME_FAMILY, EISConstants.NAME_FIRST_GIVEN, EISConstants.NAME_SECOND_GIVEN, };

        public static readonly List<string> PATIENT_OUTPUT_TRAITS = new List<string>() { EISConstants.UID, EISConstants.GENDER, EISConstants.PAT, EISConstants.TREATING_SPECIALTY, 
            EISConstants.NAME_FAMILY, EISConstants.NAME_FIRST_GIVEN, EISConstants.NAME_SECOND_GIVEN, EISConstants.IS_VISTA_VALIDATED, EISConstants.ADMITTING_DIAGNOSIS , EISConstants.SERVICE_CONNECTED_PERCENTAGE, 
            EISConstants.DATE_OF_BIRTH, EISConstants.ATTENDING_PHYSICIAN, EISConstants.IEN_LIST, EISConstants.PERSON_WEIGHT };

        public static readonly List<string> VISN_OUTPUT_TRAITS = new List<string>() {  EISConstants.NAME, EISConstants.UID, 
            EISConstants.CODE, EISConstants.MBR, EISConstants.IS_ACTIVE, EISConstants.MBR };

        public static readonly List<string> VISTA_SITE_OUTPUT_TRAITS = new List<string>() {  EISConstants.NAME, EISConstants.UID, 
            EISConstants.CODE, EISConstants.MBR, EISConstants.IS_ACTIVE };

        public static readonly List<string> REGION_OUTPUT_TRAITS = new List<string>() { EISConstants.NAME, EISConstants.UID, 
            EISConstants.CODE, EISConstants.IS_ACTIVE };

        public static readonly List<string> FACILITY_OUTPUT_TRAITS = new List<string>() { EISConstants.MBR, EISConstants.CODE, 
            EISConstants.NAME, EISConstants.UID, EISConstants.IS_ACTIVE, EISConstants.ADDRESS1, EISConstants.PHONE, EISConstants.ADDRESS2, EISConstants.IS_ALIVE, EISConstants.PERSON_CONTACT_NAME, EISConstants.EMAIL, EISConstants.TELEPHONE, EISConstants.SITE_NUMBER };

        public static readonly List<string> HOSPITAL_LOCATION_OUTPUT_TRAITS = new List<string>() { EISConstants.MBR, EISConstants.CODE, 
            EISConstants.NAME, EISConstants.UID, EISConstants.IS_ACTIVE, EISConstants.PHONE, EISConstants.WARD_LOCATION_ID };

        public static readonly List<string> DIVISION_OUTPUT_TRAITS = new List<string>() { EISConstants.MBR, EISConstants.CODE, 
            EISConstants.NAME, EISConstants.UID, EISConstants.IS_ACTIVE, EISConstants.WHITEBOARD_REPORT_PATH };

        public static readonly List<string> WARD_OUTPUT_TRAITS = new List<string>() { EISConstants.LOCE, EISConstants.SDLOC, EISConstants.CODE, EISConstants.OWN, 
            EISConstants.NAME, EISConstants.UID, EISConstants.VISTASPECIALTY, EISConstants.IS_ACTIVE, EISConstants.TYPEGROUP, EISConstants.IS_CURRENTLY_OOS };

        public static readonly List<string> BED_OUTPUT_TRAITS = new List<string>() { EISConstants.LOCE, EISConstants.MNT, EISConstants.CODE, 
            EISConstants.NAME, EISConstants.UID, EISConstants.DESCRIPTION,
            EISConstants.NURSE_ASSIGNMENT, EISConstants.COMMENT, EISConstants.REASON, EISConstants.BED_STATUS, EISConstants.IS_BED_HOLD };


        public static Place GetParent(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return null;
                if (place is Visn)
                {
                    Visn visn = place as Visn;
                    if (visn != null)
                    {
                        return visn.Region;
                    }
                }
                if (place is VistaSite)
                {
                    VistaSite vistaSite = place as VistaSite;
                    if (vistaSite != null)
                    {
                        return vistaSite.Visn;
                    }
                }
                if (place is Facility)
                {
                    Facility facility = place as Facility;
                    if (facility != null)
                    {
                        return facility.VistaSite;
                    }
                }
                if (place is HospitalLocation)
                {
                    HospitalLocation hospitalLocation = place as HospitalLocation;
                    if (hospitalLocation != null)
                    {
                        return hospitalLocation.VistaSite;
                    }
                }
                if (place is Division)
                {
                    Division division = place as Division;
                    if (division != null)
                    {
                        return division.Facility;
                    }
                }
                if (place is Ward)
                {
                    Ward ward = place as Ward;
                    if (ward != null)
                    {
                        return ward.Division;
                    }
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static List<string> GetOutputTraits(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return REGION_OUTPUT_TRAITS;
                if (place is Visn)
                    return VISN_OUTPUT_TRAITS;
                if (place is VistaSite)
                    return VISTA_SITE_OUTPUT_TRAITS;
                if (place is HospitalLocation)
                    return HOSPITAL_LOCATION_OUTPUT_TRAITS;
                if (place is Facility)
                    return FACILITY_OUTPUT_TRAITS;
                if (place is Division)
                    return DIVISION_OUTPUT_TRAITS;
                if (place is Ward)
                    return WARD_OUTPUT_TRAITS;
                if (place is Bed)
                    return BED_OUTPUT_TRAITS;
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetEntityType(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return EISConstants.ORGANIZATON;
                if (place is Visn)
                    return EISConstants.ORGANIZATON;
                if (place is VistaSite)
                    return EISConstants.ORGANIZATON;
                if (place is HospitalLocation)
                    return EISConstants.ORGANIZATON;
                if (place is Facility)
                    return EISConstants.ORGANIZATON;
                if (place is Division)
                    return EISConstants.ORGANIZATON;
                if (place is Ward)
                    return EISConstants.PLACE;
                if (place is Bed)
                    return EISConstants.PLACE;
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetParentEntityType(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return EISConstants.ORGANIZATON;
                if (place is Visn)
                    return EISConstants.ORGANIZATON;
                if (place is VistaSite)
                    return EISConstants.ORGANIZATON;
                if (place is HospitalLocation)
                    return EISConstants.ORGANIZATON;
                if (place is Facility)
                    return EISConstants.ORGANIZATON;
                if (place is Division)
                    return EISConstants.ORGANIZATON;
                if (place is Ward)
                    return EISConstants.ORGANIZATON;
                if (place is Bed)
                    return EISConstants.PLACE;
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetPlaceType(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return EISConstants.IDENTIFIER_FOR_REGION_CODE;
                if (place is Visn)
                    return EISConstants.IDENTIFIER_FOR_VISN_CODE;
                if (place is VistaSite)
                    return EISConstants.IDENTIFIER_FOR_VISTA_SITE_CODE;
                if (place is HospitalLocation)
                    return EISConstants.IDENTIFIER_FOR_HOSPITAL_LOCATION_CODE;
                if (place is Facility)
                    return EISConstants.IDENTIFIER_FOR_FACILITY_CODE;
                if (place is Division)
                    return EISConstants.IDENTIFIER_FOR_WARDGROUP_CODE;
                if (place is Ward)
                    return EISConstants.IDENTIFIER_FOR_WARD_CODE;
                if (place is Bed)
                    return EISConstants.IDENTIFIER_FOR_BED_CODE;
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetParentRoleType(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return null;
                if (place is Visn)
                    return EISConstants.MBR;
                if (place is VistaSite)
                    return EISConstants.MBR;
                if (place is HospitalLocation)
                    return EISConstants.MBR;
                if (place is Facility)
                    return EISConstants.MBR;
                if (place is Division)
                    return EISConstants.MBR;
                if (place is Ward)
                    return EISConstants.SDLOC;
                if (place is Bed)
                    return EISConstants.LOCE;
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static R_Base GetParentRole(this Place place, II parentId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                R_Base role = null;
                if (place is Region)
                    return null;
                if (place is Visn)
                    role = new R_Member() { Code = TraitManager.CreateRoleCode(EISConstants.VISN_CODE, EISConstants.VISN_NAME) };
                else if (place is VistaSite)
                    role = new R_Member() { Code = TraitManager.CreateRoleCode(EISConstants.VISTA_SITE_CODE, EISConstants.VISTA_SITE_NAME) };
                else if (place is HospitalLocation)
                    role = new R_Member() { Code = TraitManager.CreateRoleCode(EISConstants.HOSPITAL_LOCATION_CODE, EISConstants.HOSPITAL_LOCATION_NAME) };
                else if (place is Facility)
                    role = new R_Member() { Code = TraitManager.CreateRoleCode(EISConstants.FACILITY_CODE, EISConstants.FACILITY_NAME) };
                else if (place is Division)
                    role = new R_Member() { Code = TraitManager.CreateRoleCode(EISConstants.WARD_GROUP_CODE, EISConstants.WARD_GROUP_NAME) };
                else if (place is Ward)
                    role = new R_ServiceDeliveryLocation() { Code = TraitManager.CreateRoleCode(EISConstants.WARD_CODE, EISConstants.WARD_NAME) };
                else if (place is Bed)
                    role = new R_LocatedEntity();
                else
                    return null;

                if (parentId != null)
                    role.Scoper = new EntityInRole()
                    {
                        EntityTypeIdentifier = place.GetParentEntityType(),
                        DomainIdentifier = parentId.root,
                        Identifier = parentId.extension
                    };
                if (place.Id != null)
                    role.Player = new EntityInRole()
                    {
                        EntityTypeIdentifier = place.GetEntityType(),
                        DomainIdentifier = place.Id.root,
                        Identifier = place.Id.extension
                    };

                return role;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static EntityInRole GetEntityInRole(this Place place)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place.Id == null || place.Id.extension == null)
                    return null;
                return new EntityInRole()
                {
                    DomainIdentifier = place.Id.root,
                    Identifier = place.Id.extension,
                    EntityTypeIdentifier = place.GetEntityType()
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void TranslateToEntity(this Place place, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    Translator.TranslateToEntity(place as Region, entity);
                if (place is Visn)
                    Translator.TranslateToEntity(place as Visn, entity);
                if (place is VistaSite)
                    Translator.TranslateToEntity(place as VistaSite, entity);
                if (place is HospitalLocation)
                    Translator.TranslateToEntity(place as HospitalLocation, entity);
                if (place is Facility)
                    Translator.TranslateToEntity(place as Facility, entity);
                if (place is Division)
                    Translator.TranslateToEntity(place as Division, entity);
                if (place is Ward)
                    Translator.TranslateToEntity(place as Ward, entity);
                if (place is Bed)
                    Translator.TranslateToEntity(place as Bed, entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static Place TranslateFromEntity(this Place place, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (place is Region)
                    return Translator.TranslateRegion(entity);
                if (place is Visn)
                    return Translator.TranslateVisn(entity);
                if (place is VistaSite)
                    return Translator.TranslateVistaSite(entity);
                if (place is HospitalLocation)
                    return Translator.TranslateHospitalLocation(entity);
                if (place is Facility)
                    return Translator.TranslateFacility(entity);
                if (place is Division)
                    return Translator.TranslateDivision(entity);
                if (place is Ward)
                    return Translator.TranslateWard(entity);
                if (place is Bed)
                    return Translator.TranslateBed(entity);
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static EntityInRole GetEntityInRole(this Person person)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (person.Id == null || person.Id.extension == null)
                    return null;
                return new EntityInRole()
                {
                    DomainIdentifier = person.Id.root,
                    Identifier = person.Id.extension,
                    EntityTypeIdentifier = EISConstants.PERSON
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
