﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.Utils;
using FC = BMS.Facade.Data;
using SC = BMS.DataContracts;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.WF;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.ServicesWrapper.BMService;
using BMS.FaultContracts;

namespace BMS.VistaWorker2.Writer.Implementation.Concrete.WF
{
    /// <summary>
    /// The patient movement writer.
    /// </summary>
    class PatientMovementWriter : BaseWorkFlowWriter<PatientMovement>
    {
        /// <summary>
        /// Sends the event.
        /// </summary>
        /// <param name="entity">The entity.</param>
        protected override bool SendEvent(PatientMovement entity)
        {            
            decimal transaction = 0;
            decimal.TryParse(entity.TransactionTypeId, out transaction);

            if (transaction > 0)
            {
                string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                CD movementType = InsertIfNullOrUpdateIfDirty<FacilityMovementType, CD>(entity.TypeOfMovement);
                FC.Ward ward = InsertIfNullOrUpdateIfDirty<WardLocation, FC.Ward>(entity.Ward);
                FC.Bed bed = InsertIfNullOrUpdateIfDirty<RoomBed, FC.Bed>(entity.Bed);                

                //admission
                if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MAX)
                {
                    SC.AdmissionEvent wfAdmission = new SC.AdmissionEvent()
                    {
                        Id = new II(domainId, null),
                        PatientId = MakePatient(entity.Patient),
                        EnteredDate = entity.EnteredOnDateTime.HasValue ? entity.EnteredOnDateTime.Value : DateTime.Now,
                        Ien = entity.IEN,
                        BedId = (bed != null) ? bed.Id : null,
                        WardId = (ward != null) ? ward.Id : null,
                        EnteredById = MakePerson(entity.EnteredBy),
                        AdmissionDate = entity.DateTime,
                        MovementType = movementType,
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null                        
                    };
                    ConvertToUtc(wfAdmission);
                    BMSFactory.BedManagerOperationsClientWindows.CreateAdmissionEvent(wfAdmission);
                    Utils.AutoRemoveWaitingListItem(wfAdmission.PatientId);
                    Logger.LogFormat(BmsLogger.Level.Info, "Create admission event for patient {0} with succes", entity.Patient.Name);                    
                    return true;
                }

                //transfer
                if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MAX)                    
                {                    
                    SC.MovementEvent wfMovement = new SC.MovementEvent()
                    {
                        Id = new II(domainId, null),
                        PatientId = MakePatient(entity.Patient),                        
                        EnteredDate = entity.EnteredOnDateTime.HasValue ? entity.EnteredOnDateTime.Value : DateTime.Now,
                        Ien = entity.IEN,
                        BedId = (bed != null) ? bed.Id : null,
                        WardId = (ward != null) ? ward.Id : null,
                        EnteredById = MakePerson(entity.EnteredBy),
                        MovementDate = entity.DateTime,
                        MovementType = movementType,                        
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                        AdmissionIen = entity.CurrentAdmissionIen
                    };
                    ConvertToUtc(wfMovement);
                    SC.AdmissionEvent admission = null;
                    try
                    {
                        admission = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEventForPatient(null, wfMovement.VistaSiteId, wfMovement.AdmissionIen, null);
                    }
                    catch { }
                    if (admission == null)                        
                        throw new Exception("Associated admission with ien " + entity.CurrentAdmissionIen + " was not found.");                    
                    wfMovement.OldWardId = admission.WardId;
                    wfMovement.OldBedId = admission.BedId;                    
                    II movId = BMSFactory.BedManagerOperationsClientWindows.CreateMovementEvent(wfMovement);                    
                    if (!SameIdentifier(wfMovement.BedId, wfMovement.OldBedId))
                        BMSFactory.BedManagerOperationsClientFromWCF.DeleteIconAssociation(Guid.Parse(wfMovement.PatientId.extension));                    
                    if (movementType == null || (movementType != null && !movementType.displayName.ToUpper().Contains(Constants.UN_AUTHORIZED_ABSENCE) && !movementType.displayName.ToUpper().Contains(Constants.AUTHORIZED_ABSENCE)
                                && !movementType.displayName.ToUpper().Contains(Constants.FROM_ASIH) && !movementType.displayName.ToUpper().Contains(Constants.RESUME_ASIH)))
                    {
                        if (movId.extension.Equals("-2"))
                        {
                            // exists a movement with a greater date in the DB
                            if (wfMovement.BedId != null)
                                UtilsInstance.GenerateBedCleanRequest(wfMovement.Ien, wfMovement.BedId, wfMovement.WardId, wfMovement.VistaSiteId, VistATimeZone, wfMovement.EnteredDate, wfMovement.MovementDate.Value, "TRA");
                        }
                        else if (admission.BedId != null && !SameIdentifier(wfMovement.BedId, wfMovement.OldBedId))
                            UtilsInstance.GenerateBedCleanRequest(wfMovement.Ien, admission.BedId, admission.WardId, wfMovement.VistaSiteId, VistATimeZone, wfMovement.EnteredDate, wfMovement.MovementDate.Value, "TRA");
                    }
                    Logger.LogFormat(BmsLogger.Level.Info, "Create movement event for patient {0} with succes", entity.Patient.Name);                    
                    return true;
                }

                // specialty transfer
                if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MAX)
                {
                    SC.SpecialtyTransfer wfSpecialtyTransfer = new SC.SpecialtyTransfer()
                    {
                        Id = new II(domainId, null),
                        Ien = entity.IEN,
                        PatientId = MakePatient(entity.Patient),
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                        DateTime = entity.DateTime,
                        TypeOfMovement = movementType,
                        EnteredDate = entity.EnteredOnDateTime,
                        EnteredById = MakePerson(entity.EnteredBy),
                        AdmissionIen = entity.CurrentAdmissionIen
                    };
                    ConvertToUtc(wfSpecialtyTransfer);
                    BMSFactory.BedManagerOperationsClientWindows.CreateSpecialtyTransfer(wfSpecialtyTransfer);
                    Logger.LogFormat(BmsLogger.Level.Info, "Create specialty transfer event for patient {0} with succes", entity.Patient.Name);                    
                    return true;
                }

                //discharge
                if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MAX)
                {
                    SC.DischargeEvent wfDischarge = new SC.DischargeEvent()
                    {
                        Id = new II(domainId, null),
                        PatientId = MakePatient(entity.Patient),
                        EnteredDate = entity.EnteredOnDateTime.HasValue ? entity.EnteredOnDateTime.Value : DateTime.Now,
                        Ien = entity.IEN,
                        BedId = (bed != null) ? bed.Id : null,
                        WardId = (ward != null) ? ward.Id : null,
                        EnteredById = MakePerson(entity.EnteredBy),
                        DischargedDate = entity.DateTime,
                        MovementType = movementType,
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                        AdmissionIen = entity.CurrentAdmissionIen
                    };
                    ConvertToUtc(wfDischarge);
                    SC.AdmissionEvent admission = null;
                    try
                    {
                        admission = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEventForPatient(null, wfDischarge.VistaSiteId, wfDischarge.AdmissionIen, null);
                    }
                    catch { }
                    if (admission == null)
                        throw new Exception("Associated admission with ien " + entity.CurrentAdmissionIen + " was not found.");
                    wfDischarge.WardId = admission.WardId;
                    wfDischarge.BedId = admission.BedId;
                    BMSFactory.BedManagerOperationsClientWindows.CreateDischargeEvent(wfDischarge);
                    BMSFactory.BedManagerOperationsClientFromWCF.DeleteIconAssociation(Guid.Parse(wfDischarge.PatientId.extension));
                    Logger.LogFormat(BmsLogger.Level.Info, "Create discharge event for patient {0} with succes", entity.Patient.Name);
                    if (admission.BedId != null)
                    {
                        if (movementType == null || (movementType != null && !movementType.displayName.ToUpper().Contains(Constants.WHILE_ASIH)))
                            UtilsInstance.GenerateBedCleanRequest(wfDischarge.Ien, admission.BedId, admission.WardId, wfDischarge.VistaSiteId, VistATimeZone, wfDischarge.EnteredDate, wfDischarge.DischargedDate.Value, "DIS");
                    }                    
                    return true;
                }
            }
            return false;
        }

        protected override SC.VistaIntegrationLog MakeLog(PatientMovement entity)
        {
            return new SC.VistaIntegrationLog()
            {
                File = SC.VistaFiles.PatientMovement,
                Ien = entity.IEN
            };
        }

        private static bool SameIdentifier(II var1, II var2)
        {
            if (var1 == null && var2 == null) return true;
            if (var1 == null || var2 == null) return false;
            return var1.extension.Equals(var2.extension, StringComparison.InvariantCultureIgnoreCase) && var1.root.Equals(var2.root, StringComparison.InvariantCultureIgnoreCase);
        }
    }
}
