﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Exceptions;

namespace BMS.VistaWorker.Writer.Concrete.CatchExceptions
{
    /// <summary>
    /// Map different types of exceptions with AbortEntityProcessingException , RetryProcessingExeption and StopProcessingException.
    /// When an exception is caught  than the corresponding exception type is throw.
    /// </summary>
    abstract class BaseCatchException : ICatchExceptions
    {
        private Dictionary<Type, List<MapItem>> _dictionary = new Dictionary<Type, List<MapItem>>();

        private static readonly Func<Exception, string, Exception> RetryProcessingExeptionFunc = (e, msj) => new RetryProcessingException(msj, e);
        private static readonly Func<Exception, string, Exception> StopProcessingExeptionFunc = (e, msj) => new StopProcessingException(msj, e);
        private static readonly Func<Exception, string, Exception> AbortProcessingExeptionFunc = (e, msj) => new AbortEntityProcessingException(msj, e);
        private static readonly Func<Exception, bool> EmptyPredicate = (e) => true;

        protected virtual string Message { get { return string.Empty; } }

        protected void MapRetry<T>(Func<T, string> msjFunc = null) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Add<T>(RetryProcessingExeptionFunc, EmptyPredicate, CastFunc(msjFunc));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected void MapRetry<T>(Func<T, bool> predicate, Func<T, string> msjFunc = null) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Add<T>(RetryProcessingExeptionFunc, CastFunc(predicate), CastFunc(msjFunc));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected void MapStop<T>(Func<T, string> msjFunc = null) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Add<T>(StopProcessingExeptionFunc, EmptyPredicate, CastFunc(msjFunc));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected void MapStop<T>(Func<T, bool> predicate, Func<T, string> msjFunc = null) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Add<T>(StopProcessingExeptionFunc, CastFunc(predicate), CastFunc(msjFunc));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private Func<Exception, V> CastFunc<T, V>(Func<T, V> func) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (func == null)
                    return null;
                else
                    return (e) => func(e as T);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected void MapAbort<T>(Func<T, string> msjFunc = null) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Add<T>(AbortProcessingExeptionFunc, EmptyPredicate, CastFunc(msjFunc));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected void MapAbort<T>(Func<T, bool> predicate, Func<T, string> msjFunc = null) where T : Exception
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Add<T>(AbortProcessingExeptionFunc, CastFunc(predicate), CastFunc(msjFunc));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void Add<T>(Func<Exception, string, Exception> func, Func<Exception, bool> predicate, Func<Exception, string> msjFunc)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Type type = typeof(T);
                List<MapItem> list = null;
                if (!_dictionary.TryGetValue(type, out list))
                {
                    list = new List<MapItem>(1);
                    _dictionary[type] = list;
                }
                list.Add(new MapItem(msjFunc, predicate, func));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        public void RunAction(Action action)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    action();
                }
                catch (Exception e)
                {
                    if (!Resolve(e))
                        throw;

                    MapItem? mapItem = FindException(e.GetType(), e);
                    if (!mapItem.HasValue)
                        throw;
                    MapItem item = mapItem.Value;
                    string msj = item.MsjFunc != null ? item.MsjFunc(e) : Message;
                    Exception exception = item.Func(e, msj);
                    throw exception;

                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private bool Resolve(Exception e)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return !(
                    e is RetryProcessingException ||
                    e is AbortEntityProcessingException ||
                    e is StopProcessingException);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private MapItem? FindException(Type type, Exception e)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (type == null)
                    return null;
                List<MapItem> list = null;
                bool find = _dictionary.TryGetValue(type, out list);
                if (!find)
                    return FindException(type.BaseType, e);
                MapItem item = list.FirstOrDefault(i => i.Predicate(e));
                return item.Func != null ? item : FindException(type.BaseType, e);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private struct MapItem
        {
            public MapItem(Func<Exception, string> msjFunc, Func<Exception, bool> predicate, Func<Exception, string, Exception> func)
            {
                Predicate = predicate;
                Func = func;
                MsjFunc = msjFunc;
            }
            public readonly Func<Exception, string, Exception> Func;
            public readonly Func<Exception, bool> Predicate;
            public readonly Func<Exception, string> MsjFunc;
        }

    }
}
