﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Abstract;
using BMS.VistaWorker.Data;
using BMS.Facade.Data;
using BMS.ServicesWrapper.EIS;
using System.Configuration;
using BMS.Facade;
using BMS.Utils;
using BMS.ServicesWrapper.EVS;

namespace BMS.VistaWorker.Writer.Abstract
{
    /// <summary>
    /// The base writer manager implementation.
    /// </summary>
    public abstract class BaseWriterManager : IWriterManager
    {
        public VistaSite VistASite { get; private set; }

        private TimeZoneInfo VistATimeZoneInfo { get; set; }

        public BmsLogger Logger { get; private set; }

        private bool isInit = false;
        private bool isOpen = false;

        protected BaseWriterManager(string vistACode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VistASite = EISFactory.InstanceWindows.GetVistaSiteByCode(vistACode);
                if (VistASite == null)
                {
                    String msj = string.Format("No vistA site for {0}", vistACode);
                    throw new InvalidOperationException(msj);
                }
                string appendInfo = string.Format("VISTA {0} WRITER MESSAGE: ", VistASite.Name);
                Logger = new BmsLogger(appendInfo);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void InitVistATimeZoneInfo()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                VistASettings settings = Facade.FacadeManager.ConfigurationInterface.GetVistASettings(VistASite.Id);
                if (settings != null && settings.SiteTimeZone != null)
                {
                    TimeZoneInfo timeZoneInfo = TimeZoneInfo.FindSystemTimeZoneById(settings.SiteTimeZone.displayName);
                    if (settings.LocalTimeAdjust == 0)
                        VistATimeZoneInfo = timeZoneInfo;
                    else
                    {
                        TimeSpan utcOffset = timeZoneInfo.BaseUtcOffset;
                        utcOffset = utcOffset.Add(TimeSpan.FromHours(settings.LocalTimeAdjust));
                        VistATimeZoneInfo = TimeZoneInfo.CreateCustomTimeZone("CTZ", utcOffset, "Custom Time Zone", "CustomTimeZone");
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        class EntityWriterWrapper<T> : IEntityWriter<IEntity> where T : class , IEntity
        {
            private IEntityWriter<T> _entityWriter;

            public EntityWriterWrapper(IEntityWriter<T> entityWriter)
            {
                this._entityWriter = entityWriter;
            }

            public void Insert(IEntity entity)
            {
                T e = entity as T;
                if (e == null)
                    throw new InvalidCastException();
                _entityWriter.Insert(e);
            }

            public void Update(IEntity entity)
            {
                T e = entity as T;
                if (e == null)
                    throw new InvalidCastException();
                _entityWriter.Update(e);
            }

            public void Delete(IEntity entity)
            {
                T e = entity as T;
                if (e == null)
                    throw new InvalidCastException();
                _entityWriter.Delete(e);
            }



            public void InitData(VistaSite vistaSite, TimeZoneInfo vistATimeZone, BmsLogger logger)
            {
                throw new NotImplementedException();
            }
        }

        private Dictionary<Type, IEntityWriter<IEntity>> _entityWriters;

        protected abstract void CreateEntityWriters();

        /// <summary>
        /// Inserts the specified entity.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="entity">The entity.</param>
        public void Insert(IEntity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Init();
                _entityWriters[entity.GetType()].Insert(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the specified entity.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="entity">The entity.</param>
        public void Update(IEntity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Init();
                _entityWriters[entity.GetType()].Update(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the specified entity.
        /// </summary>
        /// <typeparam name="T">Entity type.</typeparam>
        /// <param name="entity">The entity.</param>
        public void Delete(IEntity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Init();
                _entityWriters[entity.GetType()].Delete(entity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Adds an entity writer to the internal dictionary.
        /// </summary>
        /// <typeparam name="T">Entity type.</typeparam>
        /// <param name="entityWriter">Specilized entity writer for the entity type.</param>
        protected void Add<T>(IEntityWriter<T> entityWriter) where T : class , IEntity
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                entityWriter.InitData(VistASite, VistATimeZoneInfo, Logger);
                _entityWriters[typeof(T)] = new EntityWriterWrapper<T>(entityWriter);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Opens the writer.
        /// </summary>
        public void Open()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (isOpen) return;
                isOpen = true;
                isInit = false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void Init()
        {
            if (!isOpen)
                throw new InvalidOperationException();

            if (isInit) return;
            EVSFactory.InstanceWindows.FlushCache();
            _entityWriters = new Dictionary<Type, IEntityWriter<IEntity>>();
            CreateEntityWriters();
            InitVistATimeZoneInfo();
            isInit = true;
        }

        /// <summary>
        /// Closes the writer.
        /// </summary>
        public void Close()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                isInit = false;
                isOpen = false;
                if (_entityWriters != null)
                {
                    _entityWriters.Clear();
                    _entityWriters = null;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }




    }
}
