﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Abstract;
using BMS.VistaWorker.Reader.Abstract;
using System.Timers;

namespace BMS.VistaWorker.Reader
{
    /// <summary>
    /// Wrapper on the reader manager that adds a custom behavior for timer.
    /// </summary>
    public class ReaderManagerTimer : IReaderManager
    {
        private readonly ReaderManager _readerManager;
        private readonly Timer _timer = new Timer();
        private bool _isClose = true;
        private readonly TimeSpan _delay;
        private readonly TimeSpan _period;
        private object _lockTimer = new object();
        private Controller _controller;

        /// <summary>
        /// Initializes a new instance of the <see cref="ReaderManagerTimer"/> class.
        /// </summary>
        /// <param name="readerManager">The reader manager.</param>
        /// <param name="delay">The delay.</param>
        /// <param name="period">The period.</param>
        public ReaderManagerTimer(ReaderManager readerManager, TimeSpan delay, TimeSpan period)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this._readerManager = readerManager;
                this._delay = delay;
                this._period = period;
                _timer.Elapsed += new ElapsedEventHandler(timer_Elapsed);
                _timer.AutoReset = false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void timer_Elapsed(object sender, ElapsedEventArgs e)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                lock (_lockTimer)
                {
                    bool startTimer = RunTimer();
                    if (startTimer)
                        StartTimer();
                    else
                        _controller.Stop();
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool RunTimer()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    if (_isClose)
                        return false;
                    OnStarted();
                    _readerManager.Start(_controller);
                    bool startTimerAgain = _readerManager.IsRunning;
                    _readerManager.Stop();
                    return startTimerAgain;
                }
                catch (Exception e)
                {
                    _controller.WriterManager.Logger.LogError("Unknown error: " + e.ToString());
                    return false;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void StartTimer()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _timer.Interval = _period.TotalMilliseconds;
                _timer.Start();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Starts de timer with a delay witch in turn starts when elapsed the reading process.
        /// </summary>
        /// <param name="controller">The controller.</param>
        /// <param name="writerManager">The writer session.</param>
        public void Start(Controller controller)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this._controller = controller;
                if (!_isClose)
                    throw new InvalidOperationException("vista already started");
                _isClose = false;
                _timer.Interval = _delay.TotalMilliseconds;
                _timer.Start();

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public event EventHandler Started;

        // Invoke the Changed event; called whenever list changes:
        protected virtual void OnStarted()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (Started != null)
                    Started(this, EventArgs.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Stops the timer and the reading process.
        /// </summary>
        public void Stop()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _readerManager.Stop();
                lock (_lockTimer)
                {
                    if (_isClose)
                        return;
                    _isClose = true;
                    _timer.Stop();
                    _timer.Dispose();
                    _controller = null;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }



    }
}
