﻿using System;

using System.Collections.Generic;
using System.Collections.ObjectModel;

using System.IdentityModel.Policy;
using System.IdentityModel.Tokens;

using System.IO;

using System.ServiceModel;
using System.ServiceModel.Security;
using System.ServiceModel.Security.Tokens;

using System.Text;

using System.Xml;
using BMS.Security.Configuration;

namespace BMS.Security.DurableIssuedToken
{
    /// <summary>
    /// Abstract class for how an IssuedToken should be cached like.
    /// </summary>
    public abstract class IssuedTokenCache
    {
        /// <summary>
        /// Addes the specific security token to the cache repository.
        /// </summary>
        /// <param name="token">Security token.</param>
        /// <param name="sessionId">Current token session Id.</param>
        public abstract void AddToken(GenericXmlSecurityToken token, String sessionId);
        /// <summary>
        /// Attempts to retrieve a token from the cache.
        /// </summary>
        /// <param name="sessionId">Session id key to look-up for the token.</param>
        /// <param name="cachedToken">The token that was retrieved.</param>
        /// <returns>True if the operation retrieved a security token from the cache, false otherwise.</returns>
        public abstract bool TryGetToken(String sessionId, out GenericXmlSecurityToken cachedToken);
        /// <summary>
        /// Removes a token from the cache.
        /// </summary>
        /// <param name="sessionId">Session id key to look-up for the token.</param>
        public abstract void RemoveToken(String sessionId);

        /// <summary>
        /// Tries the get user sid from token.
        /// </summary>
        /// <param name="sessionId">The session id.</param>
        /// <returns></returns>
        public abstract string TryGetUserSidFromToken(String sessionId);
    }

    /// <summary>
    /// Base implementation of the IssuedTokenCache abstract class.
    /// </summary>
    public abstract class IssuedTokenCacheBase : IssuedTokenCache
    {
        private IDurableIssuedTokenConfiguration _configuration;
        /// <summary>
        /// Configuration on how to instantiate the Cache.
        /// </summary>
        protected IDurableIssuedTokenConfiguration Configuration
        {
            get { return _configuration; }
        }

        protected IssuedTokenCacheBase(IDurableIssuedTokenConfiguration configuration)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _configuration = configuration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        private IssuedTokenCacheBase() { }

        /// <summary>
        /// Addes the specific security token to the cache repository.
        /// </summary>
        /// <param name="key">The internal key used to identify the token.</param>
        /// <param name="token">Security token.</param>
        protected abstract void AddToken(Key key, GenericXmlSecurityToken token);
        /// <summary>
        /// Attempts to retrieve a token from the cache.
        /// </summary>
        /// <param name="key">The internal key used to identify the token.</param>
        /// <param name="token">Security token.</param>
        /// <returns>True if the operation retrieved a security token from the cache, false otherwise.</returns>
        protected abstract bool TryGetToken(Key key, out GenericXmlSecurityToken cachedToken);
        /// <summary>
        /// Removes a token from the cache.
        /// </summary>
        /// <param name="sessionId">Session id key to look-up for the token.</param>
        protected abstract void RemoveToken(Key key);

        /// <summary>
        /// Tries the get user sid from token.
        /// </summary>
        /// <param name="key">The key.</param>
        /// <returns></returns>
        protected abstract string TryGetUserSidFromToken(Key key);

        public sealed override void RemoveToken(string sessionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (String.IsNullOrEmpty(sessionId))
                {
                    throw new ArgumentNullException("sessionId");
                }
                RemoveToken(new Key(sessionId));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        public sealed override void AddToken(GenericXmlSecurityToken token, String sessionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (token == null)
                {
                    throw new ArgumentNullException("token");
                }
                if (String.IsNullOrEmpty(sessionId))
                {
                    throw new ArgumentNullException("sessionId");
                }
                AddToken(new Key(sessionId), token);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        public sealed override bool TryGetToken(String sessionId, out GenericXmlSecurityToken cachedToken)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (String.IsNullOrEmpty(sessionId))
                {
                    throw new ArgumentNullException("sessionId");
                }
                cachedToken = null;

                GenericXmlSecurityToken tmp;
                Key key = new Key(sessionId);
                if (this.TryGetToken(key, out tmp))
                {
                    // if the cached token is going to expire soon, remove it from the cache and return a cache miss
                    if (tmp.ValidTo < DateTime.UtcNow + TimeSpan.FromMinutes(1))
                    {
                        this.RemoveToken(key);
                    }
                    else
                    {
                        cachedToken = tmp;
                    }
                }
                return (cachedToken != null);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public sealed override string TryGetUserSidFromToken(String sessionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (String.IsNullOrEmpty(sessionId))
                {
                    throw new ArgumentNullException("sessionId");
                }
                return this.TryGetUserSidFromToken(new Key(sessionId));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Security token key class. Currently limited to Http Session ID.
        /// </summary>
        protected class Key
        {
            String _sessionId;
            public String SessionId
            {
                get { return _sessionId; }
            }
            public Key(String sessionId)
            {
                this._sessionId = sessionId;
            }
            public override bool Equals(object obj)
            {
                Key other = obj as Key;
                if (other == null)
                {
                    return false;
                }
                bool result = (_sessionId == null && other._sessionId == null) || _sessionId.Equals(other._sessionId);
                return result;
            }
            public override int GetHashCode()
            {
                int hc1 = (this._sessionId == null) ? 0 : this._sessionId.GetHashCode();
                return (hc1);
            }
        }

        #region token persistence methods
        // Since the issued token is an endorsing token it will not be used for identity checks
        // hence we do not need to persist the authorization policies that represent the target service.
        /// <summary>
        /// Serialized a given Securty token.
        /// </summary>
        /// <param name="stream">The stream where to serialize to.</param>
        /// <param name="sessionId">The token key (session id).</param>
        /// <param name="token">The actual token.</param>
        protected void SerializeCachedToken(Stream stream, String sessionId, GenericXmlSecurityToken token)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                XmlTextWriter writer = new XmlTextWriter(stream, Encoding.UTF8);
                writer.WriteStartElement("Entry");
                writer.WriteElementString("SessionID", sessionId);

                writer.WriteStartElement("Token");
                writer.WriteStartElement("XML");
                token.TokenXml.WriteTo(writer);
                writer.WriteEndElement();
                SymmetricSecurityKey key = (SymmetricSecurityKey)(token.SecurityKeys[0]);
                writer.WriteElementString("Key", Convert.ToBase64String(key.GetSymmetricKey()));
                writer.WriteElementString("Id", token.Id);
                writer.WriteElementString("ValidFrom", Convert.ToString(token.ValidFrom));
                writer.WriteElementString("ValidTo", Convert.ToString(token.ValidTo));
                WSSecurityTokenSerializer serializer = new WSSecurityTokenSerializer();
                writer.WriteStartElement("InternalTokenReference");
                serializer.WriteKeyIdentifierClause(writer, token.InternalTokenReference);
                writer.WriteEndElement();
                writer.WriteStartElement("ExternalTokenReference");
                serializer.WriteKeyIdentifierClause(writer, token.ExternalTokenReference);
                writer.WriteEndElement();

                writer.WriteEndElement(); // token
                writer.WriteEndElement(); // entry

                writer.Flush();
                stream.Flush();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Attemps to deserilize a token.
        /// </summary>
        /// <param name="stream">The stream where to deserialize from.</param>
        /// <param name="tokenKey">The token key (session id).</param>
        /// <param name="token">The deserialized token.</param>
        /// <returns>True if the stream was deserialized succesfully, false otherwise.</returns>
        protected bool DeserializeCachedToken(Stream stream, IssuedTokenCacheBase.Key tokenKey, out GenericXmlSecurityToken token)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                XmlTextReader reader = new XmlTextReader(stream);
                try
                {
                    while (reader.IsStartElement("Entry"))
                    {
                        reader.ReadStartElement();
                        String sessionId = reader.ReadElementString("SessionID");

                        reader.ReadStartElement("Token");
                        reader.ReadStartElement("XML");
                        XmlDocument doc = new XmlDocument();
                        XmlElement tokenXml = doc.ReadNode(reader) as XmlElement;
                        reader.ReadEndElement();
                        byte[] key = Convert.FromBase64String(reader.ReadElementString("Key"));
                        reader.ReadElementString("Id");
                        DateTime validFrom = Convert.ToDateTime(reader.ReadElementString("ValidFrom"));
                        DateTime validTo = Convert.ToDateTime(reader.ReadElementString("ValidTo"));
                        WSSecurityTokenSerializer serializer = new WSSecurityTokenSerializer();
                        reader.ReadStartElement("InternalTokenReference");
                        SecurityKeyIdentifierClause internalReference = serializer.ReadKeyIdentifierClause(reader);
                        reader.ReadEndElement();
                        reader.ReadStartElement("ExternalTokenReference");
                        SecurityKeyIdentifierClause externalReference = serializer.ReadKeyIdentifierClause(reader);
                        reader.ReadEndElement();
                        reader.ReadEndElement(); // token
                        reader.ReadEndElement(); // entry

                        if (tokenKey.SessionId != sessionId)
                            continue;
                        else
                        {
                            List<IAuthorizationPolicy> policies = new List<IAuthorizationPolicy>();
                            token = new GenericXmlSecurityToken(tokenXml,
                                new BinarySecretSecurityToken(key), validFrom, validTo, internalReference, externalReference,
                                new ReadOnlyCollection<IAuthorizationPolicy>(policies));
                            return true;
                        }
                    }
                    token = null;
                    return false;
                }
                catch (System.Xml.XmlException)
                {
                    token = null;
                    return false;
                }
                finally { reader.Close(); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        #endregion
    }
}