﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using BMS.DataContracts;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using BMS.Facade.Service;
using BMS.ServicesWrapper.BMService;
using BMS.Utils;
using BMS.Utils.Properties;
using BMS.VistaIntegration.FacadeContracts.Admin;
using InfoWorld.HL7.ITS;
using VI = BMS.VistaIntegration.FacadeContracts;
using System.ServiceModel;

namespace BMS.Facade.Implementation
{
    /// <summary>
    /// Implementation class for configuration operations.
    /// </summary>
    public class ConfigurationImplementation : IConfigurationOperations
    {
        #region VistaIntegrationAdmin

        public IList<VI.VistASite> GetVistaSites(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<VI.VistASite> vistaSites = new List<VI.VistASite>();
                IList<VI.Admin.JobCalendar> availableCaledars = BuildCalendarsFromConfig(domainId);
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.VISTA_SITE_SETTINGS, string.Empty);
                if (config.SectionList != null)
                {
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {
                        VI.VistASite vistaSite = new VI.VistASite();
                        vistaSite.Id = kvp.Key.Replace(Constants.VISTA_SITE_SETTINGS, string.Empty);
                        vistaSite.DataRetrievalDetails = new Dictionary<VI.Admin.VistaDataType, VI.Admin.DataRetrieval>();

                        foreach (Element element in kvp.Value)
                            BuildVistASiteFromConfiguration(element, vistaSite, availableCaledars);
                        vistaSite.Number = FacadeManager.EntityInterface.GetVistaSite(new II(domainId, vistaSite.Id)).Number;
                        vistaSites.Add(vistaSite);
                    }
                }
                return vistaSites;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<VI.Admin.JobCalendar> GetDefinedCalendars(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildCalendarsFromConfig(domainId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveCalendar(VI.Admin.JobCalendar calendar, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrWhiteSpace(calendar.Id))
                    calendar.Id = Guid.NewGuid().ToString();

                string configName = Constants.JOB_CALENDAR + calendar.Id;

                Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.JOB_CALENDAR, configName);
                oldConfig.ConfigType = ConfigType.APP;
                oldConfig.UserName = string.Empty;
                oldConfig.Domain = domainId;

                if (oldConfig.SectionList == null)
                    oldConfig.SectionList = new Dictionary<string, List<Element>>();

                StringBuilder sb = new StringBuilder();
                XmlSerializer serializer = new XmlSerializer(typeof(VI.Admin.JobCalendar));
                using (StringWriter sw = new StringWriter(sb))
                {
                    serializer.Serialize(sw, calendar);
                    sw.Flush();

                    List<Element> elements = new List<Element>();
                    elements.Add(new Element() { Key = calendar.Name, Value = new ST(sb.ToString()), FriendlyName = string.Empty });

                    if (oldConfig.SectionList.ContainsKey(configName))
                        oldConfig.SectionList[configName] = elements;
                    else
                        oldConfig.SectionList.Add(configName, elements);
                }

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, Constants.JOB_CALENDAR, configName);

                try
                {
                    //refresh the schedulers used by the vista integration component
                    BMSFactory.VistaQueryClient.SetRefreshSchedulersFlag();
                }
                catch (Exception e) { Tracer.TraceException(e); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteCalendar(string schedulerId, string domainId)
        {
            string configName = Constants.JOB_CALENDAR + schedulerId;
            Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.JOB_CALENDAR, configName);
            if (oldConfig.SectionList == null)
                oldConfig.SectionList = new Dictionary<string, List<Element>>();

            oldConfig.SectionList.Remove(configName);
            oldConfig.ConfigType = ConfigType.APP;
            oldConfig.UserName = string.Empty;
            oldConfig.Domain = domainId;
            BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, Constants.JOB_CALENDAR, configName);

            try
            {
                //refresh the schedulers used by the vista integration component
                BMSFactory.VistaQueryClient.SetRefreshSchedulersFlag();
            }
            catch (Exception e) { Tracer.TraceException(e); }
        }

        public void SaveVistASite(VI.VistASite vistaSite, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrWhiteSpace(vistaSite.Id))
                    vistaSite.Id = Guid.NewGuid().ToString();
                string configName = Constants.VISTA_SITE_SETTINGS + vistaSite.Id;

                Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.VISTA_SITE_SETTINGS, configName);
                oldConfig.ConfigType = ConfigType.APP;
                oldConfig.UserName = string.Empty;
                oldConfig.Domain = domainId;

                if (oldConfig.SectionList == null)
                    oldConfig.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = new List<Element>();

                if (vistaSite.DataRetrievalDetails == null)
                {
                    //old retrieval details, new connection configurations
                    if (oldConfig.SectionList.ContainsKey(configName))
                        elementList.AddRange(GetVistaSiteSettingsRetrievalDetails(oldConfig.SectionList[configName]));

                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_NAME, Value = new ST(vistaSite.Name), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_VISTA_NUMBER, Value = new ST(vistaSite.Number), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_ODBC_CONNECTION_STRING, Value = new ST(vistaSite.OdbcConnectionString), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_ODBC_PASSWORD, Value = new ST(vistaSite.OdbcPassword), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_ODBC_USER, Value = new ST(vistaSite.OdbcUser), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_TIMEZONE, Value = new ST(vistaSite.TimeZone.Id), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.VISTA_SITE_SETTINGS_MDWS_URL_CONFIG_KEY, Value = new ST(vistaSite.MdwsEndpointConfigKey), FriendlyName = string.Empty });
                }
                else
                {
                    //old connection configurations, new retrieval details
                    if (oldConfig.SectionList.ContainsKey(configName))
                        elementList.AddRange(GetVistaSiteSettingsConfiguration(oldConfig.SectionList[configName]));

                    foreach (KeyValuePair<VI.Admin.VistaDataType, VI.Admin.DataRetrieval> kvp in vistaSite.DataRetrievalDetails)
                        if (kvp.Value.DataRetrievalMethod != (DataRetrievalMethod)0)
                        {
                            elementList.Add(new Element()
                            {
                                Key = Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_METHOD + kvp.Key,
                                Value = new ST(kvp.Value.DataRetrievalMethod + ""),
                                FriendlyName = string.Empty
                            });
                            elementList.Add(new Element()
                            {
                                Key = Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_CALENDAR + kvp.Key,
                                Value = new ST(kvp.Value.JobCalendar == null ? string.Empty : kvp.Value.JobCalendar.Id),
                                FriendlyName = string.Empty
                            });
                            elementList.Add(new Element()
                            {
                                Key = Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_IS_HL7_ENABLED + kvp.Key,
                                Value = new ST(kvp.Value.IsHL7Enabled.ToString()),
                                FriendlyName = string.Empty
                            });
                        }
                }

                oldConfig.SectionList[configName] = elementList;
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, Constants.VISTA_SITE_SETTINGS, configName);
                try
                {
                    //refresh the schedulers used by the vista integration component
                    BMSFactory.VistaQueryClientWindows.SetRefreshVistACommandsFlag();
                }
                catch (Exception e) { Tracer.TraceException(e); }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteVistaASite(string domainId, string vistSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string configName = Constants.VISTA_SITE_SETTINGS + vistSiteId;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.VISTA_SITE_SETTINGS, configName);
                if (config.SectionList != null)
                {
                    List<Element> elements = config.SectionList[configName];
                    Element retrievalDetailMethodWaitingList = elements.Where(a => a.Key == Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_METHOD_WAITINGLIST).FirstOrDefault();
                    Element retrievalDetailCalendarWaitingList = elements.Where(a => a.Key == Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_CALENDAR_WAITINGLIST).FirstOrDefault();
                    Element retrievalDetailHL7EnabledWaitingList = elements.Where(a => a.Key == Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_HL7_ENABLED_WAITINGLIST).FirstOrDefault();
                    if (retrievalDetailMethodWaitingList != null)
                        elements.Remove(retrievalDetailMethodWaitingList);
                    if (retrievalDetailCalendarWaitingList != null)
                        elements.Remove(retrievalDetailCalendarWaitingList);
                    if (retrievalDetailHL7EnabledWaitingList != null)
                        elements.Remove(retrievalDetailHL7EnabledWaitingList);
                    config.SectionList[configName] = elements;
                    BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.VISTA_SITE_SETTINGS, configName);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }

        }

        private IEnumerable<Element> GetVistaSiteSettingsRetrievalDetails(List<Element> list)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Element> retrievalDetails = new List<Element>();
                if (list == null)
                    return retrievalDetails;
                foreach (Element element in list)
                    if (element.Key.StartsWith(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL))
                        retrievalDetails.Add(element);
                return retrievalDetails;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IEnumerable<Element> GetVistaSiteSettingsConfiguration(List<Element> list)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Element> nonRetrievalDetails = new List<Element>();
                if (list == null)
                    return nonRetrievalDetails;
                foreach (Element element in list)
                    if (!element.Key.StartsWith(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL))
                        nonRetrievalDetails.Add(element);
                return nonRetrievalDetails;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region EMSNotification

        /// <summary>
        /// Gets the EMS notifications list.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// EMS Notification list.
        /// </returns>
        public List<EMSNotification> GetEMSNotifications(string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string emsFacilityKey = Constants.EMS_NOTIFICATION + " " + facilityId.extension.ToUpper();
                List<EMSNotification> notificationList = new List<EMSNotification>();
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, emsFacilityKey, string.Empty);
                EMSNotification notification = null;
                if (config.SectionList != null)
                {
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {

                        notification = new EMSNotification();
                        notification.Location = kvp.Key.Replace(emsFacilityKey + " ", string.Empty);

                        foreach (Element element in kvp.Value)
                            BuildEMSNotificationFromConfiguration(element, notification);

                        notificationList.Add(notification);

                    }
                }
                return notificationList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets an EMS notification defined for a location.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// EMS notification object.
        /// </returns>
        public EMSNotification GetEMSNotification(string location, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                EMSNotification notification = null;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.EMS_NOTIFICATION);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(location);
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());

                if (config.SectionList != null)
                {
                    notification = new EMSNotification();
                    notification.Location = location;
                    foreach (Element element in config.SectionList[sb.ToString()])
                        BuildEMSNotificationFromConfiguration(element, notification);
                }

                return notification;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves an EMS notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="isNew">The is new flag.</param>
        public void SaveEMSNotification(EMSNotification notification, string domainId, bool isNew, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.EMS_NOTIFICATION);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(notification.Location);
                Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());
                if (oldConfig.SectionList != null)
                {
                    if (isNew)
                        throw new ConfigurationFault(Resources.CONFIGURATION_EMS_NOTIFICATION_EXISTS);
                }
                else
                {
                    if (!isNew)
                        throw new ConfigurationFault(Resources.CONFIGURATION_EMS_NOTIFICATION_NOT_EXISTS);
                }

                oldConfig.ConfigType = ConfigType.APP;
                oldConfig.UserName = string.Empty;
                oldConfig.Domain = domainId;
                if (oldConfig.SectionList == null)
                    oldConfig.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = new List<Element>();
                elementList.Add(new Element() { Key = Constants.EMS_EMAIL, Value = new ST(notification.EMSEmail), FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_DIRTY, Value = notification.EMSDirty, FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_CLEANED, Value = notification.EMSCleaned, FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_BED_CONTROLLER_EMAIL, Value = new ST(notification.BedControllerEmail), FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_BED_CONTROLLER_DIRTY, Value = notification.BedControllerDirty, FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_BED_CONTROLLER_CLEANED, Value = notification.BedControllerCleaned, FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_VISTA_GROUP_MAIL, Value = new ST(notification.VistaGroupMail), FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_VISTA_GROUP_DIRTY, Value = notification.VistaGroupDirty, FriendlyName = string.Empty });
                elementList.Add(new Element() { Key = Constants.EMS_VISTA_GROUP_CLEANED, Value = notification.VistaGroupCleaned, FriendlyName = string.Empty });

                if (isNew)
                    oldConfig.SectionList.Add(sb.ToString(), elementList);
                else
                    oldConfig.SectionList[sb.ToString()] = elementList;
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, parentSection, sb.ToString());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes an EMS notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        public void DeleteEMSNotification(EMSNotification notification, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.EMS_NOTIFICATION);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(notification.Location);
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());

                if (config.SectionList.ContainsKey(sb.ToString()))
                {
                    config.SectionList.Remove(sb.ToString());
                    BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, parentSection, sb.ToString());
                }
                else
                    throw new ConfigurationFault(Resources.CONFIGURATION_EMS_NOTIFICATION_NOT_EXISTS);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Checks whether an EMS Notification defined for the provided location exists.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// True if exists or False otherwise
        /// </returns>
        public bool EMSNotificationExists(string location, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EMSNotification notification = GetEMSNotification(location, domainId, facilityId);
                if (notification == null)
                    return false;
                else
                    return true;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region EventNotification

        /// <summary>
        /// Gets the event notifications list.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// Event notification list.
        /// </returns>
        public List<EventNotification> GetEventNotifications(string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder eventFacilityKey = new StringBuilder();
                eventFacilityKey.Append(Constants.EVENT_NOTIFICATION);
                eventFacilityKey.Append(" ");
                eventFacilityKey.Append(facilityId.extension.ToUpper());
                List<EventNotification> notificationList = new List<EventNotification>();

                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, eventFacilityKey.ToString(), string.Empty);
                if (config.SectionList != null)
                {
                    var eventNotificationList = (from item in config.SectionList
                                                 select item);
                    foreach (KeyValuePair<string, List<Element>> kvp in eventNotificationList)
                    {
                        var elementList = (from item in config.SectionList[kvp.Key]
                                           group item by item.Key.Split(new char[] { '_' }, 2)[1] into grp
                                           select grp);
                        string wardGroupName = kvp.Key.Substring(eventFacilityKey.Length + 1, kvp.Key.Length - eventFacilityKey.Length - 1);

                        foreach (IGrouping<string, Element> elementGroup in elementList)
                        {
                            notificationList.Add(BuildEventNotificationFromConfiguration(elementGroup, wardGroupName));
                        }
                    }

                }
                return notificationList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets an event notification defined for an event.
        /// </summary>
        /// <param name="eventName">The event.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// Event notification object.
        /// </returns>
        public IList<EventNotification> GetEventNotification(string wardGroupName, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                IList<EventNotification> notifications = new List<EventNotification>();
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.EVENT_NOTIFICATION);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(wardGroupName);

                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());
                if (config.SectionList != null)
                {
                    var elementList = (from item in config.SectionList[sb.ToString()]
                                       group item by item.Key.Split(new char[] { '_' }, 2)[1] into grp
                                       select grp);
                    foreach (IGrouping<string, Element> elementGroup in elementList)
                    {
                        notifications.Add(BuildEventNotificationFromConfiguration(elementGroup, wardGroupName));
                    }
                }
                return notifications;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves an event notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="isNew">The is new flag.</param>
        /// <param name="favilityId">The favility id.</param>
        public void SaveEventNotification(IList<EventNotification> eventsNotifications, string domainId, bool isNew, II facilityId, string wardGroupName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.EVENT_NOTIFICATION);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(wardGroupName);
                Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());
                oldConfig.ConfigType = ConfigType.APP;
                oldConfig.UserName = string.Empty;
                oldConfig.Domain = domainId;


                List<Element> elementList = new List<Element>();
                foreach (EventNotification eventNotification in eventsNotifications)
                    elementList.AddRange(BuildEventNotification(eventNotification));
                if (oldConfig.SectionList == null)
                {
                    oldConfig.SectionList = new Dictionary<string, List<Element>>();
                    oldConfig.SectionList.Add(sb.ToString(), elementList);
                }
                else
                    oldConfig.SectionList[sb.ToString()] = elementList;
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, parentSection, sb.ToString());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes an event notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        public void DeleteEventNotification(string domainId, II facilityId, string wardGroupName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.EVENT_NOTIFICATION);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(wardGroupName);
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());
                if (config.SectionList != null)
                {
                    config.SectionList.Remove(sb.ToString());
                    BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, parentSection, sb.ToString());
                }
                else
                    throw new ConfigurationFault(Resources.CONFIGURATION_EVENT_NOTIFICATION_NOT_EXISTS);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Checks whether an Event Notification defined for the provided event name exists.
        /// </summary>
        /// <param name="eventName">The name of the event.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// True if exists or False otherwise
        /// </returns>
        public bool EventNotificationExists(string wardGroupName, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<EventNotification> notification = GetEventNotification(wardGroupName, domainId, facilityId);
                if (notification == null || notification.Count == 0)
                    return false;
                else
                    return true;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Settings

        /// <summary>
        /// Gets the list of sister sites groups.
        /// </summary> 
        /// <returns>A list of sister sites groups.</returns>
        public IList<SisterSitesGroup> GetSisterSites()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = FacadeManager.UserInterface.GetProfile().Domain;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.SETTINGS_SisterSites, string.Empty);

                List<SisterSitesGroup> groups = GetSisterSites(config);
                return groups;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves a group of sister sites.
        /// </summary> 
        /// <param name="group">A group of sister sites.</param> 
        public void SaveSisterSites(int recordNumber, SisterSitesGroup group)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = FacadeManager.UserInterface.GetProfile().Domain;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.SETTINGS_SisterSites, string.Empty);
                if (string.IsNullOrEmpty(config.Domain))
                {
                    config.ConfigType = ConfigType.APP;
                    config.UserName = string.Empty;
                    config.Domain = domainId;
                }
                List<SisterSitesGroup> groups = GetSisterSites(config);
                int i = groups.FindIndex(x => x.RecordNumber == recordNumber);
                bool add = i < 0 && (group.SiteCodeList ?? "") != "";
                bool delete = i >= 0 && (group.SiteCodeList ?? "") == "";
                if (add)
                    groups.Add(group);
                else if (delete)
                    groups.RemoveAt(i);
                else
                    groups[i] = group;

                SetSisterSites(groups, config);

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.SETTINGS_SisterSites, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private void SetSisterSites(List<SisterSitesGroup> groups, Configuration config)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (config != null && config.SectionList != null)
                    config.SectionList.Remove(Constants.SETTINGS_SisterSites);

                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();
                config.SectionList.Add(Constants.SETTINGS_SisterSites,
                    groups.Select(g => new Element()
                    {
                        Key = g.RecordNumber.ToString(),
                        FriendlyName = "Sister Sites Group " + g.RecordNumber,
                        Value = new ST() { text = g.SiteCodeList }
                    }).ToList());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<SisterSitesGroup> GetSisterSites(Configuration config)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<SisterSitesGroup> groups = new List<SisterSitesGroup>();
                if (config != null && config.SectionList != null)
                {
                    groups.AddRange(
                      config.SectionList[Constants.SETTINGS_SisterSites].Select(x => new SisterSitesGroup()
                      {
                          RecordNumber = int.Parse(x.Key),
                          SiteCodeList = (x.Value as ST).text
                      }));
                }
                return groups;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public User GetUserProfile(string username, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    if (domainId == null)
                        domainId = BMS.ServicesWrapper.Security.SecurityFactory.Instance.GetCurrentDomain();
                    Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetUserProfile(username, domainId);

                    User user = new User();
                    user.Domain = domainId;
                    user.UserName = username;
                    bool containsProfile = config.SectionList.ContainsKey("Profile");
                    List<Element> settings = containsProfile ?
                        config.SectionList["Profile"] :
                        GetDefaultSettings();

                    if (settings != null)
                        SetUserSettings(settings, user);
                    return user;
                }
                catch (System.ServiceModel.FaultException<BMS.FaultContracts.DefaultConfigException>)
                {
                    CD noCD = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision).First(x => x.code == Constants.No);
                    ;

                    User user = new User()
                    {
                        UserName = username,
                        Domain = domainId,
                    };
                    List<Element> settings = GetDefaultSettings();
                    if (settings != null)
                        SetUserSettings(settings, user);
                    return user;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<Element> GetDefaultSettings()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    string domainId = BMS.ServicesWrapper.Security.SecurityFactory.Instance.GetCurrentDomain();
                    var config = BMSFactory.ConfigurationOperationsClientFromWCF.GetUserProfile(Constants.DEFAULT_USER_PROFILE, domainId);
                    return config.SectionList.ContainsKey("Profile") ? config.SectionList["Profile"] : null;
                }
                catch
                {
                    return null;
                }

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SetUserSettings(List<Element> settings, User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EntityIdentification ei = settings.Where(e => e.Key == "DefaultFacility").Select(e => e.Value as EntityIdentification).FirstOrDefault();
                if (ei != null)
                {
                    II facilityId = new II() { root = ei.DomainIdentifier, extension = ei.EntityIdentifier };
                    try
                    {
                        user.Facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                        user.VistaSite = FacadeManager.EntityInterface.GetVistaSite(user.Facility.VistaSite.Id);
                        user.Visn = FacadeManager.EntityInterface.GetVisn(user.VistaSite.Visn.Id);
                        user.Region = FacadeManager.EntityInterface.GetRegion(user.Visn.Region.Id);
                        user.DefaultFacility = user.Facility;
                    }
                    catch { }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveUserProfile(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = new Configuration();
                config.UserName = user.UserName;
                config.Domain = user.Domain ?? BMS.ServicesWrapper.Security.SecurityFactory.Instance.GetCurrentDomain();
                config.ConfigType = user.Id.extension.Equals(Constants.DEFAULT_USER_PROFILE, StringComparison.InvariantCultureIgnoreCase) ? ConfigType.DEFUSR : ConfigType.USR;
                config.SectionList = new Dictionary<string, List<Element>>();
                List<Element> settings = new List<Element>();
                config.SectionList.Add("Profile", settings);

                Facility f = user.Facility;
                if (f != null)
                {
                    EntityIdentification ei = new EntityIdentification()
                    {
                        DomainIdentifier = f.Id.root,
                        EntityIdentifier = f.Id.extension,
                        EntityTypeIdentifier = EISConstants.ORGANIZATON,
                        EntityName = f.Name
                    };
                    settings.Add(new Element() { Key = "DefaultFacility", FriendlyName = "Default Facility", Value = ei });

                    BMSFactory.ConfigurationOperationsClientFromWCF.SetUserProfile(config);
                }

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Strict decision (yes/no values)

        private static ANY CreateStrictDecisionValue(CD cd)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool value = cd != null && string.Equals(cd.code, Constants.Yes, System.StringComparison.InvariantCultureIgnoreCase);
                return new BL() { value = value };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        private static CD CreateStrictDecisionCD(CD cd)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool value = cd != null && string.Equals(cd.code, Constants.Yes, System.StringComparison.InvariantCultureIgnoreCase);
                return CreateStrictDecisionCD(value);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static CD CreateStrictDecisionCD(ANY value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BL bl = value as BL;
                return CreateStrictDecisionCD(bl != null && bl.value);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static CD CreateStrictDecisionCD(bool value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return value ?
                    YesNoValues.First(x => x.code == Constants.Yes) :
                    YesNoValues.First(x => x.code == Constants.No);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static CDList YesNoValues
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    return FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        #endregion

        #region Bed Board Module

        /// <summary>
        /// Gets the bed board modules of BMS.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// List of application modules.
        /// </returns>
        public IList<Module> GetModules(string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string moduleFacilityKey = Constants.BED_BOARD_MODULE + " " + facilityId.extension.ToUpper();
                IList<Module> moduleList = new List<Module>();
                CDList moduleCDList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.BedBoardModule);
                CD yes = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision).Find(a => a.code == Constants.Yes);

                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, moduleFacilityKey, string.Empty);
                Module module = null;
                //get modules from config table
                if (config.SectionList != null)
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {
                        if (kvp.Key.StartsWith(moduleFacilityKey)) //Constants.BED_BOARD_MODULE
                        {
                            module = new Module();
                            module.Name = kvp.Key.Replace(moduleFacilityKey + " ", string.Empty); //Constants.BED_BOARD_MODULE
                            module.Code = moduleCDList.Where(cd => cd.displayName == module.Name)
                                                      .Select(cd => cd.code)
                                                      .FirstOrDefault() ?? string.Empty;

                            foreach (Element element in kvp.Value)
                                module.CurrentlyInUse = (CD)element.Value;

                            moduleList.Add(module);
                        }
                    }

                foreach (CD moduleCD in moduleCDList)
                {
                    //add the module in config if it is newly added in EVS
                    if (moduleList.ToList<Module>().Find(a => a.Name == moduleCD.displayName) == null)
                    {
                        module = new Module();
                        module.Name = moduleCD.displayName;
                        module.Code = moduleCDList.Where(cd => cd.displayName == module.Name).Select(cd => cd.code).FirstOrDefault() ?? string.Empty;
                        module.CurrentlyInUse = yes;
                        moduleList.Add(module);
                        AddEditModule(module, domainId, facilityId);
                    }
                }

                for (int i = moduleList.Count; i > 0; i--)
                {
                    Module mod = moduleList[i - 1];
                    //if a cd is in the config module list and is not in the cd list -> delete from module list
                    if (moduleCDList.Find(a => a.displayName == mod.Name) == null)
                    {
                        moduleList.Remove(mod);
                        DeleteModule(mod, domainId, facilityId);
                    }
                }

                return moduleList.OrderBy(a => a.Name).ToList<Module>();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the module.
        /// </summary>
        /// <param name="module">The module object.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        public void DeleteModule(Module module, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.BED_BOARD_MODULE);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(module.Name);
                Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());
                if (oldConfig.SectionList != null && oldConfig.SectionList.ContainsKey(sb.ToString()))
                {
                    oldConfig.SectionList.Remove(sb.ToString());
                    BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, parentSection, sb.ToString());
                }
                else
                    throw new ConfigurationFault(Resources.BED_BOARD_MODULE_NOT_EXISTS);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the module.
        /// </summary>
        /// <param name="module">The module object.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        public void AddEditModule(Module module, string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string parentSection = string.Empty;
                StringBuilder sb = new StringBuilder();
                sb.Append(Constants.BED_BOARD_MODULE);
                sb.Append(" ");
                sb.Append(facilityId.extension.ToUpper());
                parentSection = sb.ToString();
                sb.Append(" ");
                sb.Append(module.Name);
                Configuration oldConfig = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, parentSection, sb.ToString());
                oldConfig.ConfigType = ConfigType.APP;
                oldConfig.UserName = string.Empty;
                oldConfig.Domain = domainId;
                if (oldConfig.SectionList == null)
                    oldConfig.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = new List<Element>();
                elementList.Add(new Element() { Key = Constants.BED_BOARD_MODULE_CURRENTLY_IN_USE, Value = module.CurrentlyInUse, FriendlyName = string.Empty });

                //if (isNew)
                //    oldConfig.SectionList.Add(sb.ToString(), elementList);
                //else
                oldConfig.SectionList[sb.ToString()] = elementList;

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(oldConfig, parentSection, sb.ToString());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        #endregion

        #region FacilitySettings

        /// <summary>
        /// Gets the settings defined for a facility.
        /// </summary> 
        /// <param name="facilityId">The internal identifier of the facility.</param>
        /// <returns>Facility Settings object.</returns>
        public FacilitySettings GetFacilitySettings(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = facilityId.root;
                string facilityKey = Constants.FACILITY_SETTINGS + " " + facilityId.extension.ToUpper();
                Configuration config = BMSFactory.ConfigurationOperationsClientWindows.GetApplicationConfiguration(domainId, facilityKey, string.Empty);

                FacilitySettings facilitySettings = new FacilitySettings();
                facilitySettings.SisterSitesGroups = new List<SisterSitesGroup>();

                if (config.SectionList != null)
                {
                    foreach (Element element in config.SectionList[facilityKey])
                        BuildFacilitySettingsFromConfiguration(element, facilitySettings);
                }

                config = BMSFactory.ConfigurationOperationsClientWindows.GetApplicationConfiguration(domainId, Constants.SETTINGS_SisterSites, string.Empty);
                //get sister sites group
                if (config.SectionList != null)
                {
                    foreach (Element element in config.SectionList[Constants.SETTINGS_SisterSites])
                    {
                        string siteSisterIdentifiers = ((ST)element.Value).text;
                        if (siteSisterIdentifiers.Contains(facilityId.extension))
                            facilitySettings.SisterSitesGroups.Add(new SisterSitesGroup() { RecordNumber = Convert.ToInt32(element.Key), SiteCodeList = siteSisterIdentifiers });
                    }
                }
                return facilitySettings;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves into configuration section the settings for a facility.
        /// </summary>
        /// <param name="facilitySettings">The facility settings.</param>
        /// <param name="domainId">The domain id of the facility.</param> 
        public void SaveFacilitySettings(II facilityId, FacilitySettings facilitySettings)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = facilityId.root;
                string facilityKey = Constants.FACILITY_SETTINGS + " " + facilityId.extension;

                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, facilityKey, facilityKey);
                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = new List<Element>();
                BuildConfigurationFromFacilitySettings(facilitySettings, elementList);

                if (config.SectionList.ContainsKey(facilityKey))
                    config.SectionList[facilityKey] = elementList;
                else
                    config.SectionList.Add(facilityKey, elementList);
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, facilityKey, facilityKey);
                TimeZoneUtil.RemoveTimeZoneForFacility(facilityId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void BuildConfigurationFromFacilitySettings(FacilitySettings facilitySettings, List<Element> elementList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (facilitySettings.AutoRemovalWaitingList != null)
                    facilitySettings.AutoRemovalWaitingList = CreateStrictDecisionCD(facilitySettings.AutoRemovalWaitingList);
                if (facilitySettings.IntegratedFacility != null)
                    facilitySettings.IntegratedFacility = CreateStrictDecisionCD(facilitySettings.IntegratedFacility);
                facilitySettings.FacilitySiteTimeZone = facilitySettings.FacilitySiteTimeZone == null ? null :
                    FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone).FirstOrDefault(x => x.code == facilitySettings.FacilitySiteTimeZone.code);
                if (facilitySettings.ADTPrefix != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_ADT_PREFIX, Value = new ST(facilitySettings.ADTPrefix), FriendlyName = string.Empty });
                if (facilitySettings.ADTSuffix != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_ADT_SUFFIX, Value = new ST(facilitySettings.ADTSuffix), FriendlyName = string.Empty });
                if (!string.IsNullOrWhiteSpace(facilitySettings.IntegratedSiteSisterList) && facilitySettings.IntegratedSiteSisterGroupId != 0)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_ALLOWED_ACCESS, Value = new ST(facilitySettings.IntegratedSiteSisterList), FriendlyName = string.Empty });
                if (facilitySettings.IntegratedSiteSisterGroupId != 0)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_INTEGRATED_SITE_SISTER_ID, Value = new INT(facilitySettings.IntegratedSiteSisterGroupId), FriendlyName = string.Empty });
                if (facilitySettings.AutoRemovalWaitingList != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_AUTO_REMOVAL_WAITING_LIST, Value = facilitySettings.AutoRemovalWaitingList, FriendlyName = string.Empty });
                if (facilitySettings.BMSServerTimeZone != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_BMS_SERVER_TIME_ZONE, Value = new ST(facilitySettings.BMSServerTimeZone), FriendlyName = string.Empty });
                if (facilitySettings.IntegratedFacility != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_INTEGRATED_FACILITY, Value = facilitySettings.IntegratedFacility, FriendlyName = string.Empty });
                if (facilitySettings.MedicalCenterID != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_MEDICAL_CENTER_ID, Value = new ST(facilitySettings.MedicalCenterID), FriendlyName = string.Empty });
                if (facilitySettings.FacilitySiteTimeZone != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_SITE_TIME_ZONE, Value = facilitySettings.FacilitySiteTimeZone, FriendlyName = string.Empty });
                if (facilitySettings.WardPrefix != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_WARD_PREFIX, Value = new ST(facilitySettings.WardPrefix), FriendlyName = string.Empty });
                if (facilitySettings.WardSuffix != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_WARD_SUFFIX, Value = new ST(facilitySettings.WardSuffix), FriendlyName = string.Empty });
                if (facilitySettings.SiteAlias1 != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_SITEALIAS1, Value = new ST(facilitySettings.SiteAlias1), FriendlyName = string.Empty });
                if (facilitySettings.SiteAlias2 != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_SITEALIAS2, Value = new ST(facilitySettings.SiteAlias2), FriendlyName = string.Empty });
                if (facilitySettings.SiteAlias3 != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_SITEALIAS3, Value = new ST(facilitySettings.SiteAlias3), FriendlyName = string.Empty });
                if (facilitySettings.EMSMailSender != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_EMSMAILSENDER, Value = new ST(facilitySettings.EMSMailSender), FriendlyName = string.Empty });
                if (facilitySettings.EventMailSender != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_EVENTMAILSENDER, Value = new ST(facilitySettings.EventMailSender), FriendlyName = string.Empty });
                if (facilitySettings.BPMailMonitor != null)
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_BPMAILMONITOR, Value = new ST(facilitySettings.BPMailMonitor), FriendlyName = string.Empty });
                if (!string.IsNullOrEmpty(facilitySettings.EMSDefaultUserName))
                {
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_EMS_USER_NAME, Value = new ST(facilitySettings.EMSDefaultUserName), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_EMS_PASSWORD, Value = new ST(facilitySettings.EMSDefaultPassword), FriendlyName = string.Empty });
                }
                if (!string.IsNullOrEmpty(facilitySettings.KioskDefaultUserName))
                {
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_KIOSK_USER_NAME, Value = new ST(facilitySettings.KioskDefaultUserName), FriendlyName = string.Empty });
                    elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_KIOSK_PASSWORD, Value = new ST(facilitySettings.KioskDefaultPassword), FriendlyName = string.Empty });
                }
                elementList.Add(new Element() { Key = Constants.FACILITY_SETTINGS_LOCALTIMEADJUST, Value = new INT(facilitySettings.LocalTimeAdjust), FriendlyName = string.Empty });

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void BuildFacilitySettingsFromConfiguration(Element element, FacilitySettings facilitySettings)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                switch (element.Key)
                {
                    case Constants.FACILITY_SETTINGS_ADT_PREFIX:
                        facilitySettings.ADTPrefix = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_ADT_SUFFIX:
                        facilitySettings.ADTSuffix = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_AUTO_REMOVAL_WAITING_LIST:
                        facilitySettings.AutoRemovalWaitingList = (CD)element.Value;
                        break;
                    case Constants.FACILITY_SETTINGS_BMS_SERVER_TIME_ZONE:
                        facilitySettings.BMSServerTimeZone = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_INTEGRATED_SITE_SISTER_ID:
                        facilitySettings.IntegratedSiteSisterGroupId = ((INT)element.Value).IntValue;
                        break;
                    case Constants.FACILITY_SETTINGS_ALLOWED_ACCESS:
                        facilitySettings.IntegratedSiteSisterList = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_INTEGRATED_FACILITY:
                        facilitySettings.IntegratedFacility = (CD)element.Value;
                        break;
                    case Constants.FACILITY_SETTINGS_MEDICAL_CENTER_ID:
                        facilitySettings.MedicalCenterID = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_SITE_TIME_ZONE:
                        facilitySettings.FacilitySiteTimeZone = (CD)element.Value;
                        break;
                    case Constants.FACILITY_SETTINGS_LOCALTIMEADJUST:
                        facilitySettings.LocalTimeAdjust = ((INT)element.Value).IntValue;
                        break;
                    case Constants.FACILITY_SETTINGS_WARD_PREFIX:
                        facilitySettings.WardPrefix = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_WARD_SUFFIX:
                        facilitySettings.WardSuffix = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_SITEALIAS1:
                        facilitySettings.SiteAlias1 = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_SITEALIAS2:
                        facilitySettings.SiteAlias2 = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_SITEALIAS3:
                        facilitySettings.SiteAlias3 = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_EMSMAILSENDER:
                        facilitySettings.EMSMailSender = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_EVENTMAILSENDER:
                        facilitySettings.EventMailSender = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_BPMAILMONITOR:
                        facilitySettings.BPMailMonitor = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_EMS_USER_NAME:
                        facilitySettings.EMSDefaultUserName = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_EMS_PASSWORD:
                        facilitySettings.EMSDefaultPassword = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_KIOSK_USER_NAME:
                        facilitySettings.KioskDefaultUserName = ((ST)element.Value).text;
                        break;
                    case Constants.FACILITY_SETTINGS_KIOSK_PASSWORD:
                        facilitySettings.KioskDefaultPassword = ((ST)element.Value).text;
                        break;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region VistASettings

        private void BuildVistASettingsFromConfiguration(Element element, VistASettings settings)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                switch (element.Key)
                {
                    case Constants.VISTA_SETTINGS_SITE_TIME_ZONE:
                        settings.SiteTimeZone = (CD)element.Value;
                        break;
                    case Constants.VISTA_SETTINGS_LOCALTIMEADJUST:
                        settings.LocalTimeAdjust = ((INT)element.Value).IntValue;
                        break;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void BuildConfigurationFromVistASettings(VistASettings settings, List<Element> elementList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                settings.SiteTimeZone = settings.SiteTimeZone == null ? null :
                    FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone).FirstOrDefault(x => x.code == settings.SiteTimeZone.code);

                if (settings.SiteTimeZone != null)
                    elementList.Add(new Element() { Key = Constants.VISTA_SETTINGS_SITE_TIME_ZONE, Value = settings.SiteTimeZone, FriendlyName = string.Empty });

                elementList.Add(new Element() { Key = Constants.VISTA_SETTINGS_LOCALTIMEADJUST, Value = new INT(settings.LocalTimeAdjust), FriendlyName = string.Empty });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        public VistASettings GetVistASettings(II vistAId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = vistAId.root;
                string vistAKey = Constants.VISTA_SETTINGS + " " + vistAId.extension;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.VISTA_SETTINGS, vistAKey);

                VistASettings vistASettings = null;

                if (config.SectionList != null)
                {
                    vistASettings = new VistASettings();
                    foreach (Element element in config.SectionList[vistAKey])
                        BuildVistASettingsFromConfiguration(element, vistASettings);
                }


                return vistASettings;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveVistASettings(VistASettings settings, II vistAId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = vistAId.root;
                string vistAKey = Constants.VISTA_SETTINGS + " " + vistAId.extension;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.VISTA_SETTINGS, vistAKey);
                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = new List<Element>();
                BuildConfigurationFromVistASettings(settings, elementList);

                config.SectionList[vistAKey] = elementList;
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.VISTA_SETTINGS, vistAKey);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        /// <summary>
        /// Gets the records number per page.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Records number per page</returns>
        public int GetRecordsNumberPerPage(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return 400;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Builds the event notification.
        /// </summary>
        /// <param name="eventNotification">The event notification.</param>
        /// <returns></returns>
        private List<Element> BuildEventNotification(EventNotification eventNotification)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Element> eventNotificationList = new List<Element>();
                eventNotificationList.Add(new Element() { Key = Constants.EVENT_BED_CONTROLLER_EMAIL + "_" + eventNotification.Name, Value = new ST(eventNotification.BedControllerEmailAddress), FriendlyName = string.Empty });
                eventNotificationList.Add(new Element() { Key = Constants.EVENT_BED_CONTROLLER_NOTIFY + "_" + eventNotification.Name, Value = eventNotification.BedControllerNotification, FriendlyName = string.Empty });
                return eventNotificationList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private EventNotification BuildEventNotificationFromConfiguration(IGrouping<string, Element> elementGroup, string wardGroupName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotification notification = new EventNotification();
                notification.WardGroup = wardGroupName;
                notification.Name = elementGroup.Key;
                notification.BedControllerEmailAddress = ((ST)elementGroup.Where(a => a.Key.Contains(Constants.EVENT_BED_CONTROLLER_EMAIL)).First().Value).text;
                notification.BedControllerNotification = (CD)elementGroup.Where(a => a.Key.Contains(Constants.EVENT_BED_CONTROLLER_NOTIFY)).First().Value;
                return notification;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region internal

        internal IList<VI.Admin.JobCalendar> BuildCalendarsFromConfig(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<VI.Admin.JobCalendar> calendars = new List<VI.Admin.JobCalendar>();
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.JOB_CALENDAR, string.Empty);
                if (config.SectionList != null)
                {
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {

                        VI.Admin.JobCalendar calendar;
                        XmlSerializer serializer = new XmlSerializer(typeof(VI.Admin.JobCalendar));

                        if (kvp.Value[0].Value != null)
                        {
                            ST st = null;
                            st = kvp.Value[0].Value as ST;
                            if (st != null)
                            {
                                using (StringReader sr = new StringReader(st.text))
                                    calendar = (VI.Admin.JobCalendar)serializer.Deserialize(sr);
                                calendars.Add(calendar);
                            }

                        }
                    }
                }
                return calendars;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal void BuildEMSNotificationFromConfiguration(Element element, EMSNotification notification)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                switch (element.Key)
                {
                    case Constants.EMS_EMAIL:
                        notification.EMSEmail = ((ST)element.Value).text;
                        break;
                    case Constants.EMS_DIRTY:
                        notification.EMSDirty = (CD)element.Value;
                        break;
                    case Constants.EMS_CLEANED:
                        notification.EMSCleaned = (CD)element.Value;
                        break;
                    case Constants.EMS_BED_CONTROLLER_EMAIL:
                        notification.BedControllerEmail = ((ST)element.Value).text;
                        break;
                    case Constants.EMS_BED_CONTROLLER_DIRTY:
                        notification.BedControllerDirty = (CD)element.Value;
                        break;
                    case Constants.EMS_BED_CONTROLLER_CLEANED:
                        notification.BedControllerCleaned = (CD)element.Value;
                        break;
                    case Constants.EMS_VISTA_GROUP_MAIL:
                        notification.VistaGroupMail = ((ST)element.Value).text;
                        break;
                    case Constants.EMS_VISTA_GROUP_DIRTY:
                        notification.VistaGroupDirty = (CD)element.Value;
                        break;
                    case Constants.EMS_VISTA_GROUP_CLEANED:
                        notification.VistaGroupCleaned = (CD)element.Value;
                        break;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal void BuildVistASiteFromConfiguration(Element element, VI.VistASite vistaSite, IList<VI.Admin.JobCalendar> jobCalendars)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (element.Key.StartsWith(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_METHOD))
                {
                    ST value = null;
                    value = element.Value as ST;
                    VI.Admin.VistaDataType dataType = (VI.Admin.VistaDataType)Enum.Parse(typeof(VI.Admin.VistaDataType), element.Key.Replace(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_METHOD, ""));
                    DataRetrieval dataRetrieval = null;
                    if (vistaSite.DataRetrievalDetails.ContainsKey(dataType))
                        dataRetrieval = vistaSite.DataRetrievalDetails[dataType];
                    else
                    {
                        dataRetrieval = new DataRetrieval();
                        vistaSite.DataRetrievalDetails.Add(dataType, dataRetrieval);
                    }
                    DataRetrievalMethod drm;
                    bool success = false;
                    if (value != null)
                    {
                        success = Enum.TryParse<DataRetrievalMethod>(value.text, out drm);
                        if (success)
                            dataRetrieval.DataRetrievalMethod = drm;
                        else
                            vistaSite.DataRetrievalDetails.Remove(dataType);
                    }

                    return;
                }
                if (element.Key.StartsWith(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_CALENDAR))
                {
                    ST value = element.Value as ST;
                    VI.Admin.VistaDataType dataType = (VI.Admin.VistaDataType)Enum.Parse(typeof(VI.Admin.VistaDataType), element.Key.Replace(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_CALENDAR, ""));
                    DataRetrieval dataRetrieval = null;
                    if (vistaSite.DataRetrievalDetails.ContainsKey(dataType))
                        dataRetrieval = vistaSite.DataRetrievalDetails[dataType];
                    else
                    {
                        dataRetrieval = new DataRetrieval();
                        vistaSite.DataRetrievalDetails.Add(dataType, dataRetrieval);
                    }
                    dataRetrieval.JobCalendar = jobCalendars.FirstOrDefault(calendar => calendar.Id.Equals(value.text, StringComparison.InvariantCultureIgnoreCase));
                }
                if (element.Key.StartsWith(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_IS_HL7_ENABLED))
                {
                    ST value = null;
                    value = element.Value as ST;
                    VI.Admin.VistaDataType dataType = (VI.Admin.VistaDataType)Enum.Parse(typeof(VI.Admin.VistaDataType), element.Key.Replace(Constants.VISTA_SITE_SETTINGS_RETRIEVAL_DETAIL_IS_HL7_ENABLED, ""));
                    DataRetrieval dataRetrieval = null;
                    if (vistaSite.DataRetrievalDetails.ContainsKey(dataType))
                        dataRetrieval = vistaSite.DataRetrievalDetails[dataType];
                    else
                    {
                        dataRetrieval = new DataRetrieval();
                        vistaSite.DataRetrievalDetails.Add(dataType, dataRetrieval);
                    }
                    if (dataRetrieval != null && value != null)
                    {
                        dataRetrieval.IsHL7Enabled = bool.Parse(value.text);
                    }
                }

                ST x = null;
                x = element.Value as ST;
                switch (element.Key)
                {
                    case Constants.VISTA_SITE_SETTINGS_NAME:
                        if (x != null)
                        {
                            vistaSite.Name = x.text;
                        }
                        break;
                    case Constants.VISTA_SITE_SETTINGS_VISTA_NUMBER:
                        if (x != null)
                        {
                            vistaSite.Number = x.text;
                        }
                        break;
                    case Constants.VISTA_SITE_SETTINGS_ODBC_CONNECTION_STRING:
                        if (x != null)
                        {
                            vistaSite.OdbcConnectionString = x.text;
                        }
                        break;
                    case Constants.VISTA_SITE_SETTINGS_ODBC_PASSWORD:
                        if (x != null)
                        {
                            vistaSite.OdbcPassword = x.text;
                        }
                        break;
                    case Constants.VISTA_SITE_SETTINGS_ODBC_USER:
                        if (x != null)
                        {
                            vistaSite.OdbcUser = x.text;
                        }
                        break;
                    case Constants.VISTA_SITE_SETTINGS_TIMEZONE:
                        if (x != null)
                        {
                            vistaSite.TimeZone = TimeZoneInfo.FindSystemTimeZoneById(x.text);
                        }
                        break;
                    case Constants.VISTA_SITE_SETTINGS_MDWS_URL_CONFIG_KEY:
                        if (x != null)
                        {
                            vistaSite.MdwsEndpointConfigKey = x.text;
                        }
                        break;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Marquee

        /// <summary>
        /// Gets the marquee.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Marquee text or empty string.</returns>
        public string GetMarquee(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                MarqueeSettings settings = GetMarquees(domainId);
                return settings.ActiveMarqueeText;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the marquee.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="index">The index of the previously saved common marquee texts.</param>
        /// <returns>Marquee text or empty string.</returns>
        public string GetMarquee(string domainId, string index)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                MarqueeSettings settings = GetMarquees(domainId);
                if (!settings.Items.ContainsKey(index))
                    return String.Empty;
                else
                    return settings.Items[index].Text;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the marquee.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Marquee text or empty string.</returns>
        public MarqueeSettings GetMarquees(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                MarqueeSettings settings = null;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.MARQUEE_TEXT, String.Empty);
                if (config.SectionList != null && config.SectionList.ContainsKey(Constants.MARQUEE_TEXT))
                {
                    settings = new MarqueeSettings();
                    foreach (Element element in config.SectionList[Constants.MARQUEE_TEXT])
                    {
                        if (element != null)
                        {
                            if (element.Key == Constants.MARQUEE_ACTIVE_INDEX)
                                settings.ActiveIndex = ((ST)element.Value).text;
                            else if (element.Key == Constants.MARQUEE_TEXT)
                            {
                                // load legacy marquee format
                                MarqueeItem item = new MarqueeItem() { Index = "1", Text = ((ST)element.Value).text };
                                settings.Items.Add(item.Index, item);
                                settings.ActiveIndex = item.Index;
                            }
                            else
                            {
                                MarqueeItem item = new MarqueeItem() { Index = element.Key, Text = ((ST)element.Value).text };
                                settings.Items.Add(item.Index, item);
                            }
                            
                        }
                    }

                    return settings;
                }

                return settings;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        
        /// <summary>
        /// Stores a common marquee for later easy retrieval as the main marquee text
        /// </summary>
        /// <param name="marquee">The marquee text.</param>
        /// <param name="index">The index of the common marquee text to save.</param>
        /// <param name="domainId">The domain id.</param>
        public void SaveMarquee(string marquee, string domainId, string index)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.MARQUEE_TEXT, String.Empty);

                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = null;
                if (!config.SectionList.ContainsKey(Constants.MARQUEE_TEXT))
                {
                    elementList = new List<Element>();
                    config.SectionList.Add(Constants.MARQUEE_TEXT, elementList);
                }
                else
                    elementList = config.SectionList[Constants.MARQUEE_TEXT];

                if (marquee != null)
                {
                    Element element = config.SectionList[Constants.MARQUEE_TEXT].Where(el => el.Key == index).FirstOrDefault();
                    if (element != null)
                        element.Value = new ST(marquee);
                    else
                        elementList.Add(new Element() { Key = Constants.MARQUEE_TEXT + index.ToString(), Value = new ST(marquee), FriendlyName = string.Empty });
                }

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.MARQUEE_TEXT, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Saves the marquee settings including all marquee history and the active marquee index.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="settings">The marquee settings to save.</param>
        public void SaveMarquees(string domainId, MarqueeSettings settings)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.MARQUEE_TEXT, String.Empty);

                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = null;
                if (!config.SectionList.ContainsKey(Constants.MARQUEE_TEXT))
                {
                    elementList = new List<Element>();
                    config.SectionList.Add(Constants.MARQUEE_TEXT, elementList);
                }
                else
                {
                    elementList = config.SectionList[Constants.MARQUEE_TEXT];

                    // check for legacy marquee format and remove
                    Element legacyElem = elementList.Find(elem => elem.Key == Constants.MARQUEE_TEXT);
                    if (legacyElem != null)
                        elementList.Remove(legacyElem);
                }

                if (settings != null)
                {
                    // store the active index
                    ConfigureSectionListItem(config.SectionList, Constants.MARQUEE_TEXT, Constants.MARQUEE_ACTIVE_INDEX, settings.ActiveIndex);

                    // store the marquee items
                    foreach (MarqueeItem item in settings.Items.Values)
                    {
                        ConfigureSectionListItem(config.SectionList, Constants.MARQUEE_TEXT, item.Index.ToString(), item.Text);
                    }
                }

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.MARQUEE_TEXT, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        private void ConfigureSectionListItem(Dictionary<string, List<Element>> sectionList, string sectionKey, string itemKey, string itemValue)
        {
            Element element = sectionList[sectionKey].Where(el => el.Key == itemKey).FirstOrDefault();
            if (element != null)
                element.Value = new ST(itemValue);
            else
            {
                element = new Element() { Key = itemKey, Value = new ST(itemValue), FriendlyName = string.Empty };
                sectionList[sectionKey].Add(element);
            }
        }

        /// <summary>
        /// Gets the application parameters used to configure various site-level settings
        /// </summary>
        /// <param name="domainId"></param>
        public ApplicationParameters GetApplicationParameters(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ApplicationParameters settings = null;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.APPLICATION_PARAMETERS, String.Empty);
                if (config.SectionList != null && config.SectionList.ContainsKey(Constants.APPLICATION_PARAMETERS))
                {
                    settings = new ApplicationParameters();
                    foreach (Element element in config.SectionList[Constants.APPLICATION_PARAMETERS])
                    {
                        if (element != null)
                        {
                            ApplicationParameter item = new ApplicationParameter() { Key = element.Key, Value = ((ST)element.Value).text };
                            settings.Items.Add(item.Key, item);
                        }
                    }

                    return settings;
                }

                return settings;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the application parameters to the BMS config table.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="settings">The application parameters to save.</param>
        public void SaveApplicationParameters(string domainId, ApplicationParameters settings)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.APPLICATION_PARAMETERS, String.Empty);

                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = null;
                if (!config.SectionList.ContainsKey(Constants.APPLICATION_PARAMETERS))
                {
                    elementList = new List<Element>();
                    config.SectionList.Add(Constants.APPLICATION_PARAMETERS, elementList);
                }
                else
                    elementList = config.SectionList[Constants.APPLICATION_PARAMETERS];

                if (settings != null)
                {
                    // store the application parameter items
                    foreach (ApplicationParameter item in settings.Items.Values)
                    {
                        ConfigureSectionListItem(config.SectionList, Constants.APPLICATION_PARAMETERS, item.Key, item.Value);
                    }
                }

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.APPLICATION_PARAMETERS, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /* public string GetMedicalCenterIds(string domainId, IList<string> facilityExtension)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string result = string.Empty;
                Configuration config = ConfigurationOperationsClient.GetApplicationConfiguration(domainId);
                foreach (string extension in facilityExtension)
                {
                    string facilityKey = Constants.FACILITY_SETTINGS + " " + extension.ToUpper();
                    if (config.SectionList.ContainsKey(facilityKey))
                    {
                        Element element = config.SectionList[facilityKey].Where(a => a.Key == Constants.FACILITY_SETTINGS_MEDICAL_CENTER_ID).FirstOrDefault();
                        if (element == null)
                        {
                            if (string.IsNullOrEmpty(result))
                                result = extension + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                            else
                            {
                                result = result + ";" + extension + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                            }
                        }
                        else
                        {
                            string medicalCenterCode = string.IsNullOrWhiteSpace(((ST)element.Value).text) == true ? Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE : ((ST)element.Value).text;
                            if (string.IsNullOrEmpty(result))
                                result = extension + "," + medicalCenterCode;
                            else
                            {
                                result = result + ";" + extension + "," + medicalCenterCode;
                            }
                        }
                    }
                    else
                    {
                        if (string.IsNullOrEmpty(result))
                            result = extension + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                        else
                        {
                            result = result + ";" + extension + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                        }
                    }
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }*/

        public string GetFacMedicalCenterIds(string domainId, IList<II> facilityIIList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string result = string.Empty;
                Configuration config = null;
                foreach (II facII in facilityIIList)
                {
                    string facilityKey = Constants.FACILITY_SETTINGS + " " + facII.extension.ToUpper();
                    config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, facilityKey, facilityKey);

                    //get and concatenate the facility's offset
                    DateTime dtUtc = DateTime.UtcNow;
                    TimeZoneInfo tzFac = FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(facII);
                    DateTime dtTz = TimeZoneInfo.ConvertTimeFromUtc(dtUtc, tzFac);
                    string offset = dtTz.Subtract(dtUtc).TotalMinutes.ToString();
                    string facOffset = facII.extension + "," + offset;

                    if (config.SectionList != null && config.SectionList.ContainsKey(facilityKey))
                    {
                        Element element = config.SectionList[facilityKey].Where(a => a.Key == Constants.FACILITY_SETTINGS_MEDICAL_CENTER_ID).FirstOrDefault();
                        if (element == null)
                        {
                            if (string.IsNullOrEmpty(result))
                                result = facOffset + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                            else
                            {
                                result = result + ";" + facOffset + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                            }
                        }
                        else
                        {
                            string medicalCenterCode = string.IsNullOrWhiteSpace(((ST)element.Value).text) == true ? Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE : ((ST)element.Value).text;
                            if (string.IsNullOrEmpty(result))
                                result = facOffset + "," + medicalCenterCode;
                            else
                            {
                                result = result + ";" + facOffset + "," + medicalCenterCode;
                            }
                        }
                    }
                    else
                    {
                        if (string.IsNullOrEmpty(result))
                            result = facOffset + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                        else
                        {
                            result = result + ";" + facOffset + "," + Constants.FACILITY_SENTTINGS_EMPTY_MEDICAL_CENTER_CODE;
                        }
                    }
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Evacuation

        /// <summary>
        /// Gets the is evacuation.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns></returns>
        public bool GetIsEvacuation(string domainId, string facilityExtension)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string isEvacuationKey = Constants.IS_EVACUATION + " " + facilityExtension.ToUpper();
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.IS_EVACUATION, isEvacuationKey);
                if (config.SectionList != null)
                {
                    Element element = config.SectionList[isEvacuationKey].FirstOrDefault();
                    if (element != null)
                        if (((ST)element.Value).text == Constants.ON)
                            return true;
                }
                return false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the is evacuation.
        /// </summary>
        /// <param name="isEvacuation">if set to <c>true</c> [is evacuation].</param>
        /// <param name="domainId">The domain id.</param>
        public void SaveIsEvacuation(bool isEvacuation, string domainId, string facilityExtension)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string evacuation = string.Empty;
                if (isEvacuation)
                    evacuation = Constants.ON;
                else
                    evacuation = Constants.OFF;
                string isEvacuationKey = Constants.IS_EVACUATION + " " + facilityExtension.ToUpper();

                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.IS_EVACUATION, isEvacuationKey);
                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = new List<Element>();
                elementList.Add(new Element() { Key = Constants.IS_EVACUATION, Value = new ST(evacuation), FriendlyName = string.Empty });

                config.SectionList[isEvacuationKey] = elementList;
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.IS_EVACUATION, isEvacuationKey);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion


        private List<NumiEntry> BuildNumiEntries(Configuration config)
        {
            List<NumiEntry> numiEntries = new List<NumiEntry>();
            IList<VI.Admin.JobCalendar> availableCaledars = BuildCalendarsFromConfig(config.Domain);
            IList<VistaSite> vistaSites = FacadeManager.EntityInterface.GetVistaSitesUsingWindowsAuthentication();
            if (config.SectionList != null)
                foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                {
                    NumiEntry numiEntry = new NumiEntry();
                    string schedulerId = kvp.Key.Replace(Constants.NUMI_ENTRY, string.Empty);
                    numiEntry.Calendar = availableCaledars.First(s => s.Id == schedulerId);
                    numiEntry.Sites = new List<VistaSite>();
                    foreach (Element element in kvp.Value)
                    {
                        if (element.Value != null)
                        {
                            ST value = null;
                            value = element.Value as ST;
                            if (value != null)
                            {
                                II vistASiteId = new II(value.text, element.Key);
                                VistaSite site = vistaSites.First(s => s.Id.extension.Equals(vistASiteId.extension, StringComparison.InvariantCultureIgnoreCase) && s.Id.root.Equals(vistASiteId.root, StringComparison.InvariantCultureIgnoreCase));
                                numiEntry.Sites.Add(site);
                            }
                        }
                    }

                    numiEntries.Add(numiEntry);
                }
            return numiEntries;
        }

        private void SaveNumiEntries(Configuration config, List<NumiEntry> numiEntries)
        {
            if (config.SectionList != null)
            {
                List<string> keysToRemove = config.SectionList.Keys
                    .Where(s => s.StartsWith(Constants.NUMI_ENTRY))
                    .ToList();
                keysToRemove.ForEach(s => config.SectionList.Remove(s));
            }

            foreach (var numiEntry in numiEntries)
            {
                List<Element> sectionList = numiEntry.Sites
                    .Select(s => new Element() { Key = s.Id.extension, Value = new ST(s.Id.root), FriendlyName = "" })
                    .ToList();

                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();
                config.SectionList[Constants.NUMI_ENTRY + numiEntry.Calendar.Id] = sectionList;
            }
        }

        public IList<NumiEntry> GetNumiEntries(string domainId)
        {
            Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.NUMI_ENTRY, string.Empty);
            List<NumiEntry> numiEntries = (config != null && config.SectionList != null && config.SectionList.Count > 0) ? BuildNumiEntries(config) : new List<NumiEntry>();
            return numiEntries;
        }

        public void SaveNumiEntry(NumiEntry numiEntry, string domainId)
        {
            Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.NUMI_ENTRY, string.Empty);
            if (string.IsNullOrEmpty(config.Domain))
            {
                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
            }
            List<NumiEntry> numiEntries = BuildNumiEntries(config);
            int index = numiEntries.FindIndex(s => s.Calendar.Id == numiEntry.Calendar.Id);
            if (index == -1) numiEntries.Add(numiEntry);
            else numiEntries[index] = numiEntry;
            SaveNumiEntries(config, numiEntries);
            BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.NUMI_ENTRY, string.Empty);
            RefreshNumiCommands();
        }

        private void RefreshNumiCommands()
        {
            try
            {
                //refresh the schedulers used by the vista integration component
                BMSFactory.VistaQueryClient.SetRefreshNumiCommandsFlag();
            }
            catch (Exception e) { Tracer.TraceException(e); }
        }

        public void DeleteNumiEntry(string schedulerId, string domainId)
        {
            Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.NUMI_ENTRY, string.Empty);
            if (string.IsNullOrEmpty(config.Domain))
                config.Domain = domainId;
            List<NumiEntry> numiEntries = BuildNumiEntries(config);
            int index = numiEntries.FindIndex(s => s.Calendar.Id == schedulerId);
            if (index != -1)
            {
                numiEntries.RemoveAt(index);
                SaveNumiEntries(config, numiEntries);
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.NUMI_ENTRY, string.Empty);
            }
            RefreshNumiCommands();
        }

        #region WhiteboardReportEntries

        public List<WhiteboardReportConfigEntry> GetWhiteboardReportEntries(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                return (config != null && config.SectionList != null && config.SectionList.Count > 0) ? BuildWhiteboardReportEntries(config) : new List<WhiteboardReportConfigEntry>();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public JobCalendar GetFacilityWhiteboardReportScheduler(string domainId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                if (config.SectionList != null)
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {
                        foreach (Element element in kvp.Value)
                        {
                            if (element.Key.Equals(facilityId.extension, StringComparison.InvariantCultureIgnoreCase))
                            {
                                string schedulerId = kvp.Key.Replace(Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                                string schedulerKey = Constants.JOB_CALENDAR + schedulerId;
                                config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.JOB_CALENDAR, schedulerKey);
                                if (config.SectionList != null)
                                {
                                    Element schedulerElement = null;
                                    schedulerElement = config.SectionList[schedulerKey].FirstOrDefault();
                                    if (schedulerElement != null)
                                    {
                                        XmlSerializer serializer = new XmlSerializer(typeof(VI.Admin.JobCalendar));
                                        ST st = null;
                                        st = schedulerElement.Value as ST;
                                        if (st != null)
                                        {
                                            using (StringReader sr = new StringReader(st.text))
                                                return (VI.Admin.JobCalendar)serializer.Deserialize(sr);
                                        }
                                    }
                                }
                                return null;
                            }
                        }
                    }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveFacilityWhiteboardReportScheduler(string domainId, string newSchedulerId, string oldSchedulerId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                if (string.IsNullOrEmpty(config.Domain))
                {
                    config.ConfigType = ConfigType.APP;
                    config.UserName = string.Empty;
                    config.Domain = domainId;
                }
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();
                StringBuilder oldSectionKey = new StringBuilder();
                StringBuilder newSectionKey = new StringBuilder();
                newSectionKey.Append(Constants.WHITEBOARD_REPORT_ENTRY);
                newSectionKey.Append(newSchedulerId);
                if (!string.IsNullOrEmpty(oldSchedulerId))
                {
                    oldSectionKey.Append(Constants.WHITEBOARD_REPORT_ENTRY);
                    oldSectionKey.Append(oldSchedulerId);
                    if (config.SectionList != null && config.SectionList.ContainsKey(oldSectionKey.ToString()))
                    {
                        List<Element> kvp = config.SectionList[oldSectionKey.ToString()];
                        Element deleteElement = kvp.Where(a => a.Key.Equals(facilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        kvp.Remove(deleteElement);
                        config.SectionList[oldSectionKey.ToString()] = kvp;
                    }
                }

                if (!string.IsNullOrEmpty(newSchedulerId))
                {
                    List<Element> kvp = new List<Element>();
                    if (config.SectionList.ContainsKey(newSectionKey.ToString()))
                        kvp = config.SectionList[newSectionKey.ToString()];
                    Element saveElement = new Element() { Key = facilityId.extension, Value = new ST(facilityId.root), FriendlyName = "" };
                    kvp.Add(saveElement);
                    config.SectionList[newSectionKey.ToString()] = kvp;
                }
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }

        }

        public void SaveWhiteboardReportEntry(WhiteboardReportConfigEntry whiteboardReportEntry, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                if (string.IsNullOrEmpty(config.Domain))
                {
                    config.ConfigType = ConfigType.APP;
                    config.UserName = string.Empty;
                    config.Domain = domainId;
                }
                List<WhiteboardReportConfigEntry> wrEntries = BuildWhiteboardReportEntries(config);
                int index = wrEntries.FindIndex(s => s.Calendar.Id == whiteboardReportEntry.Calendar.Id);
                if (index == -1)
                    wrEntries.Add(whiteboardReportEntry);
                else
                    wrEntries[index] = whiteboardReportEntry;
                SaveWhiteboardReportEntries(config, wrEntries);
                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                RefreshWhiteboardReportCommands();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteWhiteboardReportEntry(string schedulerId, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                List<WhiteboardReportConfigEntry> wrEntries = BuildWhiteboardReportEntries(config);
                int index = wrEntries.FindIndex(s => s.Calendar.Id == schedulerId);
                if (index != -1)
                {
                    wrEntries.RemoveAt(index);
                    SaveWhiteboardReportEntries(config, wrEntries);
                    BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                }
                RefreshWhiteboardReportCommands();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<WhiteboardReportConfigEntry> BuildWhiteboardReportEntries(Configuration config)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<WhiteboardReportConfigEntry> wrEntries = new List<WhiteboardReportConfigEntry>();
                List<VI.Admin.JobCalendar> availableCaledars = BuildCalendarsFromConfig(config.Domain).ToList();
                List<Facility> facilities = FacadeManager.EntityInterface.GetFacilitiesUsingWindowsAuthentication().ToList();
                if (config.SectionList != null)
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {
                        WhiteboardReportConfigEntry wrEntry = new WhiteboardReportConfigEntry();
                        string schedulerId = kvp.Key.Replace(Constants.WHITEBOARD_REPORT_ENTRY, string.Empty);
                        wrEntry.Calendar = availableCaledars.First(s => s.Id == schedulerId);
                        wrEntry.Facilities = new List<Facility>();
                        foreach (Element element in kvp.Value)
                        {
                            ST value = element.Value as ST;
                            Facility facility = facilities.FirstOrDefault(s => s.Id.extension.Equals(element.Key, StringComparison.InvariantCultureIgnoreCase) && s.Id.root.Equals(value.text, StringComparison.InvariantCultureIgnoreCase));
                            
                            if (facility != null)
                                wrEntry.Facilities.Add(facility);
                        }
                        wrEntries.Add(wrEntry);
                    }
                return wrEntries;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SaveWhiteboardReportEntries(Configuration config, List<WhiteboardReportConfigEntry> wrEntries)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (config.SectionList != null)
                {
                    List<string> keysToRemove = config.SectionList.Keys.Where(s => s.StartsWith(Constants.WHITEBOARD_REPORT_ENTRY)).ToList();
                    keysToRemove.ForEach(s => config.SectionList.Remove(s));
                }
                foreach (var wrEntry in wrEntries)
                {
                    List<Element> sectionList = wrEntry.Facilities.Select(s => new Element() { Key = s.Id.extension, Value = new ST(s.Id.root), FriendlyName = "" }).ToList();
                    if (config.SectionList == null)
                        config.SectionList = new Dictionary<string, List<Element>>();
                    config.SectionList[Constants.WHITEBOARD_REPORT_ENTRY + wrEntry.Calendar.Id] = sectionList;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RefreshWhiteboardReportCommands()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.VistaQueryClient.SetRefreshWhiteboardReportCommandsFlag();
            }
            catch (Exception e) { Tracer.TraceException(e); }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        public void GetConfigurationsForHomePage(II facilityId, out bool isEvacuation, out string marqueeText, out FacilitySettings facilitySettings, out List<Module> flags)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = facilityId.root;
                string facilityKey = Constants.FACILITY_SETTINGS + " " + facilityId.extension.ToUpper();
                string isEvacuationKey = Constants.IS_EVACUATION + " " + facilityId.extension.ToUpper();
                string moduleFacilityKey = Constants.BED_BOARD_MODULE + " " + facilityId.extension.ToUpper();
                List<string> configKeys = new List<string>();
                configKeys.Add(facilityKey);
                configKeys.Add(Constants.SETTINGS_SisterSites);
                configKeys.Add(Constants.MARQUEE_TEXT);
                configKeys.Add(isEvacuationKey);
                configKeys.Add(moduleFacilityKey);
                Dictionary<string, Configuration> result = BMSFactory.ConfigurationOperationsClientWindows.GetApplicationConfigurationForHomePage(domainId, configKeys);

                // facilitySettings
                Configuration config = result[facilityKey];
                facilitySettings = new FacilitySettings();
                facilitySettings.SisterSitesGroups = new List<SisterSitesGroup>();

                if (config.SectionList != null)
                {
                    foreach (Element element in config.SectionList[facilityKey])
                        BuildFacilitySettingsFromConfiguration(element, facilitySettings);
                }

                //get sister sites group
                config = result[Constants.SETTINGS_SisterSites];
                if (config.SectionList != null)
                {
                    foreach (Element element in config.SectionList[Constants.SETTINGS_SisterSites])
                    {
                        string siteSisterIdentifiers = ((ST)element.Value).text;
                        if (siteSisterIdentifiers.Contains(facilityId.extension))
                            facilitySettings.SisterSitesGroups.Add(new SisterSitesGroup() { RecordNumber = Convert.ToInt32(element.Key), SiteCodeList = siteSisterIdentifiers });
                    }
                }

                //Marquee text
                config = result[Constants.MARQUEE_TEXT];
                marqueeText = string.Empty;
                if (config.SectionList != null)
                {
                    Element activeIndexElem = config.SectionList[Constants.MARQUEE_TEXT].Where(elem => elem.Key == Constants.MARQUEE_ACTIVE_INDEX).FirstOrDefault();
                    if (activeIndexElem != null)
                    {
                        string activeIndex = ((ST)activeIndexElem.Value).text;
                        Element marqueeElem = config.SectionList[Constants.MARQUEE_TEXT].Where(elem => elem.Key == activeIndex).FirstOrDefault();

                        if (marqueeElem != null)
                            marqueeText = ((ST)marqueeElem.Value).text;
                    }
                    else
                    {
                        // legacy style marquee config
                        Element element = config.SectionList[Constants.MARQUEE_TEXT].FirstOrDefault();
                        if (element != null)
                            marqueeText = ((ST)element.Value).text;
                    }
                }

                // Is Evacuation
                config = result[isEvacuationKey];
                isEvacuation = false;
                if (config.SectionList != null)
                {
                    Element element = config.SectionList[isEvacuationKey].FirstOrDefault();
                    if (element != null)
                        if (((ST)element.Value).text == Constants.ON)
                            isEvacuation = true;
                        else
                            isEvacuation = false;
                }

                // Bed Board Module flags
                flags = new List<Module>();
                CDList moduleCDList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.BedBoardModule);
                CD yes = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision).Find(a => a.code == Constants.Yes);

                Module module = null;
                config = result[moduleFacilityKey];

                //get modules from config table
                if (config.SectionList != null)
                {
                    foreach (KeyValuePair<string, List<Element>> kvp in config.SectionList)
                    {
                        module = new Module();
                        module.Name = kvp.Key.Replace(moduleFacilityKey + " ", string.Empty); //Constants.BED_BOARD_MODULE
                        module.Code = moduleCDList.Where(cd => cd.displayName == module.Name)
                                                  .Select(cd => cd.code)
                                                  .FirstOrDefault() ?? string.Empty;

                        foreach (Element element in kvp.Value)
                            module.CurrentlyInUse = (CD)element.Value;

                        flags.Add(module);
                    }
                }

                foreach (CD moduleCD in moduleCDList)
                {
                    //add the module in config if it is newly added in EVS
                    if (flags.ToList<Module>().Find(a => a.Name == moduleCD.displayName) == null)
                    {
                        module = new Module();
                        module.Name = moduleCD.displayName;
                        module.Code = moduleCDList.Where(cd => cd.displayName == module.Name).Select(cd => cd.code).FirstOrDefault() ?? string.Empty;
                        module.CurrentlyInUse = yes;
                        flags.Add(module);
                        AddEditModule(module, domainId, facilityId);
                    }
                }

                for (int i = flags.Count; i > 0; i--)
                {
                    Module mod = flags[i - 1];
                    //if a cd is in the config module list and is not in the cd list -> delete from module list
                    if (moduleCDList.Find(a => a.displayName == mod.Name) == null)
                    {
                        flags.Remove(mod);
                        DeleteModule(mod, domainId, facilityId);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveUserPPBPSettings(User user, string ppbpKey, string settings)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = user.Domain;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WAITING_LIST_USER_SETTINGS, String.Empty);

                config.ConfigType = ConfigType.APP;
                config.UserName = string.Empty;
                config.Domain = domainId;
                if (config.SectionList == null)
                    config.SectionList = new Dictionary<string, List<Element>>();

                List<Element> elementList = null;
                if (!config.SectionList.ContainsKey(Constants.WAITING_LIST_USER_SETTINGS))
                {
                    elementList = new List<Element>();
                    config.SectionList.Add(Constants.WAITING_LIST_USER_SETTINGS, elementList);
                }
                else
                    elementList = config.SectionList[Constants.WAITING_LIST_USER_SETTINGS];

                Element element = config.SectionList[Constants.WAITING_LIST_USER_SETTINGS].Where(el => el.Key == ppbpKey).FirstOrDefault();
                if (element != null)
                    element.Value = new ST(settings);
                else
                    elementList.Add(new Element() { Key = ppbpKey, Value = new ST(settings), FriendlyName = string.Empty });

                BMSFactory.ConfigurationOperationsClientFromWCF.SetApplicationConfiguration(config, Constants.WAITING_LIST_USER_SETTINGS, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string GetUserPPBPSettings(User user, string ppbpKey)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string domainId = user.Domain;
                string settings = String.Empty;
                Configuration config = BMSFactory.ConfigurationOperationsClientFromWCF.GetApplicationConfiguration(domainId, Constants.WAITING_LIST_USER_SETTINGS, String.Empty);
                if (config.SectionList != null && config.SectionList.ContainsKey(Constants.WAITING_LIST_USER_SETTINGS))
                {
                    foreach (Element element in config.SectionList[Constants.WAITING_LIST_USER_SETTINGS])
                    {
                        if (element != null && element.Key == ppbpKey)
                        {
                            settings = ((ST)element.Value).text;
                            break;
                        }
                    }

                    return settings;
                }

                return settings;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
