﻿using System;
using System.Collections.Generic;
using System.Linq;
using InfoWorld.HL7.ITS;
using BMS.Facade.Service;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.ServicesWrapper.BMService;
using DC = BMS.DataContracts;
using BMS.Facade.Translators;
using BMS.Facade.Fault;
using BMS.Utils.Properties;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.Security;

namespace BMS.Facade.Implementation
{
    public class ADTOperationsImplementation : IADTOperations
    {
        /// <summary>
        /// Gets the patients waiting list.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="filterAdmissionCreationTime">The filter admission creation time.</param>
        /// <param name="isOnlyCurrentWaitingAdmission">if set to true get all the admission where remove date is null esle get all admission where admission creation date is GTE the filter admission date.</param>        
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        public IList<AdmissionInfo> GetWaitingList(Guid facilityUid, DateTime filterAdmissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission, bool isMentalHealth, bool isEmergencyManagement, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Bed> roomBedAssignedList = new List<Bed>();
                IList<AdmissionInfo> admissionInfoList = BMSFactory.BedManagerQueryClient.FilterAdmissions(facilityUid, filterAdmissionCreationTime, isOnlyCurrentWaitingAdmission, isOnlyInHouseWaitingAdmission, isMentalHealth, isEmergencyManagement).Select<DC.AdmissionInfo, AdmissionInfo>(dcAI => DateTimeConverter.ConvertDateFromUTC(dcAI.ToFacadeContract(), facilityId)).ToList();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                roomBedAssignedList = EISFactory.InstanceFromWCF.GetBeds(admissionInfoList.Select(bi => bi.RoomBedAssignedId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id);
                List<II> patientIds = (from item in admissionInfoList
                                       where !string.IsNullOrEmpty(item.RootPatientId) && !string.IsNullOrEmpty(item.ExtensionPatientId) && item.ExtensionPatientId != Guid.Empty.ToString()
                                       select new II()
                                       {
                                           root = item.RootPatientId,
                                           extension = item.ExtensionPatientId
                                       }).ToList();
                IList<Patient> patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(patientIds.Distinct().ToList(), null);
                Bed roomBedAssigned = null;
                Patient assignedPatient = null;
                List<CD> genderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Gender);
                Dictionary<string, string> genders = new Dictionary<string,string>();
                genderList.ForEach(a => genders.Add(a.code, a.displayName.Substring(0, 1)));                
                foreach (AdmissionInfo admission in admissionInfoList)
                {
                    if (admission.RoomBedAssignedId != null)
                    {
                        roomBedAssigned = roomBedAssignedList.Where(b => b.Id.extension.Equals(admission.RoomBedAssignedId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RoomBedAssignedId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (roomBedAssigned != null)
                            admission.RoomBedAssigned = roomBedAssigned.Name;
                    }

                    assignedPatient = patientList.Where(b => b.Id.extension.Equals(admission.ExtensionPatientId, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RootPatientId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (assignedPatient != null)
                    {
                        admission.RootPatientId = assignedPatient.Id.root;
                        admission.PatientName = string.Format("{0}, {1}", assignedPatient.LastName, assignedPatient.FirstName);
                        admission.Gender = (assignedPatient.Gender != null && !string.IsNullOrEmpty(assignedPatient.Gender.code)) ? genders[assignedPatient.Gender.code] : string.Empty;
                        admission.PatientLastFourDigitsOfSSN = string.Format("{0}{1}", assignedPatient.LastName.Substring(0, 1), assignedPatient.SSN.extension.Substring(assignedPatient.SSN.extension.Length - 4));
                        admission.DateOfBirth = assignedPatient.DateOfBirth;
                        admission.Weight = assignedPatient.Weight;
                    }
                }
                return admissionInfoList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the patients in community hospitals.
        /// </summary>
        /// <param name="facility">The facility.</param>
        /// <param name="facilityVISN">The facility VISN.</param>
        /// <returns></returns>
        public IList<TransferInfo> GetPatientsInCommunityHospitals(Facility facility, Visn facilityVISN)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Guid? facilityUid = null;
                if (facility != null && facility.Id != null && !string.IsNullOrEmpty(facility.Id.extension))
                    facilityUid = Guid.Parse(facility.Id.extension);
                List<Facility> eisFacilities = FacadeManager.EntityInterface.GetFacilitiesByVisn(facilityVISN.Id).ToList();
                List<TransferInfo> transfers = BMSFactory.BedManagerQueryClient.FilterTransfers(facilityUid, facilityVISN.Number).
                    Select<DC.TransferInfo, TransferInfo>(dcTI => DateTimeConverter.ConvertDateFromUTC(dcTI.ToFacadeContract(eisFacilities), dcTI.FacilityId == null ? null : dcTI.FacilityId)).ToList();
                IList<Patient> patientList = new List<Patient>();
                patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(transfers.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);

                Dictionary<string, int> nameAndSSNIndex;
                Dictionary<string, Patient> patientMap;
                BuildSimilarNameSSNIndex(patientList, out nameAndSSNIndex, out patientMap);

                foreach (TransferInfo transfer in transfers)
                {
                    Patient patient = patientMap.ContainsKey(transfer.PatientId.extension.ToUpperInvariant()) ? patientMap[transfer.PatientId.extension.ToUpperInvariant()] : null;
                    if (patient != null)
                    {
                        transfer.PatientName = string.Format("{0}, {1}", patient.LastName, patient.FirstName);
                        transfer.SCRating = patient.ServiceConnectedPercentage;
                        /*HHG: Show only last four SSN*/
                        transfer.PatientLastFourDigitsOfSSN = patient.SSN.extension.Remove(0, 7);

                        string first4LastName = patient.LastName.Length > 3 ? patient.LastName.ToLower().Substring(0, 4) : patient.LastName;
                        transfer.HasSimilarNameOrSSN = (nameAndSSNIndex.ContainsKey(transfer.PatientLastFourDigitsOfSSN) && nameAndSSNIndex[transfer.PatientLastFourDigitsOfSSN] > 1) || (nameAndSSNIndex.ContainsKey(first4LastName) && nameAndSSNIndex[first4LastName] > 1);
                    }
                }
                return transfers;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<ValidationAlert> TransferRequest(Transfer transfer)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Transfer GetTransferById(II id, User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.Transfer result = BMSFactory.BedManagerOperationsClient.GetTransfer(id);
                if (result == null)
                    throw new BedManagerFault(Resources.BMS_TRANSFER_NOT_EXISTS);
                Transfer transfer = result.ToFacadeContract();
                transfer = DateTimeConverter.ConvertDateFromUTC(transfer, transfer.FacilityFrom.Id);
                if (transfer.FacilityFrom != null && transfer.FacilityFrom.Id != null)
                    transfer.FacilityFrom = FacadeManager.EntityInterface.GetFacility(transfer.FacilityFrom.Id);
                if (transfer.FacilityTo != null && transfer.FacilityTo.Id != null)
                    transfer.FacilityTo = FacadeManager.EntityInterface.GetFacility(transfer.FacilityTo.Id);
                if (transfer.PatientId != null)
                    transfer.Patient = FacadeManager.EntityInterface.GetPatientById(transfer.PatientId, null);
                return transfer;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<TransferInfo> GetPatientsInCommunityHospitalsStandardView(bool isDisplayOnlyNationalPatients, string regionCode, string VISNCode, CD specialty, bool isCurrentWaitingList, DateTime transferDateTime)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<TransferInfo> transfers = new List<TransferInfo>();
                string specialtyCode = specialty == null ? null : specialty.code;
                string specialtyCodeSystem = specialty == null ? null : specialty.codeSystem;
                IList<DC.PatientWaitingStandardView> patientWaitingList = BMSFactory.BedManagerQueryClient.FilterNationalPatientWaitingStandardView(isDisplayOnlyNationalPatients, regionCode, VISNCode, specialtyCode, specialtyCodeSystem, transferDateTime, isCurrentWaitingList);
                if (patientWaitingList != null && patientWaitingList.Count > 0)
                {
                    Dictionary<string, string> facilities = new Dictionary<string, string>(StringComparer.InvariantCultureIgnoreCase);
                    List<CD> timeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                    foreach (DC.PatientWaitingStandardView a in patientWaitingList)
                    {
                        if (a.FacilityId != null && !string.IsNullOrEmpty(a.FacilityId.extension) && !a.FacilityId.extension.Equals(Guid.Empty.ToString()) && !facilities.ContainsKey(a.FacilityId.extension))
                            facilities.Add(a.FacilityId.extension, timeZoneList.Where(x => x.displayName.Equals(FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(a.FacilityId).Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().code);
                    }

                    transfers = patientWaitingList.Select<DC.PatientWaitingStandardView, TransferInfo>(dcTI => DateTimeConverter.ConvertDateFromUTC(dcTI.ToFacadeContract(), dcTI.FacilityId == null ? null : dcTI.FacilityId)).ToList();
                    IList<Patient> patientList = new List<Patient>();
                    patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(transfers.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);
                    foreach (TransferInfo ti in transfers)
                    {
                        if (ti.FacilityFrom != null && ti.FacilityFrom.Id != null && facilities.ContainsKey(ti.FacilityFrom.Id.extension))
                            ti.FacilityTimeZoneCode = facilities[ti.FacilityFrom.Id.extension];

                        Patient patient = patientList.Where(b => b.Id.extension.Equals(ti.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(ti.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (patient != null)
                        {
                            ti.PatientName = string.Format("{0}, {1}", patient.LastName, patient.FirstName);
                            ti.SCRating = patient.ServiceConnectedPercentage;
                            ti.PatientLastFourDigitsOfSSN = patient.SSN.extension.Substring(patient.SSN.extension.Length - 4);
                        }
                    }
                }
                return transfers;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<DateView> GetPatientsInCommunityHospitalsDateView(bool isDisplayOnlyNationalPatients, string regionCode, string VISNCode, CD specialty, bool isCurrentWaitingList, DateTime transferDateTime)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<DateView> datesView = new List<DateView>();
                string specialtyCode = specialty == null ? null : specialty.code;
                string specialtyCodeSystem = specialty == null ? null : specialty.codeSystem;
                IList<DC.PatientWaitingDateView> patientWaitingList = BMSFactory.BedManagerQueryClient.FilterNationalPatientWaitingDateView(isDisplayOnlyNationalPatients, regionCode, VISNCode, specialtyCode, specialtyCodeSystem, transferDateTime, isCurrentWaitingList);

                if (patientWaitingList != null && patientWaitingList.Count > 0)
                {
                    Dictionary<string, string> facilities = new Dictionary<string, string>(StringComparer.InvariantCultureIgnoreCase);
                    List<CD> timeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                    foreach (DC.PatientWaitingDateView a in patientWaitingList)
                    {
                        if (a.FacilityId != null && !string.IsNullOrEmpty(a.FacilityId.extension) && !a.FacilityId.extension.Equals(Guid.Empty.ToString()) && !facilities.ContainsKey(a.FacilityId.extension))
                            facilities.Add(a.FacilityId.extension, timeZoneList.Where(x => x.displayName.Equals(FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(a.FacilityId).Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().code);
                        if (a.AdmissionFacilityId != null && !string.IsNullOrEmpty(a.AdmissionFacilityId.extension) && !a.AdmissionFacilityId.extension.Equals(Guid.Empty.ToString()) && !facilities.ContainsKey(a.AdmissionFacilityId.extension))
                            facilities.Add(a.AdmissionFacilityId.extension, timeZoneList.Where(x => x.displayName.Equals(FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(a.AdmissionFacilityId).Id, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().code);
                    }

                    List<Facility> eisFacilities = FacadeManager.EntityInterface.GetFacilities().ToList();
                    datesView = patientWaitingList.Select<DC.PatientWaitingDateView, DateView>(dcDV => dcDV.ToFacadeContract(eisFacilities)).ToList();
                    List<DateView> datesViewList = new List<DateView>();
                    datesView.ToList().ForEach((item) =>
                    {
                        datesViewList.Add(new DateView() { DispositionDate = item.DispositionDate, FacilityTo = item.FacilityTo });
                    });

                    datesView = datesView.Select<DateView, DateView>(dcDV => DateTimeConverter.ConvertDateFromUTC(dcDV, (dcDV.FacilityFrom == null) ? null : dcDV.FacilityFrom.Id)).ToList();
                    datesViewList = datesViewList.Select<DateView, DateView>(dcDV => DateTimeConverter.ConvertDateFromUTC(dcDV, (dcDV.FacilityTo == null) ? null : dcDV.FacilityTo.Id)).ToList();
                    for (int i = 0; i < datesView.Count; i++)
                        datesView[i].DispositionDate = datesViewList[i].DispositionDate;
                    IList<Patient> patientList = new List<Patient>();
                    patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(datesView.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);

                    Dictionary<string, int> nameAndSSNIndex;
                    Dictionary<string, Patient> patientMap;
                    BuildSimilarNameSSNIndex(patientList, out nameAndSSNIndex, out patientMap);

                    foreach (DateView dw in datesView)
                    {
                        if (dw.FacilityFrom != null && dw.FacilityFrom.Id != null && facilities.ContainsKey(dw.FacilityFrom.Id.extension))
                            dw.FacilityFromTimeZoneCode = facilities[dw.FacilityFrom.Id.extension.ToLower()];
                        if (dw.FacilityTo != null && dw.FacilityTo.Id != null && facilities.ContainsKey(dw.FacilityTo.Id.extension))
                            dw.FacilityToTimeZoneCode = facilities[dw.FacilityTo.Id.extension.ToLower()];

                        Patient patient = patientMap.ContainsKey(dw.ExtensionPatientId.ToUpperInvariant()) ? patientMap[dw.ExtensionPatientId.ToUpperInvariant()] : null;
                        if (patient != null)
                        {
                            dw.PatientName = string.Format("{0}, {1}", patient.LastName, patient.FirstName);
                            dw.PatientLastFourDigitsOfSSN = patient.SSN.extension.Remove(0, 7);

                            string first4LastName = patient.LastName.Length > 3 ? patient.LastName.ToLower().Substring(0, 4) : patient.LastName;
                            dw.HasSimilarNameOrSSN = (nameAndSSNIndex.ContainsKey(dw.PatientLastFourDigitsOfSSN) && nameAndSSNIndex[dw.PatientLastFourDigitsOfSSN] > 1) || (nameAndSSNIndex.ContainsKey(first4LastName) && nameAndSSNIndex[first4LastName] > 1);
                        }
                    }
                }
                return datesView;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public WaitingListItem GetWaitingListItem(II id, User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.WaitingListItem result = BMSFactory.BedManagerOperationsClient.GetWaitingListItem(id, null);

                if (result == null)
                    throw new BedManagerFault(Resources.BMS_ADMISSION_NOT_EXISTS);
                Facility facility = null;
                Patient patient = null;
                Bed bed = null;
                if (result.FacilityId != null)
                    facility = FacadeManager.EntityInterface.GetFacility(result.FacilityId);
                if (result.PatientId != null)
                    patient = FacadeManager.EntityInterface.GetPatientById(result.PatientId, null);
                if (result.RequestedBedId != null)
                    bed = EISFactory.InstanceFromWCF.GetBed(result.RequestedBedId, user.VistaSite.Id);
                WaitingListItem wli = result.ToFacadeContract(facility, patient, bed);   
                wli = DateTimeConverter.ConvertDateFromUTC(wli, user.Facility.Id);
                return wli;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public PatientLocation GetPatientLocation(II patientId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.PatientLocation patientLocation = BMSFactory.BedManagerOperationsClient.GetPatientLocation(patientId);
                if (patientLocation == null)
                    return null;

                return new PatientLocation()
                    {
                        CurrentAdmissionBed = (patientLocation.CurrentAdmissionBedId != null) ? EISFactory.InstanceFromWCF.GetBed(patientLocation.CurrentAdmissionBedId, vistaSiteId) : null,
                        CurrentAdmissionWard = (patientLocation.CurrentAdmissionWardId != null) ? EISFactory.InstanceFromWCF.GetWard(patientLocation.CurrentAdmissionWardId) : null
                    };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void ModifyWaitingListItem(WaitingListItem waitingListItem, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DateTimeConverter.ConvertDateToUTC(waitingListItem, facility.Id);
                DC.WaitingListItem result = waitingListItem.ToDataContract();
                BMSFactory.BedManagerOperationsClient.UpdateWaitingListItem(result);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Act GetLastAct(string patientSSN, ActType actType, string vistaSiteNumber)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string querySSN = patientSSN;
                //if the ssn is without dashes, they must be added before making the query in EIS.
                if (patientSSN.Length == 9)
                    querySSN = patientSSN.Insert(3, "-").Insert(6, "-");
                Patient patient = EISFactory.InstanceFromWCF.GetPatientBySsn(new II(Constants.SSNROOT, querySSN), null);
                if (patient == null)
                    return null;
                IList<Facility> facilities = EISFactory.InstanceFromWCF.GetFacilitiesBySiteNumber(vistaSiteNumber);
                if (facilities == null || facilities.Count == 0)
                    return null;

                return BMSFactory.BedManagerQueryClientFromWCF.GetLastAct(patient.Id, actType, facilities[0].VistaSite.Id).ToFacadeContract();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public PatientWaitingCount GetPatientWaitingCount(II facilityId, II vistaSiteId, string vistaDivisionCode, int timeZoneOffset)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.PatientWaitingCount result = BMSFactory.BedManagerQueryClient.GetPatientsWaitingCount(Guid.Parse(facilityId.extension), Guid.Parse(vistaSiteId.extension), vistaDivisionCode, timeZoneOffset);
                return new PatientWaitingCount()
                {
                    WaitingListCount = result.WaitingListCount,
                    ScheduledAdmissionsCount = result.ScheduledAdmissionsCount
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<WaitingListReport> GetWaitingListReport(II facilityId, string filterBy, string filterPatientName, string filterGender)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<WaitingListReport> result = new List<WaitingListReport>();
                List<WaitingListReport> waitingListReport = BMSFactory.BedManagerQueryClient.FilterWaitingListReport(Guid.Parse(facilityId.extension), filterBy).Select<DC.WaitingListReport, WaitingListReport>(dcAI => DateTimeConverter.ConvertDateFromUTC(dcAI.ToFacadeContract(), facilityId)).ToList();
                IList<Bed> roomBedAssignedList = new List<Bed>();
                IList<Patient> patientList = new List<Patient>();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                roomBedAssignedList = EISFactory.InstanceFromWCF.GetBeds(waitingListReport.Select(bi => bi.RoomBedAssignedId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id);
                patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(waitingListReport.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);

                string filterPatientFirstName = null, filterPatientLastName = null;
                if (filterPatientName.Contains(","))
                {
                    filterPatientLastName = filterPatientName.Substring(0,filterPatientName.IndexOf(",")).Trim();
                    filterPatientFirstName = filterPatientName.Substring(filterPatientName.IndexOf(",") + 1).Trim();
                }
                foreach (WaitingListReport item in waitingListReport)
                {
                    if (item.RoomBedAssignedId != null)
                    {
                        Bed roomBedAssigned = roomBedAssignedList.Where(b => b.Id.extension.Equals(item.RoomBedAssignedId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(item.RoomBedAssignedId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (roomBedAssigned != null)
                            item.RoomBedAssigned = roomBedAssigned.Name;
                    }

                    Patient assignedPatient = patientList.Where(b => b.Id.extension.Equals(item.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(item.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (!string.IsNullOrEmpty(filterPatientName) && assignedPatient != null)
                    {
                        if (!filterPatientName.Contains(","))
                        {
                            if (!assignedPatient.LastName.Contains(filterPatientName) && !assignedPatient.FirstName.Contains(filterPatientName))
                                continue;
                        }
                        else
                        {
                            if (!assignedPatient.LastName.Contains(filterPatientLastName) || !assignedPatient.FirstName.Contains(filterPatientFirstName))
                                continue;
                        }
                    }
                    if (!string.IsNullOrEmpty(filterGender) && assignedPatient != null && assignedPatient.Gender != null && assignedPatient.Gender.code != filterGender.Substring(0,1))
                        continue;

                    if (assignedPatient != null)
                    {
                        item.PatientGender = (assignedPatient.Gender != null) ? assignedPatient.Gender.GetDisplayName() : string.Empty;
                        item.PatientName = string.Format("{0}{1}{2}{3}", assignedPatient.LastName, ",", assignedPatient.FirstName, assignedPatient.SSN.extension.Substring(assignedPatient.SSN.extension.Length - 4));
                        result.Add(item);
                    }
                }

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<EvacuationPatient> GetEvacuationPatients(II facilityId, string wardList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                IList<EvacuationPatient> evacuationPatientList = BMSFactory.BedManagerQueryClientFromWCF.GetEvacuationPatients(wardList, facilityId).Select<DC.EvacuationPatient, EvacuationPatient>(dcEP => dcEP.ToFacadeContract()).ToList();
                IList<Patient> patientList = new List<Patient>();
                patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(evacuationPatientList.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);
                foreach (EvacuationPatient patient in evacuationPatientList)
                {
                    Patient evacuationPatient = patientList.Where(b => b.Id.extension.Equals(patient.PatientId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(patient.PatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (evacuationPatient != null)
                        patient.Patient = evacuationPatient;
                }
                return evacuationPatientList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public WaitingListItem GetWaitingListItem(II waitingListItemId, string patientUid)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.WaitingListItem result = BMSFactory.BedManagerOperationsClient.GetWaitingListItem(null, patientUid);

                if (result == null)
                    throw new BedManagerFault(Resources.BMS_ADMISSION_NOT_EXISTS);
                Facility facility = null;
                Patient patient = null;
                Bed bed = null;
                if (result.FacilityId != null)
                    facility = FacadeManager.EntityInterface.GetFacility(result.FacilityId);
                if (result.PatientId != null)
                    patient = FacadeManager.EntityInterface.GetPatientById(result.PatientId, null);
                if (result.RequestedBedId != null)
                    bed = EISFactory.InstanceFromWCF.GetBed(result.RequestedBedId, result.VistaSiteId);
                WaitingListItem wli = result.ToFacadeContract(facility, patient, bed);
                wli = DateTimeConverter.ConvertDateFromUTC(wli, FacadeManager.UserInterface.GetProfile().Facility.Id);
                return wli;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void GetHomePageBMSData(II vistaSiteId, Facility facility, string wardIdList, List<Module> flags, int timeZoneMinutesOffset, DateTime filterAdmissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission, bool isMentalHealth, bool isEmergencyManagement,
                                         out List<BedOccupancyCount> bedOccupancylist, out List<NewEventInfo> newEventsInfo, out IList<AdmissionInfo> facilityWaitingList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<DC.BedOccupancyCount> bmsOccupancyCount = new List<DC.BedOccupancyCount>();
                List<DC.AdmissionInfo> admissionInfoList = new List<DC.AdmissionInfo>();
                Guid facilityGuid = Guid.Parse(facility.Id.extension);

                BMSFactory.BedManagerQueryClient.GetHomePageBMSData(facilityGuid, vistaSiteId.extension, filterAdmissionCreationTime, isOnlyCurrentWaitingAdmission, isOnlyInHouseWaitingAdmission, isMentalHealth, isEmergencyManagement, out bmsOccupancyCount, out admissionInfoList);
                List<NewEventInfo> newEvents = Facade.FacadeManager.NewEventInterface.GetNewEvents(wardIdList, facilityGuid, timeZoneMinutesOffset);

                #region bed occupancy percent

                bedOccupancylist = GetBedOccupancyForFacilityOld(facility, bmsOccupancyCount);

                #endregion

                #region new events count

                newEventsInfo = new List<NewEventInfo>();
                foreach (NewEventInfo newEvent in newEvents)
                {
                    if (newEvent.EventCount == 0)
                        continue;

                    // filter out events that are in deactivated modules and assign report parameters
                    switch (newEvent.EventName)
                    {
                        case Utils.NewEvent.SignedAdmission:
                            if (IsModuleActive(flags, Constants.ADM_ORD))
                                newEventsInfo.Add(newEvent);
                            break;
                        case Utils.NewEvent.SignedTransfer:
                            if (IsModuleActive(flags, Constants.TRSF_ORD))
                                newEventsInfo.Add(newEvent);
                            break;
                        case Utils.NewEvent.SignedDischarge:
                            if (IsModuleActive(flags, Constants.DISCH_ORD))
                                newEventsInfo.Add(newEvent);
                            break;
                        case Utils.NewEvent.SignedAnticipatedDischarge:
                            if (IsModuleActive(flags, Constants.ANTIC_DISCH))
                                newEventsInfo.Add(newEvent);
                            break;
                        case Utils.NewEvent.DischargeAppointment:
                            if (IsModuleActive(flags, Constants.DISCHARGE_APPOINTMENT))
                                newEventsInfo.Add(newEvent);
                            break;
                        case Utils.NewEvent.VacatedBed:
                        case Utils.NewEvent.CurrentlyCleaningBed:
                        case Utils.NewEvent.CompletedCleaningBed:
                            if (IsModuleActive(flags, Constants.EMS_MODULE))
                                newEventsInfo.Add(newEvent);
                            break;
                        default:
                            newEventsInfo.Add(newEvent);
                            break;
                    }
                }

                newEventsInfo.Sort((a, b) => { return -1 * a.EventCount.CompareTo(b.EventCount); });

                #endregion

                #region admission waiting list

                facilityWaitingList = GetAdmissionList(facility.Id, filterAdmissionCreationTime, isOnlyCurrentWaitingAdmission, isOnlyInHouseWaitingAdmission, isMentalHealth, isEmergencyManagement);

                #endregion
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<BedOccupancyCount> GetBedOccupancyForFacility(Facility facility, out Dictionary<string, List<BedInfo>> bedsByDivisionId)
        {
            List<BedOccupancyCount> bedOccupancylist;
            IList<Division> divisions = EISFactory.InstanceFromWCF.GetDivisions(facility.Id);

            Dictionary<string, Ward> wards = FacadeManager.EntityInterface.GetWardsByFacility(facility.Id).ToDictionary(item => item.Id.extension);
            II selectedDivision = new II("", (default(Guid)).ToString());
            IList<BedInfo> beds = FacadeManager.BedInterface.GetBedOccupancy(wards.Values, selectedDivision, facility, null, true, null);

            Dictionary<string, BedOccupancyCount> wardGroupBedCounts = new Dictionary<string, BedOccupancyCount>();
            BedOccupancyCount allCount = new BedOccupancyCount() { FacilityId = facility.Id, DivisionName = "All" };
            wardGroupBedCounts.Add(Guid.Empty.ToString(), allCount);
            List<BedInfo> facilityBeds = new List<BedInfo>();

            bedsByDivisionId = new Dictionary<string, List<BedInfo>>();
            bedsByDivisionId.Add(Guid.Empty.ToString(), facilityBeds);

            foreach (Division division in divisions)
            {
                BedOccupancyCount boc = new BedOccupancyCount() { FacilityId = facility.Id, DivisionName = division.Name };
                wardGroupBedCounts.Add(division.Id.extension, boc);

                List<BedInfo> divisionBeds = new List<BedInfo>();

                if (!bedsByDivisionId.ContainsKey(division.Id.extension))
                    bedsByDivisionId.Add(division.Id.extension, divisionBeds);

                foreach (BedInfo bed in beds)
                {
                    string idExtension = bed.BedId.extension;
                    if (bed.DivisionList.Count(d => d.Id.extension == division.Id.extension) > 0) // get beds in this division
                    {
                        divisionBeds.Add(bed);
                        facilityBeds.Add(bed);

                        bool bedHasEmptyGuid = bed.BedId.extension.Equals(Guid.Empty.ToString(), StringComparison.InvariantCultureIgnoreCase);
                        bool hasPatient = !String.IsNullOrEmpty(bed.PatientId.extension) || bedHasEmptyGuid;

                        if (!bed.BedAvailabilityStatusCode.Equals(Constants.BED_OOS_FROM_BEDBOARD, StringComparison.InvariantCultureIgnoreCase) || hasPatient)
                            ++boc.TotalBeds;

                        if (hasPatient)
                            ++boc.TotalOccupiedBeds;

                        if (bed.WardId != null && wards.ContainsKey(bed.WardId.extension) && wards[bed.WardId.extension].CensusCategory != null && wards[bed.WardId.extension].Division.Name == boc.DivisionName)
                            boc.CensusCategory = wards[bed.WardId.extension].CensusCategory.displayName;
                    }
                }

                allCount.TotalBeds += boc.TotalBeds;
                allCount.TotalOccupiedBeds += boc.TotalOccupiedBeds;
            }

            // repeat loop through divisions and build virtual wards
            foreach (Division division in divisions)
            {
                // find wards for this division
                IEnumerable<Ward> divisionWards = wards.Values.Where(w => w.Division.Id.extension == division.Id.extension);
                foreach (Ward divisionWard in divisionWards)
                {
                    if (divisionWard.AssignedVirtualWard != null)
                    {
                        Ward assignedVirtualWard = null;
                        if (wards.TryGetValue(divisionWard.AssignedVirtualWard.Id.extension, out assignedVirtualWard))
                        {
                            bedsByDivisionId[assignedVirtualWard.Division.Id.extension].AddRange(bedsByDivisionId[division.Id.extension]);
                        }
                    }
                }
            }

            bedOccupancylist = wardGroupBedCounts.Values.ToList();
            return bedOccupancylist;
        }

        public List<BedOccupancyCount> GetBedOccupancyForFacilityOld(Facility facility, List<DC.BedOccupancyCount> bmsOccupancyCount)
        {
            List<BedOccupancyCount> bedOccupancylist;
            IList<Division> divisions = EISFactory.InstanceFromWCF.GetDivisions(facility.Id);

            Dictionary<string, Bed> beds = new Dictionary<string, Bed>();
            Dictionary<string, Division> bedIdDivisionMap = new Dictionary<string, Division>();
            Dictionary<string, BedOccupancyCount> wardGroupBedCounts = new Dictionary<string, BedOccupancyCount>();

            foreach (Division division in divisions)
            {
                IList<Ward> wards = FacadeManager.EntityInterface.GetWardsByDivisionId(division.Id);
                if (wards.Count == 1)
                {
                    Ward ward = wards.FirstOrDefault();
                    if (ward == null || ward.IsVirtual)
                        continue;
                }

                IList<Bed> divisionBeds = EISFactory.InstanceFromWCF.GetBedsInDivision(division.Id, facility.VistaSite.Id);
                BedOccupancyCount boc = new BedOccupancyCount() { FacilityId = facility.Id, DivisionName = division.Name, TotalBeds = divisionBeds.Count };
                wardGroupBedCounts.Add(division.Id.extension, boc);

                foreach (Bed divisionBed in divisionBeds)
                {
                    string idExtension = divisionBed.Id.extension;
                    if (!beds.ContainsKey(idExtension))
                    {
                        beds.Add(idExtension, divisionBed);
                        bedIdDivisionMap.Add(idExtension, division);
                    }
                }
            }

            foreach (DC.BedOccupancyCount bedOccupancyCountItem in bmsOccupancyCount)
            {
                if (!beds.ContainsKey(bedOccupancyCountItem.BedId.extension))
                    continue;

                string divisionUid = bedIdDivisionMap[bedOccupancyCountItem.BedId.extension].Id.extension;
                BedOccupancyCount boc = wardGroupBedCounts[divisionUid];

                Bed bed = beds[bedOccupancyCountItem.BedId.extension];
                foreach (Ward ward in bed.WardList)
                {
                    Ward completeWard = EISFactory.Instance.GetWard(ward.Id);
                    if (completeWard != null && completeWard.CensusCategory != null && completeWard.Division != null) // if the division is null, then this ward has been "deleted" in the ward configuration page
                        boc.CensusCategory = completeWard.CensusCategory.displayName;
                }

                if (bedOccupancyCountItem.IsOccupiedBed)
                    ++boc.TotalOccupiedBeds;

                if (bedOccupancyCountItem.IsUnavailableBed)
                    ++boc.TotalUnavailableBeds;
            }

            BedOccupancyCount allCount = new BedOccupancyCount() { FacilityId = facility.Id, DivisionName = "All" };
            allCount.TotalBeds = beds.Count;
            foreach (BedOccupancyCount bocItem in wardGroupBedCounts.Values)
            {
                allCount.TotalOccupiedBeds += bocItem.TotalOccupiedBeds;
                allCount.TotalUnavailableBeds += bocItem.TotalUnavailableBeds;
            }

            wardGroupBedCounts.Add(Guid.Empty.ToString(), allCount);
            bedOccupancylist = wardGroupBedCounts.Values.ToList();

            return bedOccupancylist;
        }

        public IList<AdmissionInfo> GetAdmissionList(II facilityId, DateTime filterAdmissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission, bool isMentalHealth, bool isEmergencyManagement)
        {
            Guid facilityGuid = Guid.Parse(facilityId.extension);
            IList<DC.AdmissionInfo> admissionInfoList = BMSFactory.BedManagerQueryClient.FilterAdmissions(facilityGuid, filterAdmissionCreationTime, isOnlyCurrentWaitingAdmission, isOnlyInHouseWaitingAdmission, isMentalHealth, isEmergencyManagement);

            IList<Bed> roomBedAssignedList = new List<Bed>();
            List<AdmissionInfo> facilityWaitingList = admissionInfoList.Select<DC.AdmissionInfo, AdmissionInfo>(dcAI => DateTimeConverter.ConvertDateFromUTC(dcAI.ToFacadeContract(), facilityId)).ToList();
            Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
            roomBedAssignedList = EISFactory.InstanceFromWCF.GetBeds(facilityWaitingList.Select(bi => bi.RoomBedAssignedId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id);
            
            List<II> patientIds = (from item in facilityWaitingList
                                   where !string.IsNullOrEmpty(item.RootPatientId) && !string.IsNullOrEmpty(item.ExtensionPatientId) && item.ExtensionPatientId != Guid.Empty.ToString()
                                   select new II()
                                   {
                                       root = item.RootPatientId,
                                       extension = item.ExtensionPatientId
                                   }).ToList();

            List<string> patientIdsString = new List<string>();
            patientIdsString = patientIds.Select(a => a.extension).ToList();
            Dictionary<Guid, DC.PatientIcon> patientIcons = BMSFactory.BedManagerQueryClient.GetPatientIcons(Constants.WHITEBOARD_ICON, Constants.EMERGENCY_ICON, String.Join(",", patientIdsString));
            Dictionary<int, IconInfo> whiteboardIcons = FacadeManager.IconInterface.GetWhiteboardIcons(facilityGuid).ToDictionary(ic => ic.IconId);

            IList<Patient> patientList = EISFactory.InstanceFromWCF.GetPatientsByIds(patientIds.Distinct().ToList(), null);

            // index patients and collect same/similar name and SSN info
            Dictionary<string, int> nameAndSSNIndex;
            Dictionary<string, Patient> patientMap;
            BuildSimilarNameSSNIndex(patientList, out nameAndSSNIndex, out patientMap);

            Bed roomBedAssigned = null;
            Patient assignedPatient = null;
            List<CD> genderList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Gender);
            Dictionary<string, string> genders = new Dictionary<string, string>();
            genderList.ForEach(a => genders.Add(a.code, a.displayName.Substring(0, 1)));

            foreach (AdmissionInfo admission in facilityWaitingList)
            {
                admission.EmergencyIcons = new List<IconInfo>();

                if (admission.RoomBedAssignedId != null)
                {
                    roomBedAssigned = roomBedAssignedList.Where(b => b.Id.extension.Equals(admission.RoomBedAssignedId.extension, StringComparison.InvariantCultureIgnoreCase) && b.Id.root.Equals(admission.RoomBedAssignedId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (roomBedAssigned != null)
                        admission.RoomBedAssigned = roomBedAssigned.Name;
                }

                assignedPatient = patientMap.ContainsKey(admission.ExtensionPatientId.ToUpperInvariant()) ? patientMap[admission.ExtensionPatientId.ToUpperInvariant()] : null;
                if (assignedPatient != null)
                {
                    admission.RootPatientId = assignedPatient.Id.root;
                    admission.PatientName = string.Format("{0}, {1}", assignedPatient.LastName, assignedPatient.FirstName);
                    admission.Gender = (assignedPatient.Gender != null && !string.IsNullOrEmpty(assignedPatient.Gender.code)) ? genders[assignedPatient.Gender.code] : string.Empty;

                    string last4SSN = assignedPatient.SSN.extension.Substring(assignedPatient.SSN.extension.Length - 4);
                    string first4LastName = assignedPatient.LastName.Length > 3 ? assignedPatient.LastName.ToLower().Substring(0, 4) : assignedPatient.LastName;

                    admission.PatientLastFourDigitsOfSSN = string.Format("{0}{1}", assignedPatient.LastName.Substring(0, 1), last4SSN);
                    admission.HasSimilarNameOrSSN = (nameAndSSNIndex.ContainsKey(last4SSN) && nameAndSSNIndex[last4SSN] > 1) || (nameAndSSNIndex.ContainsKey(first4LastName) && nameAndSSNIndex[first4LastName] > 1);

                    Guid patientId = Guid.Parse(assignedPatient.Id.extension);
                    if (patientIcons.ContainsKey(patientId))
                    {       
                        foreach (int patientIconId in patientIcons[patientId].IconFlagIds)
                        {
                            if (!whiteboardIcons.ContainsKey(patientIconId))
                                continue;

                            admission.EmergencyIcons.Add(whiteboardIcons[patientIconId]);
                        }
                    }
                    
                }
            }
            return facilityWaitingList;
        }

        public static void BuildSimilarNameSSNIndex(IList<Patient> patientList, out Dictionary<string, int> nameAndSSNIndex, out Dictionary<string, Patient> patientMap)
        {
            nameAndSSNIndex = new Dictionary<string, int>();
            patientMap = new Dictionary<string, Patient>();
            if (patientList != null)
            {
                foreach (Patient patient in patientList)
                {
                    if (patientMap.ContainsKey(patient.Id.extension))
                        continue;

                    patientMap.Add(patient.Id.extension.ToUpperInvariant(), patient);

                    string first4LastName = patient.LastName.Length > 3 ? patient.LastName.ToLower().Substring(0, 4) : patient.LastName;
                    if (!nameAndSSNIndex.ContainsKey(first4LastName))
                        nameAndSSNIndex.Add(first4LastName, 0);

                    ++nameAndSSNIndex[first4LastName];

                    string last4SSN = patient.SSN.extension.Substring(patient.SSN.extension.Length - 4);
                    if (!nameAndSSNIndex.ContainsKey(last4SSN))
                        nameAndSSNIndex.Add(last4SSN, 0);

                    ++nameAndSSNIndex[last4SSN];
                }
            }
        }

        

        #region Private Methods

        private bool IsModuleActive(IList<Module> flags, string moduleCode)
        {
            Module flag = flags.Where(a => a.Code.Equals(moduleCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
            return IsModuleActive(flag);
        }

        private bool IsModuleActive(Module flag)
        {
            return flag != null && flag.CurrentlyInUse.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase);
        }

        private string GetFacilitiesUids(IList<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string facilitiesUids = string.Empty;
                foreach (Facility facility in facilities)
                    facilitiesUids += facility.Id.extension + ";";
                return facilitiesUids;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
