﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.DirectoryServices.ActiveDirectory;
using System.DirectoryServices;
using System.Security.Principal;
using System.Diagnostics;
using System.IO;
using System.Reflection;

namespace ImportUsers
{
    class Program
    {
        private const string SUPPORT_ROLE = "SupportUsers";
        private const string NATIONAL_ROLE = "NationalUsers";
        private const string REGIONAL_ROLE = "RegionalUsers";
        private const string VISN_ROLE = "VisnUsers";
        private const string ADMIN_ROLE = "Administrators";
        private const string SITE_ROLE = "SiteUsers";
        private const string EMS_ROLE = "EmsStaff";

        private const string READ_DATA_PERMISSION = "BMS, Read";
        private const string WRITE_DATA_PERMISSION = "BMS, Write";

        private const string AUTHZ_INSERT_RESOURCE = "EXEC INS_RESOURCE_POLICIES @RESOURCE_ROOT=N'InfoWorld',@RESOURCE_EXTENSION=N'@FACILITY_ID',@RESOURCE_TYPE=N'HL7-RIM-V3-ORGANIZATION',@ROLE_ID=N'@USER_ROLE_ID',@ACTION_ID=N'@ACTION_PERMISSION_ID'";
        private const string AUTHZ_INSERT_ASSIGNED_ROLE = "EXEC INS_USER_ROLE @USER_ID='@USER_SID',@ROLE_ID='@ASSIGNED_ROLE_ID'";
        private static string RESOURCES_ALL = null;
        private static Dictionary<int, string> RESOURCES_REGION = new Dictionary<int,string>();
        private static Dictionary<int, string> RESOURCES_VISN = new Dictionary<int, string>();
        private static int TotalUsers = 0;

        public static void Main(string[] args)
        {
            Tracer.TraceMessage("Start import users...");
            InsertUsers();
            Tracer.TraceMessage("End import users...");
        }
        
        private static List<User> GetUsersFromAD()
        {
            try
            {
                List<User> result = SqlQueries.GetUsers();
                if (result == null || result.Count == 0)
                    return null;
                TotalUsers = result.Count;
                //query AD for sid and domain
                Forest currentForest = Forest.GetCurrentForest();
                GlobalCatalog globalCatalog = currentForest.FindGlobalCatalog();
                using (var searcher = globalCatalog.GetDirectorySearcher())
                {
                    searcher.PropertiesToLoad.Add("objectsid");
                    searcher.PropertiesToLoad.Add("distinguishedname");
                    SearchResult searchResult = null;
                    string dn = null;
                    StringBuilder domain = null;
                    foreach (User u in result)
                    {
                        searcher.Filter = ("(&(objectClass=user)(objectCategory=user) " + "(SamAccountName=" + u.UserName + "))");
                        searchResult = searcher.FindOne();
                        if (searchResult != null)
                        {
                            u.Sid = new SecurityIdentifier((byte[])(searchResult.Properties["objectsid"][0]), 0).ToString();
                            dn = searchResult.Properties["distinguishedname"][0].ToString();
                            domain = new StringBuilder();
                            foreach (string item in dn.Split(','))
                            {
                                if (item.StartsWith("DC="))
                                {
                                    domain.Append(item.Replace("DC=", string.Empty));
                                    domain.Append(".");
                                }
                            }
                            if (domain.Length > 0) domain = domain.Remove(domain.Length - 1, 1);
                            u.Domain = domain.ToString();
                            u.FullUserName = u.Domain + "\\" + u.UserName;
                        }
                        else
                        {
                            Tracer.TraceMessage("UserName not found in AD: " + u.UserName);
                        }
                    }
                }
                //remove users without sid
                result.RemoveAll(a => string.IsNullOrEmpty(a.Sid));
                return result;
            }
            catch (Exception ex)
            {
                Tracer.TraceException(ex);
                return null;
            }
        }              

        private static void InsertUsers()
        {
            List<Facility> facilities = SqlQueries.GetFacilities();
            if (facilities == null || facilities.Count == 0)
            {
                Tracer.TraceMessage("Facilities not found!");
                return;
            }
            List<User> users = GetUsersFromAD();
            if (users == null || users.Count == 0)
            {
                Tracer.TraceMessage("Users for import not found!");
                return;
            }
            Dictionary<string, string> actions = SqlQueries.GetActions();
            if (actions == null || actions.Count == 0)
            {
                Tracer.TraceMessage("Permissions not found in AUTHZ!");
                return;
            }
            else
            {
                if (!(actions.ContainsKey(READ_DATA_PERMISSION) && actions.ContainsKey(WRITE_DATA_PERMISSION)))
                {
                    Tracer.TraceMessage("Read and Write permissions not found in AUTHZ!");
                    return;
                }
            }            
            Dictionary<string, string> roles = SqlQueries.GetRoles();
            
            if (roles == null || roles.Count == 0)
            {
                Tracer.TraceMessage("Roles not found in AUTHZ!");
                return;
            }
            else
            {
                if (!(roles.ContainsKey(SUPPORT_ROLE) && roles.ContainsKey(NATIONAL_ROLE) && roles.ContainsKey(REGIONAL_ROLE) && roles.ContainsKey(VISN_ROLE) && roles.ContainsKey(ADMIN_ROLE)
                        && roles.ContainsKey(SITE_ROLE) && roles.ContainsKey(EMS_ROLE)))
                {
                    Tracer.TraceMessage("Application roles not defined in AUTHZ!");
                    return;
                }
            }

            string theDirectory = Path.GetDirectoryName(new Uri(Assembly.GetExecutingAssembly().CodeBase).LocalPath);
            //get sql for authz insert
            string authzInsertSQL = "";
            string filePath = theDirectory + "/SqlScripts/InsertUserAUTHZ.sql";
            using (FileStream strm = File.OpenRead(filePath))
            {
                StreamReader reader = new StreamReader(strm);
                authzInsertSQL = reader.ReadToEnd();
            }
            //get sql for config insert
            string configInsertSQL = "";
            filePath = theDirectory + "/SqlScripts/InsertUserConfig.sql";
            using (FileStream strm = File.OpenRead(filePath))
            {
                StreamReader reader = new StreamReader(strm);
                configInsertSQL = reader.ReadToEnd();
            }            
            StringBuilder authzQuery = new StringBuilder();
            StringBuilder configQuery = new StringBuilder();
            Facility defaultFacility = null;
            int usersNotImported = 0;
            bool roleFound = false;
            foreach (User user in users)
            {
                roleFound = false;
                defaultFacility = facilities.Where(a => a.Code.Equals(user.DefaultSite, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (defaultFacility == null)
                {
                    Tracer.TraceMessage("User " + user.FullUserName + " has no default facility. Code: " + (string.IsNullOrEmpty(user.DefaultSite) ? "NULL" : user.DefaultSite));
                    usersNotImported = usersNotImported + 1;
                    continue;
                }
                if (user.SupportUser || user.NationalUser || user.RegionalUser || user.VisnUser || user.AdminUser || user.SiteUser || user.EmsUser)
                {
                    if (!roles.ContainsKey(user.FullUserName))
                    {
                        try
                        {
                            #region AUTHZ

                            authzQuery.Clear();
                            authzQuery.AppendLine(authzInsertSQL);
                            if (user.SupportUser || user.NationalUser)
                            {
                                if (user.SupportUser)
                                {
                                    authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[SUPPORT_ROLE]);
                                    roleFound = true;
                                }
                                if (user.NationalUser)
                                {
                                    if (roleFound)
                                        authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                    authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[NATIONAL_ROLE]);                                    
                                    roleFound = true;
                                }
                                if (user.ReadAccess)
                                {
                                    authzQuery.AppendLine(GetAllPermissionFacilities(facilities));
                                    authzQuery.Replace("@ACTION_PERMISSION_ID", actions[READ_DATA_PERMISSION]);
                                }
                                if (user.WriteAccess)
                                {
                                    authzQuery.AppendLine(GetAllPermissionFacilities(facilities));
                                    authzQuery.Replace("@ACTION_PERMISSION_ID", actions[WRITE_DATA_PERMISSION]);
                                }
                            }
                            if (user.RegionalUser)
                            {
                                if (roleFound)
                                {
                                    authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                    authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[REGIONAL_ROLE]);
                                }
                                else
                                {
                                    authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[REGIONAL_ROLE]);
                                    roleFound = true;
                                    if (user.ReadAccess)
                                    {
                                        authzQuery.AppendLine(GetPermissionFacilitiesForRegion(facilities, user.DefaultRegion));
                                        authzQuery.Replace("@ACTION_PERMISSION_ID", actions[READ_DATA_PERMISSION]);
                                    }
                                    if (user.WriteAccess)
                                    {
                                        authzQuery.AppendLine(GetPermissionFacilitiesForRegion(facilities, user.DefaultRegion));
                                        authzQuery.Replace("@ACTION_PERMISSION_ID", actions[WRITE_DATA_PERMISSION]);
                                    }
                                }
                            }
                            if (user.VisnUser)
                            {
                                if (roleFound)
                                {
                                    authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                    authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[VISN_ROLE]);
                                }
                                else
                                {
                                    authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[VISN_ROLE]);
                                    roleFound = true;
                                    if (user.ReadAccess)
                                    {
                                        authzQuery.AppendLine(GetPermissionFacilitiesForVisn(facilities, user.DefaultVisn));
                                        authzQuery.Replace("@ACTION_PERMISSION_ID", actions[READ_DATA_PERMISSION]);
                                    }
                                    if (user.WriteAccess)
                                    {
                                        authzQuery.AppendLine(GetPermissionFacilitiesForVisn(facilities, user.DefaultVisn));
                                        authzQuery.Replace("@ACTION_PERMISSION_ID", actions[WRITE_DATA_PERMISSION]);
                                    }
                                }
                            }
                            if (user.AdminUser || user.SiteUser || user.EmsUser)
                            {
                                if (roleFound)
                                {
                                    if (user.AdminUser)
                                    {
                                        authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                        authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[ADMIN_ROLE]);
                                    }
                                    if (user.SiteUser)
                                    {
                                        authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                        authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[SITE_ROLE]);
                                    }
                                    if (user.EmsUser)
                                    {
                                        authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                        authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[EMS_ROLE]);
                                    }
                                }
                                else
                                {
                                    if (user.AdminUser)
                                    {
                                        authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[ADMIN_ROLE]);
                                        roleFound = true;
                                    }
                                    if (user.SiteUser)
                                    {
                                        if (roleFound)
                                            authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                        authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[SITE_ROLE]);
                                        roleFound = true;
                                    }
                                    if (user.EmsUser)
                                    {
                                        if (roleFound)
                                            authzQuery.AppendLine(AUTHZ_INSERT_ASSIGNED_ROLE);
                                        authzQuery.Replace("@ASSIGNED_ROLE_ID", roles[EMS_ROLE]);
                                        roleFound = true;
                                    }
                                    if (user.ReadAccess)
                                    {
                                        authzQuery.AppendLine(AUTHZ_INSERT_RESOURCE.Replace("@FACILITY_ID", defaultFacility.Uid));
                                        authzQuery.Replace("@ACTION_PERMISSION_ID", actions[READ_DATA_PERMISSION]);
                                    }
                                    if (user.WriteAccess)
                                    {
                                        if (!user.ReadAccess)
                                        {
                                            authzQuery.AppendLine(AUTHZ_INSERT_RESOURCE.Replace("@FACILITY_ID", defaultFacility.Uid));
                                            authzQuery.Replace("@ACTION_PERMISSION_ID", actions[READ_DATA_PERMISSION]);
                                        }
                                        authzQuery.AppendLine(AUTHZ_INSERT_RESOURCE.Replace("@FACILITY_ID", defaultFacility.Uid));
                                        authzQuery.Replace("@ACTION_PERMISSION_ID", actions[WRITE_DATA_PERMISSION]);
                                    }
                                }
                            }
                            authzQuery.Replace("@USER_ROLE_ID", Guid.NewGuid().ToString());
                            authzQuery.Replace("@USER_NAME", user.FullUserName);
                            authzQuery.Replace("@USER_SID", user.Sid);
                            SqlQueries.InsertAuthzUser(authzQuery.ToString());

                            #endregion

                            #region Config

                            configQuery.Clear();
                            configQuery.AppendLine(configInsertSQL);
                            configQuery.Replace("@FULL_USER_NAME", user.FullUserName);
                            configQuery.Replace("@FACILITY_ID", defaultFacility.Uid);
                            configQuery.Replace("@FACILITY_NAME", defaultFacility.Name.Replace("'", "''"));
                            SqlQueries.InsertConfigUser(configQuery.ToString());

                            #endregion
                        }
                        catch (Exception ex)
                        {
                            Tracer.TraceMessage("Exception on insert for user: " + user.FullUserName);
                            Tracer.TraceException(ex);
                            usersNotImported = usersNotImported + 1;
                        }
                    }
                    else
                    {
                        Tracer.TraceMessage("UserName already exists in AUTHZ: " + user.FullUserName);
                        usersNotImported = usersNotImported + 1;
                    }
                }
                else
                {
                    Tracer.TraceMessage("User " + user.FullUserName + " has no role!");
                    usersNotImported = usersNotImported + 1;
                }
            }
            GlobalConnections.CloseConnections();

            Tracer.TraceMessage("Total users found for import: " + TotalUsers.ToString());
            Tracer.TraceMessage("Users found in AD: " + users.Count.ToString());
            Tracer.TraceMessage("Users not imported(already exists in AUTHZ/exception): " + usersNotImported.ToString());
            Tracer.TraceMessage("Users import finished!");
        }

        private static string GetAllPermissionFacilities(List<Facility> facilities)
        {
            if (string.IsNullOrEmpty(RESOURCES_ALL))                
            {
                StringBuilder sb = new StringBuilder();
                foreach (Facility f in facilities)                
                    sb.AppendLine(AUTHZ_INSERT_RESOURCE.Replace("@FACILITY_ID", f.Uid));
                
                RESOURCES_ALL = sb.ToString();                
            }
            return RESOURCES_ALL;
        }

        private static string GetPermissionFacilitiesForRegion(List<Facility> facilities, int regionNumber)
        {
            if (!RESOURCES_REGION.ContainsKey(regionNumber) || string.IsNullOrEmpty(RESOURCES_REGION[regionNumber]))
            {
                StringBuilder sb = new StringBuilder();
                foreach (Facility f in facilities)
                {
                    if (f.RegionNumber == regionNumber)
                        sb.AppendLine(AUTHZ_INSERT_RESOURCE.Replace("@FACILITY_ID", f.Uid));
                }
                if (RESOURCES_REGION.ContainsKey(regionNumber))
                    RESOURCES_REGION.Remove(regionNumber);
                RESOURCES_REGION.Add(regionNumber, sb.ToString());

            }
            return RESOURCES_REGION[regionNumber];
        }

        private static string GetPermissionFacilitiesForVisn(List<Facility> facilities, int visnNumber)
        {
            if (!RESOURCES_VISN.ContainsKey(visnNumber) || string.IsNullOrEmpty(RESOURCES_VISN[visnNumber]))
            {
                StringBuilder sb = new StringBuilder();
                foreach (Facility f in facilities)
                {
                    if (f.VisnNumber == visnNumber)
                        sb.AppendLine(AUTHZ_INSERT_RESOURCE.Replace("@FACILITY_ID", f.Uid));
                }
                if (RESOURCES_VISN.ContainsKey(visnNumber))
                    RESOURCES_VISN.Remove(visnNumber);
                RESOURCES_VISN.Add(visnNumber, sb.ToString());

            }
            return RESOURCES_VISN[visnNumber];
        }
    }
}
