﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Timers;

namespace BMS.Schedulers
{
    public class SchedulerTimer
    {
        private const string SchedulerSecondsLateKey = "VistaIntegration.SchedulerSecondsLate";

        private Timer timer;
        private bool stoped = true;
        private int schedulerMilliSecondsLate;
        private object _lock = new object();

        private static bool IsRefreshSchedulers = false;
        private static bool IsRefreshVistaCommands = false;
        private static bool IsRefreshNumiCommands = false;
        private static bool IsRefreshWhiteboardCommands = false;

        public IEnumerable<Scheduler> Schedulers
        {
            get;
            private set;
        }

        public void SetRefreshSchedulersFlag()
        {
            lock (_lock) { IsRefreshSchedulers = true; }
        }

        public void SetRefreshVistACommandsFlag()
        {
            lock (_lock) { IsRefreshVistaCommands = true; }
        }

        public void SetRefreshNumiCommandsFlag()
        {
            lock (_lock) { IsRefreshNumiCommands = true; }
        }

        public void SetRefreshWhiteboardReportCommandsFlag()
        {
            lock (_lock) { IsRefreshWhiteboardCommands = true; }
        }

        public SchedulerTimer()
        {
            Schedulers = Enumerable.Empty<Scheduler>();
            schedulerMilliSecondsLate = 1000 * int.Parse(System.Configuration.ConfigurationManager.AppSettings[SchedulerSecondsLateKey]);
        }

        private void CheckDistinctSchedulers(List<Scheduler> list)
        {
            HashSet<Scheduler> hash = new HashSet<Scheduler>(list);
            if (hash.Count != list.Count)
                throw new InvalidOperationException("No distinct schedulers");
        }

        public void Setchedulers(IEnumerable<Scheduler> schedulers)
        {
            List<Scheduler> list = schedulers.ToList();
            CheckDistinctSchedulers(list);

            if (!stoped)
                InitSchedulers(list);

            Schedulers = list;
        }

        private double GetTimerInterval()
        {
            DateTime now = DateTime.Now;
            return ((60 - now.Second) * 1000 - now.Millisecond) + schedulerMilliSecondsLate;
        }

        public void Start()
        {
            if (!stoped)
                throw new InvalidOperationException();

            stoped = false;
            timer = new Timer();
            timer.AutoReset = false;
            timer.Elapsed += new ElapsedEventHandler(timer_Elapsed);
            InitSchedulers(Schedulers);
            timer.Interval = GetTimerInterval();
            timer.Start();
        }


        private void InitSchedulers(IEnumerable<Scheduler> schedulers)
        {
            DateTime now = DateTime.UtcNow;
            foreach (var scheduler in schedulers)
            {
                scheduler.Init(now);
            }
        }

        private void timer_Elapsed(object sender, ElapsedEventArgs e)
        {
            timer.Dispose();
            try
            {
                bool isRefreshSchedulers, isRefreshVistaCommands, isRefreshNumiCommands, isRefreshWhiteboardCommands = false;
                lock (_lock)
                {
                    isRefreshSchedulers = IsRefreshSchedulers;
                    isRefreshVistaCommands = IsRefreshVistaCommands;
                    isRefreshNumiCommands = IsRefreshNumiCommands;
                    isRefreshWhiteboardCommands = IsRefreshWhiteboardCommands;
                    IsRefreshSchedulers = IsRefreshVistaCommands = IsRefreshNumiCommands = IsRefreshWhiteboardCommands = false;
                }
                if (isRefreshSchedulers)
                {
                    BMS.ServicesWrapper.BMService.BMSFactory.VistaQueryClientWindows.RefreshSchedulers();
                    isRefreshSchedulers = isRefreshVistaCommands = isRefreshNumiCommands = isRefreshWhiteboardCommands = false;
                }
                if (isRefreshVistaCommands)
                    BMS.ServicesWrapper.BMService.BMSFactory.VistaQueryClientWindows.RefreshVistACommands();
                if (isRefreshNumiCommands)
                    BMS.ServicesWrapper.BMService.BMSFactory.VistaQueryClientWindows.RefreshNumiCommands();
                if (isRefreshWhiteboardCommands)
                    BMS.ServicesWrapper.BMService.BMSFactory.VistaQueryClientWindows.RefreshWhiteboardReportCommands();
            }
            catch (Exception ex)
            {
                BMS.Utils.Tracer.TraceMessage("Exception on SchedulerTimer RefreshSchedulers");
                BMS.Utils.Tracer.TraceException(ex);
            }
            finally
            {
                timer = new Timer();
                timer.AutoReset = false;
                timer.Elapsed += new ElapsedEventHandler(timer_Elapsed);
                timer.Interval = GetTimerInterval();
                timer.Start();
                DateTime now = DateTime.UtcNow;
                IEnumerable<Scheduler> schedulers = Schedulers;
                var validSchedulers = schedulers.Where(s => s.IsValid(now)).ToList();
                foreach (Scheduler scheduler in validSchedulers)
                {
                    scheduler.Execute();
                }
            }
        }

        public void Stop()
        {
            if (stoped) return;
            stoped = true;
            timer.Stop();
            timer = null;
        }
    }
}
