﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Mdws2ORM.Maps;
using Mdws2ORM.Maps.BaseFields;
using Mdws2ORM.Core;
using Mdws2ORM.Exceptions;
using BMS.VistaIntegration.Data;

namespace Mdws2ORM.Impl
{
    public class EntityQuery : IEntityQuery
    {
        public ICache Cache { get; private set; }
        public IEntryQuery Query { get; private set; }
        public VistASite VistaSite { get; private set; }

        public EntityQuery(ICache cache, IEntryQuery query, VistASite vistaSite)
        {
            this.Cache = cache;
            this.Query = query;
            this.VistaSite = vistaSite;
        }

        private void AddMultipleFields<T>(BaseEntityMap<T> entityMap, T parent, string iens) where T : class
        {
            BaseMultipleFieldMap<T>[] maps = entityMap.MultipleFieldsMap;
            for (int i = 0; i < maps.Length; i++)
                maps[i].Map(this, parent, iens);
        }

        private void AddGetFetchOnlyFields<T>(BaseEntityMap<T> entityMap, T entity, string iens) where T : class
        {
            if (entityMap.GetForListFieldsMap.Length == 0) return;
            Entry? getFetchOnlyEntry = Query.Get(entityMap.GetForListQueryParam, GetParamBuilder.DefaultParam, VistaSite, iens);
            if (getFetchOnlyEntry.HasValue)
            {
                Translator.PopulateFields(this, entity, getFetchOnlyEntry.Value, entityMap.GetForListFieldsMap);
            }
        }


        public T Get<T>(BaseEntityMap<T> entityMap, GetParam getParam, string iens) where T : class
        {
            try
            {
                string ien = Utilities.GetIen(iens);
                T entity = Cache.Get<T>(ien);
                if (entity != null) return entity;
                Entry? entry = Query.Get(entityMap.GetQueryParam, getParam, VistaSite, iens);
                if (entry == null) return null;
                entity = entityMap.NewEntity(ien);
                Translator.PopulateFields(this, entity, entry.Value, entityMap.GetFieldsMap);
                AddMultipleFields(entityMap, entity, "," + iens);
                Cache.Add(ien, entity);
                return entity;
            }
            catch (MdwsException me)
            {
                me.AddGetRpc<T>(entityMap.GetQueryParam, getParam, iens);
                throw;
            }
            catch (Exception e)
            {
                MdwsException me = new MdwsException(e);
                me.AddGetRpc<T>(entityMap.GetQueryParam, getParam, iens);
                throw me;
            }
        }


        private IList<Entry> GetEntries<T>(BaseEntityMap<T> entityMap, ListParam listParam, string iens) where T : class
        {
            try
            {
                IList<Entry> entries = Query.List(entityMap.ListQueryParam, listParam, VistaSite, iens);
                return entries;
            }
            catch (MdwsException me)
            {
                me.AddListRpc<T>(entityMap.ListQueryParam, listParam, iens);
                throw;
            }
            catch (Exception e)
            {
                MdwsException me = new MdwsException(e);
                me.AddListRpc<T>(entityMap.ListQueryParam, listParam, iens);
                throw me;
            }
        }

        private T GetEntity<T>(Entry entry, BaseEntityMap<T> entityMap, ListParam listParam, string iens) where T : class
        {
            try
            {
                T entity = Cache.Get<T>(entry.Ien);
                if (entity == null)
                {
                    entity = entityMap.NewEntity(entry.Ien);
                    Translator.PopulateFields(this, entity, entry, entityMap.ListFieldsMap);
                    if (entityMap.MultipleFieldsMap.Length > 0)
                        AddMultipleFields(entityMap, entity, Utilities.MakeListIens(entry, iens));
                    if (entityMap.GetForListFieldsMap.Length > 0)
                        AddGetFetchOnlyFields(entityMap, entity, Utilities.MakeGetIens(entry, iens));
                    Cache.Add(entry.Ien, entity);
                }
                return entity;
            }
            catch (MdwsException me)
            {
                me.AddListRpc<T>(entityMap.ListQueryParam, listParam, iens);
                throw;
            }
            catch (Exception e)
            {
                MdwsException me = new MdwsException(e);
                me.AddListRpc<T>(entityMap.ListQueryParam, listParam, iens);
                throw me;
            }
        }


        public IEnumerable<ListItem<T>> List<T>(BaseEntityMap<T> entityMap, ListParam listParam, string iens = "") where T : class
        {
            IEnumerable<Entry> entries = GetEntries(entityMap, listParam, iens);
            foreach (Entry entry in entries)
            {
                T entity = GetEntity(entry, entityMap, listParam, iens);
                yield return new ListItem<T>(entity, entry);
            }
        }

        public bool ThrowExceptionWhenReferenceEntityNotFound { get; set; }

        public T GetNotNull<T>(BaseEntityMap<T> entityMap, GetParam getParam, string iens = "") where T : class
        {
            T entity = Get(entityMap, getParam, iens);
            if (entity == null && ThrowExceptionWhenReferenceEntityNotFound)
            {
                MdwsException me =  new EntryNotFoundException(entityMap.GetQueryParam.File, iens);
                me.AddGetRpc<T>(entityMap.GetQueryParam, getParam, iens);
                throw me;
            }
            return entity;
        }
    }
}
